#include "xgl.h"

#include "html/html-layout-text-analysis.h"
#include "html/html-layout-text-flow.h"
#include "xgl/skia/include/effects/SkBlurMaskFilter.h"

#if defined(WINDOWS)
  #define USE_USP_ON_WINDOWS
#endif

#if defined(USE_USP_ON_WINDOWS)

#include <usp10.h>
#pragma comment( lib, "usp10" )
#include <stdexcept>

#endif

uint32 estimate_glyph_count(uint32 text_length);

namespace xgl {

  using namespace html;

  class text_analysis
    : public html::tflow::text_analysis
  {

  public:
    // the only public method that wraps text analysis functionality:
    static bool exec(view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes);

  private:

    text_analysis() {}

    bool generate_results(
      view& v, element* el, slice<hnode> nodes,
      OUT tool::array<wchar>& text,
      OUT tool::array<tflow::text_run>& runs,
      OUT tool::array<tflow::LINE_BREAKPOINT>& breakpoints
      );

    void shape_glyph_runs(view& v, element* elem, tflow::text_flow& tf);
    void shape_glyph_run(view& v, tflow::text_flow& tf, element* elem, uint run_index, IN OUT uint& glyph_start);

    void analyze_line_breakpoints();
    void init_runs(view& v, element* el, const ustring& lang);

  };


  bool text_analysis::exec(
    view& v, element* el, tflow::text_flow& tf, slice<hnode> nodes)
  {
    text_analysis inst;

    if (inst.generate_results(v, el, nodes, tf._text, tf._runs, tf._breakpoints))
    {
      inst.shape_glyph_runs(v, el, tf);
      return true;
    }
    return false;
  }

  bool text_analysis::generate_results(
    view& v, element* elem, slice<hnode> nodes,
    OUT tool::array<wchar>& text,
    OUT tool::array<tflow::text_run>& runs,
    OUT tool::array<tflow::LINE_BREAKPOINT>& breakpoints)
  {
    hstyle es = elem->get_style(v);
    ustring lang = elem->get_lang();

    _reading_direction = es->direction == direction_rtl ?
      tflow::READING_DIRECTION_RIGHT_TO_LEFT :
      tflow::READING_DIRECTION_LEFT_TO_RIGHT;

    uint  next_run_index = 0;
    text.clear();
    runs.clear();
    breakpoints.clear();

    auto push_run = [&](node* pnode, uint start, uint length, uint dom_start, char_mark_t marks)
    {
      linked_text_run& run = _runs.push();
      run.node = pnode;
      run.next_run_index = ++next_run_index;
      run.text_start = start;
      run.text_length = length;
      run.bidi_level(pnode->get_style(v)->direction == direction_rtl ? 1 : 0);
      run.text_node_start = dom_start;
      run.marks = marks;
    };

    if (elem->c_style->collapse_ws())
    {
      auto receiver = [&](wchars span, char_marks marks, node* pnode)
      {
        uint start = static_cast<uint>(text.length());
        if (span.length == 1 && span[0] == 0)
        {
          text.push(span); // inline block
          push_run(pnode, start, static_cast<::uint32>(text.length()) - start, 0, char_mark_t());
        }
        else
          tflow::visible_chars_collapse_ws(pnode, span, marks, text, push_run, lang);
      };
      tflow::flatten_nodes(v, elem, nodes, receiver);
    }
    else
    {
      auto receiver = [&](wchars span, char_marks marks, node* pnode)
      {
        uint start = static_cast<uint>(text.length());
        if (span.length == 1 && span[0] == 0)
        {
          text.push(span); // inline block
          push_run(pnode, start, static_cast<uint>(text.length()) - start, 0, char_mark_t());
        }
        else
          tflow::visible_chars_break_nl(pnode, span, marks, text, push_run, lang);

        //push_run(pnode, start, static_cast<uint32>(text.length()) - start,0);
      };
      tflow::flatten_nodes(v, elem, nodes, receiver);
    }

    if (text.length() == 0
      && nodes.length == 1
      && nodes[0]->is_text()
      && nodes[0].ptr_of<html::text>()->chars.length() == 0)
    {
      // special case for empty paragraphs with forced flow:text
      // note: needed for content-editable
      text.push(EMPTY_STRING_PLACEHOLDER /* zero width space*/);
      push_run(nodes[0], 0, 1, 0, char_mark_t());
    }

    _text = text();
    if (_text.length == 0)
      return true;

    // Analyzes the text using each of the analyzers and returns
    // their results as a series of runs.

    try
    {
      // Initially start out with one result that covers the entire range.
      // This result will be subdivided by the analysis processes.

      // Allocate enough room to have one breakpoint per code unit.
      _breakpoints.length(_text.length);

      init_runs(v,elem,lang);
      analyze_line_breakpoints();


      //if (FAILED(hr = textAnalyzer->AnalyzeLineBreakpoints(   this, 0, UINT32(_text.length), this))) return false;
      //if (FAILED(hr = textAnalyzer->AnalyzeBidi(              this, 0, UINT32(_text.length), this))) return false;
      //if (FAILED(hr = textAnalyzer->AnalyzeScript(            this, 0, UINT32(_text.length), this))) return false;


      for (index_t i = _runs.last_index(); i >= 0; --i) // new runs (if any) will have numbers total_runs
      {
        //uint rl = (uint)_runs.length();
        linked_text_run& r = _runs[i];
        html::style *rs = r.get_style(v);
        if (r.is_inline_block_pos(_text))
        {
          r.used_font = v.get_font(rs);
          r.no_visual(false);
          //r.script.shapes = DWRITE_SCRIPT_SHAPES_NO_VISUAL;
          continue;
        }

        if (rs != es && !rs->can_wrap() && r.text_length)
        {
          uint nm = r.text_start + r.text_length - 1;
          for (uint n = r.text_start + 1; n < nm; ++n)
          {
            tflow::LINE_BREAKPOINT &bp = _breakpoints[n];
            if (bp.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
              bp.break_condition_after = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
            if (bp.break_condition_before != tflow::BREAK_CONDITION_MUST_BREAK)
              bp.break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
          }
          tflow::LINE_BREAKPOINT &bp1 = _breakpoints[r.text_start];
          tflow::LINE_BREAKPOINT &bp2 = _breakpoints[nm];
          if (bp1.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
            bp1.break_condition_after = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
          if (bp2.break_condition_after != tflow::BREAK_CONDITION_MUST_BREAK)
            bp2.break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
        }

        wchars run_chars = r.chars(_text.start);

        //if(!setup_used_run_font(v, rs, r, run_chars, lang))
        //  view::debug_printf(OT_DOM, OS_ERROR,"Font fallback failure for font %S fallback %S.\n",
        //    r.decl_font->name.c_str(),
        //    r.used_font->name.c_str());

        //r.decl_font = v.get_font(rs);
        wchars font_list = rs->font_family();

        handle<xgl::font> decl_font = static_cast<xgl::font*>(v.get_font(rs));

        // check each character in the run for supported glyphs, split the run and set substitution font.

        while (run_chars.length)
        {
          const wchar* start = run_chars.start;
          const wchar* end = start;

          uint ucp = u16::getc(run_chars);

          if (decl_font->has_glyph_for(ucp))
            continue;
          end = run_chars.start;

          handle<gool::font> fb_font;

          auto script = writing_script(ucp);

          if (!app()->get_used_font(fb_font, font_list, decl_font, lang, script, ucp))
            continue;

          while (run_chars.length)
          {
            end = run_chars.start;
            auto t = run_chars;
            ucp = u16::getc(run_chars);
            if (decl_font->has_glyph_for(ucp))
              break;
            if (script != writing_script(ucp) || !fb_font->has_glyph_for(ucp)) {
              run_chars = t;
              break;
            }
            end = run_chars.start;
          }
          uint text_position = uint(start - _text.start);
          uint text_length = uint(end - start);
          set_current_run(text_position);
          split_current_run(text_position);
          while (text_length > 0)
          {
            linked_text_run& run = fetch_next_run(&text_length);
            run.used_font = fb_font;
          }
        }
      }

      // Exchange our results with the caller's.
      breakpoints.swap(_breakpoints);

      size_t total_runs = _runs.length();

      // Resequence the resulting runs in order before returning to caller.
      runs.length(total_runs);

      uint nextRunIndex = 0;
      for (int i = 0; i < int(total_runs); ++i)
      {
        runs[i] = _runs[nextRunIndex];
        //wchars tt = runs[i].chars(_text.start);
        nextRunIndex = _runs[nextRunIndex].next_run_index;
      }

    }
    catch (...)
    {
      return false;
    }
    return true;
  }

  // VERY basic

  void text_analysis::analyze_line_breakpoints() {
    wchars text = _text;
    tflow::LINE_BREAKPOINT* bp = _breakpoints.begin();

    tflow::LINE_BREAKPOINT _pbp = { 0 };
    tflow::LINE_BREAKPOINT* pbp = &_pbp;

    bool p_is_punct = false;

    for (wchar pc = 0; !!text; ++text)
    {
      wchar c = *text;
      bool is_punct = ucispunct(c) != 0;
      bp->is_whitespace = ucisspace(*text);
      bp->is_soft_hyphen = *text == 0x00AD;
      bp->break_condition_after = bp->break_condition_before = tflow::BREAK_CONDITION_MAY_NOT_BREAK;
      if (pbp->is_whitespace && !bp->is_whitespace) {
        pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
        bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
      }
      else if (pc == '\r' && c != '\n') {
        pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
        bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
      }
      else if (pc == '\n') {
        pbp->break_condition_after = tflow::BREAK_CONDITION_MUST_BREAK;
        bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
      }
      else if (p_is_punct && !is_punct) {
        pbp->break_condition_after = tflow::BREAK_CONDITION_CAN_BREAK;
        bp->break_condition_before = tflow::BREAK_CONDITION_CAN_BREAK;
      }
      pbp = bp++;
      pc = c;
      p_is_punct = is_punct;
    }
  }

#if defined(USE_USP_ON_WINDOWS)

  inline void check_hr(HRESULT hr) {
    if (SUCCEEDED(hr))
      return;
    else if (hr == E_INVALIDARG)
      throw tool::error("invalid arg");
    else if (hr == E_OUTOFMEMORY)
      throw tool::error("out of memory");
    else if (hr == USP_E_SCRIPT_NOT_IN_FONT)
      throw tool::error("script not in the font");
    else
      throw tool::error(tool::string::format("text_analysis failure %X", hr).c_str());
  }

  void text_analysis::init_runs(view& v, element* el, const ustring& lang) {

    //SCRIPT_CACHE   scache;
    SCRIPT_CONTROL sc;
    SCRIPT_STATE   ss;
    array<SCRIPT_ITEM> sitems;
    memzero(sc);
    memzero(ss);

    ss.uBidiLevel = _reading_direction == tflow::READING_DIRECTION_RIGHT_TO_LEFT ? 1 : 0;
    sc.uDefaultLanguage = LANGIDFROMLCID(locale_name_2_LCID(lang));

    HRESULT hr = ScriptApplyDigitSubstitution(NULL, &sc, &ss); check_hr(hr);

    sitems.length(_text.length + 1);

    int nitems = 0;
    hr = ScriptItemize(
      _text.start,        // _In_ const WCHAR *pwcInChars,
      _text.size(),
      sitems.size(),      //_In_      int cMaxItems,
      &sc,                //_In_opt_  const SCRIPT_CONTROL *psControl,
      &ss,                //_In_opt_  const SCRIPT_STATE *psState,
      sitems.head(),
      &nitems         //_Out_     int *pcItems
    );
    check_hr(hr);

    sitems.size(nitems + 1);

    for (uint n = 0; n < sitems.length() - 1; ++n)
    {
      auto sitem = sitems[n];
      uint start = uint(sitem.iCharPos);
      uint end = uint(sitems[n + 1].iCharPos);

      set_current_run(start);
      split_current_run(start);
      uint text_length = uint(end - start);
      while (text_length > 0)
      {
        linked_text_run& run = fetch_next_run(&text_length);
        run.script_analysis = sitem.a;
      }
    }
  }

#else

  void text_analysis::init_runs(view& v, element* el, const ustring& lang) {
    wchars text = _text;

    for (index_t i = _runs.last_index(); i >= 0; --i) // new runs (if any) will have numbers total_runs
    {
      linked_text_run& r = _runs[i];
      if (r.is_inline_block_pos(_text))
      {
        r.no_visual(false);
        continue;
      }

      wchars run_chars = r.chars(_text.start);

      while (run_chars.length)
      {
        const wchar* start = run_chars.start;
        const wchar* end = start;

        uint ucp = u16::getc(run_chars);
        if (ucp >= ' ')
          continue;

        end = run_chars.start;

        while (run_chars.length)
        {
          end = run_chars.start;
          if (u16::getc(run_chars) >= ' ')
            break;
          end = run_chars.start;
        }
        uint text_position = uint(start - _text.start);
        uint text_length = uint(end - start);
        set_current_run(text_position);
        split_current_run(text_position);
        while (text_length > 0)
        {
          linked_text_run& run = fetch_next_run(&text_length);
          run.no_visual(true);
        }
      }
    }
  }
#endif

  uint estimate_glyph_count(uint text_length)
  {
    return 3 * text_length / 2 + 16;
  }

#if defined(USE_USP_ON_WINDOWS)

  class screen_dc
  {
  public:
    HDC   hdc;
    HFONT hfont;
    HFONT hfont_old;
    float uratio;

    screen_dc() : hfont(0), hfont_old(0), uratio(0) { hdc = ::GetDC(NULL); }
    ~screen_dc() {
      if (hfont) {
        ::SelectObject(hdc, hfont_old);
        ::DeleteObject(hfont);
      }
      ::ReleaseDC(NULL, hdc);
    }
    operator HDC() { return hdc; }
    void select(gool::font* pf) {

      if (hfont) {
        ::SelectObject(hdc, hfont_old);
        ::DeleteObject(hfont);
      }

      hfont =
        CreateFont(
          -int(pf->size + 0.5f),      // height of the font
          0,                          // average character width
          0,                          // angle of escapement
          0,                          // base-line orientation angle
          pf->weight,                 // font weight
          pf->italic ? TRUE : FALSE,  // italic attribute option
          FALSE,                      // underline attribute option
          FALSE,                      // strikeout attribute option
          DEFAULT_CHARSET,            // character set identifier
          OUT_DEFAULT_PRECIS,         // output precision
          CLIP_DEFAULT_PRECIS,        // clipping precision
          CLEARTYPE_QUALITY,          // output quality
          DEFAULT_PITCH | FF_DONTCARE,// pitch and family
          pf->used_font_name()        // typeface name
        );

      hfont_old = (HFONT)SelectObject(hdc, hfont);

      OUTLINETEXTMETRIC otm;
      GetOutlineTextMetrics(hdc,sizeof(otm),&otm);
      uratio = pf->size / otm.otmEMSquare;
    }
  };

  void text_analysis::shape_glyph_runs(view& v, element* elem, tflow::text_flow& tf)
  {
    // Shapes all the glyph runs in the layout.

    handle<gool::font>     cfont;
    SCRIPT_CACHE           scache = nullptr;
    screen_dc              sdc;

    auto do_shape_glyph_run = [&]( uint32 run_index, IN OUT uint32& glyph_start)
    {

        tflow::text_run& run = tf._runs[run_index];

        uint32 text_start = run.text_start;
        uint32 text_length = run.text_length;
        uint32 maxglyph_count = static_cast<uint32>(tf._glyph_indices.size() - glyph_start);
        int actual_glyph_count = 0;

        element* el = run.get_element();

        if (run.is_inline_block_pos(tf._text()))
        {
          el->check_layout(v);
          //assert(el->is_inline_block_element(v));
          run.glyph_start = glyph_start;
          run.glyph_count = 1;
          if (text_length > maxglyph_count)
          {
            uint32 total_glyphs_array_count = glyph_start + 1;
            tf._glyph_indices.size(total_glyphs_array_count);
          }
          tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_advances.length()));
          tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_offsets.length()));
          tf._glyph_advances[glyph_start] = static_cast<float>(el->min_inline_margin_box_width(v));
          tf._glyph_indices[glyph_start] = 0;
          glyph_start += 1;
          return;
        }

        ustring  locale_name = el->get_lang();

        gool::font* pf = run.get_used_font(v);

        if (pf != cfont)
        {
          cfont = pf;
          sdc.select(cfont);
          if (scache) { ScriptFreeCache(&scache); scache = NULL; }
        }

        run.glyph_start = glyph_start;
        run.glyph_count = 0;

        if (text_length == 0)
          return; // Nothing to do..


                  ////////////////////
                  // Allocate space for shaping to fill with glyphs and other information,
                  // with about as many glyphs as there are text characters. We'll actually
                  // need more glyphs than codepoints if they are decomposed into separate
                  // glyphs, or fewer glyphs than codepoints if multiple are substituted
                  // into a single glyph. In any case, the shaping process will need some
                  // room to apply those rules to even make that determintation.

        if (text_length > maxglyph_count)
        {
          maxglyph_count = estimate_glyph_count(text_length);
          uint32 total_glyphs_array_count = glyph_start + maxglyph_count;
          tf._glyph_indices.size(total_glyphs_array_count);
        }

        ////////////////////
        // Get the glyphs from the text

        SCRIPT_ANALYSIS *psan = (SCRIPT_ANALYSIS *)&run.script_analysis;

        const wchar_t* pc = &tf._text[text_start];
        pc = pc;

        array<SCRIPT_VISATTR> vattr(maxglyph_count);
        HRESULT hr = ScriptShape(sdc, &scache,
          pc,                                        //_In_     const WCHAR *pwcChars,
          text_length,                               //_In_     int cChars,
          maxglyph_count,                            //_In_     int cMaxGlyphs,
          psan,                                      //_Inout_  SCRIPT_ANALYSIS *psa,
          &tf._glyph_indices[glyph_start],           //_Out_    WORD *pwOutGlyphs,
          &tf._glyph_clusters[text_start],           //_Out_    WORD *pwLogClust,
          vattr.head(),                              //_Out_    SCRIPT_VISATTR *psva,
          &actual_glyph_count                        //_Out_    int *pcGlyphs
        ); check_hr(hr);

        vattr.size(actual_glyph_count);

        ////////////////////
        // Get the placement of the all the glyphs.

        tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_advances.length()));
        tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + actual_glyph_count), tf._glyph_offsets.length()));

        array<int>     advances(actual_glyph_count);
        array<GOFFSET> offsets(actual_glyph_count);
        ABC            abc;

        hr = ScriptPlace(
          sdc, &scache,                             //_In_     HDC hdc, _Inout_  SCRIPT_CACHE *psc,
          &tf._glyph_indices[glyph_start],          //_In_     const WORD *pwGlyphs,
          actual_glyph_count,                       //_In_     int cGlyphs,
          vattr.head(),                             //_In_     const SCRIPT_VISATTR *psva,
          (SCRIPT_ANALYSIS *)&run.script_analysis,  //_Inout_  SCRIPT_ANALYSIS *psa,
          advances.head(),                          //_Out_    int *piAdvance,
          offsets.head(),                           //_Out_    GOFFSET *pGoffset,
          &abc                                      //_Out_    ABC *pABC
        ); check_hr(hr);

        uint16*               pindices = &tf._glyph_indices[glyph_start];
        uint16*               pclusters = &tf._glyph_clusters[text_start];
        float*                pwidths = &tf._glyph_advances[glyph_start];
        tflow::GLYPH_OFFSET*  poffset = &tf._glyph_offsets[glyph_start];

        for (int n = 0; n < actual_glyph_count; ++n)
        {
          float width = float(advances[n]);
          pwidths[n] = width;
          GOFFSET goff = offsets[n];
          poffset[n].advance_offset = sdc.uratio * goff.du;
          poffset[n].ascender_offset = sdc.uratio * goff.dv;
        }

        if (run.bidi_level() & 1) {
          // d2d uses visual order, but uniscribe uses logical order of these elements
          // so we need to reverse them here to match the rest:
          std::reverse(pindices, pindices + actual_glyph_count);
          std::reverse(pclusters, pclusters + text_length);
          std::reverse(pwidths, pwidths + actual_glyph_count);
          std::reverse(poffset, poffset + actual_glyph_count);
        }

        ////////////////////
        // Certain fonts, like Batang, contain glyphs for hidden control
        // and formatting characters. So we'll want to explicitly force their
        // advance to zero.
        if (run.no_visual())
          tf._glyph_advances.set(glyph_start, glyph_start + actual_glyph_count, 0.0f);

        ////////////////////
        // Set the final glyph count of this run and advance the starting glyph.
        run.glyph_count = actual_glyph_count;
        glyph_start += actual_glyph_count;

    };

    uint32 text_length = static_cast<uint32>(tf._text.length());
    if (!text_length) {
      if (scache) {
        ScriptFreeCache(&scache);
      }
      return;
    }


    // Estimate the maximum number of glyph indices needed to hold a string.
    uint32 estimated_glyph_count = estimate_glyph_count(text_length);

    tf._glyph_indices.size(estimated_glyph_count);
    tf._glyph_offsets.size(estimated_glyph_count);
    tf._glyph_advances.size(estimated_glyph_count);
    tf._glyph_clusters.size(text_length);

    uint32 glyph_start = 0;

    //uint total = _runs.length();

    // Shape each run separately. This is needed whenever script, locale,
    // or reading direction changes.
    for (uint32 run_index = 0; run_index < tf._runs.length(); ++run_index) {
      try {
        do_shape_glyph_run(run_index, glyph_start);
      }
      catch (tool::error&) {}
    }

    tf._glyph_indices.size(glyph_start);
    tf._glyph_offsets.size(glyph_start);
    tf._glyph_advances.size(glyph_start);
    tf._glyph_justified_advances = tf._glyph_advances;

    if (scache) { ScriptFreeCache(&scache); }

  }


#else
  void text_analysis::shape_glyph_runs(view& v, element* elem, tflow::text_flow& tf)
  {
    // Shapes all the glyph runs in the layout.

    uint text_length = static_cast<uint>(tf._text.length());
    if (!text_length)
      return;

    // Estimate the maximum number of glyph indices needed to hold a string.
    uint estimated_glyph_count = estimate_glyph_count(text_length);

    tf._glyph_indices.size(estimated_glyph_count);
    tf._glyph_offsets.size(estimated_glyph_count);
    tf._glyph_advances.size(estimated_glyph_count);
    tf._glyph_clusters.size(text_length);

    uint glyph_start = 0;

    //uint total = _runs.length();

    // Shape each run separately. This is needed whenever script, locale,
    // or reading direction changes.
    for (uint runIndex = 0; runIndex < tf._runs.length(); ++runIndex)
      shape_glyph_run(v, tf, elem, runIndex, glyph_start);

    tf._glyph_indices.size(glyph_start);
    tf._glyph_offsets.size(glyph_start);
    tf._glyph_advances.size(glyph_start);
    tf._glyph_justified_advances = tf._glyph_advances;

  }

  void text_analysis::shape_glyph_run(
    view& v, tflow::text_flow& tf, element* elem,
    uint run_index,
    IN OUT uint& glyph_start
    )
  {
    // Shapes a single run of text into glyphs.
    // Alternately, you could iteratively interleave shaping and line
    // breaking to reduce the number glyphs held onto at once. It's simpler
    // for this demostration to just do shaping and line breaking as two
    // separate processes, but realize that this does have the consequence that
    // certain advanced fonts containing line specific features (like Gabriola)
    // will shape as if the line is not broken.


    tflow::text_run& run = tf._runs[run_index];

    uint text_start = run.text_start;
    uint text_length = run.text_length;
    uint maxglyph_count = static_cast<uint>(tf._glyph_indices.size() - glyph_start);
    uint actual_glyph_count = 0;

    element* el = run.get_element();

    if (run.is_inline_block_pos(tf._text()))
    {
      el->check_layout(v);
      //assert(el->is_inline_block_element(v));
      run.glyph_start = glyph_start;
      run.glyph_count = 1;
      if (text_length > maxglyph_count)
      {
        uint totalGlyphsArrayCount = glyph_start + 1;
        tf._glyph_indices.size(totalGlyphsArrayCount);
      }
      tf._glyph_advances.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_advances.length()));
      tf._glyph_offsets.length(max(static_cast<size_t>(glyph_start + 1), tf._glyph_offsets.length()));
      tf._glyph_advances[glyph_start] = static_cast<float>(el->min_inline_margin_box_width(v));
      tf._glyph_indices[glyph_start] = 0;
      glyph_start += 1;
      return;
    }


    ustring  locale_name = el->get_lang();
    //html::style* style = el->get_style(v);
    xgl::font* pf = static_cast<xgl::font*>(run.get_used_font(v));

    //float  font_em_size = run.decl_font->size;

    //pf = used_font_face(pf,run);
    //d2d::asset<IDWriteFontFace> font_face = dw_font_face(pf, run);

    run.glyph_start = glyph_start;
    run.glyph_count = 0;

    if (text_length == 0)
      return; // Nothing to do..

    ////////////////////
    // Allocate space for shaping to fill with glyphs and other information,
    // with about as many glyphs as there are text characters. We'll actually
    // need more glyphs than codepoints if they are decomposed into separate
    // glyphs, or fewer glyphs than codepoints if multiple are substituted
    // into a single glyph. In any case, the shaping process will need some
    // room to apply those rules to even make that determintation.

    if (text_length > maxglyph_count)
    {
      maxglyph_count = estimate_glyph_count(text_length);
      uint totalGlyphsArrayCount = glyph_start + maxglyph_count;
      tf._glyph_indices.size(totalGlyphsArrayCount);
    }

    auto push_glyph = [&](uint16 gi, float x, float y) {
      tf._glyph_indices.push(gi);
      tf._glyph_advances.push(x);
      tflow::GLYPH_OFFSET goff = { 0 };
      goff.ascender_offset = y;
      tf._glyph_offsets.push(goff);
    };

    tf._glyph_indices.length(glyph_start);
    tf._glyph_advances.length(glyph_start);
    tf._glyph_offsets.length(glyph_start);

    actual_glyph_count = pf->get_glyph_indices_and_advances(tf._text(text_start, text_start + text_length), push_glyph);

    for (uint i = 0; i < text_length; ++i)
      tf._glyph_clusters[text_start + i] = i;

    ////////////////////
    // Certain fonts, like Batang, contain glyphs for hidden control
    // and formatting characters. So we'll want to explicitly force their
    // advance to zero.
    if (run.no_visual())
      tf._glyph_advances.set(glyph_start, glyph_start + actual_glyph_count, 0.0f);

    ////////////////////
    // Set the final glyph count of this run and advance the starting glyph.
    run.glyph_count = actual_glyph_count;
    glyph_start += actual_glyph_count;
  }
#endif

#if !defined(CORETEXT_LAYOUT)
  void window::setup_text_flow(html::element* elem, html::tflow::text_flow& tf, tool::slice< tool::handle<html::node> > nodes)
  {
    text_analysis::exec(*this, elem, tf, nodes);
  }
#endif

  void window::draw_glyph_run(gool::graphics* gfx, const html::tflow::text_flow& tf, const html::tflow::glyph_run& gr, gool::pointf at, argb color, const style* run_style)
  {
    //SkPaint paint;
    xgl::graphics* pg = static_cast<xgl::graphics*>(gfx);
    xgl::font* pf = gr.font.ptr_of<xgl::font>();
    buffer<uint16_t, 255> glyphs(gr.glyph_count);
    buffer<SkPoint, 255>  points(gr.glyph_count);
    SkScalar x = 0;
#ifdef OSX
    at.x = int(at.x) + 0.5f; // starting from middle of pixel,
                             // NOTE: without this Skia/OpenGL may produce strange artifacts on Retina
#endif

    if ((gr.bidi_level & 1) == 0)
      for (uint n = 0; n < gr.glyph_count; ++n)
      {
        glyphs[n] = tf._glyph_indices[gr.glyph_start + n];
        points[n].fY = tf._glyph_offsets[gr.glyph_start + n].ascender_offset;
        points[n].fX = x;
        x += tf._glyph_advances[gr.glyph_start + n];
      }
    else
      for (uint n = 0; n < gr.glyph_count; ++n) {
        glyphs[n] = tf._glyph_indices[gr.glyph_start + n];
        x -= tf._glyph_advances[gr.glyph_start + n];
        points[n].fY = tf._glyph_offsets[gr.glyph_start + n].ascender_offset;
        points[n].fX = x;
      }

    pg->text_paint.setTypeface(pf->_xgl_typeface);
    pg->text_paint.setTextSize(pf->size);
    pg->text_paint.setARGB(color.alfa, color.red, color.green, color.blue);

    pg->canvas()->save();
    pg->canvas()->translate(at.x, at.y);

      if (run_style->text_shadow.is_defined()) {
        element* pel = gr.node->get_element();
        for (shadow_def* it = run_style->text_shadow; it; it = it->next)
        {
          pointf shadow_offset;
            shadow_offset.x = pixels(*this, pel ,it->offset_x).width_f();
            shadow_offset.y = pixels(*this, pel,it->offset_y).height_f();

          pg->canvas()->translate(shadow_offset.x, shadow_offset.y);

          gool::argb c = it->color.val(run_style->font_color).to_argb() ;
          element* pel = gr.node->get_element();
          float radius = pixels(*this, pel,it->radius).width_f();
          //float spread = it->spread.pixels_width_f(*this, pel );
          SkPaint blur(pg->text_paint);
          blur.setARGB(c.alfa, c.red, c.green, c.blue);
          blur.setLCDRenderText(false);
          blur.setMaskFilter(SkBlurMaskFilter::Create( kNormal_SkBlurStyle, SkBlurMaskFilter::ConvertRadiusToSigma(radius), 0));
          pg->canvas()->drawPosText(glyphs.cbegin(), gr.glyph_count * 2, points.cbegin(), blur);

          pg->canvas()->translate(-shadow_offset.x, -shadow_offset.y);
        }
      }

    pg->canvas()->drawPosText(glyphs.cbegin(), gr.glyph_count * 2, points.cbegin(), pg->text_paint);
    pg->canvas()->restore();
  }


}


