#ifndef __xgl_graphics_h__
#define __xgl_graphics_h__

#include "config.h"
#include "xgl-types.h"
#include "tool/tool.h"
#include "gool/gool.h"

#include "xgl/skia/include/core/SkPath.h"
#include "xgl/skia/include/core/SkGraphics.h"
#include "xgl/skia/include/core/SkTypeface.h"
#include "xgl/skia/include/core/SkCanvas.h"
#include "xgl/skia/include/effects/SkGradientShader.h"

namespace xgl
{
  using namespace tool;
  using namespace gool;

  class graphics;


  struct font: public gool::font
  {
    typedef gool::font super;

    font():super() {}
    font(const font& f):super(f) { _xgl_typeface = f._xgl_typeface;  }
    virtual ~font() { }

    virtual bool has_glyph_for(uint ucodepoint) const override;
    virtual uint glyph_index(uint ucodepoint) const override;
    virtual bool glyph_metrics(::uint16 glyph_index, float em_size, float& width, float& height) const  override;

    virtual ustring used_font_name() const override;

    uint get_glyph_indices_and_advances( slice<wchar> text , function<void(uint16,float,float)> cb );

    mutable ref<SkTypeface> _xgl_typeface;

    ref<SkTypeface> xgl_typeface() const { assert(_xgl_typeface); return _xgl_typeface; }

  };

  //extern cairo_surface_t* cairo_surface( gool::bitmap* bmp );

  struct brush : public resource {

    brush() {}
    virtual ~brush() { ; }

    virtual bool is_solid() const { return false; }

    virtual void fill(graphics* gfx);
    virtual void stroke(graphics* gfx);

    ref<SkShader> _shader;

    ref<SkShader> shader() { return _shader; }

  };

  struct solid_brush : public brush {
    solid_brush(gool::argb c) {
      _shader = SkShader::CreateColorShader(cvt(c));
    }
    solid_brush(const gool::solid_brush& c) {
      _shader = SkShader::CreateColorShader(cvt(c.color));
    }
    virtual bool is_solid() const override { return true; }
    //virtual void fill(graphics* gfx);
    //virtual void stroke(graphics* gfx);
  };

  struct gradient_brush : public brush
  {
    //affine_mtx_f mx;
    void get_stops(const gool::gradient_brush& bin, array<SkColor>& colors, array<SkScalar>& positions);

  };

  struct linear_gradient_brush : public gradient_brush
  {
    linear_gradient_brush( const gool::linear_brush& bin  )
    {
      SkPoint fromto[2] = { cvt(bin.start), cvt(bin.end) };
      array<SkColor> colors;
      array<SkScalar> positions;
      get_stops(bin, colors, positions);

      auto m = bin.transform();
      SkMatrix mtx;
      if (m.is_identity())
        mtx.setIdentity();
      else {
        SkScalar affine[6] = { m.sx, m.shy, m.shx, m.sy, m.tx, m.ty };
        mtx.setAffine(affine);
      }
      _shader = SkGradientShader::CreateLinear(fromto, colors.cbegin(), positions.cbegin(), colors.size(), SkShader::kMirror_TileMode, 0, &mtx);
    }
  };

  struct radial_gradient_brush : public gradient_brush
  {
    radial_gradient_brush( const gool::radial_brush& bin  )
    {
      array<SkColor> colors;
      array<SkScalar> positions;
      get_stops(bin, colors, positions);

      auto m = bin.transform();
      SkMatrix mtx; 
      if (m.is_identity())
        mtx.setIdentity();
      else {
        SkScalar affine[6] = { m.sx, m.shy, m.shx, m.sy, m.tx, m.ty };
        mtx.setAffine(affine);
      } 
      if (bin.radius.x != bin.radius.y) {
        SkMatrix matrix;
        matrix.setScale(1, bin.radius.x / bin.radius.y);
        mtx.preConcat(matrix);
      }
      _shader = SkGradientShader::CreateRadial(cvt(bin.center), max(bin.radius.x, bin.radius.y), colors.cbegin(), positions.cbegin(), colors.size(), SkShader::kClamp_TileMode, 0, &mtx);
    }
  };

  struct image_brush : public brush
  {
    image_brush(const image* im)
    {
#pragma TODO("bitmap brush!")
      //SkShader::CreateBitmapShader();
    }
  };

  struct state_registers
  {
    handle<brush>     stroke_brush;
    handle<brush>     fill_brush;
    bool              even_odd;
    SkPaint           state;

    state_registers() : even_odd(false) { state.setAntiAlias(true); }

    bool drawable() const { return stroke_drawable() || fill_drawable(); }
    bool stroke_drawable() const { return stroke_brush.is_defined(); }
    bool fill_drawable() const { return fill_brush.is_defined(); }
  };


  class graphics : public gool::graphics
  {
    typedef gool::graphics super;
  protected:
    gool::point            _offset;
    SkCanvas*              _canvas = nullptr;
    state_registers        _registers;
    array<state_registers> _registers_stack;
    //int                    _n_layers = 0;
    argb                   _clear_color;

    struct layer_registers {
      handle<bitmap> mask;
      tristate_v     draw1a;
    };

    array<layer_registers> _layers_stack;

    graphics(bool transparent): super(transparent) {}
  public:
    SkPaint                text_paint;

  public:
    graphics(SkCanvas* canvas, argb cc): super(!cc.is_opaque()), _clear_color(cc) {
       _canvas = canvas;
       text_paint.setTextEncoding(SkPaint::kGlyphID_TextEncoding);
       text_paint.setAntiAlias(true);
       text_paint.setSubpixelText(true);
       text_paint.setLCDRenderText(!cc.is_opaque() ? false : true);
       _registers.state.setFilterQuality(kLow_SkFilterQuality);
    }

    graphics(SkCanvas* canvas, bool is_transparent) : super(is_transparent) {
      _canvas = canvas;
      text_paint.setTextEncoding(SkPaint::kGlyphID_TextEncoding);
      text_paint.setAntiAlias(true);
      text_paint.setSubpixelText(true);
      text_paint.setLCDRenderText(is_transparent ? false : true);
      _registers.state.setFilterQuality(kLow_SkFilterQuality);
      _clear_color = argb::no_color();
    }

    virtual ~graphics() {
      while (pop_state());
      //_canvas will be unref'ed by xgl::ref
    }
      
    virtual void begin_drawing() override {
      if(_canvas)
        _canvas->clear(cvt(_clear_color)); // SK_ColorTRANSPARENT
    }
      
    virtual void clear(const rect& rc) override {
      do_push_clip(rc);
      canvas()->clear(SK_ColorTRANSPARENT);
      do_pop_clip();
    }
      
    virtual gool::application* app() const override;

    virtual void push_opacity() override {
      super::push_opacity();
      text_paint.setLCDRenderText(true);
    }
    virtual void pop_opacity() override {
      text_paint.setLCDRenderText(false);
      super::pop_opacity();
    }

	  //SkSurface* target() const { return _target; }
    virtual SkCanvas*  canvas() const { return _canvas; }

    virtual void flush() override { _canvas->flush(); }

    virtual void fill( argb c, const rect& dst) override;
    virtual void fill( argb c, const rect& dst, const size& radius) override;
    virtual void fill( argb c, const gool::path* dst) override;
    virtual void fill( const gool::brush* lb, const rect& dst) override;
    virtual void fill( image* img, const rect& dst, const rect& src,point offset = point(0,0), size celldim = size()) override;
    virtual void fill( image* img, const rect& dst, point offset = point(0,0)) { fill(img, dst,rect(img->dim()),offset); }
    virtual void fill( image* img, const gool::path* dst) override;

    //virtual image_renderer_f image_renderer(image* img) override;
    virtual gool::path* create_path() const override;

    virtual void draw( const gool::path* dst, argb c, float width = 1.0f ) override;

    virtual void draw_line_v(const rect& rc, argb c, int stroke, int step, int_v off) override;
    virtual void draw_line_h(const rect& rc, argb c, int stroke, int step, int_v off) override;

    virtual void draw_line(pointf p1, pointf p2 , argb c, int width) override;

    //virtual void draw( gool::text_layout& tl, point  dst, argb c) override { draw( tl, pointf(dst),c); }
    //virtual void draw( gool::text_layout& tl, pointf dst, argb c) override;
    //virtual void draw( gool::text_layout& tl, pointf dst ) override; // using current fill and stroke

    virtual bool push_clip(const rect& r, clipper* cl) override { do_push_clip(r); super::push_clip(r,cl); return true;  }
    virtual void pop_clip() { super::pop_clip(); do_pop_clip(); }

            void do_push_clip(const rect& r);
            void do_pop_clip();

    virtual point offset(point noff) override;

    virtual bool set_antialiasing(bool onoff) override;

    virtual uint push_state() override;
    virtual bool pop_state() override;
    virtual void restore_state(uint level) override;

    //virtual const text_format& get_text_format();
    //virtual void               set_text_format(const text_format& tf);

	  virtual void push_layer(const gool::rect& clip, byte opacity = 255, function<bool(gool::filter_graph_builder*)> filters = nullptr) override;
    virtual void push_layer(const gool::path* clip, byte opacity = 255) override;
    virtual void push_layer(const gool::bitmap* clip, bool draw1a, byte opacity = 255) override;
    virtual void pop_layer() override;
    virtual uint n_layers() override;

    virtual void transform(const affine_mtx_f& m) override;
    virtual affine_mtx_f transform() const override;
    virtual void reset_transform() override;


    //|
    //| Graphin stuff
    //|

    virtual    CAP_STYLE cap_style() override;
    virtual         void cap_style(CAP_STYLE ncs) override;
    virtual    LINE_JOIN line_join() override;
    virtual         void line_join(LINE_JOIN ncs, float miterlimit = 4.0f) override;
    virtual   DASH_STYLE dash_style() override;
    virtual         void dash_style( DASH_STYLE st ) override;
    virtual         void custom_dash_style(slice<float> pattern, float offset) override;

    virtual void set_stroke( argb c = argb::no_color()) override ;
    virtual void set_stroke( const linear_brush& lb ) override;
    virtual void set_stroke( const radial_brush& rb ) override;
    virtual void set_stroke_width( float w ) override;

    virtual void set_fill( argb c = argb::no_color() ) override;
    virtual void set_fill( const linear_brush& lb ) override;
    virtual void set_fill( const radial_brush& rb ) override;
    virtual void set_fill( const image* ib ) override ;
    virtual void set_fill_even_odd( bool even_odd ) ;

    virtual void draw_ellipse(pointf center, sizef radius, bool stroke = true, bool fill = true)  override;
    virtual void draw_rectangle(pointf org, sizef dim, bool stroke = true, bool fill = true) override;
	  virtual void draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr, sizef rbr, sizef rbl, bool stroke = true, bool fill = true) override { super::draw_rectangle(org,dim,rtl, rtr, rbr, rbl); }
    
    virtual void translate(pointf pt) override;
    virtual void rotate(float radians, pointf center = pointf()) override;
    virtual void scale(sizef sz, pointf center = pointf()) override;
    virtual void skew(sizef sz, pointf center = pointf()) override;

    virtual pointf world_to_screen(pointf pt) const override;
    virtual pointf screen_to_world(pointf pt) const override;

    virtual void draw_path( const gool::path* dst, bool stroke = true, bool fill = true) override;
    virtual void draw_path( const gool::polygonf& p, bool stroke = true, bool fill = true) override;
    virtual void draw_line(pointf start, pointf end) override;
    virtual void draw_line(pointf origin, pointf end, line_end_style sorigin, line_end_style send) override { super::draw_line(origin, end, sorigin, send); }

    virtual void do_draw( image* img, const rectf&  dst, const rect& src, byte opacity = 255) override;
    virtual void do_expand(image* img, const rect& dst, const SECTION_DEFS& sds, rect area) override;
            //void render_current_path(bool fill = true, bool stroke = true);
            void render_figure(bool fill, bool stroke, function<void(SkCanvas* canvas, const SkPaint& paint)> worker);

    virtual void            image_rendering_mode(IMAGE_RENDERING irm);
    virtual IMAGE_RENDERING image_rendering_mode() const;

  };

  class path : public gool::path
  {
    public:
        path() { set_even_odd(true); }
        virtual ~path() {}

        virtual bool is_inside(pointf pt) override;

        virtual void start(pointf start, bool filled) override;
        virtual void move_to(pointf pt, bool rel = false) override;
        virtual void line_to(pointf pt, bool rel = false) override;
        virtual void hline_to(float x, bool rel = false) override;
        virtual void vline_to(float y, bool rel = false) override;

        virtual void arc_to(pointf to, sizef r, float rotation_angle, bool large_arc, bool clockwise, bool rel_to) override;

        virtual void add_arc(pointf c, sizef r, float angle_start, float angle_swipe) override;
        virtual void quadratic_to(pointf pt, pointf cp, bool rel = false) override;
        virtual void cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel = false) override; // bezier
        virtual void close() override;

        virtual bool is_empty() const override { return _path.isEmpty(); }
        virtual void reset() override { _path.reset(); }
        virtual void seal() override { ; }

        bool even_odd() const { return _path.getFillType() == SkPath::kEvenOdd_FillType; }
        virtual void set_even_odd(bool yes) { _path.setFillType(yes ? SkPath::kEvenOdd_FillType : SkPath::kWinding_FillType); }

        virtual rectf bounds() const override { return cvt(_path.getBounds()); }

        virtual gool::path* combine(gool::path::COMBINE_MODE mode, const gool::path* other) override;

        const SkPath& sk_path() const { return _path; }

        virtual pointf lastp() { SkPoint p; _path.getLastPt(&p); return pointf(p.x(),p.y()); }
        virtual void   lastp(pointf p) { _path.setLastPt(p.x, p.y); }
    protected:
        SkPath           _path;
  };

  inline gool::path* graphics::create_path() const { return new xgl::path(); }

  class bitmap_graphics : public graphics
  {
	   typedef graphics super;

    protected:
     SkCanvas _canvas;

     bitmap_graphics(bool is_transparent):super(is_transparent) {}
	   bitmap_graphics(SkBitmap& bmp, bool is_transparent);

    public:

	   static bitmap_graphics* create(gool::graphics* pg, gool::bitmap* pb, argb initc = argb());
     virtual ~bitmap_graphics() {}
     virtual SkCanvas*  canvas() const { return &const_cast<bitmap_graphics*>(this)->_canvas; }

	   virtual bool is_bitmap_graphics() const override { return true; }

  };

  class bitmap_bits_graphics : public bitmap_graphics
  {
		typedef bitmap_graphics super;
		handle<bitmap> pbmp;
  protected:
		 bitmap_bits_graphics(bool transparent): super(transparent) {}
     bitmap_bits_graphics(SkBitmap& bmp, bool transparent) : bitmap_graphics(bmp,transparent) {}
     public:
 		 static bitmap_bits_graphics* create(gool::bitmap* pb, argb initc = argb());
     virtual ~bitmap_bits_graphics();
  };

}

#endif
