
#include "xgl.h"

#include "SkGraphics.h"
#include "SkEvent.h"
#include "SkTypeface.h"
#include "SkFontMgr.h"
#include "SkData.h"

#if defined(WINDOWS)
#include "SkTypes.h"
#include "SkTypeface_win.h"

SkFontMgr* SkFontMgr::Factory() {
#if defined(USE_DWRITE_SKIA)
  if( tool::environment::get_os_version() > environment::WIN_VISTA) {
    if(SkFontMgr* t = SkFontMgr_New_DirectWrite())
      return t;
  }
#endif
  return SkFontMgr_New_GDI();
}

#endif

#if defined(OSX)
  extern CTFontRef SkTypeface_GetCTFontRef(const SkTypeface* face);
  extern SkTypeface* SkCreateTypefaceFromCTFont(CTFontRef fontRef);
#endif

//extern xgl::view* xgl_create_frame(const window_params& params);

namespace xgl {

  ref<SkFontMgr> sk_font_mgr;

  static SkFontStyle sk_fontstyle(const gool::font* pf) {
    return SkFontStyle(int(pf->weight), SkFontStyle::kNormal_Width, pf->italic ? SkFontStyle::kItalic_Slant : SkFontStyle::kUpright_Slant);
  }

  static SkTypeface::Style sk_typefacestyle(const gool::font* pf) {
    uint s = 0;
    if (pf->weight > 600) s |= SkTypeface::Style::kBold;
    if (pf->italic) s |= SkTypeface::Style::kItalic;
    return SkTypeface::Style(s);
  }



  tool::wchars application::system_fontname()
  {
    static tool::ustring fn;
    if( fn.is_undefined() ) {

      ref<SkTypeface> tf = SkTypeface::RefDefault();
      assert(tf);
      if(!tf)
        return WCHARS("Tahoma");
      SkString tfn;
      tf->getFamilyName(&tfn);
      fn = u8::cvt( chars(tfn.c_str(), tfn.size()) );
    }
    return fn();
 }

 //static tool::pool<ustring> names;
 application*  application::instance = nullptr;

  application::application() {

    SkGraphics::Init();
    //SkEvent::Init();
#if defined(WINDOWS) && !defined(WINDOWLESS)
    frame::init();
    dialog::init();
#endif
    sk_font_mgr = ref<SkFontMgr>(SkFontMgr::RefDefault());
  }

  application::~application() {
      _fonts.clear();
      sk_font_mgr = nullptr;
      //SkEvent::Term();
      SkGraphics::Term();
      //html::behavior::init(false);
      instance = 0;
    }

  gool::font* application::do_create_font(const ustring& name
                  , float size
                  , uint weight
                  , bool italic
                  , gool::FONT_RENDERING  mode
                  , ref<SkTypeface> sktf ) {
    xgl::font key;
    key.name = name;
    key.size = size;
    key.weight = weight;
    key.italic = italic;
    xgl::font* pf = _fonts.intern(key);
    if( pf ) do_init_font(pf,sktf);
    return pf;
    //return nullptr;
  }

  void application::clear_font_cache() {
      //auto fonts = _fonts.elements()();
      //for( uint n = 0; n < fonts.length; ++n )
      //    fonts[n]->_ct_font = nullptr;
  }
#if defined(OSX)
  gool::font* application::register_font( CTFontRef ctf ) {

     ref<SkTypeface> ptf = ref<SkTypeface>(SkCreateTypefaceFromCTFont(ctf));
     
     xgl::font key;
     SkString name; ptf->getFamilyName(&name);
     key.name = ustring(name.c_str());
     key.size = CTFontGetSize(ctf);
     key.weight = ptf->fontStyle().weight();
     key.italic = ptf->fontStyle().slant() >= SkFontStyle::kItalic_Slant;
     xgl::font* pf = _fonts.intern(key);
     if( pf && !pf->_xgl_typeface) {
          //pf->ascent = ceil(CTFontGetAscent(ctf));
          //pf->descent = ceil(CTFontGetDescent(ctf));
          //pf->_xgl_typeface = ptf;
          do_init_font(pf, ptf);
     }
          
     return pf;
      
  }

  
  CF::ref<CTFontRef>  application::ctfont(gool::font* pf) {
    CTFontRef rt = SkTypeface_GetCTFontRef( static_cast<xgl::font*>(pf)->xgl_typeface().ptr() );
    return CF::ref<CTFontRef>(CTFontCreateCopyWithAttributes(rt, pf->size, NULL, NULL));
  }
#endif
    

   wchars application::fallback_list(wchars lang_id, WRITING_SCRIPT script, bool serif)
   {
#if defined(WINDOWS)
     switch (script)
     {
     default:
     case WS_UNKNOWN:
     case WS_CYRILLIC: return serif
       ? WCHARS("Times New Roman,!")
       : WCHARS("Segoe UI,Arial,!");
     case WS_HANZI:
       if (lang_id.starts_with(WCHARS("ja")))
         return serif
         ? WCHARS("MS Mincho,MS明朝,!")
         : WCHARS("Meiryo UI,メイリオ UI,Yu Gothic UI,!");
       else if (lang_id.ends_with(WCHARS("tw")) || lang_id.ends_with(WCHARS("hant")))
         return serif  // traditional
         ? WCHARS("MingLiU,名流,MingLiU-ExtB,名流-ExtB,!")
         : WCHARS("Microsoft JhengHei,微軟正黑體,MingLiU-ExtB,名流-ExtB,!");
       else
         return serif  // simplified
         ? WCHARS("NSimSun,SimSun-ExtB,!")
         : WCHARS("Microsoft YaHei UI,微软雅黑,SimSun-ExtB,!");
     case WS_KANA:
       return serif
         ? WCHARS("MS Mincho,MS明朝,!")
         : WCHARS("Meiryo UI,メイリオ UI,Yu Gothic UI,!");
     case WS_ARABIC:
       return serif
         ? WCHARS("Traditional Arabic,Arabic Typesetting,!")
         : WCHARS("Arabic Simplified,!");
     case WS_HEBREW:
       return serif
         ? WCHARS("Narkisim,!")
         : WCHARS("Miriam,!");
     case WS_HANGUL:
       return serif
         ? WCHARS("Batang,!")
         : WCHARS("Malgun Gothic,!");
     }
#elif defined(OSX)
    switch(script)
    {
        default:
        case WS_UNKNOWN:
            return serif
            ? WCHARS("Georgia,Times,Times New Roman, Bangla MN,Devanagari MN,Euphemia UCAS,Kannad MN,Kefa,Khmer MN,Lao MN,Malayalam MN,Mshtakan,Myanmar MN,Oriya MN, Sathu,Sinhala MN,Tamil MN,!")
            : WCHARS("Helvetica,Arial Unicode MS,Ayuthaya,Bangla Sangam MN,Devanagari Sangam MN,Euphemia UCAS,Kannad Sangam MN,Kefa,Khmer Sangam MN,Lao Sangam MN,Malayalam Sangam MN,Mshtakan,Myanmar Sangam MN,Oriya Sangam MN, Sathu,Sinhala Sangam MN,Tamil Sangam MN,!");
        case WS_CYRILLIC: return serif
            ? WCHARS("Georgia,Times,!")
            : WCHARS("Helvetica,Arial Unicode MS,!");
        case WS_HANZI:
            if(lang_id.starts_with(WCHARS("ja")))
                return serif
                ? WCHARS("Hiragino Kaku Gothic Pro,Hiragino Kaku Gothic ProN,!")
                : WCHARS("Hiragino Mincho Pro,Hiragino Mincho ProN,!");
            else if(lang_id.ends_with(WCHARS("tw")))
                return serif  // traditional
                ? WCHARS("Kaiti TC,Apple LiGothic,!")
                : WCHARS("Heiti TC,LiHei Pro,BiauKai,!");
            else
                return serif  // simplified
                ? WCHARS("Kaiti SC,Hei,!")
                : WCHARS("Heiti SC,Hannotate SC,ST Song,Hei,!");
        case WS_KANA:
            return serif
            ? WCHARS("Hiragino Kaku Gothic Pro,Hiragino Kaku Gothic ProN,!")
            : WCHARS("Hiragino Mincho Pro,Hiragino Mincho ProN,!");
        case WS_ARABIC:
            return serif
            ? WCHARS("Nadeem,Geeza Pro,!")
            : WCHARS("Baghdad,Geezah,!");
        case WS_HEBREW:
            return serif
            ? WCHARS("New Peninim,Corsiva Hebrew,!")
            : WCHARS("Arial Hebrew,Raanana,!");
        case WS_HANGUL:
            return serif
            ? WCHARS("Apple Myungjo,Hiragino Kaku Gothic Pro,Hiragino Kaku Gothic ProN,!")
            : WCHARS("Apple Gothic,Hiragino Mincho Pro,Hiragino Mincho ProN,!");
    }
#elif defined(LINUX)
   switch(script)
    {
    default:
    case WS_UNKNOWN:
      return serif
        ? WCHARS("Georgia,Times,Times New Roman, Bangla MN,Devanagari MN,Euphemia UCAS,Kannad MN,Kefa,Khmer MN,Lao MN,Malayalam MN,Mshtakan,Myanmar MN,Oriya MN, Sathu,Sinhala MN,Tamil MN,!")
        : WCHARS("Helvetica,Arial Unicode MS,Ayuthaya,Bangla Sangam MN,Devanagari Sangam MN,Euphemia UCAS,Kannad Sangam MN,Kefa,Khmer Sangam MN,Lao Sangam MN,Malayalam Sangam MN,Mshtakan,Myanmar Sangam MN,Oriya Sangam MN, Sathu,Sinhala Sangam MN,Tamil Sangam MN,!");
    case WS_CYRILLIC: return serif
                        ? WCHARS("Georgia,Times,!")
                        : WCHARS("Helvetica,Arial Unicode MS,!");
    case WS_HANZI:
      if(lang_id.starts_with(WCHARS("ja")))
        return serif
        ? WCHARS("Hiragino Kaku Gothic Pro,Hiragino Kaku Gothic ProN,!")
        : WCHARS("Hiragino Mincho Pro,Hiragino Mincho ProN,!");
      else if(lang_id.ends_with(WCHARS("tw")))
        return serif  // traditional
        ? WCHARS("Kaiti TC,Apple LiGothic,!")
        : WCHARS("Heiti TC,LiHei Pro,BiauKai,!");
      else
        return serif  // simplified
        ? WCHARS("Kaiti SC,Hei,!")
        : WCHARS("Heiti SC,Hannotate SC,ST Song,Hei,!");
    case WS_KANA:
      return serif
        ? WCHARS("Hiragino Kaku Gothic Pro,Hiragino Kaku Gothic ProN,!")
        : WCHARS("Hiragino Mincho Pro,Hiragino Mincho ProN,!");
    case WS_ARABIC:
      return serif
        ? WCHARS("Nadeem,Geeza Pro,!")
        : WCHARS("Baghdad,Geezah,!");
    case WS_HEBREW:
      return serif
        ? WCHARS("New Peninim,Corsiva Hebrew,!")
        : WCHARS("Arial Hebrew,Raanana,!");
    case WS_HANGUL:
      return serif
        ? WCHARS("Apple Myungjo,Hangangche,!")
        : WCHARS("Apple Gothic,Jung Gothic,!");
    }
#endif

    return wchars();
  }

#if defined(WINDOWS)
  #define MONOSPACE WCHARS("consolas,courier new")
  #define SERIF     WCHARS("times new roman")
  #define SANSSERIF WCHARS("segoe ui,tahoma,arial")
  #define CURSIVE   WCHARS("vladimir script,freestyle script,serif")
  #define FANTASY   WCHARS("jokeman,comic sans ms,sans-serif")
  #define LAST_RESORT   WCHARS("Arial Unicode MS")
#elif defined(LINUX)
  #define MONOSPACE WCHARS("ubuntu mono,courier")
  #define SERIF     WCHARS("dejavu serif,times new roman")
  #define SANSSERIF WCHARS("ubuntu,lucida sans unicode,arial")
  #define CURSIVE   WCHARS("zapf chancery,freestyle script,brush script,serif")
  #define FANTASY   WCHARS("comic sans,zapfino,sans-serif")
  #define LAST_RESORT   WCHARS("system")
#elif defined(OSX)
  #define MONOSPACE WCHARS("menlo")
  #define SERIF     WCHARS("times new roman")
  #define SANSSERIF WCHARS("lucida grande,lucida sans unicode,arial")
  #define CURSIVE   WCHARS("apple chancery,zapf chancery,freestyle script,brush script,serif")
  #define FANTASY   WCHARS("comic sans,zapfino,sans-serif")
  #define LAST_RESORT   WCHARS("Helvetica")
#endif

  #define SYSTEM        WCHARS("system")

  bool font_exists(wchars name)
  {
    tool::string as = u8::cvt(name);
    SkFontStyleSet* ss = sk_font_mgr->matchFamily(as.c_str());
    if (ss) {
      int cnt = ss->count();
      ss->unref();
      return cnt > 0;
    }
    return false;
  }

  bool application::each_font_family(function<bool(wchars name)>  cb, function<void(ucode, ucode)> cb_ranges_or_null)
  {
    int n = sk_font_mgr->countFamilies();
    if (cb_ranges_or_null)
      assert(false); // not implemented
    else {
      for (int i = 0; i < n; ++i) {
        SkString fn;
        sk_font_mgr->getFamilyName(i, &fn);
        if (cb(ustring(fn.c_str())))
          return true;
      }
    }
    return false;
  }

  tool::ustring application::get_supported_font_family(const tool::ustring& family_name_list)
  {
    wtokens tz(family_name_list(), WCHARS(",;"));
    wchars name;
    while (tz.next(name))
    {
      name = trim(name);
      if (icmp(name, WCHARS("monospace")))
        return get_supported_font_family(MONOSPACE);
      else if (icmp(name, WCHARS("sans-serif")))
        return get_supported_font_family(SANSSERIF);
      else if (icmp(name, WCHARS("serif")))
        return get_supported_font_family(SERIF);
      else if (icmp(name, WCHARS("cursive")))
        return get_supported_font_family(CURSIVE);
      else if (icmp(name, WCHARS("fantasy")))
        return get_supported_font_family(FANTASY);
      //else if (name.like(W("system*"))) { - wrong - recursive call
      //  return get_supported_font_family(SYSTEM);
      //}
      else if (name == WCHARS("!"))
        return LAST_RESORT;
      else if (name == SYSTEM)
        return system_fontname();
      else if (has_memory_family(name))
        return name;
      else if (font_exists(name))
        return name;
      else if (icmp(name, system_fontname()))
        return name;
    }
    return LAST_RESORT;
  }

  
  bool application::get_supported_font_family(wchars family_name_list,
                                  const function<bool(wchars)>& cb,
                                  tool::wchars lang_id,
                                  WRITING_SCRIPT script,
                                  uint char_code,
                                  tristate_v serif) // cb shall return true if it accepts the family
  {

    wtokens tz( family_name_list,WCHARS(",;") );
    wchars name;

    while(tz.next(name))
    {
      name = trim(name);
      if( icmp(name,WCHARS("monospace"))  )
        return get_supported_font_family( MONOSPACE, cb, lang_id, script, char_code, false);
      else if( icmp(name,WCHARS("sans-serif")) )
        return get_supported_font_family( SANSSERIF, cb, lang_id, script, char_code, false);
      else if( icmp(name,WCHARS("serif")) )
        return get_supported_font_family( SERIF,cb,  lang_id, script, char_code, true );
      else if( icmp(name,WCHARS("cursive")) )
        return get_supported_font_family( CURSIVE,cb, lang_id, script, char_code, false);
      else if( icmp(name,WCHARS("fantasy")) )
        return get_supported_font_family(FANTASY,cb, lang_id, script, char_code, false);
      else if( name.like(W("system*"))) {
        if(cb(system_fontname()))
            return true;
        break;
      }
      else if(name == WCHARS("!"))
        return cb(LAST_RESORT);
      else if (has_memory_family(name) && cb(name))
        return true;
      else if(font_exists(name) && cb( name ))
        return true;
      else if( icmp(name, system_fontname()) && cb( name ) )
        return true;
    }
    //return false;
    if (get_supported_font_family(fallback_list(lang_id, script, !!serif), cb, lang_id, script, char_code, serif))
      return true;

    array<ustring> list;
    if (font_families_for_char_code(lang_id, script, char_code, list))
    {
      for (int i = 0; i < list.size(); ++i) {
        if (cb(list[i]))
          return true;
      }
    }
    return each_font_family(cb,nullptr);
  }

  bool application::construct_font_collection_for(memory_font_variant& fv)
  {
    ref<SkData> data = SkData::NewWithCopy(fv.data.cbegin(),fv.data.length());
    fv.typeface = sk_font_mgr->createFromData(data);
    return true;
  }

  bool application::get_used_font(
                              tool::handle<gool::font>& pf,
                              tool::wchars family_name_list, const gool::font* proto,
                              tool::wchars lang_id,
                              WRITING_SCRIPT script,
                              uint char_code)
  {
      handle<gool::font> rf;
      char_code = max(char_code,' ');

      auto acceptor = [&]( wchars ff ) -> bool
      {
        string s_lang_id = lang_id;
        const char* p = s_lang_id.c_str();
        if( !ff ) {
          ref<SkTypeface> tf = ref<SkTypeface>(sk_font_mgr->matchFamilyStyle(NULL,sk_fontstyle(proto)));
          ustring us_name;
          if (tf) {
            SkString s; tf->getFamilyName(&s);
            us_name = u8::cvt(tool::chars(s.c_str(), s.size()));
          }
          rf = this->do_create_font(us_name, proto->size, proto->weight, proto->italic, proto->mode, tf);
        }
        else {
#if 0
          string s_name = ff;
          ref<SkTypeface> tf = ref<SkTypeface>(sk_font_mgr->matchFamilyStyleCharacter(s_name.c_str(), sk_fontstyle(proto), &p, 1, char_code));
          if(!tf)
                return false;
          SkString s; tf->getFamilyName(&s);
          ustring us_name = ustring::utf8(tool::chars(s.c_str(),s.size()));
          rf = this->do_create_font(us_name, proto->size, proto->weight, proto->italic, proto->mode, tf);
#else 
          string s_name = u8::cvt(ff);
          ref<SkTypeface> tf = ref<SkTypeface>(sk_font_mgr->matchFamilyStyle(s_name.c_str(), sk_fontstyle(proto)));
          if(!tf)
              return false;
          SkString s; tf->getFamilyName(&s);
            ustring us_name = u8::cvt(tool::chars(s.c_str(), s.size()));
          rf = this->do_create_font(us_name, proto->size, proto->weight, proto->italic, proto->mode, tf);
#endif 

        }
        if(char_code > ' ' && !rf->has_glyph_for(char_code))
          return false;
        return true;
      };
      bool r =  get_supported_font_family(family_name_list, acceptor, lang_id, script, char_code);
      pf = rf;
      return r;
  }

  gool::path* application::create_path()
  {
    return new xgl::path();
  }

  handle<html::view> application::create_frame(const window_params& params)
  {
#if defined(WINDOWS) && !defined(WINDOWLESS)
	  return xgl::frame::construct(params);
#else
    return super::create_frame(params);
#endif
  }

  handle<html::view> application::create_dialog(const window_params& params)
  {
#if defined(WINDOWS) && !defined(WINDOWLESS)
      return xgl::dialog::construct(params);
#else
      return super::create_dialog(params);
#endif
  }

  /*gool::text_layout* application::create_text_layout(tool::wchars text, const gool::text_format& tf)
  {
    return nullptr;
  }*/

  window_processor_t* application::create_window_processor(const window_params& params)
  {
    return new xgl::window(params);
  }


  bool application::do_init_font(gool::font* pfont, ref<SkTypeface> tf)
  {
    xgl::font* pf = static_cast<xgl::font*>(pfont);
	  if(pf && !pf->_xgl_typeface)
	  {

        if(!tf)
        {
          memory_font_variant* fv = closest_memory_font_collection(pf->name, pf->weight, pf->italic);
          if(fv)
            tf = fv->typeface;
          else {
            tf = SkTypeface::CreateFromName(string(pf->name), sk_typefacestyle(pf));
          }
        }
        SkPaint paint;
        paint.setAntiAlias(true);
        paint.setSubpixelText(true);
        paint.setTypeface(tf);
//#ifdef OSX
//        paint.setTextSize(floorf(pfont->size));
//#else
        paint.setTextSize(pfont->size);
//#endif
        SkPaint::FontMetrics fm;
        paint.getFontMetrics(&fm);

#if defined(OSX)
        //pf->ascent = (int)(-fm.fAscent * 1.2f + 0.5f);
        //pf->descent = (int)(fm.fDescent * 1.2f + 0.5f);
        //pf->ascent = (int)floorf(-fm.fTop + 0.5f);
        //pf->descent = (int)ceilf(fm.fBottom + 0.5f);
        pf->ascent = (int)floorf(-fm.fTop);
        pf->descent = (int)ceilf(fm.fDescent);
        pf->x_height = (int)ceilf(fm.fXHeight); //pf->ascent - pf->descent;
#else
        if (fm.fLeading > 0) {
          pf->ascent = (int)floorf(-fm.fTop + 0.5f);
          pf->descent = (int)ceilf(fm.fBottom + 0.5f);
          pf->x_height = (int)ceilf(fm.fXHeight); //pf->ascent - pf->descent;
        }
        else {
          pf->ascent = (int)(-fm.fAscent + 0.5f);
          pf->descent = (int)(fm.fDescent + 0.5f);
          pf->x_height = (int)(fm.fXHeight + 0.5f); //pf->ascent - pf->descent;
        }
#endif
        
        pf->_xgl_typeface = tf;

	  }
    return true;
  }

  void  application::get_system_font(/*inout*/ ustring& name, /*out*/int& size, /*out*/ uint& weight, /*out*/bool& italic)
  {
      static  ustring sfn;
      static  int ssize = 0;
      static  uint sweight = 0;
      static  bool sitalic = false;
      if(sfn.is_undefined()) {
        super::get_system_font(sfn, ssize, sweight, sitalic);
#ifndef OSX
        ref<SkTypeface> tf = SkTypeface::RefDefault();
        SkString t;
        tf->getFamilyName(&t);
        sfn = u8::cvt(chars(t.c_str(),t.size()));
#endif
      }
      name = sfn;
      size = ssize;
      weight = sweight;
      italic = sitalic;
  }

  gool::graphics* application::create_bitmap_graphics( gool::graphics *proto, gool::bitmap *bmp, gool::argb initc )
  {
    return bitmap_graphics::create(proto,bmp,initc);
  }
  gool::graphics* application::create_bitmap_bits_graphics(gool::bitmap* bmp, gool::argb initc, bool high_quality)
  {
    return bitmap_bits_graphics::create(bmp,initc);
  }

  application* app_factory()
  {
    critical_section _(application::guard);
    if (!application::instance)
      application::instance = new application();
    return application::instance;
  }


}








