//--------------------------------------------------------------------------------------
// File: dxsdk\Samples\C++\Direct3D11\Tutorials\Tutorial07\Tutorial07.cpp
//
// This application demonstrates Sciter/DirectX integration.
//
// Iriginal Tutorial07 Copyright (c) Microsoft Corporation. All rights reserved.
// Sciter additions: Terra Informatica Software, Inc.
// marked by SCITER+/SCITER- brackets.
//--------------------------------------------------------------------------------------
#include <windows.h>
#include <d3d11.h>
#include <d3dx11.h>
#include <d3dcompiler.h>
#include <xnamath.h>
#include "resource.h"

// SCITER+
#include "sciter-x.h"
#include "sciter-x-lite-keycodes.h"

#define SciterInstanceID  ((void*)1) // could be any pointer identifying uniquely the instance

unsigned translate_key(unsigned vk);

// SCITER-

//--------------------------------------------------------------------------------------
// Structures
//--------------------------------------------------------------------------------------
struct SimpleVertex
{
    XMFLOAT3 Pos;
    XMFLOAT2 Tex;
};

struct CBNeverChanges
{
    XMMATRIX mView;
};

struct CBChangeOnResize
{
    XMMATRIX mProjection;
};

struct CBChangesEveryFrame
{
    XMMATRIX mWorld;
    XMFLOAT4 vMeshColor;
};


//--------------------------------------------------------------------------------------
// Global Variables
//--------------------------------------------------------------------------------------
HINSTANCE                           g_hInst = NULL;
HWND                                g_hWnd = NULL;
D3D_DRIVER_TYPE                     g_driverType = D3D_DRIVER_TYPE_NULL;
D3D_FEATURE_LEVEL                   g_featureLevel = D3D_FEATURE_LEVEL_11_0;
ID3D11Device*                       g_pd3dDevice = NULL;
ID3D11DeviceContext*                g_pImmediateContext = NULL;
IDXGISurface*                       g_pSurface = NULL; // for SCITER
IDXGISwapChain*                     g_pSwapChain = NULL;
ID3D11RenderTargetView*             g_pRenderTargetView = NULL;
ID3D11Texture2D*                    g_pDepthStencil = NULL;
ID3D11DepthStencilView*             g_pDepthStencilView = NULL;
ID3D11VertexShader*                 g_pVertexShader = NULL;
ID3D11PixelShader*                  g_pPixelShader = NULL;
ID3D11InputLayout*                  g_pVertexLayout = NULL;
ID3D11Buffer*                       g_pVertexBuffer = NULL;
ID3D11Buffer*                       g_pIndexBuffer = NULL;
ID3D11Buffer*                       g_pCBNeverChanges = NULL;
ID3D11Buffer*                       g_pCBChangeOnResize = NULL;
ID3D11Buffer*                       g_pCBChangesEveryFrame = NULL;
ID3D11ShaderResourceView*           g_pTextureRV = NULL;
ID3D11SamplerState*                 g_pSamplerLinear = NULL;
XMMATRIX                            g_World;
XMMATRIX                            g_View;
XMMATRIX                            g_Projection;
XMFLOAT4                            g_vMeshColor( 0.7f, 0.7f, 0.7f, 1.0f );

FLOAT                               g_rotationSpeed = 1.0;
FLOAT                               g_colorSpeed = 1.0;

//SCITER+
sciter::dom::element                g_backLayer; // layer
sciter::dom::element                g_foreLayer; // elements
//SCITER-

//--------------------------------------------------------------------------------------
// Forward declarations
//--------------------------------------------------------------------------------------
HRESULT InitWindow( HINSTANCE hInstance, int nCmdShow );
HRESULT InitDevice();
void CleanupDevice();
LRESULT CALLBACK    WndProc( HWND, UINT, WPARAM, LPARAM );
void Render();


//--------------------------------------------------------------------------------------
// Entry point to the program. Initializes everything and goes into a message processing 
// loop. Idle time is used to render the scene.
//--------------------------------------------------------------------------------------
int WINAPI wWinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPWSTR lpCmdLine, int nCmdShow )
{
    UNREFERENCED_PARAMETER( hPrevInstance );
    UNREFERENCED_PARAMETER( lpCmdLine );

//SCITER+
    SciterSetOption(NULL, SCITER_SET_GFX_LAYER, GFX_LAYER_D2D); // use Direct2D targets in this demo
    SciterSetOption(NULL, SCITER_SET_UX_THEMING, TRUE);         // use "unisex" theming

    // these two calls are optional
    //SciterSetOption(NULL, SCITER_SET_SCRIPT_RUNTIME_FEATURES,
    //  ALLOW_FILE_IO |
    //  ALLOW_SOCKET_IO |
    //  ALLOW_EVAL |
    //  ALLOW_SYSINFO);
    //SciterSetOption(NULL, SCITER_SET_DEBUG_MODE, TRUE);

 //SCITER-

    if( FAILED( InitWindow( hInstance, nCmdShow ) ) )
        return 0;

    if( FAILED( InitDevice() ) )
    {
        CleanupDevice();
        return 0;
    }

    // Main message loop
    MSG msg = {0};
    while( WM_QUIT != msg.message )
    {
        if( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            DispatchMessage( &msg );
        }
        else
        {
            Render();
        }
    }

    CleanupDevice();

    return ( int )msg.wParam;
}


//--------------------------------------------------------------------------------------
// Register class and create window
//--------------------------------------------------------------------------------------
HRESULT InitWindow( HINSTANCE hInstance, int nCmdShow )
{
    // Register class
    WNDCLASSEX wcex;
    wcex.cbSize = sizeof( WNDCLASSEX );
    wcex.style = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc = WndProc;
    wcex.cbClsExtra = 0;
    wcex.cbWndExtra = 0;
    wcex.hInstance = hInstance;
    wcex.hIcon = LoadIcon( hInstance, ( LPCTSTR )IDI_SCITER_DIRECTX );
    wcex.hCursor = LoadCursor( NULL, IDC_ARROW );
    wcex.hbrBackground = ( HBRUSH )( COLOR_WINDOW + 1 );
    wcex.lpszMenuName = NULL;
    wcex.lpszClassName = L"TutorialWindowClass";
    wcex.hIconSm = LoadIcon( wcex.hInstance, ( LPCTSTR )IDI_SCITER_DIRECTX );
    if( !RegisterClassEx( &wcex ) )
        return E_FAIL;

    // Create window
    g_hInst = hInstance;
    RECT rc = { 0, 0, 1000, 1000 };
    AdjustWindowRect( &rc, WS_OVERLAPPEDWINDOW, FALSE );
    g_hWnd = CreateWindow( L"TutorialWindowClass", L"Sciter with Direct3D 11 Tutorial 7", WS_OVERLAPPEDWINDOW,
                           CW_USEDEFAULT, CW_USEDEFAULT, rc.right - rc.left, rc.bottom - rc.top, NULL, NULL, hInstance,
                           NULL );
    if( !g_hWnd )
        return E_FAIL;

    ShowWindow( g_hWnd, nCmdShow );

    return S_OK;
}


//--------------------------------------------------------------------------------------
// Helper for compiling shaders with D3DX11
//--------------------------------------------------------------------------------------
HRESULT CompileShaderFromFile( WCHAR* szFileName, LPCSTR szEntryPoint, LPCSTR szShaderModel, ID3DBlob** ppBlobOut )
{
    HRESULT hr = S_OK;

    DWORD dwShaderFlags = D3DCOMPILE_ENABLE_STRICTNESS;
#if defined( DEBUG ) || defined( _DEBUG )
    // Set the D3DCOMPILE_DEBUG flag to embed debug information in the shaders.
    // Setting this flag improves the shader debugging experience, but still allows 
    // the shaders to be optimized and to run exactly the way they will run in 
    // the release configuration of this program.
    dwShaderFlags |= D3DCOMPILE_DEBUG;
#endif

    ID3DBlob* pErrorBlob;
    hr = D3DX11CompileFromFile( szFileName, NULL, NULL, szEntryPoint, szShaderModel, 
        dwShaderFlags, 0, NULL, ppBlobOut, &pErrorBlob, NULL );
    if( FAILED(hr) )
    {
        if( pErrorBlob != NULL )
            OutputDebugStringA( (char*)pErrorBlob->GetBufferPointer() );
        if( pErrorBlob ) pErrorBlob->Release();
        return hr;
    }
    if( pErrorBlob ) pErrorBlob->Release();

    return S_OK;
}

// SCITER +
BOOL InitSciterEngineInstance(HWND hwnd, IDXGISwapChain* pSwapChain );
// SCITER -


//--------------------------------------------------------------------------------------
// Create Direct3D device and swap chain
//--------------------------------------------------------------------------------------
HRESULT InitDevice()
{
    HRESULT hr = S_OK;

    RECT rc;
    GetClientRect( g_hWnd, &rc );
    UINT width = rc.right - rc.left;
    UINT height = rc.bottom - rc.top;

//SCITER +
    // createDeviceFlags must contain D3D10_CREATE_DEVICE_BGRA_SUPPORT/D3D11_CREATE_DEVICE_BGRA_SUPPORT flag
    UINT createDeviceFlags = D3D11_CREATE_DEVICE_BGRA_SUPPORT;
//SCITER -
#ifdef _DEBUG
    createDeviceFlags |= D3D11_CREATE_DEVICE_DEBUG;
#endif

    D3D_DRIVER_TYPE driverTypes[] =
    {
        D3D_DRIVER_TYPE_HARDWARE,
        D3D_DRIVER_TYPE_WARP,
        D3D_DRIVER_TYPE_REFERENCE,
    };
    UINT numDriverTypes = ARRAYSIZE( driverTypes );

    D3D_FEATURE_LEVEL featureLevels[] =
    {
        D3D_FEATURE_LEVEL_11_0,
        D3D_FEATURE_LEVEL_10_1,
        D3D_FEATURE_LEVEL_10_0,
    };
    UINT numFeatureLevels = ARRAYSIZE( featureLevels );

    DXGI_SWAP_CHAIN_DESC sd;
    ZeroMemory( &sd, sizeof( sd ) );
    sd.BufferCount = 1;
    sd.BufferDesc.Width = width;
    sd.BufferDesc.Height = height;
    sd.BufferDesc.Format = DXGI_FORMAT_B8G8R8A8_UNORM;
    sd.BufferDesc.RefreshRate.Numerator = 60;
    sd.BufferDesc.RefreshRate.Denominator = 1;
    sd.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    sd.OutputWindow = g_hWnd;
    sd.SampleDesc.Count = 1;
    sd.SampleDesc.Quality = 0;
    sd.Windowed = TRUE;

    for( UINT driverTypeIndex = 0; driverTypeIndex < numDriverTypes; driverTypeIndex++ )
    {
        g_driverType = driverTypes[driverTypeIndex];
        hr = D3D11CreateDeviceAndSwapChain( NULL, g_driverType, NULL, createDeviceFlags, featureLevels, numFeatureLevels,
                                            D3D11_SDK_VERSION, &sd, &g_pSwapChain, &g_pd3dDevice, &g_featureLevel, &g_pImmediateContext );
        if( SUCCEEDED( hr ) )
            break;
    }
    if( FAILED( hr ) )
        return hr;

    // Create a render target view
    ID3D11Texture2D* pBackBuffer = NULL;
    hr = g_pSwapChain->GetBuffer( 0, __uuidof( ID3D11Texture2D ), ( LPVOID* )&pBackBuffer );
    if( FAILED( hr ) )
        return hr;

    hr = g_pSwapChain->GetBuffer(0, __uuidof(IDXGISurface), (void**)&g_pSurface);
    if (FAILED(hr))
      return hr;

    hr = g_pd3dDevice->CreateRenderTargetView( pBackBuffer, NULL, &g_pRenderTargetView );
    pBackBuffer->Release();
    if( FAILED( hr ) )
        return hr;

    // Create depth stencil texture
    D3D11_TEXTURE2D_DESC descDepth;
    ZeroMemory( &descDepth, sizeof(descDepth) );
    descDepth.Width = width;
    descDepth.Height = height;
    descDepth.MipLevels = 1;
    descDepth.ArraySize = 1;
    descDepth.Format = DXGI_FORMAT_D24_UNORM_S8_UINT;
    descDepth.SampleDesc.Count = 1;
    descDepth.SampleDesc.Quality = 0;
    descDepth.Usage = D3D11_USAGE_DEFAULT;
    descDepth.BindFlags = D3D11_BIND_DEPTH_STENCIL;
    descDepth.CPUAccessFlags = 0;
    descDepth.MiscFlags = 0;
    hr = g_pd3dDevice->CreateTexture2D( &descDepth, NULL, &g_pDepthStencil );
    if( FAILED( hr ) )
        return hr;

    // Create the depth stencil view
    D3D11_DEPTH_STENCIL_VIEW_DESC descDSV;
    ZeroMemory( &descDSV, sizeof(descDSV) );
    descDSV.Format = descDepth.Format;
    descDSV.ViewDimension = D3D11_DSV_DIMENSION_TEXTURE2D;
    descDSV.Texture2D.MipSlice = 0;
    hr = g_pd3dDevice->CreateDepthStencilView( g_pDepthStencil, &descDSV, &g_pDepthStencilView );
    if( FAILED( hr ) )
        return hr;

    g_pImmediateContext->OMSetRenderTargets( 1, &g_pRenderTargetView, g_pDepthStencilView );

    // Setup the viewport
    D3D11_VIEWPORT vp;
    vp.Width = (FLOAT)width;
    vp.Height = (FLOAT)height;
    vp.MinDepth = 0.0f;
    vp.MaxDepth = 1.0f;
    vp.TopLeftX = 0;
    vp.TopLeftY = 0;
    g_pImmediateContext->RSSetViewports( 1, &vp );

// SCITER+
    // create Sciter engine instance for the window:
    InitSciterEngineInstance(g_hWnd, g_pSwapChain );
// SCITER-

    // Compile the vertex shader
    ID3DBlob* pVSBlob = NULL;
    hr = CompileShaderFromFile( L"..\\..\\demos.lite\\sciter-directx\\shaders.fx", "VS", "vs_4_0", &pVSBlob );
    if( FAILED( hr ) )
    {
        MessageBox( NULL,
                    L"The FX file cannot be compiled.  Please run this executable from sciter-sdk/bin folder.", L"Error", MB_OK );
        return hr;
    }

    // Create the vertex shader
    hr = g_pd3dDevice->CreateVertexShader( pVSBlob->GetBufferPointer(), pVSBlob->GetBufferSize(), NULL, &g_pVertexShader );
    if( FAILED( hr ) )
    {    
        pVSBlob->Release();
        return hr;
    }

    // Define the input layout
    D3D11_INPUT_ELEMENT_DESC layout[] =
    {
        { "POSITION", 0, DXGI_FORMAT_R32G32B32_FLOAT, 0, 0, D3D11_INPUT_PER_VERTEX_DATA, 0 },
        { "TEXCOORD", 0, DXGI_FORMAT_R32G32_FLOAT, 0, 12, D3D11_INPUT_PER_VERTEX_DATA, 0 },
    };
    UINT numElements = ARRAYSIZE( layout );

    // Create the input layout
    hr = g_pd3dDevice->CreateInputLayout( layout, numElements, pVSBlob->GetBufferPointer(),
                                          pVSBlob->GetBufferSize(), &g_pVertexLayout );
    pVSBlob->Release();
    if( FAILED( hr ) )
        return hr;

    // Set the input layout
    g_pImmediateContext->IASetInputLayout( g_pVertexLayout );

    // Compile the pixel shader
    ID3DBlob* pPSBlob = NULL;
    hr = CompileShaderFromFile(L"..\\..\\demos.lite\\sciter-directx\\shaders.fx", "PS", "ps_4_0", &pPSBlob );
    if( FAILED( hr ) )
    {
        MessageBox( NULL, L"The FX file cannot be compiled.  Please run this executable from sciter-sdk/bin folder.", L"Error", MB_OK );
        return hr;
    }

    // Create the pixel shader
    hr = g_pd3dDevice->CreatePixelShader( pPSBlob->GetBufferPointer(), pPSBlob->GetBufferSize(), NULL, &g_pPixelShader );
    pPSBlob->Release();
    if( FAILED( hr ) )
        return hr;

    // Create vertex buffer
    SimpleVertex vertices[] =
    {
        { XMFLOAT3( -1.0f, 1.0f, -1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, -1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, 1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( -1.0f, 1.0f, 1.0f ), XMFLOAT2( 0.0f, 1.0f ) },

        { XMFLOAT3( -1.0f, -1.0f, -1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, -1.0f, -1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, -1.0f, 1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( -1.0f, -1.0f, 1.0f ), XMFLOAT2( 0.0f, 1.0f ) },

        { XMFLOAT3( -1.0f, -1.0f, 1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( -1.0f, -1.0f, -1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( -1.0f, 1.0f, -1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( -1.0f, 1.0f, 1.0f ), XMFLOAT2( 0.0f, 1.0f ) },

        { XMFLOAT3( 1.0f, -1.0f, 1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, -1.0f, -1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, -1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, 1.0f ), XMFLOAT2( 0.0f, 1.0f ) },

        { XMFLOAT3( -1.0f, -1.0f, -1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, -1.0f, -1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, -1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( -1.0f, 1.0f, -1.0f ), XMFLOAT2( 0.0f, 1.0f ) },

        { XMFLOAT3( -1.0f, -1.0f, 1.0f ), XMFLOAT2( 0.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, -1.0f, 1.0f ), XMFLOAT2( 1.0f, 0.0f ) },
        { XMFLOAT3( 1.0f, 1.0f, 1.0f ), XMFLOAT2( 1.0f, 1.0f ) },
        { XMFLOAT3( -1.0f, 1.0f, 1.0f ), XMFLOAT2( 0.0f, 1.0f ) },
    };

    D3D11_BUFFER_DESC bd;
    ZeroMemory( &bd, sizeof(bd) );
    bd.Usage = D3D11_USAGE_DEFAULT;
    bd.ByteWidth = sizeof( SimpleVertex ) * 24;
    bd.BindFlags = D3D11_BIND_VERTEX_BUFFER;
    bd.CPUAccessFlags = 0;
    D3D11_SUBRESOURCE_DATA InitData;
    ZeroMemory( &InitData, sizeof(InitData) );
    InitData.pSysMem = vertices;
    hr = g_pd3dDevice->CreateBuffer( &bd, &InitData, &g_pVertexBuffer );
    if( FAILED( hr ) )
        return hr;

    // Set vertex buffer
    UINT stride = sizeof( SimpleVertex );
    UINT offset = 0;
    g_pImmediateContext->IASetVertexBuffers( 0, 1, &g_pVertexBuffer, &stride, &offset );

    // Create index buffer
    // Create vertex buffer
    WORD indices[] =
    {
        3,1,0,
        2,1,3,

        6,4,5,
        7,4,6,

        11,9,8,
        10,9,11,

        14,12,13,
        15,12,14,

        19,17,16,
        18,17,19,

        22,20,21,
        23,20,22
    };

    bd.Usage = D3D11_USAGE_DEFAULT;
    bd.ByteWidth = sizeof( WORD ) * 36;
    bd.BindFlags = D3D11_BIND_INDEX_BUFFER;
    bd.CPUAccessFlags = 0;
    InitData.pSysMem = indices;
    hr = g_pd3dDevice->CreateBuffer( &bd, &InitData, &g_pIndexBuffer );
    if( FAILED( hr ) )
        return hr;

    // Set index buffer
    g_pImmediateContext->IASetIndexBuffer( g_pIndexBuffer, DXGI_FORMAT_R16_UINT, 0 );

    // Set primitive topology
    g_pImmediateContext->IASetPrimitiveTopology( D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST );

    // Create the constant buffers
    bd.Usage = D3D11_USAGE_DEFAULT;
    bd.ByteWidth = sizeof(CBNeverChanges);
    bd.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    bd.CPUAccessFlags = 0;
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBNeverChanges );
    if( FAILED( hr ) )
        return hr;
    
    bd.ByteWidth = sizeof(CBChangeOnResize);
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBChangeOnResize );
    if( FAILED( hr ) )
        return hr;
    
    bd.ByteWidth = sizeof(CBChangesEveryFrame);
    hr = g_pd3dDevice->CreateBuffer( &bd, NULL, &g_pCBChangesEveryFrame );
    if( FAILED( hr ) )
        return hr;

    // Load the Texture
    hr = D3DX11CreateShaderResourceViewFromFile( g_pd3dDevice, L"..\\..\\demos.lite\\sciter-directx\\seafloor.dds", NULL, NULL, &g_pTextureRV, NULL );
    if( FAILED( hr ) )
        return hr;

    // Create the sample state
    D3D11_SAMPLER_DESC sampDesc;
    ZeroMemory( &sampDesc, sizeof(sampDesc) );
    sampDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
    sampDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
    sampDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
    sampDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
    sampDesc.ComparisonFunc = D3D11_COMPARISON_NEVER;
    sampDesc.MinLOD = 0;
    sampDesc.MaxLOD = D3D11_FLOAT32_MAX;
    hr = g_pd3dDevice->CreateSamplerState( &sampDesc, &g_pSamplerLinear );
    if( FAILED( hr ) )
        return hr;

    // Initialize the world matrices
    g_World = XMMatrixIdentity();

    // Initialize the view matrix
    XMVECTOR Eye = XMVectorSet( 0.0f, 3.0f, -6.0f, 0.0f );
    XMVECTOR At = XMVectorSet( 0.0f, 1.0f, 0.0f, 0.0f );
    XMVECTOR Up = XMVectorSet( 0.0f, 1.0f, 0.0f, 0.0f );
    g_View = XMMatrixLookAtLH( Eye, At, Up );

    CBNeverChanges cbNeverChanges;
    cbNeverChanges.mView = XMMatrixTranspose( g_View );
    g_pImmediateContext->UpdateSubresource( g_pCBNeverChanges, 0, NULL, &cbNeverChanges, 0, 0 );

    // Initialize the projection matrix
    g_Projection = XMMatrixPerspectiveFovLH( XM_PIDIV4, width / (FLOAT)height, 0.01f, 100.0f );
    
    CBChangeOnResize cbChangesOnResize;
    cbChangesOnResize.mProjection = XMMatrixTranspose( g_Projection );
    g_pImmediateContext->UpdateSubresource( g_pCBChangeOnResize, 0, NULL, &cbChangesOnResize, 0, 0 );

    return S_OK;
}


//--------------------------------------------------------------------------------------
// Clean up the objects we've created
//--------------------------------------------------------------------------------------
void CleanupDevice()
{
    if( g_pImmediateContext ) g_pImmediateContext->ClearState();

    if( g_pSamplerLinear ) g_pSamplerLinear->Release();
    if( g_pTextureRV ) g_pTextureRV->Release();
    if( g_pCBNeverChanges ) g_pCBNeverChanges->Release();
    if( g_pCBChangeOnResize ) g_pCBChangeOnResize->Release();
    if( g_pCBChangesEveryFrame ) g_pCBChangesEveryFrame->Release();
    if( g_pVertexBuffer ) g_pVertexBuffer->Release();
    if( g_pIndexBuffer ) g_pIndexBuffer->Release();
    if( g_pVertexLayout ) g_pVertexLayout->Release();
    if( g_pVertexShader ) g_pVertexShader->Release();
    if( g_pPixelShader ) g_pPixelShader->Release();
    if( g_pDepthStencil ) g_pDepthStencil->Release();
    if( g_pDepthStencilView ) g_pDepthStencilView->Release();
    if( g_pRenderTargetView ) g_pRenderTargetView->Release();
    if( g_pSwapChain ) g_pSwapChain->Release();
    if( g_pImmediateContext ) g_pImmediateContext->Release();
    if (g_pSurface) g_pSurface->Release();
    if( g_pd3dDevice ) g_pd3dDevice->Release();
}


void Resize()
{
  if (!g_pSwapChain)
    return;

  RECT rc;
  GetClientRect(g_hWnd,&rc);

  int width = rc.right - rc.left;
  int height = rc.bottom - rc.top;
// SCITER +
  SciterProcX(SciterInstanceID, SCITER_X_MSG_SIZE(width,height));
// SCITER -

  g_pImmediateContext->OMSetRenderTargets(0, 0, 0);

  // SCITER+
  if (g_pSurface) { 
    g_pSurface->Release(); 
    g_pSurface = nullptr; 
  }
  // SCITER-


  // Release all outstanding references to the swap chain's buffers.
  g_pRenderTargetView->Release();

  HRESULT hr;
  // Preserve the existing buffer count and format.
  // Automatically choose the width and height to match the client rect for HWNDs.
  hr = g_pSwapChain->ResizeBuffers(0, 0, 0, DXGI_FORMAT_UNKNOWN, 0);
                                            
  // Perform error handling here!

  // Get buffer and create a render-target-view.
  ID3D11Texture2D* pBuffer;
  hr = g_pSwapChain->GetBuffer(0, __uuidof( ID3D11Texture2D),
                                (void**) &pBuffer );
  // Perform error handling here!

  hr = g_pd3dDevice->CreateRenderTargetView(pBuffer, NULL,
                                            &g_pRenderTargetView);

  // SCITER+
  if (g_pSurface) { g_pSurface->Release(); g_pSurface = nullptr; }
  hr = g_pSwapChain->GetBuffer(0, __uuidof(IDXGISurface),(void**)&g_pSurface);
  // SCITER-

  // Perform error handling here!
  pBuffer->Release();

  g_pImmediateContext->OMSetRenderTargets(1, &g_pRenderTargetView, NULL );

  // Set up the viewport.
  D3D11_VIEWPORT vp;
  vp.Width = float(width);
  vp.Height = float(height);
  vp.MinDepth = 0.0f;
  vp.MaxDepth = 1.0f;
  vp.TopLeftX = 0;
  vp.TopLeftY = 0;
  g_pImmediateContext->RSSetViewports( 1, &vp );

}

//--------------------------------------------------------------------------------------
// Called every time the application receives a message
//--------------------------------------------------------------------------------------
LRESULT CALLBACK WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{

    PAINTSTRUCT ps;
    HDC hdc;
    unsigned mb = 0;
    unsigned ks = 0;
    unsigned code = 0;
    unsigned evt = 0;

    switch( message )
    {
        case WM_PAINT:
          hdc = BeginPaint( hWnd, &ps );
          EndPaint( hWnd, &ps );
          break;

        case WM_SIZE:
          Resize();
          break;

        case WM_DESTROY:
          // SCITER +
          SciterProcX(SciterInstanceID, SCITER_X_MSG_DESTROY());
          // SCITER -
          PostQuitMessage( 0 );
          break;

          // SCITER +
        case WM_LBUTTONDOWN:    evt = MOUSE_EVENTS::MOUSE_DOWN; mb |= MOUSE_BUTTONS::MAIN_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_LBUTTONUP:      evt = MOUSE_EVENTS::MOUSE_UP; mb |= MOUSE_BUTTONS::MAIN_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_LBUTTONDBLCLK:  evt = MOUSE_EVENTS::MOUSE_DCLICK; mb |= MOUSE_BUTTONS::MAIN_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_RBUTTONDOWN:    evt = MOUSE_EVENTS::MOUSE_DOWN; mb |= MOUSE_BUTTONS::PROP_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_RBUTTONUP:      evt = MOUSE_EVENTS::MOUSE_UP; mb |= MOUSE_BUTTONS::PROP_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_RBUTTONDBLCLK:  evt = MOUSE_EVENTS::MOUSE_DCLICK; mb |= MOUSE_BUTTONS::PROP_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_MBUTTONDOWN:    evt = MOUSE_EVENTS::MOUSE_DOWN; mb |= MOUSE_BUTTONS::MIDDLE_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_MBUTTONUP:      evt = MOUSE_EVENTS::MOUSE_UP; mb |= MOUSE_BUTTONS::MIDDLE_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_MBUTTONDBLCLK:  evt = MOUSE_EVENTS::MOUSE_DCLICK; mb |= MOUSE_BUTTONS::MIDDLE_MOUSE_BUTTON; goto SEND_MOUSE_EVENT;
        case WM_MOUSEMOVE: 
          evt = MOUSE_EVENTS::MOUSE_MOVE;
          if (wParam & MK_LBUTTON) mb |= MOUSE_BUTTONS::MAIN_MOUSE_BUTTON;
          if (wParam & MK_RBUTTON) mb |= MOUSE_BUTTONS::PROP_MOUSE_BUTTON;
          if (wParam & MK_MBUTTON) mb |= MOUSE_BUTTONS::MIDDLE_MOUSE_BUTTON;
          goto SEND_MOUSE_EVENT;

        case WM_MOUSELEAVE:
          evt = MOUSE_EVENTS::MOUSE_LEAVE;
          goto SEND_MOUSE_EVENT;


        case WM_SETFOCUS:
        case WM_KILLFOCUS:
          SciterProcX(SciterInstanceID, SCITER_X_MSG_FOCUS(message == WM_SETFOCUS));
          break;

        case WM_DPICHANGED:
          SciterProcX(SciterInstanceID, SCITER_X_MSG_RESOLUTION(LOWORD(wParam)));
          break;

        case WM_KEYDOWN: evt = KEY_EVENTS::KEY_DOWN; code = translate_key(wParam);  goto SEND_KEY_EVENT;
        case WM_KEYUP:   evt = KEY_EVENTS::KEY_UP; code = translate_key(wParam);  goto SEND_KEY_EVENT;
        case WM_CHAR:    evt = KEY_EVENTS::KEY_CHAR; code = wParam;  goto SEND_KEY_EVENT;

          // SCITER -

        default:
          return DefWindowProc( hWnd, message, wParam, lParam );
    }

    return 0;

  SEND_MOUSE_EVENT:

    POINT pos = { LOWORD(lParam), HIWORD(lParam) };
    SciterProcX(SciterInstanceID, SCITER_X_MSG_MOUSE(MOUSE_EVENTS(evt), MOUSE_BUTTONS(mb), KEYBOARD_STATES(ks), pos));

    return DefWindowProc(hWnd, message, wParam, lParam);

  SEND_KEY_EVENT:
    if (GetKeyState(VK_CONTROL) < 0)
      ks |= KEYBOARD_STATES::CONTROL_KEY_PRESSED;
    if (GetKeyState(VK_SHIFT) < 0)
      ks |= KEYBOARD_STATES::SHIFT_KEY_PRESSED;
    if (GetKeyState(VK_MENU))
      ks |= KEYBOARD_STATES::ALT_KEY_PRESSED;

    SciterProcX(SciterInstanceID, SCITER_X_MSG_KEY(KEY_EVENTS(evt), code, KEYBOARD_STATES(ks)));
    return DefWindowProc(hWnd, message, wParam, lParam);

}


//--------------------------------------------------------------------------------------
// Render a frame
//--------------------------------------------------------------------------------------
void Render()
{
    // Update our time
    static float t = 0.0f;
    if( g_driverType == D3D_DRIVER_TYPE_REFERENCE )
    {
        t += ( float )XM_PI * 0.0125f;
    }
    else
    {
        static DWORD dwTimeStart = 0;
        DWORD dwTimeCur = GetTickCount();
        if( dwTimeStart == 0 )
            dwTimeStart = dwTimeCur;
        t = ( dwTimeCur - dwTimeStart ) / 1000.0f;
    }

    // Rotate cube around the origin
    g_World = XMMatrixRotationY( t * g_rotationSpeed );

    // Modify the color
    g_vMeshColor.x = ( sinf( t * g_colorSpeed * 1.0f ) + 1.0f ) * 0.5f;
    g_vMeshColor.y = ( cosf( t * g_colorSpeed * 3.0f ) + 1.0f ) * 0.5f;
    g_vMeshColor.z = ( sinf( t * g_colorSpeed * 5.0f ) + 1.0f ) * 0.5f;

    //
    // Clear the back buffer
    //
    //float ClearColor[4] = { 0.0f, 0.125f, 0.3f, 1.0f }; // red, green, blue, alpha
    //g_pImmediateContext->ClearRenderTargetView( g_pRenderTargetView, ClearColor );

// SCITER +
    // render HTML document, back layer (if any)
    {
      UINT ticks = GetTickCount();
      SciterProcX(SciterInstanceID, SCITER_X_MSG_HEARTBIT(ticks));

      SCITER_X_MSG_PAINT msg_paint_back(g_backLayer, FALSE);
      msg_paint_back.targetType = SPT_SURFACE;
      msg_paint_back.target.pSurface = g_pSurface;

      BOOL rpaint = SciterProcX(SciterInstanceID, msg_paint_back);
      assert(rpaint);
    }
// SCITER -
    
    //
    // Clear the depth buffer to 1.0 (max depth)
    //
    g_pImmediateContext->ClearDepthStencilView( g_pDepthStencilView, D3D11_CLEAR_DEPTH, 1.0f, 0 );

    //
    // Update variables that change once per frame
    //
    CBChangesEveryFrame cb;
    cb.mWorld = XMMatrixTranspose( g_World );
    cb.vMeshColor = g_vMeshColor;
    g_pImmediateContext->UpdateSubresource( g_pCBChangesEveryFrame, 0, NULL, &cb, 0, 0 );

    //
    // Render the cube
    //
    g_pImmediateContext->VSSetShader( g_pVertexShader, NULL, 0 );
    g_pImmediateContext->VSSetConstantBuffers( 0, 1, &g_pCBNeverChanges );
    g_pImmediateContext->VSSetConstantBuffers( 1, 1, &g_pCBChangeOnResize );
    g_pImmediateContext->VSSetConstantBuffers( 2, 1, &g_pCBChangesEveryFrame );
    g_pImmediateContext->PSSetShader( g_pPixelShader, NULL, 0 );
    g_pImmediateContext->PSSetConstantBuffers( 2, 1, &g_pCBChangesEveryFrame );
    g_pImmediateContext->PSSetShaderResources( 0, 1, &g_pTextureRV );
    g_pImmediateContext->PSSetSamplers( 0, 1, &g_pSamplerLinear );
    g_pImmediateContext->DrawIndexed( 36, 0, 0 );

// SCITER +
    // render fore layer on top of 3D scene (if any)
    if (g_backLayer && g_foreLayer) {
      SCITER_X_MSG_PAINT msg_paint_fore(g_foreLayer, TRUE);
      msg_paint_fore.targetType = SPT_SURFACE;
      msg_paint_fore.target.pSurface = g_pSurface;
      BOOL rpaint = SciterProcX(SciterInstanceID, msg_paint_fore);
      assert(rpaint);
    }

// SCITER -

    //
    // Present our back buffer to our front buffer
    //
    g_pSwapChain->Present( 0, 0 );
}

// SCITER +

  // view_dom_event_handler, provides native methods to be used in script

  struct view_dom_event_handler : public sciter::event_handler_raw {
    BEGIN_FUNCTION_MAP
      FUNCTION_1("setRotationSpeed",setRotationSpeed)
      FUNCTION_1("setColorSpeed",setColorSpeed)
    END_FUNCTION_MAP

    sciter::value setRotationSpeed(sciter::value speed)
    {
      g_rotationSpeed = (FLOAT) speed.get(1.0);
      return sciter::value(true);
    }
    sciter::value setColorSpeed(sciter::value speed)
    {
      g_colorSpeed = (FLOAT) speed.get(1.0);
      return sciter::value(true);
    }

  };

  view_dom_event_handler g_ViewDomEventHandler;

  UINT SC_CALLBACK SciterCallback( LPSCITER_CALLBACK_NOTIFICATION pns, LPVOID callbackParam );

  // create Sciter engine instance for the window:
  BOOL InitSciterEngineInstance(HWND hWnd, IDXGISwapChain *pSwapChain )
  {
    // 1. create engine instance:
    BOOL r = SciterProcX(SciterInstanceID, SCITER_X_MSG_CREATE(GFX_LAYER_D2D, FALSE));
    if (!r) return FALSE;

    // 2. setup callback (resource loading, etc):
    SciterSetCallback(SciterInstanceID, SciterCallback, NULL);

    // 2b. setup DOM event handler:
    sciter::attach_dom_event_handler(SciterInstanceID, &g_ViewDomEventHandler);

    // set device resoultion ( pixels per inch )
    UINT resolution = GetDpiForWindow(hWnd);
    SciterProcX(SciterInstanceID, SCITER_X_MSG_RESOLUTION(resolution));

    RECT rc;
    GetClientRect(g_hWnd, &rc);

    int width = rc.right - rc.left;
    int height = rc.bottom - rc.top;

    SciterProcX(SciterInstanceID, SCITER_X_MSG_SIZE(width, height));

    // 3. load HTML content in it:
    SciterLoadFile(SciterInstanceID,L"res:FACADE.HTML");

    // 4. get layer elements:
    sciter::dom::element root = sciter::dom::element::root_element(SciterInstanceID);
    g_backLayer = root.find_first("section#back-layer");
    g_foreLayer = root.find_first("section#fore-layer");
    assert(g_backLayer && g_foreLayer);

    // done
    return true;
  }

  // handle SC_LOAD_DATA requests - get data from resources of this application
  UINT DoLoadData(LPSCN_LOAD_DATA pnmld)
  {
    LPCBYTE pb = 0; UINT cb = 0;
    aux::wchars wu = aux::chars_of(pnmld->uri);
    if(wu.like(WSTR("res:*")))
    {
      // then by calling possibly overloaded load_resource_data method
      if(sciter::load_resource_data(g_hInst,wu.start+4, pb, cb))
        ::SciterDataReady( pnmld->hwnd, pnmld->uri, pb,  cb);
    } else if(wu.like(WSTR("this://app/*"))) {
      // try to get them from archive (if any, you need to call sciter::archive::open() first)
      aux::bytes adata = sciter::archive::instance().get(wu.start+11);
      if(adata.length)
        ::SciterDataReady( pnmld->hwnd, pnmld->uri, adata.start, adata.length);
    }
    return LOAD_OK;
  }

  // fulfill SC_ATTACH_BEHAVIOR request 
  UINT DoAttachBehavior( LPSCN_ATTACH_BEHAVIOR lpab )
  {
    sciter::event_handler *pb = sciter::behavior_factory::create(lpab->behaviorName, lpab->element);
    if(pb)
    {
      lpab->elementTag  = pb;
      lpab->elementProc = sciter::event_handler::element_proc;
      return TRUE;
    }
    return FALSE;
  }

  UINT SC_CALLBACK SciterCallback( LPSCITER_CALLBACK_NOTIFICATION pns, LPVOID callbackParam )
  {
    // here are all notifiactions
    switch(pns->code) 
    {
      case SC_LOAD_DATA:          return DoLoadData((LPSCN_LOAD_DATA) pns);
      case SC_ATTACH_BEHAVIOR:    return DoAttachBehavior((LPSCN_ATTACH_BEHAVIOR)pns );
    }
    return 0;
  }

  unsigned translate_key(unsigned vk)
  {
    auto in_range = [vk](unsigned vk_start, unsigned vk_end, unsigned kb_first) -> unsigned
    {
      if (vk >= vk_start && vk <= vk_end)
        return (vk - vk_start) + kb_first;

      return 0;
    };

    if (auto kb = in_range('0', '9', KB_0))
      return kb;

    if (auto kb = in_range('A', 'Z', KB_A))
      return kb;

    if (auto kb = in_range(VK_NUMPAD0, VK_NUMPAD9, KB_NUMPAD0))
      return kb;

    if (auto kb = in_range(VK_F1, VK_F24, KB_F1))
      return kb;

    switch (vk)
    {
    case VK_ESCAPE: return KB_ESCAPE;
    case VK_BACK:   return KB_BACK;
    case VK_TAB:    return KB_TAB;
    case VK_SPACE:  return KB_SPACE;
    case VK_SHIFT:  return KB_SHIFT;
    case VK_CONTROL:return KB_CONTROL;
    case VK_MENU:   return KB_MENU;
    case VK_LEFT:   return KB_LEFT;
    case VK_RIGHT:  return KB_RIGHT;
    case VK_UP:     return KB_UP;
    case VK_DOWN:   return KB_DOWN;
    case VK_PRIOR:  return KB_PAGE_UP;
    case VK_NEXT:   return KB_PAGE_DOWN;
    case VK_END:    return KB_END;
    case VK_HOME:   return KB_HOME;
    case VK_DELETE: return KB_DELETE;
    }
    return 0;
  }

// SCITER -

