
#include "tool/tool.h"

#include "sdk-headers.h"

#ifdef SCITERJS
  #include "sdk.js/include/sciter-x-window.hpp"
#else 
  #include "sdk/include/sciter-x-window.hpp"
#endif

#include "debug-console.h"

#ifdef WINDOWS
#include "windows/resource.h"
#include "winver.h"

#pragma pack(push, 2)

typedef struct GRPICONDIRENTRY
{
  BYTE  bWidth;
  BYTE  bHeight;
  BYTE  bColorCount;
  BYTE  bReserved;
  WORD  wPlanes;
  WORD  wBitCount;
  DWORD dwBytesInRes;
  DWORD nOffset;
} GRPICONDIRENTRY;

typedef struct GRPICONDIR
{
  WORD idReserved;
  WORD idType;
  WORD idCount;
  GRPICONDIRENTRY idEntries[];
} GRPICONDIR;

struct EXEICONDIRENTRY {
  BYTE  bWidth;
  BYTE  bHeight;
  BYTE  bColorCount;
  BYTE  bReserved;
  WORD  wPlanes;
  WORD  wBitCount;
  DWORD dwBytesInRes;
  WORD  nId;
};

struct EXEICONDIR
{
  WORD idReserved;
  WORD idType;
  WORD idCount;
  EXEICONDIRENTRY idEntries[];
};

struct Translate
{
  WORD wLanguage;
  WORD wCodePage;
};


#pragma pack(pop)

#endif


int  MessagePump();
void Initialize();

struct bdata {
  union {
    uint32  i;
    byte bytes[4];
  } data;

  bdata(uint32 i) { data.i = i; }
  bdata(const byte* b) { data.i = *(const uint32*)b; }

  tool::bytes as_bytes() { return tool::items_of(data.bytes); }
};

struct exedata
{
  tool::mm_file fl;

  tool::bytes head;
  tool::bytes tail;

  exedata() {
    auto this_path = tool::get_self_path();

    if (!fl.open(this_path))
      return;

    tool::bytes body = fl.bytes();

    if (body.ends_with(bdata(0xAFED).as_bytes())) {
      // seem like we have payload
      auto tail_length = bdata(body.start + body.length - 4 - 4).data.i;
      tail = tool::bytes(body.start + body.length - 4 - 4 - tail_length, tail_length);
      head = tool::bytes::range(body.start, tail.start);
    }
    else {
      // no tail payload
      head = body;
    }
  }
};

bool patch_exe(const tool::ustring& exepath, const tool::value& props) {
#ifndef WINDOWS
  return false;
#else

  // update icon
  if(tool::ustring icopath = props.get_prop("icofile").get<tool::ustring>())
  {
    tool::mm_file ico;
    if (ico.open(icopath)) {

      tool::array<byte> ver_info;

      DWORD   dwHandle, dwSize;
      // determine the size of the resource information
      dwSize = GetFileVersionInfoSize(exepath.c_str(), &dwHandle);
      if (dwSize > 0) {
        ver_info.length(dwSize);
        GetFileVersionInfo(exepath.c_str(), 0, dwSize, ver_info.begin());
      }

      HANDLE hu = BeginUpdateResource(exepath.c_str(), FALSE);
      if (!hu)
        return false;

      GRPICONDIR* pidr_src = (GRPICONDIR*)ico.data();

      size_t pdir_size = sizeof(EXEICONDIR) + pidr_src->idCount * sizeof(EXEICONDIRENTRY);
      tool::array<byte> data = ico.bytes()(0, int(pdir_size));

      EXEICONDIR* pidr_dst = (EXEICONDIR*)data.begin();

      pidr_dst->idType = 1;
      pidr_dst->idReserved = 0;
      pidr_dst->idCount = pidr_src->idCount;

      for (uint i = 0; i < pidr_src->idCount; ++i) {
        tool::bytes icodata = ico.bytes()(pidr_src->idEntries[i].nOffset, pidr_src->idEntries[i].nOffset + pidr_src->idEntries[i].dwBytesInRes);
        UpdateResource(hu, RT_ICON, MAKEINTRESOURCE(i + 1), MAKELANGID(LANG_NEUTRAL, SUBLANG_NEUTRAL), (LPVOID)icodata.start, DWORD(icodata.length));

        pidr_dst->idEntries[i].bColorCount = pidr_src->idEntries[i].bColorCount;
        pidr_dst->idEntries[i].wBitCount = pidr_src->idEntries[i].wBitCount;
        pidr_dst->idEntries[i].bHeight = pidr_src->idEntries[i].bHeight;
        pidr_dst->idEntries[i].bWidth = pidr_src->idEntries[i].bWidth;
        pidr_dst->idEntries[i].bColorCount = pidr_src->idEntries[i].bColorCount;
        pidr_dst->idEntries[i].bReserved = 0;
        pidr_dst->idEntries[i].dwBytesInRes = pidr_src->idEntries[i].dwBytesInRes;
        pidr_dst->idEntries[i].wPlanes = pidr_src->idEntries[i].wPlanes;
        pidr_dst->idEntries[i].nId = (WORD)i + 1;
      }

      UpdateResource(hu, RT_GROUP_ICON, MAKEINTRESOURCE(IDI_SCITEREXE), MAKELANGID(LANG_NEUTRAL, SUBLANG_NEUTRAL), pidr_dst, DWORD(pdir_size));
      //UpdateResource(hu, RT_GROUP_ICON, MAKEINTRESOURCE(IDI_SMALL), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), pidr_dst, pdir_size);

      if (ver_info.length()) {

        auto patch_string = [&ver_info](const wchar* key, tool::wchars val) {
          LPVOID  dptr; UINT dlen;
          if (VerQueryValue(ver_info.begin(), key, &dptr, &dlen))
          {
            memset(dptr, 0, dlen * sizeof(wchar));
            memcpy(dptr, val.start, min(val.length,dlen) * sizeof(wchar));
          }
        };

        tool::value val = props.get_prop("productName");
        if (val.is_string())
          patch_string(L"\\StringFileInfo\\040004b0\\ProductName", val.get<tool::ustring>()());

        val = props.get_prop("productVersion");
        if (val.is_string()) 
          patch_string(L"\\StringFileInfo\\040004b0\\ProductVersion", val.get<tool::ustring>()());

        val = props.get_prop("productCopyright");
        if (val.is_string()) 
          patch_string(L"\\StringFileInfo\\040004b0\\LegalCopyright", val.get<tool::ustring>()());

        val = props.get_prop("productDescription");
        if (val.is_string())
          patch_string(L"\\StringFileInfo\\040004b0\\FileDescription", val.get<tool::ustring>()());

        val = props.get_prop("productCompany");
        if (val.is_string())
          patch_string(L"\\StringFileInfo\\040004b0\\CompanyName", val.get<tool::ustring>()());

        UpdateResource(hu, RT_VERSION, MAKEINTRESOURCE(VS_VERSION_INFO), MAKELANGID(LANG_NEUTRAL, SUBLANG_NEUTRAL), ver_info.begin(), DWORD(ver_info.length()));
      }

      EndUpdateResource(hu, FALSE);
    }

  }

  return true;
#endif
}

static tool::array<tool::value> args;

class scapp : public sciter::window {
public:
  scapp(bool debug) : window(SW_TITLEBAR | SW_CONTROLS | SW_RESIZEABLE | SW_MAIN | SW_GLASSY | (debug ? SW_ENABLE_DEBUG : 0)) {}

  int assembleExe(tool::ustring scapp_path, tool::ustring dat_path, tool::ustring dst_path,  tool::value params) {
    //exedata ed;
    tool::mm_file resfl;
    if (!resfl.open(dat_path))
      return -1;
    tool::mm_file scafl;
    if (!scafl.open(scapp_path))
      return -1;

    FILE *f = fopen(tool::string(dst_path), "w+b");
    if (!f) return -2;
    bool r = fwrite(scafl.data(), 1, scafl.size(), f) == scafl.size();
    bool patched = false;
#ifdef WINDOWS    
    if (params.is_object() || params.is_proxy_of_object())
    {
      fclose(f);
      if (r) patched = patch_exe(dst_path, params); // this may change file length
      f = fopen(tool::string(dst_path), "ab");
    }
#endif
    auto res = resfl.bytes();
    r = r && (fwrite(res.start, 1, res.length, f) == res.length);
    bdata len(uint32(res.length));
    r = r && (fwrite(len.data.bytes, 1, 4, f) == 4);
    bdata marker(0xAFED);
    r = r && (fwrite(marker.data.bytes, 1, 4, f) == 4);
    fclose(f);
    return r ? (patched? 0:1) : -3;
  }

  tool::value get_argv() const {
    return tool::value::make_array(args());
  }

  SOM_PASSPORT_BEGIN(scapp)
    SOM_FUNCS(
      SOM_FUNC(assembleExe)
    )
    SOM_PROPS(SOM_RO_VIRTUAL_PROP(argv, get_argv))
    SOM_PASSPORT_END

};

#ifdef _DEBUG
extern "C" void my_function_to_handle_aborts(int signal_number)
{
  /*Your code goes here. You can output debugging info.
    If you return from this function, and it was called
    because abort() was called, your program will exit or crash anyway
    (with a dialog box on Windows).
   */
  signal_number = signal_number;
  //tool::alert("leaks!");
}
#endif
int main(int argc, char *argv[])
{
    /*Do this early in your program's initialization */
    #ifdef _DEBUG
      signal(SIGABRT, &my_function_to_handle_aborts);
    #endif

    //sciter::console _;

    Initialize();

    SciterSetOption(NULL, SCITER_SET_SCRIPT_RUNTIME_FEATURES,
      ALLOW_FILE_IO |
      ALLOW_SOCKET_IO |
      ALLOW_EVAL |
      ALLOW_SYSINFO);

    SciterSetOption(NULL, SCITER_SET_UX_THEMING, TRUE);

    tool::ustring home = tool::get_home_dir();
    tool::ustring path;
    bool path_ok = false;

    for (int n = 0; n < argc; ++n)
      args.push(tool::value(tool::ustring(argv[n])));

    if( argc == 1 )
      do
      {
//TRY_LOCAL:
        path = home + WCHARS("scapp.zip") ; if( tool::filesystem::is_file(path) ) { path_ok = true; break; }
        path = home + WCHARS("scapp.html"); if( tool::filesystem::is_file(path) ) { path_ok = true; break; }
        path = home + WCHARS("scapp.htm");  if( tool::filesystem::is_file(path) ) { path_ok = true; break; }
        path = home + WCHARS("main.htm");   if (tool::filesystem::is_file(path)) { path_ok = true; break; }
        path = home + WCHARS("main.html");  if (tool::filesystem::is_file(path)) { path_ok = true; break; }
        path = home + WCHARS("index.htm");  if (tool::filesystem::is_file(path)) { path_ok = true; break; }
        path = home + WCHARS("index.html");  if (tool::filesystem::is_file(path)) { path_ok = true; break; }
      } while(false);
    else {
      path = tool::real_path(tool::ustring(argv[1]));
      if( (path.like(W("*.htm")) || path.like(W("*.html"))) && tool::filesystem::is_file(path) )
        path_ok = true;
      //else
      //  goto TRY_LOCAL;
    }

    bool debug = false;

    for (int n = 0; n < argc; ++n) {
      //argv
      if (tool::streq(argv[n], "-d") || tool::streq(argv[n], "--debug")) {
        debug = true;
        break;
      }
    }

    sciter::window *pwin = new scapp(debug);

    if( !path_ok )
    {
      #include "resources.cpp"

      do {
        exedata ed;
        static tool::array<byte> tail = ed.tail;
        if (tail.length() && sciter::archive::instance().open(aux::bytes(tail.cbegin(), tail.length()))) {
          if(pwin->load(WSTR("this://app/index.htm")))
            break;
          if (pwin->load(WSTR("this://app/main.htm")))
            break;
          tail.destroy();
        }
        sciter::archive::instance().open(aux::elements_of(resources)); // bind resources[] (defined in "resources.cpp") with the archive
        pwin->load(WSTR("this://app/about.htm"));
      } while (false);

    }
    else
    {
      path = tool::url::path_to_file_url(path);
      pwin->load(path);
    }

    sciter::dom::element root = pwin->root();
    if (!root)
      return -1;

    if (root.get_attribute("window-state").length() == 0)
      pwin->expand(); // undefined, expand

    return MessagePump();
}

