//
//  osx-sciter-graphics.mm
//  sciter-osx
//
//  Created by andrew on 2014-03-22.
//  Copyright (c) 2014 andrew fedoniouk. All rights reserved.
//
#import <Cocoa/Cocoa.h>
#include "osx-sciter.h"
#include "gool/gool.h"

#include <ApplicationServices/ApplicationServices.h>
#include <Carbon/Carbon.h>
#import <IOKit/graphics/IOGraphicsLib.h>

namespace gool
{

cursor::cursor(void *nscursor): id(uint(-1)), pcur(nscursor) { NSCursor* pnc = (NSCursor*)pcur; [pnc retain]; }

cursor::~cursor() {
    NSCursor* pnc = (NSCursor*)pcur;
    [pnc release];
}

//static NSCursor* gcur = nullptr;

void cursor::set() {
    NSCursor* pnc = (NSCursor*)pcur;
    
    //if(gcur && pnc != gcur)
    //    return;
    
    if( [NSCursor currentCursor] != pnc)
    {
      [pnc set];
      //dbg_printf("[pnc set]\n");
    }
}

cursor* get_cursor(NSString *cursorName) {
    NSString *cursorPath = [@"/System/Library/Frameworks/ApplicationServices.framework/Versions/A/Frameworks/HIServices.framework/Versions/A/Resources/cursors" stringByAppendingPathComponent:cursorName];
    NSImage *image = [[NSImage alloc] initByReferencingFile:[cursorPath stringByAppendingPathComponent:@"cursor.pdf"]];
    if( !image )
        return new cursor([NSCursor arrowCursor ]);
    NSDictionary *info = [NSDictionary dictionaryWithContentsOfFile:[cursorPath stringByAppendingPathComponent:@"info.plist"]];
    if( !info )
        return new cursor([NSCursor arrowCursor ]);
    return new cursor( [[NSCursor alloc] initWithImage:image hotSpot:NSMakePoint([[info valueForKey:@"hotx"] doubleValue], [[info valueForKey:@"hoty"] doubleValue])] );
}

cursor* cursor::system(uint cid)
{
  switch( cid )
    {
      default:
      case html::cursor_auto:
      case html::cursor_wait:
      case html::cursor_uparrow:
      {
    DEFAULT:
          static handle<cursor> pc = new cursor( [NSCursor arrowCursor ] );
          pc->id = cid;
          return pc;
      }
      case html::cursor_text:
      {
          static handle<cursor> pc = new cursor( [NSCursor IBeamCursor ] );
          pc->id = cid;
          return pc;
      }
      case html::cursor_crosshair:
      {
          static handle<cursor> pc = new cursor( [NSCursor crosshairCursor ] );
          pc->id = cid;
          return pc;
      }
      case html::cursor_pointer:
      {
          static handle<cursor> pc = new cursor( [NSCursor pointingHandCursor ] );
          pc->id = cid;
          return pc;
      }
      case html::cursor_help:
      {
          static handle<cursor> pc = new cursor( [NSCursor contextualMenuCursor ] );
          pc->id = cid;
          return pc;
      }
      //case html::cursor_w_resize:
      //{
      //    static handle<cursor> pc = new cursor( [NSCursor resizeLeftRightCursor ] );
      //    return pc;
      //}
      //case html::cursor_n_resize:
      //{
      //    static handle<cursor> pc = new cursor( [NSCursor resizeUpDownCursor ] );
      //    return pc;
      //
      //}
      case html::cursor_move:
      {
          static handle<cursor> pc = new cursor( [NSCursor openHandCursor ] );
          pc->id = cid;
          return pc;
          
      }
      case html::cursor_drag_move:
      {
          static handle<cursor> pc = new cursor( [NSCursor closedHandCursor ] );
          pc->id = cid;
          return pc;
          
      }
      case html::cursor_drag_copy:
      {
          static handle<cursor> pc = new cursor( [NSCursor dragCopyCursor] );
          pc->id = cid;
          return pc;
          
      }
                
      case html::cursor_no:
      {
          static handle<cursor> pc = new cursor( [NSCursor disappearingItemCursor ] );
          pc->id = cid;
          return pc;
          
      }
        case html::cursor_none:
        {
            NSImage *blank = [[NSImage alloc] initWithSize:NSMakeSize(7, 7)];
            [blank lockFocus];
            NSGraphicsContext *currentContext = [NSGraphicsContext currentContext];
            [currentContext saveGraphicsState];
            
            // Clear image
            [[NSColor clearColor] set];
            NSRectFill(NSMakeRect(0, 0, 7, 7));
            [blank unlockFocus];
            static handle<cursor> pc = new cursor( [[NSCursor alloc] initWithImage:blank hotSpot:NSMakePoint(0, 0)] );
            pc->id = cid;
            return pc;
            
        }
      case html::cursor_ne_resize:
      {
          static handle<cursor> pc = get_cursor(@"resizenortheastsouthwest");
          pc->id = cid;
          return pc;
      }
      case html::cursor_se_resize:
      {
          static handle<cursor> pc = get_cursor(@"resizenorthwestsoutheast");
          pc->id = cid;
          return pc;
      }
      case html::cursor_w_resize:
      {
          static handle<cursor> pc = get_cursor(@"resizeeastwest");
          pc->id = cid;
          return pc;
      }
      case html::cursor_n_resize:
      {
          static handle<cursor> pc = get_cursor(@"resizenorthsouth");
          pc->id = cid;
          return pc;
      }
      //    cursor_no = 10,
      //    cursor_starting = 11,
          
  }
}
cursor* cursor::from_data(bytes data, string url)
{
    handle<gool::image> pimg = gool::image::create(data, url);
    
    if(!pimg)
        return nullptr;

    handle<gool::bitmap> pbmp = pimg->get_bitmap(nullptr, pimg->dim());
    if(pbmp)
      return from_bitmap(pbmp, url, size(0,0));

    return nullptr;
}


cursor* cursor::from_bitmap(bitmap* bmp, string url, size offset)
{
    NSPoint   hotspot;
    hotspot.x = offset.x;
    hotspot.y = offset.y;
    
    CGImageRef cgimg = osx_ttb_image( bmp);
    
    //CGImageRetain(cgimg);
    
    if(!cgimg)
        return nullptr;
    NSImage* nsimg = [[NSImage alloc] initWithCGImage:cgimg size:NSZeroSize];
    if(!nsimg)
        return nullptr;
    
    NSCursor* pcur = [[NSCursor alloc] initWithImage: nsimg hotSpot: hotspot];
    if(!pcur)
        return nullptr;
    
    [nsimg release];
    
    [pcur push]; [pcur pop]; // this must be called here, seems like those creative guys update something when the cursor get selected.
    
    return new cursor(pcur);
    
/*
    NSBitmapImageRep* ns_bitmap = [[NSBitmapImageRep alloc] initWithCGImage:cgimg];
    NSImage* cursor_image = [[NSImage alloc] init];
    [ns_bitmap setSize: NSMakeSize(bmp->dim().x,bmp->dim().y)];
    [cursor_image addRepresentation:ns_bitmap];
    [ns_bitmap release];
    
    NSCursor* pcursor = [[NSCursor alloc] initWithImage:cursor_image hotSpot:NSMakePoint(offset.x,offset.y)];
    [cursor_image release];

    return new cursor((NSCursor*)[pcursor autorelease]); */
    
}

}

gool::argb osx_sys_color(gool::SYSTEM_COLORS cs)
{
    //NSColorList* clist = [ NSColorList colorListNamed: @"Developer."];
    //    NSColor* nc = [clist colorWithKey:@"windowBackgroundColor"];
    NSColorSpace *colorSpace = [NSColorSpace sRGBColorSpace];
    NSColor* sysc = NULL;
    switch(cs) {
        case gool::SC_WINDOW:          sysc = [NSColor windowBackgroundColor]; break;
        case gool::SC_WINDOW_TEXT:     sysc = [NSColor windowFrameTextColor]; break;
        case gool::SC_DIALOG_BRIGHT:   sysc = [NSColor controlLightHighlightColor]; break;
        case gool::SC_DIALOG_LIGHT:    sysc = [NSColor controlHighlightColor]; break;
        case gool::SC_DIALOG_SURFACE:  sysc = [NSColor controlColor]; break;
        case gool::SC_DIALOG_SHADOW:   sysc = [NSColor controlShadowColor]; break;
        case gool::SC_DIALOG_DKSHADOW: sysc = [NSColor controlDarkShadowColor]; break;
        case gool::SC_DIALOG_TEXT:     sysc = [NSColor controlTextColor]; break;
            
        case gool::SC_SELECTION:       sysc = [NSColor selectedTextBackgroundColor]; break;
        case gool::SC_SELECTION_TEXT:  sysc = [NSColor selectedTextColor]; break;
        case gool::SC_MENU_SELECTION:       sysc = [NSColor selectedMenuItemColor]; break;
        case gool::SC_MENU_SELECTION_TEXT:  sysc = [NSColor selectedMenuItemTextColor]; break;
        case gool::SC_WORKSPACE:       sysc = [NSColor knobColor]; break;
        case gool::SC_ACTIVE_CAPTION:         sysc = [NSColor headerColor ]; break;
        case gool::SC_ACTIVE_CAPTION_TEXT:    sysc = [NSColor headerTextColor ]; break;
        case gool::SC_CAPTION:         sysc = [NSColor secondarySelectedControlColor ]; break;
        case gool::SC_CAPTION_TEXT:    sysc = [NSColor headerTextColor ]; break;
        case gool::SC_INFO:            return gool::rgba(250,249,199);
        case gool::SC_INFO_TEXT:       return gool::rgba(0,0,0);
        case gool::SC_SCROLLBAR:       sysc = [NSColor scrollBarColor ]; break;
        case gool::SC_DESKTOP:         sysc = [NSColor underPageBackgroundColor ]; break;
        case gool::SC_GRAY_TEXT:             sysc = [NSColor disabledControlTextColor]; break;
        case gool::SC_WINDOW_BORDER:         sysc = [NSColor windowFrameColor]; break;
        case gool::SC_BORDER:                sysc = [NSColor gridColor]; break;
        case gool::SC_DIALOG_LIGHTSHADOW:    sysc = [NSColor controlBackgroundColor ]; break;
        case gool::SC_HYPERLINK:             return 0x007fff;
        case gool::SC_ACCENT: {
            sysc = [NSColor selectedTextBackgroundColor ];
            sysc = [sysc colorUsingColorSpace:colorSpace];
            CGFloat h = 0,s = 0,b = 0;
            [sysc getHue:&h saturation:&s brightness:&b alpha:nullptr];
            if( s < 0.001 )
                return gool::rgb(gool::hsv(h*360,0,0.5*b));
            else
              return gool::rgb(gool::hsv(h*360,0.8,b));
            //break;
        }

        default:  return gool::argb::no_color_white();
    }
    
    sysc = [sysc colorUsingColorSpace:colorSpace];

    CGFloat r = 1.0,g = 1.0,b = 1.0;
    [sysc getRed:&r green:&g blue:&b alpha: NULL];
    
    return gool::argb(uint(r * 255.0), uint(g * 255.0), uint(b * 255.0));
}

#if defined(NATIVE_OSX_BACKEND)

gool::pointf osx_graphics_world_to_screen(const osx::graphics* gfx, gool::pointf pt)
{
    CGPoint c = CGContextConvertPointToDeviceSpace ( gfx->target(), cvt(pt));
    return  cvt<float>(c);
}

gool::pointf osx_graphics_screen_to_world(const osx::graphics* gfx, gool::pointf pt)
{
    CGPoint c = CGContextConvertPointToUserSpace ( gfx->target(), cvt(pt));
    return  cvt<float>(c);
}

void osx_graphics_fill( osx::graphics* gfx, gool::argb c, const gool::rect& dst)
{
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetFillColorWithColor ( gfx->target(),  color);
    CGContextFillRect(gfx->target(), cvt(dst));
}

void osx_graphics_fill( osx::graphics* gfx, gool::argb c, const gool::path* dst)
{
   CF::ref<CGColorRef> color = cvt(c);
    
   CGContextSaveGState ( gfx->target() );
     CGContextSetFillColorWithColor ( gfx->target(),  color);
     CGContextAddPath ( gfx->target(), static_cast<const osx::path*>(dst)->cgpath());
     if(static_cast<const osx::path*>(dst)->even_odd() )
       CGContextEOFillPath(gfx->target());
     else
       CGContextFillPath(gfx->target());
    
   CGContextRestoreGState ( gfx->target() );
}

void osx_graphics_stroke( osx::graphics* gfx, gool::argb c, float width, const gool::path* dst)
{
    CF::ref<CGColorRef> color = cvt(c);
    
    CGContextSaveGState ( gfx->target() );
    CGContextSetStrokeColorWithColor ( gfx->target(),  color);
    CGContextAddPath ( gfx->target(), static_cast<const osx::path*>(dst)->cgpath());
    CGContextStrokePath(gfx->target());
   
    CGContextRestoreGState ( gfx->target() );
}



void osx_graphics_fill_rr( osx::graphics* gfx, gool::argb c, const gool::rect& dst, float r)
{
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetFillColorWithColor ( gfx->target(),  color);
    
    CGRect rrect = cvt( dst );
    CGFloat radius = r;

    CGFloat minx = CGRectGetMinX(rrect), midx = CGRectGetMidX(rrect), maxx = CGRectGetMaxX(rrect);
    CGFloat miny = CGRectGetMinY(rrect), midy = CGRectGetMidY(rrect), maxy = CGRectGetMaxY(rrect);
    
    // Start at 1
    CGContextMoveToPoint(gfx->target(), minx, midy);
    // Add an arc through 2 to 3
    CGContextAddArcToPoint(gfx->target(), minx, miny, midx, miny, radius);
    // Add an arc through 4 to 5
    CGContextAddArcToPoint(gfx->target(), maxx, miny, maxx, midy, radius);
    // Add an arc through 6 to 7
    CGContextAddArcToPoint(gfx->target(), maxx, maxy, midx, maxy, radius);
    // Add an arc through 8 to 9
    CGContextAddArcToPoint(gfx->target(), minx, maxy, minx, midy, radius);
    // Close the path 
    CGContextClosePath(gfx->target());
    // Fill & stroke the path 
    CGContextFillPath(gfx->target());
}

void osx_graphics_line_v(osx::graphics* gfx, const gool::rect& rc, gool::argb c, int stroke, int step)
{
    if(c.alfa == 0 || rc.empty())
        return;

    auto target = gfx->target();

    CGContextSaveGState ( target );
    
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetStrokeColorWithColor ( gfx->target(),  color);
    
    CGContextSetLineJoin(target,kCGLineJoinMiter);
    
    //if(rc.width() == 1)
       CGContextSetLineCap(target, kCGLineCapButt);
    //else
    //   CGContextSetLineCap(target, kCGLineCapRound); NOTE: this needs clipping!

    float w = float(rc.width());
    
    CGContextSetLineWidth (target, w );
    
    CGFloat offset = -(rc.height() % step) + stroke - w / 2;
    
    CGFloat pat[2];
    pat[0] = stroke;
    pat[1] = (step - stroke);
    CGContextSetLineDash ( target, offset, pat, 2);
    
    gool::pointf start( rc.left() + w / 2, rc.top() ), end( rc.left() + w / 2, rc.bottom() );
    
    CGContextBeginPath( target );

    CGContextMoveToPoint( target, start.x, start.y);
    CGContextAddLineToPoint( target, end.x, end.y);
    
    //CGContextDrawPath (target, kCGPathStroke);
    CGContextStrokePath (target);
    
    
    CGContextRestoreGState ( target );
}
void osx_graphics_line_h(osx::graphics* gfx, const gool::rect& rc, gool::argb c, int stroke, int step)
{

    if(c.alfa == 0 || rc.empty())
        return;
    
    auto target = gfx->target();
    
    CGContextSaveGState ( target );
    
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetStrokeColorWithColor ( target,  color);
    
    if(rc.height() == 1)
        CGContextSetLineCap(target, kCGLineCapButt);
    //else
    //    CGContextSetLineCap(target, kCGLineCapRound);
    
    float w = float(rc.height());
    
    CGContextSetLineWidth (target, w );
    
    CGFloat offset = -(rc.width() % step) + stroke - w / 2;
    
    CGFloat pat[2];
    pat[0] = stroke;
    pat[1] = (step - stroke);
    CGContextSetLineDash ( target, offset, pat, 2);
    
    gool::pointf start( rc.left(), rc.top() + w / 2 ), end( rc.right(), rc.top() + w / 2);
    
    CGContextBeginPath( target );
    
    CGContextMoveToPoint( target, start.x, start.y);
    CGContextAddLineToPoint( target, end.x, end.y);
    
    CGContextStrokePath (target);
    
    
/*
    float h2 = w / 2.0f;
    float w2 = rc.height() == 1? 0: w / 2.0f;

    CGContextSetLineWidth (target, w );
    
    gool::pointf pts = {rc.left() + w2,rc.top() + h2};
    gool::pointf pte = {rc.left() + s1 - w2,rc.top() + h2};
    
    CGContextBeginPath( target );
    
    CGContextMoveToPoint( target, pts.x, pts.y);
    CGContextAddLineToPoint( target, pte.x, pte.y);
    
    int space = rc.width() - step - stroke + 1;
    int num_steps = space / step;
    for(int lx = rc.origin.x + step; num_steps > 0;)
    {
        pts.x = lx + w2;
        pte.x = lx + s1 - w2;
        CGContextMoveToPoint( target, pts.x, pts.y);
        CGContextAddLineToPoint( target, pte.x, pte.y);
        int delta = space / num_steps;
        lx += delta;
        space -= delta;
        --num_steps;
    }
    pte.x = rc.corner.x + 1 - w2;
    pts.x = rc.corner.x + 1 - s1 + w2;
    CGContextMoveToPoint( target, pts.x, pts.y);
    CGContextAddLineToPoint( target, pte.x, pte.y);

    CGContextDrawPath (target, kCGPathStroke); */
    
    CGContextRestoreGState ( target );
}


void osx_graphics_push_layer(osx::graphics* gfx, const gool::rect& rc, byte opacity)
{
   CGContextSaveGState(gfx->target());
   CGContextClipToRect(gfx->target(), cvt(rc));
   if( opacity < 255 ) {
     CGContextBeginTransparencyLayer(gfx->target(),NULL);
     CGContextSetAlpha(gfx->target(), opacity / 255.0);
   }
}
void osx_graphics_push_layer(osx::graphics* gfx, const gool::path* pa, byte opacity)
{
   CGContextSaveGState(gfx->target());
   CGContextAddPath ( gfx->target(), static_cast<const osx::path*>(pa)->cgpath());
   CGContextClip(gfx->target());

   if( opacity < 255 ) {
     CGContextBeginTransparencyLayer(gfx->target(),NULL);
     CGContextSetAlpha(gfx->target(), opacity / 255.0);
   }
}

CGImageRef osx_image( gool::bitmap* bmp, gool::graphics* gfx );
CGImageRef osx_ttb_image( gool::bitmap* bmp, gool::graphics* gfx );

void osx_graphics_push_layer(osx::graphics* gfx, const gool::bitmap* bmp, bool draw1a, byte opacity)
{
    CGContextSaveGState(gfx->target());
    //CGRectInfinite
    
    if( bmp->_pixels.length() ) {

        CGRect rc = cvt(gool::rect(bmp->dim()));
        
        if( draw1a ) {
            CGContextClipToMask(gfx->target(), rc, osx_image(const_cast<gool::bitmap*>(bmp),gfx));
        } else {
        
          CF::ref<CGDataProviderRef> provider = CGDataProviderCreateWithData (0, bmp->get_bits().start, bmp->get_bits().length * sizeof( gool::argb), 0);
        
          CF::ref<CGColorSpaceRef> colorSpace = CGColorSpaceCreateDeviceRGB();
        
          CF::ref<CGImageRef> cgi = CGImageMaskCreate ((size_t) bmp->dim().x,
                                                       (size_t) bmp->dim().y,
                                                       8, sizeof(gool::argb) * 8,
                                                       (size_t) bmp->stride(),
                                                       provider,
                                                       nullptr, false);
          CGContextClipToMask(gfx->target(),rc, cgi/*osx_image(const_cast<gool::bitmap*>(bmp),gfx)*/);
        }

    }
    
    if( opacity < 255 ) {
        CGContextBeginTransparencyLayer(gfx->target(),NULL);
        CGContextSetAlpha(gfx->target(), opacity / 255.0);
    }
}

void osx_graphics_pop_layer(osx::graphics* gfx, byte opacity)
{
   if( opacity < 255 )
     CGContextEndTransparencyLayer(gfx->target());
   CGContextRestoreGState(gfx->target());
}

void osx_graphics_push_clip(osx::graphics* gfx, const gool::rect& rc)
{
    CGContextSaveGState(gfx->target());
    CGContextClipToRect(gfx->target(), cvt(rc));
}
void osx_graphics_pop_clip(osx::graphics* gfx)
{
    CGContextRestoreGState(gfx->target());
}

void osx_graphics_set_transform(osx::graphics* gfx, const gool::affine_mtx_f& m)
{
    CGAffineTransform transform = CGAffineTransformMake( m.sx,m.shy,m.shx,m.sy, m.tx, m.ty );
    CGContextConcatCTM ( gfx->target(), transform );
}

void osx_graphics_reset_transform(osx::graphics* gfx) {
    // WTF! Where is CGContextSetCTM ?
    CGAffineTransform transform = CGContextGetCTM(gfx->target());
    transform = CGAffineTransformInvert(transform);
    CGContextConcatCTM ( gfx->target(), transform );
}

void osx_graphics_get_transform(const osx::graphics* gfx, gool::affine_mtx_f& m)
{
    CGAffineTransform t = CGContextGetCTM( const_cast<osx::graphics*>(gfx)->target() );
    m.sx = t.a;
    m.shy = t.b;
    m.shx = t.c;
    m.sy = t.d;
    m.tx = t.tx;
    m.ty = t.ty;
}

void osx_graphics_translate( osx::graphics* gfx, gool::pointf pt)
{
    CGContextTranslateCTM ( gfx->target(), pt.x, pt.y );
}
void osx_graphics_rotate( osx::graphics* gfx, float radians, gool::pointf center)
{
    if(center.x != 0 || center.y != 0) {
       CGContextTranslateCTM ( gfx->target(), center.x, center.y );
       CGContextRotateCTM ( gfx->target(), radians );
       CGContextTranslateCTM ( gfx->target(), -center.x, -center.y );
    } else {
       CGContextRotateCTM ( gfx->target(), radians );
    }
}
void osx_graphics_scale( osx::graphics* gfx, gool::sizef sz, gool::pointf center)
{
    if(center.x != 0 || center.y != 0) {
        CGContextTranslateCTM ( gfx->target(), center.x, center.y );
        CGContextScaleCTM ( gfx->target(), sz.x, sz.y );
        CGContextTranslateCTM ( gfx->target(), -center.x, -center.y );
    } else {
        CGContextScaleCTM ( gfx->target(), sz.x, sz.y );
    }
    
}
void osx_graphics_skew( osx::graphics* gfx, gool::sizef sz, gool::pointf center)
{
    CGAffineTransform shear = CGAffineTransformMake( 1, real_traits<CGFloat>::tan(sz.y), real_traits<CGFloat>::tan(sz.x), 1, 0,0 );
    if(center.x != 0 || center.y != 0) {
        CGContextTranslateCTM ( gfx->target(), center.x, center.y );
        CGContextConcatCTM ( gfx->target(), shear );
        CGContextTranslateCTM ( gfx->target(), -center.x, -center.y );
    } else {
        CGContextConcatCTM ( gfx->target(), shear );
    }
}


void osx_graphics_push_state(osx::graphics* gfx)
{
    CGContextSaveGState(gfx->target());
}
void osx_graphics_pop_state(osx::graphics* gfx)
{
    CGContextRestoreGState(gfx->target());
}




void osx_graphics_draw_image( osx::graphics* gfx, gool::image* img, const gool::rectf&  dst, const gool::rect& src, byte opacity)
{
   if( dst.empty() || src.empty() || !img )
        return;
    
    gool::bitmap *bmp = img->get_bitmap( gfx, src.size());
    if( !bmp )
    {
        img->draw(gfx,dst,src,opacity);
        return;
    }
    
    CGContextSaveGState ( gfx->target() );
    
    if( opacity != 255 )
        CGContextSetAlpha ( gfx->target(), opacity / 255.0f);
    
    CGImageRef cgimg = osx_image( bmp, gfx );
    
    if(src.size() == bmp->dim())
      CGContextDrawImage ( gfx->target(), cvt(dst), cgimg );
    else
    {
      CF::ref<CGImageRef> frag = CGImageCreateWithImageInRect ( cgimg, cvt(bmp->mapped(src)) );
      CGContextDrawImage ( gfx->target(), cvt(dst), frag );
    }

    CGContextRestoreGState(gfx->target());
    
}

void osx_graphics_tile_image( osx::graphics* gfx, gool::image* img, const gool::rectf&  dst, const gool::rect& src, const gool::point& offset, const gool::size& celldim)
{
    if( dst.empty() || src.empty() || !img)
        return;
    
    gool::bitmap *bmp = img->get_bitmap( gfx, src.size());
    if( !bmp )
    {
        img->draw(gfx,dst,src,255);
        return;
    }
    
    
    CGContextSaveGState ( gfx->target() );
    
    CGImageRef cgimg = osx_image( bmp, gfx );
    
    //CGContextSetInterpolationQuality(gfx->target(), kCGInterpolationLow);
    
    CGContextClipToRect (gfx->target(), cvt(dst));
    
    gool::size  sz = src.size();
    gool::point off = offset;
    
    CGRect tile;
    tile.origin = cvt( gool::pointf(off) + gool::pointf(dst.start() ));
    tile.size = cvt( celldim.empty() ? sz : celldim );
    
    if(src.size() == bmp->dim())
         CGContextDrawTiledImage ( gfx->target(), tile, cgimg );
    else
    {
        CF::ref<CGImageRef> frag = CGImageCreateWithImageInRect ( cgimg, cvt(bmp->mapped(src)) );
        CGContextDrawTiledImage ( gfx->target(), tile, frag );
    }
    
    CGContextRestoreGState(gfx->target());
    
}


void osx_graphics_fill_gradient(osx::graphics* gfx,  const gool::brush* bin, const gool::rect& dst) {
    
    if(bin->type() != gool::brush::LINEAR && bin->type() != gool::brush::RADIAL)
        return;
 
    CGContextSaveGState ( gfx->target() );
    
    
    CGContextClipToRect (gfx->target(), cvt(dst));
    
 
    switch( bin->type() )
    {
      case gool::brush::LINEAR:
        {
            osx::linear_gradient_brush br( *static_cast<const gool::linear_brush*>(bin));
            br.draw(gfx);
        }
        break;
      case gool::brush::RADIAL:
        {
            osx::radial_gradient_brush br( *static_cast<const gool::radial_brush*>(bin));
            br.draw(gfx);
        }
        break;
      default:
        break;
    }
    
    CGContextRestoreGState(gfx->target());
    
}

void osx_graphics_set_offset(osx::graphics* gfx, gool::point off )
{
   CGContextTranslateCTM(gfx->target(), off.x, off.y);
}

void osx_graphics_set_line_join( osx::graphics* gfx, gool::LINE_JOIN ncs )
{
    CGLineJoin joins[] = { kCGLineJoinMiter, kCGLineJoinBevel, kCGLineJoinRound, kCGLineJoinBevel };
    CGContextSetLineJoin(gfx->target(), joins[ncs]);
}

void osx_graphics_set_line_cap( osx::graphics* gfx, gool::CAP_STYLE ncs )
{
    CGLineCap caps[] = {  kCGLineCapButt, kCGLineCapSquare, kCGLineCapRound, kCGLineCapSquare };
    CGContextSetLineCap(gfx->target(), caps[ncs]);
}

void osx_graphics_set_stroke_style( osx::graphics* gfx, gool::DASH_STYLE ncs )
{
    switch( ncs ) {
        default:
        case gool::DASH_STYLE_SOLID: CGContextSetLineDash ( gfx->target(), 0, NULL, 0); break;
        case gool::DASH_STYLE_DOT:
        {
            float lw = gfx->_registers.stroke_width;
            CGFloat dash[] = { 0.1, 2 * lw };
            if(gfx->_registers.cap_style == gool::CAP_BUTT ) {
                dash[0] = lw;
                dash[1] = lw;
            }
            CGContextSetLineDash ( gfx->target(), 0, dash, 2);
        } break;
        case gool::DASH_STYLE_DASH:
        {
            float lw = gfx->_registers.stroke_width;
            CGFloat dash[] = { 4 * lw, 4 * lw };
            CGContextSetLineDash ( gfx->target(), 0, dash, 2);
        } break;
    }
}

void osx_graphics_set_stroke_pattern( osx::graphics* gfx, slice<float> pattern, float offset )
{
   array<CGFloat> pat; pattern.each([&pat](float f) -> bool { pat.push(f); return false; } );
   CGContextSetLineDash ( gfx->target(), offset, pat.cbegin(), pat.length());
}


void osx_graphics_set_stroke_color( osx::graphics* gfx, gool::argb c )
{
   //const CGFloat components[] = { c.red / 255.0, c.green / 255.0, c.blue / 255.0, c.alfa / 255.0 };
   //CGContextSetStrokeColor ( gfx->target(), components );
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetStrokeColorWithColor ( gfx->target(), color );
}

void osx_graphics_set_stroke_width( osx::graphics* gfx, float w )
{
   CGContextSetLineWidth ( gfx->target(), w );
}

void osx_graphics_set_fill_color( osx::graphics* gfx, gool::argb c )
{
    //const CGFloat components[] = { c.red / 255.0, c.green / 255.0, c.blue / 255.0, c.alfa / 255.0 };
    //CGContextSetFillColor ( gfx->target(), components );
    CF::ref<CGColorRef> color = cvt(c);
    CGContextSetFillColorWithColor ( gfx->target(), color );
}



/*void drawGradientPattern(void *info, CGContextRef context) {
    osx_gradient* pb = static_cast<osx_gradient* >(info);
    pb->draw(context);
}

void releaseGradientPattern(void *info)
{
    osx_gradient* pb = static_cast<osx_gradient* >(info);
    pb->release();
}


void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::linear_brush& lb )
{
    osx_gradient_linear* pbr = new osx_gradient_linear(lb);
    pbr->add_ref();
    
    CGRect rect = CGRectMake(0,0, 1000, 1000);
    static const CGPatternCallbacks callbacks = {0, &drawGradientPattern, &releaseGradientPattern};
    CGAffineTransform transform = CGAffineTransformIdentity;//CGAffineTransformMakeScale(1.0, -1.0);
    CF::ref<CGPatternRef> pattern = CGPatternCreate( static_cast<osx_gradient*>(pbr), rect, transform, rect.size.width, rect.size.height, kCGPatternTilingNoDistortion, true, &callbacks);
    CF::ref<CGColorSpaceRef> patternSpace = CGColorSpaceCreatePattern(NULL);
    CGFloat alpha = 1.0;
    CGColorRef patternColorRef = CGColorCreateWithPattern(patternSpace, pattern, &alpha);
    //CGContextSetFillColorSpace( gfx->target(), patternSpace );
    CGContextSetFillColorWithColor(gfx->target(), patternColorRef );
    //CGContextSetFillPattern (gfx->target(), pattern, &alpha);
}
void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::radial_brush& rb )
{
    osx_gradient_radial* pbr = new osx_gradient_radial(rb);
    pbr->add_ref();
    
    CGRect rect = CGRectMake(0,0, 10000, 10000);
    static const CGPatternCallbacks callbacks = {0, &drawGradientPattern, &releaseGradientPattern};
    CGAffineTransform transform = CGAffineTransformMakeScale(1.0, -1.0);
    CF::ref<CGPatternRef> pattern = CGPatternCreate( static_cast<osx_gradient*>(pbr), rect, transform, rect.size.width, rect.size.height, kCGPatternTilingNoDistortion, true, &callbacks);
    CF::ref<CGColorSpaceRef> patternSpace = CGColorSpaceCreatePattern(NULL);
    CGFloat alpha = 1.0;
    CGColorRef patternColorRef = CGColorCreateWithPattern(patternSpace, pattern, &alpha);
    //CGContextSetFillColorSpace( gfx->target(), patternSpace );
    CGContextSetFillColorWithColor(gfx->target(), patternColorRef );
    //CGContextSetFillPattern (gfx->target(), pattern, &alpha);

}
void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::image* ib )
{
    osx_graphics_set_fill_color( gfx, gool::argb(0,255, 0));
}
void osx_graphics_set_fill_even_odd( osx::graphics* gfx, bool even_odd  )
{
    
} */



namespace osx {
    
    void path::start(pointf start, bool tofill) {
        _path = CGPathCreateMutable();
        CGPathMoveToPoint ( _path, NULL, start.x, start.y);
        filled = tofill;
        //lastp() = firstp = start;
        //opened = true;
    }
    
    void path::reset()
    {
        _path = CGPathCreateMutable();
    }
    
    bool path::is_inside(pointf pt)
    {
        seal();
        if( is_empty() ) return false;
        return CGPathContainsPoint ( _path, NULL, cvt(pt), evenodd );
    }
    
    void path::set(const polygonf& vertices, bool fill_even_odd )
    {
        slice<pointf> points = vertices();
        
        if(points.length == 0) return;
        
        _path = CGPathCreateMutable();
        
        pointf p = points++;
        CGPathMoveToPoint ( _path, NULL, p.x, p.y );
        while( !!points )
        {
            p = points++;
            CGPathAddLineToPoint(_path,NULL, p.x, p.y);
        }
        CGPathCloseSubpath (_path);
        evenodd = fill_even_odd;
    }
    
    void path::set(const polygonf& v1,const polygonf& v2)
    {
        slice<pointf> points = v1();
        
        if(points.length == 0) return;
        
        _path = CGPathCreateMutable();
        pointf p = points++;
        CGPathMoveToPoint ( _path, NULL, p.x, p.y );
        while( !!points )
        {
            p = points++;
            CGPathAddLineToPoint(_path,NULL, p.x, p.y);
        }
        CGPathCloseSubpath (_path);
        
        points = v2();
        p = points++;
        CGPathMoveToPoint ( _path, NULL, p.x, p.y );
        while( !!points )
        {
            p = points++;
            CGPathAddLineToPoint(_path,NULL, p.x, p.y);
        }
        CGPathCloseSubpath (_path);
        
        evenodd = true;
    }
    
    void path::move_to(pointf pt, bool rel)
    {
        //if(rel)
        //    lastp() += pt;
        //else
        //    lastp() = pt;
        if( is_empty() )
           return start(pt,true);
        
        if(rel)
            pt += lastp();

        CGPathMoveToPoint ( _path, NULL, pt.x, pt.y );
    }
    void path::line_to(pointf pt, bool rel)
    {
        assert(_path);
        //if(is_empty())
        //    start(lastp(),true);
        
        if( !_path ) return;
        
        if( rel ) pt += lastp();
        CGPathAddLineToPoint(_path,NULL, pt.x, pt.y);
    }
    
    void path::close()
    {
        CGPathCloseSubpath (_path);
    }
    
    void path::seal()
    {
    }
    
    void path::add_arc(pointf c, sizef r, float angle_start, float angle_swipe)
    {
        //if(is_empty())
        //    start(lastp(),true);
        CGPathAddRelativeArc(_path, NULL, c.x, c.y, r.x, angle_start, angle_swipe);
    }
    
    void path::quadratic_to(pointf pt, pointf cp, bool rel)
    {
        //if(is_empty())
        //    start(lastp(),true);
        //lastp() = pt;
        if( rel ) { pt += lastp(); cp += lastp(); }
        CGPathAddQuadCurveToPoint(_path, NULL, cp.x,cp.y, pt.x,pt.y );
    }
    
    void path::cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel)
    {
        //if(is_empty())
        //    start(lastp(),true);
        if( rel ) { pt += lastp(); cp1 += lastp(); cp2 += lastp(); }
        //lastp() = pt;
        CGPathAddCurveToPoint(_path, NULL, cp1.x, cp1.y, cp2.x, cp2.y, pt.x,pt.y);
    }
    
    void path::set_even_odd(bool even_odd)
    {
        evenodd = even_odd;
    }
    
    gool::pointf path::lastp()
    {
        CGPoint cgp = CGPathGetCurrentPoint(_path);
        return pointf(cgp.x,cgp.y);
    }
  

    bool path::is_empty() const { return !_path || CGPathIsEmpty(_path); }
    
    rectf path::bounds() const {
        CGRect r = CGPathGetBoundingBox (_path);
        if(r.size.width <= 0 || r.size.height <= 0)
            return rectf();
        return cvt<float>(r);
    }
    
    gool::path* path::combine(COMBINE_MODE mode, const gool::path* other)
    {
        //@autoreleasepool {
        //
        //    SKRegion * sr = [[SKRegion alloc] initWithPath : _path];
        //
        //
        //}
        return nullptr;
    }
    
    
    /*bitmap_graphics::bitmap_graphics(gool::bitmap *bmp, gool::argb initc)
    {
        CF::ref<CGColorSpaceRef> colorSpace = CGColorSpaceCreateDeviceRGB();
                                              //CGColorSpaceCreateWithName(kCGColorSpaceGenericRGBLinear);
        CGBitmapInfo flags = kCGImageAlphaPremultipliedFirst | kCGBitmapByteOrder32Little;
                             //kCGImageAlphaFirst | kCGBitmapByteOrder32Little;
 
        
        _target = CGBitmapContextCreate((void*)bmp->get_bits().start,
                                                     bmp->dim().x,
                                                     bmp->dim().y, 8, bmp->stride(), colorSpace, flags);
        
        CGContextSetFillColorSpace(_target, colorSpace);
        CGContextSetStrokeColorSpace(_target, colorSpace);
        
        CGContextSetFillColorWithColor(_target, cvt(initc));

        CGContextFillRect(_target,cvt(rect(bmp->dim())));

        set_clip_rc(rect(bmp->dim()));
        
        CGContextTranslateCTM(_target, 0.0f, bmp->dim().y);
        CGContextScaleCTM(_target, 1.0f, -1.0f);
        
        
    }*/
    
    bitmap_graphics::bitmap_graphics(gool::bitmap *bmp, gool::argb initc): super(!initc.is_opaque())
    {
        _bitmap = bmp;
        
        CF::ref<CGColorSpaceRef> colorSpace = CGColorSpaceCreateDeviceRGB();
        CGBitmapInfo flags = kCGImageAlphaPremultipliedFirst | kCGBitmapByteOrder32Little;
        
        if( initc == argb::undefined() )
        {
            bmp->top_to_bottom_inplace();
            pixels = bmp->_pixels;
            _target = CGBitmapContextCreate((void*)pixels.begin(),
                                        bmp->dim().x,
                                        bmp->dim().y, 8, bmp->stride(), colorSpace, flags);
        }
        else
            _target = CGBitmapContextCreate(nullptr,
                                        bmp->dim().x,
                                        bmp->dim().y, 8, bmp->stride(), colorSpace, flags);
        
        
        CGContextSetFillColorSpace(_target, colorSpace);
        CGContextSetStrokeColorSpace(_target, colorSpace);
        
        if( initc != argb::undefined() ) {
          CGContextSetFillColorWithColor(_target, cvt(initc));
          CGContextFillRect(_target,cvt(rect(bmp->dim())));
        }
        
        
        set_clip_rc(rect(bmp->dim()));
        CGContextTranslateCTM(_target, 0.0f, bmp->dim().y);
        CGContextScaleCTM(_target, 1.0f, -1.0f);
        
    }
    
    void bitmap_graphics::finalize()
    {
        void *data = CGBitmapContextGetData (_target);
        if (data)
        {
            const gool::argb *pba = (gool::argb*)data;
            //assert( pba == pixels.begin() );
            //pixels.swap(_bitmap->_pixels);
            //_bitmap->top_to_bottom_inplace();
            _bitmap->set_bits(slice<gool::argb>(pba, _bitmap->dimension().x * _bitmap->dimension().y));
            //_bitmap->_pixels = slice<gool::argb>(pba, _bitmap->dimension().x * _bitmap->dimension().y);
            //++_bitmap->_generation;
            
        }
        gool::graphics::finalize();
    }

}

#endif



