#ifndef __osx_sciter_graphics_h__
#define __osx_sciter_graphics_h__

#include "osx-sciter-basics.h"
#include <ApplicationServices/ApplicationServices.h>
//#include <Frameworks/CoreGraphics/CGFont.h>

#ifdef WINDOWLESS
#error "should not be included"
#endif

namespace osx
{
   struct font;
   class graphics;
}


gool::pointf osx_graphics_world_to_screen(const osx::graphics* gfx, gool::pointf pt);
gool::pointf osx_graphics_screen_to_world(const osx::graphics* gfx, gool::pointf pt);
void osx_graphics_fill( osx::graphics* gfx, gool::argb c, const gool::rect& dst);
void osx_graphics_fill_rr( osx::graphics* gfx, gool::argb c, const gool::rect& dst, float r);
void osx_graphics_fill( osx::graphics* gfx, gool::argb c, const gool::path* dst);
void osx_graphics_stroke( osx::graphics* gfx, gool::argb c, float width, const gool::path* dst);

CGImageRef osx_image( gool::bitmap* bmp, gool::graphics* gfx = nullptr);
CGImageRef osx_ttb_image( gool::bitmap* bmp, gool::graphics* gfx = nullptr);

void osx_graphics_draw_image( osx::graphics* gfx, gool::image* img, const gool::rectf&  dst, const gool::rect& src, byte opacity);
void osx_graphics_tile_image( osx::graphics* gfx, gool::image* img, const gool::rectf&  dst, const gool::rect& src, const gool::point& offset, const gool::size& celldim);
void osx_graphics_fill_gradient(osx::graphics* gfx,  const gool::brush* bin, const gool::rect& dst);

uint osx_get_glyph_indices( osx::font* pf, slice<wchar> text , uint16* glyphs, uint glyphs_length );
void osx_get_glyph_advances( osx::font* pf, tool::slice<uint16> glyphs, float* xoffsets );

void osx_graphics_set_offset(osx::graphics* gfx, gool::point off );

void osx_graphics_push_layer(osx::graphics* gfx, const gool::rect& rc, byte opacity);
void osx_graphics_push_layer(osx::graphics* gfx, const gool::path* rc, byte opacity);
void osx_graphics_push_layer(osx::graphics* gfx, const gool::bitmap* rc, bool draw1a, byte opacity);

void osx_graphics_pop_layer(osx::graphics* gfx, byte opacity);

void osx_graphics_push_clip(osx::graphics* gfx, const gool::rect& rc);
void osx_graphics_pop_clip(osx::graphics* gfx);

void osx_graphics_set_transform(osx::graphics* gfx, const gool::affine_mtx_f& m);
void osx_graphics_get_transform(const osx::graphics* gfx, gool::affine_mtx_f& m);
void osx_graphics_reset_transform(osx::graphics* gfx);
void osx_graphics_push_state(osx::graphics* gfx);
void osx_graphics_pop_state(osx::graphics* gfx);

void osx_graphics_set_line_join( osx::graphics* gfx, gool::LINE_JOIN ncs );
void osx_graphics_set_line_cap( osx::graphics* gfx, gool::CAP_STYLE ncs );
void osx_graphics_set_stroke_style( osx::graphics* gfx, gool::DASH_STYLE ncs );
void osx_graphics_set_stroke_color( osx::graphics* gfx, gool::argb c );
void osx_graphics_set_stroke_width( osx::graphics* gfx, float w );
void osx_graphics_set_stroke_pattern( osx::graphics* gfx, slice<float> pattern, float offset );

void osx_graphics_line_v(osx::graphics* gfx, const gool::rect& rc, gool::argb c, int stroke, int step);
void osx_graphics_line_h(osx::graphics* gfx, const gool::rect& rc, gool::argb c, int stroke, int step);

//void osx_graphics_draw_line(osx::graphics* gfx, gool::pointf start, gool::pointf end );
//void osx_graphics_draw_rectangle(osx::graphics* gfx, gool::pointf start, gool::pointf end );
//void osx_graphics_draw_path( osx::graphics* gfx, const gool::path* dst, bool stroke, bool fill);
//void osx_graphics_draw_ellipse(osx::graphics* gfx, gool::pointf center, gool::sizef radius);

void osx_graphics_translate( osx::graphics* gfx, gool::pointf pt);
void osx_graphics_rotate( osx::graphics* gfx, float radians, gool::pointf center);
void osx_graphics_scale( osx::graphics* gfx, gool::sizef sz, gool::pointf center);
void osx_graphics_skew( osx::graphics* gfx, gool::sizef sz, gool::pointf center);

void osx_graphics_set_fill_color( osx::graphics* gfx, gool::argb c );
void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::linear_brush& lb );
void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::radial_brush& rb );
void osx_graphics_set_fill_brush( osx::graphics* gfx, const gool::image* ib );
void osx_graphics_set_fill_even_odd( osx::graphics* gfx, bool even_odd /*false - non-null*/ );
void osx_graphics_draw_ellipse(osx::graphics* gfx, gool::pointf center, gool::sizef radius);



namespace osx
{
  
  class graphics;

  struct font: public gool::font
  {
    typedef gool::font super;
    
    font():super(), _ct_font(NULL) {}
    font(const font& f):super(f) { _ct_font = f._ct_font;  }
    virtual ~font() { /*CGFontRelease(_ct_font)*/; }

    virtual bool has_glyph_for(uint ucodepoint) const override;
    virtual uint glyph_index(uint ucodepoint) const override;
    virtual bool glyph_metrics(::uint16 glyph_index, float em_size, float& width, float& height) const  override;

    mutable CTFontRef _ct_font;
    
    CTFontRef ct_font() const { return _ct_font; }

  };

  struct brush : public resource {
    brush() {}
    virtual ~brush() {}
    
    virtual bool is_solid() const { return false; }
    
    virtual void draw( graphics* context ) = 0;
    tristate_v even_odd;
  };

  struct solid_brush : public brush {
    
    CF::ref<CGColorRef> color;
    
    solid_brush(gool::argb c);
    
    virtual bool is_solid() const override { return true; }
    
    virtual void draw( graphics* context );
  };

  struct gradient_brush : public brush
  {
    CF::ref<CGGradientRef> gradient;
    gradient_brush( const gool::gradient_brush& bin );
    affine_mtx_f mx;
  };

  struct linear_gradient_brush : public gradient_brush
  {
    gool::point start, end;
    linear_gradient_brush( const gool::linear_brush& bin  ): gradient_brush(bin) { start = bin.start; end = bin.end; }
    
    virtual void draw( graphics* context );
  };

  struct radial_gradient_brush : public gradient_brush
  {
    gool::pointf center;
    gool::sizef  radius;
    radial_gradient_brush( const gool::radial_brush& bin  ): gradient_brush(bin) { center = bin.center; radius = bin.radius;  }
    
    virtual void draw( graphics* context );
  };

  struct image_brush : public brush {
    handle<image> img;
    image_brush(const image* im): img(im) {}
    virtual void draw( graphics* context );
  };

  struct state_registers
  {
    DASH_STYLE        stroke_style;
    float             stroke_width;
    handle<brush>     stroke_brush;
    
    CAP_STYLE         cap_style;
    LINE_JOIN         line_join;
    DASH_STYLE        dash_style;
    array<float>      dash_pattern;
    
    handle<brush>     fill_brush;
    bool              fill_even_odd;
    
    state_registers()
    : fill_even_odd(true)
    , stroke_style()
    , cap_style()
    , line_join()
    , dash_style()
    , stroke_width()
    {}
    /*state_registers(const state_registers& rs)
    : stroke_style(rs.stroke_style)
    , stroke_brush(rs.stroke_brush)
    , stroke_width(rs.stroke_width)
    , cap_style(rs.cap_style)
    , line_join(rs.line_join)
    , fill_brush(rs.fill_brush)
    , fill_even_odd(rs.fill_even_odd)
    , dash_style(rs.dash_style)
    {}*/
    
    bool drawable() const { return stroke_drawable() || fill_drawable(); }
    bool stroke_drawable() const { return stroke_brush.is_defined() && stroke_width > 0.0f; }
    bool fill_drawable() const { return fill_brush.is_defined(); }
  };


  class graphics : public gool::graphics
  {
    //friend class popup;
    friend class view;
    //friend class application;
    typedef gool::graphics super;
  protected:
    gool::point      _offset;
    CGContextRef     _target;
    graphics(bool transparent): super(transparent), _target(NULL) {}
  public:
    graphics(CGContextRef target, bool transparent): super(transparent) {
      _target = target;//CGContextRetain(target);
        CF::ref<CGColorSpaceRef> cp = CGColorSpaceCreateDeviceRGB();
                                      //CGColorSpaceCreateWithName(kCGColorSpaceGenericRGBLinear);
                                      //CGColorSpaceCreateWithName(kCGColorSpaceGenericRGB);
     
      CGContextSetFillColorSpace ( _target, cp );
      CGContextSetStrokeColorSpace ( _target, cp );
      /*CGContextSetShouldAntialias(_target, false);*/
    }
    virtual ~graphics() {
      while (pop_state());
      //CGContextRelease(_target);
    }
      
    virtual void finalize() override {} // this one gets created on stack

    CGContextRef target() const { return _target; }

    virtual path*         create_path() const override;
    virtual gool::application*  app() const override { return gool::app(); }
    
    
    virtual void            image_rendering_mode( IMAGE_RENDERING irm) override;
    virtual IMAGE_RENDERING image_rendering_mode() const override;

    virtual void fill( argb c, const rect& dst) override { return osx_graphics_fill( this, c, dst ); }
    virtual void fill( argb c, const rect& dst, const size& radius) override { return osx_graphics_fill_rr( this, c, dst, (radius.x+radius.y) / 2 ); }
    virtual void fill( argb c, const gool::path* dst) override { return osx_graphics_fill( this, c, dst ); }
    virtual void fill( const gool::brush* lb, const rect& dst) { osx_graphics_fill_gradient(this, lb, dst); }
    virtual void fill( image* img, const rect& dst, const rect& src,point offset = point(0,0), size celldim = size()) { osx_graphics_tile_image( this, img, dst,src, offset, celldim ); }
    virtual void fill( image* img, const rect& dst, point offset = point(0,0)) { fill(img, dst,rect(img->dim()),offset); }
    virtual void fill( image* img, const gool::path* dst) override;

    virtual void draw( const gool::path* dst, argb c, float width = 1.0f ) override { return osx_graphics_stroke( this, c, width, dst ); }

    virtual void draw_line_v(const rect& rc, argb c, int stroke, int step, int_v off) override { osx_graphics_line_v(this, rc,c,stroke,step); }
    virtual void draw_line_h(const rect& rc, argb c, int stroke, int step, int_v off) override { osx_graphics_line_h(this, rc,c,stroke,step); }

    virtual void draw_line(pointf p1, pointf p2 , argb c, int width) override;

    virtual bool push_clip(const rect& r, clipper* cl) override { osx_graphics_push_clip(this, r); super::push_clip(r,cl);  return true;  }
    virtual void pop_clip() { super::pop_clip(); osx_graphics_pop_clip(this); }

    virtual point offset(point noff);

    virtual bool set_antialiasing(bool onoff) override { return true; }

    virtual uint push_state() override; //{ osx_graphics_push_state( this ); return state_level++; }
    virtual bool pop_state() override;  //{ assert( state_level > 0 ); if(!state_level) return false; osx_graphics_pop_state( this ); --state_level; return true; }
    virtual void restore_state(uint level) override; //{ while( state_level > level ) pop_state(); }

    array<byte> _layers_stack;

    virtual void push_layer(const gool::rect& clip, byte opacity = 255, function<bool(filter_graph_builder*)> filters = nullptr) override { 
      _layers_stack.push(opacity);
      //super::push_layer(clip, opacity, filters); 
      osx_graphics_push_layer( this, clip,opacity ); 
    }
    virtual void push_layer(const gool::path* clip, byte opacity = 255) override { 
      _layers_stack.push(opacity);
      //super::push_layer(clip, opacity); 
      osx_graphics_push_layer( this, clip,opacity ); 
    }
    virtual void push_layer(const gool::bitmap* clip, bool draw1a, byte opacity = 255) override {
      _layers_stack.push(opacity);
      //super::push_layer(clip, opacity);
      osx_graphics_push_layer( this, clip, draw1a, opacity );
    }
    
    virtual void pop_layer() override { 
      byte opacity = _layers_stack.pop();
      osx_graphics_pop_layer( this, opacity );
    }
    virtual uint n_layers() override {
      return uint(_layers_stack.length());
    }

    virtual void transform(const affine_mtx_f& m) override        { osx_graphics_set_transform( this, m ); }
    virtual affine_mtx_f transform() const override               { affine_mtx_f m; osx_graphics_get_transform( this, m ); return m; }
      virtual void reset_transform() override                     { osx_graphics_reset_transform( this ); }
      
    //|
    //| Graphin stuff
    //| 

    virtual    CAP_STYLE cap_style() override                     { return _registers.cap_style; }
    virtual         void cap_style(CAP_STYLE ncs) override        { osx_graphics_set_line_cap (this, _registers.cap_style = ncs ); }
    virtual    LINE_JOIN line_join() override                     { return _registers.line_join; }
    virtual         void line_join(LINE_JOIN ncs, float mitter_limit = 4) override        { osx_graphics_set_line_join( this, _registers.line_join = ncs ); }
    virtual DASH_STYLE   dash_style() override                    { return _registers.stroke_style; }
    virtual         void dash_style( DASH_STYLE st ) override     { osx_graphics_set_stroke_style( this, _registers.stroke_style = st ); }
    virtual         void custom_dash_style(slice<float> pattern, float offset) override { osx_graphics_set_stroke_pattern( this, pattern, offset ); }

    virtual void set_stroke( argb c = argb::no_color()) override ;
    virtual void set_stroke( const linear_brush& lb ) override;
    virtual void set_stroke( const radial_brush& rb ) override;
    virtual void set_stroke_width( float w ) override;

    virtual void set_fill( argb c = argb::no_color() ) override;
    virtual void set_fill( const linear_brush& lb ) override;
    virtual void set_fill( const radial_brush& rb ) override;
    virtual void set_fill( const image* ib ) override ;
    virtual void set_fill_even_odd( bool even_odd ) ;

    virtual void draw_ellipse(pointf center, sizef radius, bool stroke = true, bool fill = true)  override;
    virtual void draw_arc(pointf center, sizef radius, float angle_start, float angle_sweep, bool stroke = true, bool fill = true) override;
    virtual void draw_rectangle(pointf org, sizef dim, bool stroke = true, bool fill = true)  override ;
    //virtual void draw_rectangle(rectf r)  override { draw_rectangle(r.origin, r.corner); }
    virtual void draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr, sizef rbr, sizef rbl, bool stroke = true, bool fill = true) override { super::draw_rectangle(org, dim, rtl, rtr,rbr,rbl,stroke,fill); }
    virtual void translate(pointf pt) override                                  { osx_graphics_translate( this, pt); }
    virtual void rotate(float radians, pointf center = pointf()) override       { osx_graphics_rotate( this, radians, center); }
    virtual void scale(sizef sz, pointf center = pointf()) override             { osx_graphics_scale( this, sz, center); }
    virtual void skew(sizef sz, pointf center = pointf()) override              { osx_graphics_skew( this, sz, center); }

    virtual pointf world_to_screen(pointf pt) const override { return osx_graphics_world_to_screen(this,pt); }
    virtual float  world_to_screen(float scalar) const override { return super::world_to_screen(scalar); }

    virtual pointf screen_to_world(pointf pt) const override { return osx_graphics_screen_to_world(this,pt); }
    virtual float  screen_to_world(float scalar) const override { return super::screen_to_world(scalar); }

    virtual void draw_path( const gool::path* dst, bool stroke = true, bool fill = true) override;
    virtual void draw_path( const gool::polygonf& p, bool stroke = true, bool fill = true) override;
    virtual void draw_line(pointf start, pointf end) override;
    virtual void draw_line(pointf origin, pointf end, line_end_style sorigin, line_end_style send) override { super::draw_line(origin, end, sorigin, send); }


  //protected:
    virtual void do_draw( image* img, const rectf&  dst, const rect& src, byte opacity = 255) { osx_graphics_draw_image( this, img, dst, src, opacity); }


    //juce_image

    //handle<gool::path>        _path;

    state_registers           _registers;
    array<state_registers>    _registers_stack;
    //uint state_level;


    void render_current_path(bool fill = true, bool stroke = true);

  };

  class path : public gool::path
  {
    public:
        path():evenodd(true) {}
        virtual ~path() {}
      
        virtual void set(const polygonf& vertices, bool even_odd = true);
        virtual void set(const polygonf& v1,const polygonf& v2);
        
        virtual bool is_inside(pointf pt);
        
        virtual void start(pointf start, bool filled) override;
        virtual void move_to(pointf pt, bool rel = false) override;
        virtual void line_to(pointf pt, bool rel = false) override;
        //virtual void hline_to(float x, bool rel = false) override { line_to(pointf(x,rel?0:lastp().y),rel); }
        //virtual void vline_to(float y, bool rel = false) override { line_to(pointf(rel?0:lastp().x,y),rel); }
        virtual void add_arc(pointf c, sizef r, float angle_start, float angle_swipe) override;
        virtual void quadratic_to(pointf pt, pointf cp, bool rel = false) override;
        virtual void cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel = false) override; // bezier
        virtual void set_even_odd(bool yes) override;
        virtual void close() override;
        virtual bool is_empty() const  override;
      
        virtual void reset() override;
        virtual void seal() override;
      
        CGMutablePathRef cgpath() const { return _path; }
        tristate_v even_odd() const { return evenodd; }

        virtual rectf bounds() const override;
    
        virtual pointf lastp() override;
    
        virtual gool::path* combine(COMBINE_MODE mode, const gool::path* other) override;
    
    protected:
        CF::ref<CGMutablePathRef> _path;
        tristate_v evenodd;
  };


  inline gool::path* graphics::create_path() const { return new osx::path(); }

  inline void graphics::draw_path( const gool::polygonf& p, bool stroke, bool fill) {
      path pa; pa.set(p);
      this->draw_path(&pa,stroke,fill);
  }

  class bitmap_graphics: public graphics
  {
    typedef graphics super;
    tool::handle<gool::bitmap> _bitmap;
    array<argb> pixels;
    public:
      virtual bool is_bitmap_graphics()const override { return true; }
      virtual void finalize() override;// { gool::graphics::finalize(); }
      
      bitmap_graphics(gool::bitmap *bmp, gool::argb initc);
      
  };

  class bitmap_bits_graphics: public bitmap_graphics
  {
//    tool::handle<gool::bitmap> _bitmap;
  public:
//    virtual void finalize() override;// { gool::graphics::finalize(); }
    
    bitmap_bits_graphics(gool::bitmap *bmp, gool::argb initc) : bitmap_graphics(bmp, initc) {}
    
  };

  




}

#endif
