//
//  osx-cf-ref.h
//  sciter-osx
//
//  Created by andrew on 2014-03-25.
//  Copyright (c) 2014 andrew fedoniouk. All rights reserved.
//

#ifndef sciter_osx_osx_cf_ref_h
#define sciter_osx_osx_cf_ref_h

#include "CoreFoundation/CoreFoundation.h"

namespace CF
{
    template <class T> class ref
    {
    public:
        
        // Function Members ----------------------------------------------------------------------
        
        ref (T cfo = NULL) : fRef (cfo) { }
        // Construct a temporary reference for 'cfo' or NIL.
        // This constructor does NOT raise the use count of 'cfo'. It is assumed that
        // 'cfo' already carries a use count (which is the case when the object was
        // just created and returned by an AcquireXXX method).
        
        ref (const ref& src)
        // Copy constructor. This constructor DOES raise the use count.
        : fRef (src.fRef) {
            if (fRef)
                ::CFRetain (fRef);
        }
        
        ~ref() {
            // Free the reference. To avoid any chance of recursive access to the
            // reference while it is being freed, the pointer is set to NIL first.
            T cfo = fRef;
            fRef = NULL;
            if (cfo)
                ::CFRelease (cfo);
        }
        
        ref& operator= (T cfo) {
            // Assign a new reference to be owned and free the old reference.
            // Does NOT raise the use count of 'cfo', see comment for TempCFRef (T cfo)
            // above.
            T old = fRef;
            fRef = cfo;
            if (old)
                ::CFRelease (old);
            return *this;
        }
        
        ref& operator= (const ref& src) {
            // Assign the source reference to be owned and free the old reference.
            // This assignment operator DOES raise the use count.
            T old = fRef;
            if ((fRef = src.fRef) != NULL)
                ::CFRetain (fRef);
            if (old)
                ::CFRelease (old);
            return *this;
        }
        
        T detach() {
            // Detach the reference from the owner, ownership is released.
            T cfo = fRef;
            fRef = NULL;
            return cfo;
        }
        
        //T acquire() {
        //    // Return the reference with an additional use count.
        //    if (fRef)
        //        ::CFRetain (fRef);
        //    return fRef;
        //}
        
        static ref copy(T fRef) {
            if (fRef)
               ::CFRetain (fRef);
            return ref(fRef);
        }
        
        operator T() const { return fRef; }
        // Return the pointer.
        
        T data() const { return fRef; }
        // If the operator above it not triggered automatically, writing
        // "temp.Data()" is probably more readable than "(T) temp",
        // especially if another cast is present.
        
        T* operator&() {
            T cfo = fRef;
            fRef = NULL;
            if (cfo)
                ::CFRelease (cfo);
            return &fRef;
        }
        
        operator bool() const { return fRef != 0; }
        
        // This operator is used to pass a pointer to the CF reference to some API
        // which will return a CF reference via pointer. To guarantee reference count
        // consistency, any old contents is release first.
        
        
        // Data Members --------------------------------------------------------------------------
    private:
        T fRef;	// The object reference
    };
    
}

extern tool::ustring cvt(CFStringRef theString);
extern CFStringRef   cfstr(tool::wchars text);
extern CFStringRef   cfstr(tool::chars text);



#endif
