//
//  SciterView.m
//  sciter-osx
//
//  Created by andrew on 2014-03-21.
//  Copyright (c) 2014 andrew fedoniouk. All rights reserved.
//

#import "SciterView.h"
#include "config.h"
#include "sdk-headers.h"
#include "osx-sciter-view.h"
#include "osx-sciter-application.h"
#include "api/api-defs.h"
#include "html/html.h"

gool::point get_mouse_pos(NSEvent* e, NSView* view);
uint get_alts(NSEvent* ev);
uint get_mouse_buttons();
gool::rect osx_iwindow_screen_pos(html::iwindow* iw);
gool::rect osx_screen_workarea(gool::rect rc);
gool::rect osx_screen_rect (gool::rect r);
gool::point osx_screen_point (gool::point pt);
gool::rect flipped_screen_rect (gool::rect r);
gool::size osx_pixels_per_inch(html::iwindow* iw);

gool::size  osx_iwindow_client_dim(html::iwindow* iw);
gool::size  osx_iwindow_window_dim(html::iwindow* iw);

html::clipboard::data* osx_get_clipboard_data(NSPasteboard* pb);
bool osx_set_clipboard_data(html::clipboard::data *cbdata, NSPasteboard *pasteboard );

extern NSView* viewUnderMouse;


void osx_post_event() {
    NSEvent* event = [NSEvent otherEventWithType: NSApplicationDefined
                                        location: NSMakePoint(0,0)
                                   modifierFlags: 0
                                       timestamp: 0.0
                                    windowNumber: 0
                                         context: nil
                                         subtype: 0
                                           data1: 0
                                           data2: 0];
    [NSApp postEvent: event atStart: YES];
}


NSDragOperation dd_operation( unsigned long op )
{
    if( op == html::dd_none )
        return NSDragOperationNone;
    uint ops = 0;
    if( op & html::dd_copy ) ops |= NSDragOperationCopy;
    if( op & html::dd_move ) ops |= NSDragOperationMove;
    if( op & html::dd_link ) ops |= NSDragOperationLink;
    return NSDragOperation(ops);
}

unsigned long dd_operations( NSDragOperation ops )
{
    if( ops == NSDragOperationNone )
        return html::dd_none;

    unsigned long r = 0;
    if( ops & NSDragOperationCopy ) r |= html::dd_copy;
    if( ops & NSDragOperationMove ) r |= html::dd_move;
    if( ops & NSDragOperationLink ) r |= html::dd_link;
    
    return r;
}


@implementation SciterView

//@synthesize mouseDownCanMoveWindow;

- (BOOL) thisThread {
    //return hview? hview->runloop == CFRunLoopGetCurrent() : YES;
    return [NSThread isMainThread];
}

- (id)initWithFrame:(NSRect)frame
//             ownsVm:(BOOL) ownVm
//          debugMode: (BOOL) dbgMode
         windowParams: (const window_params*) params
{
    self = [super initWithFrame:frame];
    
    if (self)
    {
        trackingArea = nullptr;
        
        markedText = [[NSMutableAttributedString alloc] init];
        
        // Initialization
        //html::view_creation_params params(html::FRAME_WINDOW);
        //params.owns_vm = ownVm;
        //params.debug_mode = dbgMode;
        hview = osx::app_factory()->create_window_processor(*params);
        if(hview) {
            hview->add_ref();
            hview->set_hwnd(self);
            //hview->load_html(CHARS("<html><body><div style='size:100px; background:yellow' /></body></html>"));
        }
        
        //preservesContentDuringLiveResize = NO;
        
        

        NSNotificationCenter*  notificationCenter = [NSNotificationCenter defaultCenter];
        
        [notificationCenter  addObserver: self
                                selector: @selector (frameChanged:)
                                    name: NSViewFrameDidChangeNotification
                                  object: self];
        
        [notificationCenter addObserver: self
                               selector: @selector(windowMouseDown:)
                                   name: @"mouseDown"
                                 object: self.window];
        
        [notificationCenter addObserver: self
                               selector: @selector(windowMouseUp:)
                                   name: @"mouseUp"
                                 object: self.window];
        
        [notificationCenter addObserver:self
                               selector:@selector(_update:)
                                   name:NSViewGlobalFrameDidChangeNotification
                                 object:self];
        
        [self setPostsFrameChangedNotifications: YES];
        
        [self registerForDraggedTypes:[NSArray arrayWithObjects:NSFilenamesPboardType, NSStringPboardType, NSHTMLPboardType, NSURLPboardType, nil]];
        
        [self setAcceptsTouchEvents:YES];
    }
    return self;
}

- (void)drawRect:(NSRect)dirtyRect
{
    if( !hview ) /*|| ![self.window isVisible] - WRONG window can be invisible at this time - offscreen rendering */
      [super drawRect:dirtyRect];
    else //if(hview->graphics_caps() != gool::SKIA_OPENGL_GRAPHICS)
    { // it uses render method below (request_render_layered) instead
        gool::rect rc = fromns<int>(dirtyRect);
#ifdef _DEBUG
        if(hview->type == html::POPUP_WINDOW)
          rc = fromns<int>(dirtyRect);
#endif
        //printf("drawRect %d %d %d %d\n", rc.left(), rc.top(), rc.width(), rc.height() );
        CGContextRef cg = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
        hview->draw(cg,rc);  // note without running animations
    }
 }

#ifdef USE_LAYERS
- (void)updateLayer
{
    if( hview )
    {
        //printf("updateLayer %d %d\n", hview->dim.x, hview->dim.y );
        CGContextRef cg = (CGContextRef)[[NSGraphicsContext currentContext] graphicsPort];
        hview->render(cg,hview->dim);
    }
    
}

- (NSViewLayerContentsPlacement) layerContentsPlacement {
    if( hview ) {
        return (NSViewLayerContentsPlacement) hview->get_resize_policy();
    }
    return NSViewLayerContentsPlacementScaleProportionallyToFill;
}

#endif

- (void)refresh
{
    if( hview /*&& [[self window] isVisible]*/) {
        NSView* pnv = (NSView*) hview->get_hwnd();
        if(pnv)
          [pnv setNeedsDisplay:TRUE];
    }
}

// -----------------------------------
// Release the View
// -----------------------------------

- (void)dealloc
{
   
    NSNotificationCenter*  notificationCenter = [NSNotificationCenter defaultCenter];
    [notificationCenter removeObserver: self];

    if( hview ) {
        handle<osx::view> pv = hview;
        pv->stop();
        pv->set_hwnd(nullptr);
        pv->release();
    }
    
    [markedText release];
    
    // call super
    [super dealloc];
}

// -----------------------------------
// (0,0) is at top left corner
// -----------------------------------

- (BOOL)isFlipped
{
    return TRUE;
}

- (BOOL)acceptsFirstMouse:(NSEvent *)theEvent
{
    return YES;
}


// -----------------------------------
// WM_SIZE kind of event
// -----------------------------------

- (void) frameChanged: (NSNotification*)evt
{
    
    if( hview ) {
        NSSize sz = self.frame.size;
//#ifdef _DEBUG
//        NSLog(@"frameChanged %f %f", sz.width, sz.height);
//#endif
        if( hview->uses_physical_size() )
          sz = [self convertSizeToBacking:sz];
        hview->on_size( fromns<int>( sz ) );
    }
}

- (void)updateTrackingAreas {
    if(trackingArea) {
      [self removeTrackingArea:trackingArea];
      [trackingArea release];
    }
        
    NSTrackingAreaOptions opts =
    //  NSTrackingMouseEnteredAndExited
    NSTrackingMouseMoved
    //| NSTrackingCursorUpdate
    | NSTrackingActiveAlways
    | NSTrackingInVisibleRect;
    
    trackingArea = [[NSTrackingArea alloc] initWithRect:self.frame
        options: opts
        owner:self userInfo:nil];
    [self addTrackingArea:trackingArea];

}

- (BOOL)isSciter
{
    return TRUE;
}

- (osx::view*) engine
{
    return hview;
}

- (BOOL) doesAllowVibrancy
{
    return TRUE;
}

#ifdef USE_LAYERS
- (BOOL) wantsUpdateLayer
{
    if(hview)
      return hview->graphics_caps() == gool::SKIA_OPENGL_GRAPHICS;
    return TRUE;
}
#endif

// -----------------------------------
// First Responder Methods
// -----------------------------------

- (BOOL)acceptsFirstResponder
{
    return YES;
}

- (void)mouseMoved:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = get_mouse_buttons();
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    auto_state<void*> _(hview->currentEvent,theEvent);
    
    if(hview->on_mouse(html::MOUSE_MOVE, buttons, alts, pt))
        return;
    
    [super mouseMoved:theEvent];
}
/*
- (void)mouseEntered:(NSEvent *)theEvent
{
    if(!hview)
      return;
    
    [[self window] setAcceptsMouseMovedEvents:YES];
    [[self window] makeFirstResponder:self];
    
    uint        buttons = get_mouse_buttons();
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);

    hview->setup_mouse_idle(true,pt);
    
    //printf("mouseEntered %d %d\n", pt.x, pt.y);
    if(hview->on_mouse(html::MOUSE_ENTER, buttons, alts, pt))
        return;
    
    [super mouseEntered:theEvent];

}

- (void)mouseExited:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = get_mouse_buttons();
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    hview->setup_mouse_idle(false,pt);
    
    //printf("mouseExited %d %d\n", pt.x, pt.y);
    if(hview->on_mouse(html::MOUSE_LEAVE, buttons, alts, pt))
        return;
    
    [super mouseExited:theEvent];
}
*/

- (void)mouseDown:(NSEvent *)theEvent
{
    if(!hview)
      return; 

    [[self inputContext] handleEvent:theEvent];
    
    uint        buttons = html::MAIN_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    //double        last_mouse_down_time;
    //point         last_mouse_down_point;
    
    //printf("mouseDown:%d %d\n", pt.x, pt.y);
    uint click_count = [theEvent clickCount];
    
    uint cmd = html::MOUSE_DOWN;
    if( click_count <= 1 );
    else if( click_count == 2 ) cmd = html::MOUSE_DCLICK;
    else cmd = html::MOUSE_TCLICK;

    auto_state<void*> _(hview->currentEvent,theEvent);
    
    if(hview->on_mouse(cmd,buttons, alts, pt))
    {
/*        if(hview && hview->mouse_down_on_caption && hview->supports_native_ui_window_move())
        {
          [self windowWillMove: nil];
          [[self window] performWindowDragWithEvent:theEvent];
        }*/
        return; // ???
    }
    
    [super mouseDown:theEvent];
}

- (void)mouseUp:(NSEvent *)theEvent
{
    if(!hview)
      return; 

    [[self inputContext] handleEvent:theEvent];
    
    uint        buttons = html::MAIN_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(uiActivatedMove)
      [self windowDidMove: nil];

    //uint click_count = [theEvent clickCount];
    
    uint cmd = html::MOUSE_UP;
    
    auto_state<void*> _(hview->currentEvent,theEvent);

    if(hview->on_mouse(cmd, buttons, alts, pt))
        return;

    [super mouseUp:theEvent];
}

- (void)mouseDragged:(NSEvent *)theEvent
{
    if(!hview)
      return; 

    [[self inputContext] handleEvent:theEvent];
    
    uint        buttons = html::MAIN_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    auto_state<void*> _(hview->currentEvent,theEvent);
    
    //printf("mouseDragged: %d %d\n", pt.x, pt.y);
    if(hview->on_mouse(html::MOUSE_MOVE, buttons, alts, pt))
        return;
    
    [super mouseDragged:theEvent];
}

- (void)otherMouseDown:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = html::MIDDLE_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(hview->on_mouse(html::MOUSE_DOWN, buttons, alts, pt))
        return;
    
    [super otherMouseDown:theEvent];
}

- (void)otherMouseUp:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = html::MIDDLE_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(hview->on_mouse(html::MOUSE_UP, buttons, alts, pt))
        return;
    
    [super otherMouseUp:theEvent];
}


- (void)rightMouseDown:(NSEvent *)theEvent
{
    if(!hview)
      return; 
   
    uint        buttons = html::PROP_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(hview->on_mouse(html::MOUSE_DOWN, buttons, alts, pt))
        return;
    
    [super rightMouseDown:theEvent];
}

- (void)rightMouseUp:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = html::PROP_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(hview->on_mouse(html::MOUSE_UP, buttons, alts, pt))
        return;
    
    [super rightMouseUp:theEvent];
}

- (void)rightMouseDragged:(NSEvent *)theEvent
{
    if(!hview)
      return; 
    
    uint        buttons = html::PROP_BUTTON;
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    if(hview->on_mouse(html::MOUSE_MOVE, buttons, alts, pt))
        return;
    
    [super rightMouseDragged:theEvent];
}

#ifdef USE_TOUCH
//$BOOL fromTrackPad = NO;
-(void)touchesBeganWithEvent:(NSEvent *)theEvent;
{
    if(!hview)
      return;
    
    uint        alts = get_alts(theEvent) | ALT_TOUCH;
    gool::point pt = get_mouse_pos(theEvent, self);
    
    //hview->on_mouse(html::MOUSE_TOUCH_START, 0, alts, pt);
    hview->handle_touch_start(ALT_TOUCH);
    //[super touchesBeganWithEvent:theEvent];
}
-(void)touchesCancelledWithEvent:(NSEvent *)theEvent;
{
    if(!hview)
        return;
      
    uint        alts = get_alts(theEvent) | ALT_TOUCH;
    gool::point pt = get_mouse_pos(theEvent, self);
      
    //hview->on_mouse(html::MOUSE_TOUCH_END, 0, alts, pt);
    hview->handle_touch_end(ALT_TOUCH);
    //[super touchesCancelledWithEvent:theEvent];
}
-(void) touchesEndedWithEvent:(NSEvent *)theEvent;
{
    if(!hview)
        return;
      
    uint        alts = get_alts(theEvent) | ALT_TOUCH;
    gool::point pt = get_mouse_pos(theEvent, self);
      
    //hview->on_mouse(html::MOUSE_TOUCH_END, 0, alts, pt);
    hview->handle_touch_end(ALT_TOUCH);
    //[super touchesEndedWithEvent:theEvent];
}
#endif

- (void)scrollWheel:(NSEvent *)theEvent {
    //NSLog(@"user scrolled %f horizontally and %f vertically", [theEvent deltaX], [theEvent deltaY]);
    
    if(!hview)
      return; 
    
    uint        alts = get_alts(theEvent);
    gool::point pt = get_mouse_pos(theEvent, self);
    
    uint  phase = (uint)[theEvent momentumPhase];

    //if(phase == NSEventPhaseNone)
    {
      gool::size delta;
        
      auto fromTouchPad = [theEvent hasPreciseScrollingDeltas];
#ifdef USE_TOUCH
      if(fromTouchPad)
        alts |= ALT_TOUCH;
#endif
        
      float deltaY = [theEvent scrollingDeltaY];
      float deltaX = [theEvent scrollingDeltaX];
        
      //NSLog(@"user scrolled %f horizontally and %f vertically", deltaX, deltaY);

      if( !fromTouchPad ) { // mouse wheel
          if( deltaY > 0.0001f ) deltaY = 120.0f;
          else if( deltaY < -0.0001f ) deltaY = -120.0f;
          else deltaY = 0;
          if( deltaX > 0.0001f ) deltaX = 120.0f;
          else if( deltaX < -0.0001f ) deltaX = -120.0f;
          else deltaX = 0;
          delta.x = deltaX;
          delta.y = deltaY;
      } else {
          delta.x = deltaX * self.window.backingScaleFactor;
          delta.y = deltaY * self.window.backingScaleFactor;
      }
        
      if(delta.x || delta.y )
        hview->on_mouse(html::MOUSE_WHEEL, tool::make_dword(short(delta.x), short(delta.y)), alts, pt);
    }
    //else if(phase & NSEventPhaseEnded){
    //    phase = phase;
    //}
    
}


//- (BOOL)wantsScrollEventsForSwipeTrackingOnAxis:(NSEventGestureAxis)axis {
//    return (axis == NSEventGestureAxisHorizontal) ? YES : NO;
//}

//- (BOOL)acceptsTouchEvents {
//    return YES;
//}

/*- (void)swipeWithEvent:(NSEvent *)event {
    CGFloat x = [event deltaX];
    CGFloat y = [event deltaY];
    NSLog(@"swipe %f %f\n",x,y);
  }

- (void)beginGestureWithEvent:(NSEvent *)event {
    NSLog(@"Gesture detected!");
}

- (void)endGestureWithEvent:(NSEvent *)event {
    NSLog(@"Gesture end detected!");
}

- (void)rotateWithEvent:(NSEvent *)event {
    NSLog(@"Gesture ROTATE detected!");
}

- (void)magnifyWithEvent:(NSEvent *)event {
    NSLog(@"Gesture MAGNIFY detected!");

 .}*/

- (void)keyDown:(NSEvent *)theEvent {

    uint key_code = [ theEvent keyCode ];
    uint alts = get_alts(theEvent);
    
    keyEventConsumed = false;
    
    //string s = cvt([theEvent characters]);
    //printf("key %x %x %s\n", key_code, alts, s.c_str());
    
    if( ![self isImeActive] ) {
        if (hview->on_key(html::KEY_DOWN, key_code, alts ))
            return;
    }
    
    if( hview->is_ime_enabled() )
      [[self inputContext] handleEvent:theEvent];
    
    if( !keyEventConsumed) {
        alts &= html::ALT_COMMAND;
        ustring us = cvt([theEvent characters]);
        u16::each_codepoint cpi(us);
        for(ucode c; cpi(c);)
          //if((c != 0) && ((c < 0xE000) || (c > 0xF8FF)))
          if(ucisgraph(c) || c == ' ' || c == '\t')
              hview->on_key(html::KEY_CHAR, c, alts);
    }
    
}

- (void)keyUp:(NSEvent *)theEvent {
    uint key_code = [ theEvent keyCode ];
    uint alts = get_alts(theEvent);
    hview->on_key(html::KEY_UP, key_code, alts);
    
}

- (void)windowWillMiniaturize:(NSNotification *)notification
{
    window_level = self.window.level;
    [self.window setLevel: NSNormalWindowLevel];
}

- (void)windowDidMiniaturize:(NSNotification *)notification
{
    assert([self thisThread]);
    if(hview) hview->handle_minimized();
}

- (void)windowDidDeminiaturize:(NSNotification *)notification
{
    assert([self thisThread]);
    
    if( window_level != 0 && window_level != NSNormalWindowLevel ) {
        [self.window setLevel: window_level];
    }
    
    if(hview) hview->handle_restored();
}

- (void)windowDidEnterFullScreen:(NSNotification *)notification
{
    assert([self thisThread]);
    if(hview) hview->handle_fullscreen();
    
    //[self.window makeKeyAndOrderFront: self];
    /*NSRect vr = self.window.frame;
    vr.origin.x = 0;
    vr.origin.y = 0;
    
    NSTrackingArea* ta = [NSTrackingArea alloc];
    
    NSTrackingAreaOptions opts = NSTrackingMouseEnteredAndExited |
    NSTrackingMouseMoved |
    NSTrackingCursorUpdate |
    NSTrackingActiveAlways |
    NSTrackingInVisibleRect;
    [ta initWithRect:vr options:opts owner:self.window userInfo:NULL];
    [self addTrackingArea:ta];
    [ta release];*/
}

- (void)windowDidExitFullScreen:(NSNotification *)notification
{
    assert([self thisThread]);
    if(hview) hview->handle_shown();
}

//- (void)windowDidExpose:(NSNotification *)notification
//{
//    if(hview) hview->handle_shown();
//}

- (BOOL)windowShouldClose:(id)sender
{
    assert([self thisThread]);
    if(hview)
      return hview->ask_unload(hview->doc(),html::view::UNLOAD_BY_CHROME) ? YES : NO;
    return YES;
}

//- (void)windowDidBecomeMain:(NSNotification *)notification
//{
//    assert([self thisThread]);
//    if(hview) hview->handle_shown();
//    //NSLog(@"window became main");
//}
- (void)windowDidChangeOcclusionState:(NSNotification *)notification
{
    if (self.window.occlusionState & NSWindowOcclusionStateVisible)
    {
        //NSLog(@"DDDDDD:window shown");
        if(hview) hview->handle_shown();
    }
    else
    {
        if(hview) hview->handle_hidden();
    }
}

- (void)windowDidBecomeKey:(NSNotification *)notification
{
    if(hview) {
      hview->on_activate(html::ACTIVATED);
      hview->on_focus( true );
    }
    //std::cout << isDialog << " got focus" << std::endl;
    //NSLog(@"DDDDDD:window got focus");
}

- (void)windowDidResignKey:(NSNotification *)notification
{
    if(hview) {
        hview->on_focus( false );
        hview->on_activate(html::INACTIVE);
    }
    //std::cout << isDialog << " lost focus" << std::endl;
    //NSLog(@"window lost focus");
}

- (void)windowWillClose:(NSNotification *)notification
{
    assert([self thisThread]);
    
    NSNotificationCenter*  notificationCenter = [NSNotificationCenter defaultCenter];
    [notificationCenter removeObserver: self];
    
    [[self window] setDelegate:nil];
    
    if( hview ) {
        handle<osx::view> pv = hview;
        pv->stop();
        pv->set_hwnd(nullptr);
        pv->release();
        hview = nullptr;
    }
    
    //if( isDialog ) {
    //   NSApplication * application = [NSApplication sharedApplication];
    //  [application stopModal];
    //}
    
    if( isMain ) {
      NSApplication * application = [NSApplication sharedApplication];
      [application stop: self];
    }
    
    //[self release];
}

/*- (NSRect) constrainFrameRect: (NSRect) frameRect toScreen:(NSScreen *)screen
{
    if(hview) {
        gool::rect rc = flipped_screen_rect(fromns<int>(frameRect));
        if(hview->on_move_request(rc))
        {
            return tons(osx_screen_rect(rc));
        }
    }
    return frameRect;
}*/

- (void)windowWillMove:(NSNotification *)notification
{
    if(!hview)
        return;
    
    assert([self thisThread]);

    window_moved = false;
    
    switch([NSEvent pressedMouseButtons]) {
        case 1 << 0: // left button
            if (!uiActivatedMove) {
                uiActivatedMove = true;
                hview->on_start_ui_replacement();
            }
            break;
    }
    
    gool::rect rc = hview->screen_place();
    if(hview->on_move_request(rc))
    {
        hview->move_window(rc);
    }
}

/*- (void)windowDidResize:(NSNotification *)notification
{
    if(!hview)
        return;
    size sz = hview->window_dim();
#ifdef _DEBUG
    NSLog(@"windowDidResize %d %d", sz.x, sz.y);
#endif
    hview->on_size(hview->client_dim());
}*/

- (void)windowDidMove:(NSNotification *)notification
{
//    std::cout << "window did move" << std::endl;
    if(!hview)
        return;

    assert([self thisThread]);
    
    
    hview->on_move();
    
    window_moved = true;
    
    switch([NSEvent pressedMouseButtons]) {
        case 0: // no buttons pressed
            if (uiActivatedMove) {
                uiActivatedMove = false;
                hview->on_end_ui_replacement();
            }
    }
}

- (void)windowWillStartLiveResize:(NSNotification *)notification
{
    if(!hview)
        return;
    
    assert([self thisThread]);
    hview->on_start_ui_replacement();
}

- (void)windowDidEndLiveResize:(NSNotification *)notification
{
    if(!hview)
        return;
    
    assert([self thisThread]);
    hview->on_end_ui_replacement();
}



- (void)windowDidChangeScreen:(NSNotification *)notification
{
    if(!hview)
        return;

    assert([self thisThread]);
    hview->on_media_changed();
    
}

- (void)windowDidChangeBackingProperties:(NSNotification *)notification
{
    if(!hview)
        return;
    hview->reset_resolution();
    hview->on_size(hview->client_dim());
    hview->on_media_changed();
}


- (void)setIsMain:(BOOL)onOff
{
    isMain = onOff;
}

//- (void)setIsDialog:(BOOL)onOff
//{
//    isDialog = onOff;
//}




// multi-delegate support
/*
- (void) setExtDelegate:(id) ed {
    extDelegate = ed;
}


- (BOOL)respondsToSelector:(SEL)selector {
    if ([super respondsToSelector:selector])
        return YES;
    
    if( extDelegate && ([extDelegate respondsToSelector:selector]))
       return YES;
    
    return NO;
}

- (NSMethodSignature *)methodSignatureForSelector:(SEL)selector {
    NSMethodSignature* signature = [super methodSignatureForSelector:selector];
    if (signature)
        return  signature;
    
   if(extDelegate)
       signature = [extDelegate methodSignatureForSelector:selector];
    
    return signature;
}

- (void)forwardInvocation:(NSInvocation *)invocation {
    SEL selector = [invocation selector];
    BOOL responded = NO;
    
    if(extDelegate) {
        if ([extDelegate respondsToSelector:selector]) {
            [invocation invokeWithTarget:extDelegate];
            responded = YES;
        }
    }
    
    if (!responded)
        [self doesNotRecognizeSelector:selector];
}*/


- (void)viewDidChangeBackingProperties {
    
    //NSLog(@"viewDidChangeBackingProperties");
    
    //[super viewDidChangeBackingProperties];
    if( !hview )
        return;
    assert([self thisThread]);
    gool::size dpi = osx_pixels_per_inch(hview);
    hview->on_dpi_changed(dpi,gool::rect());
}



- (void)update
{
    if( hview ) hview->update_request();
}

- (void)_update: (NSNotification*)evt
{
    [self update];
}


- (void) screenParametersChanged: (NSNotification*)evt
{
    assert([self thisThread]);
    hview->on_media_changed();
}

// drag-n-drop support, drop target

- (NSDragOperation)draggingEntered:(id<NSDraggingInfo>)sender
{
    NSPasteboard* pb = [sender draggingPasteboard];
    
    hview->dragging_data = osx_get_clipboard_data(pb);
    
    unsigned long dd_modes = dd_operations([sender draggingSourceOperationMask]);
    if(hview->on_drag_enter( dd_modes, hview->dragging_data, hview->cursor_pos(), nullptr))
      return dd_operation(dd_modes);
    return NSDragOperationNone;
}
- (void)draggingExited:(id<NSDraggingInfo>)sender
{
    hview->on_drag_leave(nullptr);
}

- (NSDragOperation)draggingUpdated:(id<NSDraggingInfo>)sender
{
    unsigned long dd_modes = dd_operations([sender draggingSourceOperationMask]);
    if(hview->on_drag_over( dd_modes, hview->dragging_data, hview->cursor_pos(), nullptr))
      return dd_operation(dd_modes);
    return NSDragOperationNone;
}

- (BOOL)prepareForDragOperation:(id<NSDraggingInfo>)sender
{
    return YES;
}

- (BOOL)performDragOperation:(id<NSDraggingInfo>)sender
{
    unsigned long dd_modes = dd_operations([sender draggingSourceOperationMask]);
    return hview->on_drop( dd_modes, hview->dragging_data, hview->cursor_pos(), nullptr) ? YES : NO;
}

// NSDraggingSource

- (void)draggingSession:(NSDraggingSession *)session
           endedAtPoint:(NSPoint)screenPoint
              operation:(NSDragOperation)operation
{
    if(operation && hview)
      hview->dd_completion = dd_operations( operation );
}

- (NSDragOperation)draggingSession:(NSDraggingSession *)session
sourceOperationMaskForDraggingContext:(NSDraggingContext)context
{
    if(hview)
      return dd_operation( (unsigned long) hview->dd_operation );
    return NSDragOperationNone;
}

- (BOOL)trayiconClick:(id)sender
{
    if(hview) {
       uint buttons = (uint)[NSEvent pressedMouseButtons];
       NSPoint npt = [NSEvent mouseLocation];
       gool::point pt = gool::point(npt.x,npt.y);
       pt = osx_screen_point(pt);
       return hview->trayicon_notify(pt, buttons, html::MOUSE_CLICK);
    }
   return FALSE;
}

/*- (BOOL) mouseDownCanMoveWindow
{
    NSLog(@"mouseDownCanMoveWindow %d", int(self->_mouseDownCanMoveWindow));
    return //self->_mouseDownCanMoveWindow;
      YES;
}
- (void) setMouseDownCanMoveWindow : (BOOL) onOff
{
    self->_mouseDownCanMoveWindow = onOff;
}*/

/*
- (NSView *)hitTest:(NSPoint)aPoint
{
    //NSLog(@"hitTest %d %d", int(aPoint.x), int(aPoint.y));
    if( self->_mouseDownCanMoveWindow ) {
        NSLog(@"hitTest nil %d %d", int(aPoint.x), int(aPoint.y));
        return nil;
    }
    return self;
}*/

- (BOOL)isImeActive
{
    return [markedText length] > 0 ? YES : NO;
}

// NSTextInputClient stuff:

- (BOOL)hasMarkedText
{
    return [markedText length] > 0;
}

- (NSRange)markedRange
{
    if ([markedText length] > 0)
        return NSMakeRange(0, [markedText length] - 1);
    else
        return {NSNotFound,0};
}

- (NSRange)selectedRange
{
    return {NSNotFound,0};
}


- (void)unmarkText
{
    [[markedText mutableString] setString:@""];
    [[self inputContext] discardMarkedText];
    //keyEventConsumed = TRUE;
}

- (NSArray*)validAttributesForMarkedText
{
    return [NSArray array];
}

- (NSAttributedString*)attributedSubstringForProposedRange:(NSRange)range
                                               actualRange:(NSRangePointer)actualRange
{
    //if(actualRange)
    //    *actualRange = range;
    //
    //return [markedText attributedSubstringFromRange:range];
    return nil;
}

- (NSUInteger)characterIndexForPoint:(NSPoint)point
{
    return 0;
}

- (NSRect)firstRectForCharacterRange:(NSRange)range
                         actualRange:(NSRangePointer)actualRange
{
    gool::rect rc;
    if(hview && hview->focus_element && hview->focus_element->get_caret_location(*hview, rc)) {
        if(actualRange)
            *actualRange = NSMakeRange(0,0);
        rc += hview->focus_element->screen_pos(*hview);
        return tons(osx_screen_rect(rc));
    }
    if(actualRange)
       *actualRange = NSMakeRange(NSNotFound,0);
    
    return NSMakeRect(0.0, 0.0, 0.0, 0.0);
}

- (void)setMarkedText:(id)string
        selectedRange:(NSRange)selectedRange
     replacementRange:(NSRange)replacementRange
{
    if ([string isKindOfClass:[NSAttributedString class]])
        [markedText initWithAttributedString:string];
    else
        [markedText initWithString:string];
    
    NSString* str;
    if ([string isKindOfClass:[NSAttributedString class]])
        str = [string string];
    else
        str = (NSString*) string;
    hview->on_ime_chars(false, cvt(str));

    keyEventConsumed = true;
    //printf("setMarkedText <%s> [%d,%d] [%d,%d]\n", tool::string(cvt(str)).c_str(), replacementRange.location,replacementRange.length, selectedRange.location,selectedRange.length);
    
}


- (void)insertText:(id)string replacementRange:(NSRange)replacementRange
{
    NSString* characters;
    NSEvent* ev = [NSApp currentEvent];
    uint alts = get_alts(ev);
    
    if ([string isKindOfClass:[NSAttributedString class]])
        characters = [string string];
    else
        characters = (NSString*) string;
    
    ustring us = cvt(characters);
    
    if( [markedText length] ) {
      hview->on_ime_chars(true, us);
      [[markedText mutableString] setString:@""];
    }
    else {
      alts &= html::ALT_COMMAND;
      u16::each_codepoint cpi(us);
      for(ucode c; cpi(c);)
        //if((c != 0) && ((c < 0xE000) || (c > 0xF8FF)))
        if(ucisgraph(c) || c == ' ' || c == '\t')
          hview->on_key(html::KEY_CHAR, c, alts);
    }

    keyEventConsumed = true;
    
    //printf("insertText <%s> [%d,%d] buffer length = %d\n", tool::string(cvt(characters)).c_str(), replacementRange.location,replacementRange.length, [markedText length]);
}

- (void)doCommandBySelector:(SEL)selector
{
    [super doCommandBySelector:selector];
}

- (NSInteger) windowLevel {
    return [[self window] level];
}

@end

@implementation SciterWindow

  - (BOOL) canBecomeKeyWindow {
      
      NSView * cv = self.contentView;
      
      if( ![cv isKindOfClass:[SciterView class]] )
          cv = [[cv subviews] objectAtIndex:0];
      
      assert ([cv isKindOfClass:[SciterView class]]);
      
      SciterView* sv = (SciterView*) cv;
      return (sv && sv.engine && sv.engine->is_enabled()) ?  YES : NO;
      //return YES;
  }
  - (BOOL) canBecomeMainWindow { return YES; }
  - (BOOL) acceptsFirstResponder { return YES; }
  - (BOOL) becomeFirstResponder { return YES; }
  - (BOOL) resignFirstResponder { return YES; }

@end

@implementation SciterPanelWindow

- (BOOL) canBecomeKeyWindow {
    
      //return YES;

      NSView * cv = self.contentView;
    
      if( ![cv isKindOfClass:[SciterView class]] )
          cv = [[cv subviews] objectAtIndex:0];
      
      assert ([cv isKindOfClass:[SciterView class]]);
      
      SciterView* sv = (SciterView*)cv;
      return (sv && sv.engine && sv.engine->is_enabled()) ?  YES : NO;
      //return YES;
  }
  - (BOOL) canBecomeMainWindow { return YES; }
  - (BOOL) acceptsFirstResponder { return YES; }
  - (BOOL) becomeFirstResponder { return YES; }
  - (BOOL) resignFirstResponder { return YES; }

@end



EXPORT EXTERN_C
BOOL SciterIsSciter( HWINDOW hw ) {
    id vv = (id) hw;
    return vv && [ vv class] == [SciterView class];
}

gool::size  osx_screen_dim();

osx::view* osxview( HWINDOW hw ) {
    SciterView* nv = (SciterView*)hw;
    return nv? [nv engine] : nullptr;
}

EXPORT HWINDOW SciterCreateNSView_api(LPRECT prect)
{
    NSRect frame; frame.origin.x = 0; frame.origin.y = 0; frame.size.width = 400; frame.size.height = 300;
    
    if(prect) //frame = *pframe;
    {
        frame.size.width = prect->right - prect->left; frame.size.height = prect->bottom - prect->top;
        frame.origin.x = prect->left;
        frame.origin.y = osx_screen_dim().y - prect->bottom - 1;
    }
    
    NSRect vr = frame;
    
    vr.origin.x = 0;
    vr.origin.y = 0;
   
    SciterView* nv = [ SciterView alloc ];
    if( nv ) {
        
      window_params cparams(gool::CHILD_WINDOW);
      [nv initWithFrame:vr windowParams: &cparams ];
      osxview(nv)->start(window_params(html::CHILD_WINDOW));
      //[nv initWithFrame:vr ownsVm:NO debugMode:NO];
    }
    else
      return NULL;
    
    return nv;
}

EXPORT BOOL SciterProcX_api(HWINDOW hwnd, SCITER_X_MSG* pMsg) {
    return FALSE;
}


void setup_window(NSWindow *nw, NSView* nv, osx::view* pv)
{
    bool needs_layered = false;
    bool custom_frame = false;
    
    switch(pv->get_frame_type()) {
        case html::LAYERED:
        {
          [nw setOpaque:NO];
          [nw setBackgroundColor: [NSColor clearColor]];
          [nw setHasShadow: NO];
          needs_layered = true;
          custom_frame = true;
        } break;
//        case html::GLASSY:
//        {
//          //[window setAlphaValue:1.0];
//          [nw setOpaque:NO];
//          [nw setHasShadow: YES];
////          Class vibrantClass = NSClassFromString(@"NSVisualEffectView");
////          if (vibrantClass)
////          {
////             [nw setBackgroundColor: [NSColor clearColor]];
////             NSView *vibrant=[[vibrantClass alloc] initWithFrame:nv.bounds];
////             [vibrant setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable];
////             [vibrant setBlendingMode: 0 /*NSVisualEffectBlendingModeBehindWindow*/];
////             [nv addSubview:vibrant positioned:NSWindowBelow relativeTo:nil];
////          } else
//             [nw setBackgroundColor: [NSColor colorWithCalibratedRed:0.4 green:0.4 blue:0.4 alpha: 0.5 ]];
//
//        } break;
    
        case html::SOLID:
        {
           custom_frame = true;
           [nw setHasShadow: NO];
           [nw setBackgroundColor: [NSColor whiteColor]];
        } break;
        case html::SOLID_WITH_SHADOW:
        {
           custom_frame = true;
           [nw setHasShadow: YES];
           [nw setBackgroundColor: [NSColor whiteColor]];
        } break;
        case html::STANDARD:        {
           [nw setOpaque: YES];
           [nw setHasShadow: YES]; //((windowStyleFlags & windowHasDropShadow) != 0)];
        }
    }
        
    int wlevel = NSNormalWindowLevel;
    
    switch(pv->type)
    {
       case html::POPUP_WINDOW:  wlevel = NSPopUpMenuWindowLevel; break;
       case html::TOOL_WINDOW:   wlevel = NSFloatingWindowLevel; break;
       case html::DIALOG_WINDOW: wlevel = NSModalPanelWindowLevel; break;
       default: break;
    }
    
    if(auto ppv = pv->parent())
    {
        int ppl = [[((NSView*)ppv->get_hwnd()) window] level];
        if( wlevel <= ppl ) wlevel = ppl + 1;
    }

    [nw setLevel: wlevel];
    
    if(pv->is_layered() != needs_layered)
      pv->set_layered(needs_layered);
    
    //if( custom_frame )
    //    osx::window_controls::setup(*pv);
    
}


EXPORT HWINDOW SciterCreateWindow_api(UINT creationFlags,
                           LPRECT prect,
                           SciterWindowDelegate* delegate,
                           LPVOID delegateParam,
                           HWINDOW parent)
{
    NSRect frame; frame.origin.x = 0; frame.origin.y = 0; frame.size.width = 400; frame.size.height = 300;

    if(prect) //frame = *pframe;
    {
        frame.size.width = prect->right - prect->left; frame.size.height = prect->bottom - prect->top;
        frame.origin.x = prect->left;
        frame.origin.y = osx_screen_dim().y - prect->bottom + 1;
    } else {
        frame = NSInsetRect([[NSScreen mainScreen] frame], 100,100);
    }
    
    NSRect vr = frame;
    
    vr.origin.x = 0;
    vr.origin.y = 0;
    
    window_params cparams;
    if(creationFlags & SW_OWNS_VM)
      cparams.owns_vm = TRUE;
    if(creationFlags & SW_ENABLE_DEBUG)
      cparams.debug_mode = TRUE;
    
    SciterView* nv = [ SciterView alloc ];
    
    
    if( nv ) {
      [nv initWithFrame:vr windowParams: &cparams ];
    }
    else
      return NULL;
    
    html::FRAME_TYPE ft = html::STANDARD;
    
    if( (creationFlags & SW_CHILD) == 0 ) {
        
        unsigned int window_style = NSMiniaturizableWindowMask | NSClosableWindowMask | NSResizableWindowMask;
        if( creationFlags & SW_TITLEBAR ) {
            window_style |= NSTitledWindowMask;
            ft = html::STANDARD;
        }
        if( creationFlags & SW_RESIZEABLE ) {
            window_style |= NSMiniaturizableWindowMask | NSClosableWindowMask | NSResizableWindowMask;
            [nv engine]->set_resizeable(true);
        } else {
            window_style &= ~NSResizableWindowMask;
            [nv engine]->set_resizeable(false);
        }
        if( creationFlags & SW_ALPHA ) {
            ft = html::LAYERED;
            window_style |= NSBorderlessWindowMask;
            [nv engine]->set_layered(true);
        }

        NSWindow* window;
        
        if( creationFlags & SW_POPUP )
            window = [SciterPanelWindow alloc];
        else
            window = [SciterWindow alloc];
        
        [nv setIsMain: ((creationFlags & SW_MAIN) != 0 ? YES : NO)];

        
//        frame.origin.x = 0;
//        frame.origin.y = 0;
        
        [window initWithContentRect:  frame
                          styleMask: window_style
                            backing: NSBackingStoreBuffered
                              defer: TRUE];
        
        [window setFrame:frame display:FALSE];
        
        //window.preferredBackingLocation =  NSWindowBackingLocationVideoMemory;
        
        [window orderOut: nil];
        [window setDelegate: (id<NSWindowDelegate>) nv];
        
        [window setContentView: nv];

        // ??? [nv engine]->attached(nullptr);
        
        [window setAutodisplay: YES];
        [window setAcceptsMouseMovedEvents: YES];
        
        // We'll both retain and also release this on closing because plugin hosts can unexpectedly
        // close the window for us, and also tend to get cause trouble if setReleasedWhenClosed is NO.
        [window setReleasedWhenClosed: YES];
        
        
        [window setExcludedFromWindowsMenu: NO]; //(windowStyleFlags & windowIsTemporary) != 0];
        //WRONG:  [window setIgnoresMouseEvents: NO]; //(windowStyleFlags & windowIgnoresMouseClicks) != 0];
        
        //[nv setExtDelegate: (id)delegate];
        [window setReleasedWhenClosed:YES];
        
/*        NSTrackingArea* ta = [NSTrackingArea alloc];
        
        NSTrackingAreaOptions opts = NSTrackingMouseEnteredAndExited |
                                     NSTrackingMouseMoved |
                                     NSTrackingCursorUpdate |
                                     NSTrackingActiveAlways |
                                     NSTrackingInVisibleRect;
        [ta initWithRect:vr options:opts owner:window userInfo:NULL];
        [nv addTrackingArea:ta];
        [ta release];*/
        
        bool needs_layered = false;
        
        if( creationFlags & SW_ALPHA )
        {
            //[window setAlphaValue:1.0];
            [window setOpaque:NO];
            [window setBackgroundColor: [NSColor clearColor]];
            [window setHasShadow: NO];
            needs_layered = true;
        }
        else if( creationFlags & SW_GLASSY )
        {
            //[window setBackgroundColor: [NSColor clearColor]];
            //[nv engine]->set_blurbehind( html::BLUR_BEHIND_CUSTOM );
        } else {
#ifdef USE_LAYERS
            [nv setWantsLayer: YES];
#endif
            [window setOpaque: YES];
            [window setHasShadow: YES]; //((windowStyleFlags & windowHasDropShadow) != 0)];
        }
        
        
        if( creationFlags & SW_POPUP ) {
           [window setLevel: NSPopUpMenuWindowLevel];
           [window setFloatingPanel: YES];
        }
        else if( creationFlags & SW_TOOL )
           [window setLevel: NSFloatingWindowLevel];
       
        //[nv engine]->set_layered(needs_layered);
        setup_window(window, nv, [nv engine]);
        
        [nv engine]->start(cparams);
        
    }
    
    [nv setAcceptsTouchEvents:YES];
    
    
    return nv;
}


handle<html::view>  osx_create_frame(const window_params& params)
{
    gool::rect drc(params.pos,params.dim);
    
    gool::rect wrc = osx_screen_rect(gool::rect(params.pos,params.dim));
    NSRect frame;
    frame.origin.x = wrc.s.x;
    frame.origin.y = wrc.s.y;
    frame.size.width = wrc.width();
    frame.size.height = wrc.height();
    html::view::frame_attributes fatts;
    
    NSRect vr = frame;
    vr.origin.x = 0;
    vr.origin.y = 0;
    
    //bool needs_debug = params.debug_mode.val( params.parent ? params.parent->debug_mode() : false );
    
    SciterView* nv = [ SciterView alloc ];
    if( nv ) {
        [nv initWithFrame: vr
                   //ownsVm: params.owns_vm
                //debugMode: needs_debug
             windowParams: &params];
#ifdef USE_LAYERS
        [nv setWantsLayer:YES];
#endif
    }
    else
        return nullptr;
    
    handle<osx::view> pview = osxview(nv);
    //pview->parent(params.parent);
    //pview->type = params.window_type;
    //pview->param = value_to_value(pview->vm,params.window_parameters);
    
    //if( params.window_type == html::DIALOG_WINDOW )
    //    [nv setIsDialog: TRUE];
    
    bool client_coordinates = params.client_coordinates;
    ustring caption = params.caption;
    //bool resizeable = params.window_type == html::FRAME_WINDOW;
    //bool custom_frame = false;
    bool require_layered = false;
    
    NSWindow* window;

    unsigned int window_style = 0;
    
    switch( params.window_type )
    {
        default:
        case html::FRAME_WINDOW:
            window = [SciterWindow alloc];
            window_style = NSTitledWindowMask | NSMiniaturizableWindowMask | NSClosableWindowMask | NSResizableWindowMask; break;
        case html::DIALOG_WINDOW:
            window = [SciterWindow alloc];
            window_style = NSTitledWindowMask | NSClosableWindowMask; break;
        case html::TOOL_WINDOW:
            window = [SciterWindow alloc];
            window_style = NSTitledWindowMask | NSUnifiedTitleAndToolbarWindowMask | NSClosableWindowMask; break;
        case html::POPUP_WINDOW:
            window = [[SciterPanelWindow alloc] initWithContentRect: frame
                                                     styleMask: NSNonactivatingPanelMask | NSClosableWindowMask | NSMiniaturizableWindowMask
                                                       backing: NSBackingStoreBuffered
                                                         defer: NO];
            window_style = NSNonactivatingPanelMask | NSClosableWindowMask | NSMiniaturizableWindowMask;
            [window setCollectionBehavior: [window collectionBehavior] | NSWindowCollectionBehaviorCanJoinAllSpaces | NSWindowCollectionBehaviorFullScreenAuxiliary ];
            [window setLevel:NSPopUpMenuWindowLevel];
            break;
            
    }
    
    if (params.window_type != html::POPUP_WINDOW) {
    [window initWithContentRect: frame
                      styleMask: window_style
                        backing: NSBackingStoreBuffered
                          defer: TRUE];
        [window orderOut: nil];
    }
    
    
    
    [window setDelegate: (id<NSWindowDelegate>) nv];
    
    //[nv setExtDelegate: (id)delegate];
    html::FRAME_TYPE ft = html::FRAME_TYPE::STANDARD;
    
    [window setContentView: nv];
 
    [window canBecomeKeyWindow];
    [window setAutodisplay: YES];
    [window setAcceptsMouseMovedEvents: YES];
    
    // We'll both retain and also release this on closing because plugin hosts can unexpectedly
    // close the window for us, and also tend to get cause trouble if setReleasedWhenClosed is NO.
    [window setReleasedWhenClosed: YES];
    //[window retain];
    
    [window setExcludedFromWindowsMenu: YES]; //(windowStyleFlags & windowIsTemporary) != 0];
    // NOTE calling this [window setIgnoresMouseEvents: NO]; //(windowStyleFlags & windowIgnoresMouseClicks) != 0];
    // prevents call through on transparent areas!
    
    if(params.parent && !params.is_detached) {
       NSView* ppnv = (NSView*) params.parent->get_hwnd();
       if( ppnv ) {
         NSWindow* ppnw = [ppnv window];
         [ppnw addChildWindow:window ordered: NSWindowAbove];
       }
    }
    
    pview->set_hwnd(nv);
    pview->start(params);
    
    wrc = pview->screen_place();
    
    gool::rect parentrc = osx_screen_workarea( drc );
    
    gool::rect displayrc = parentrc;
    if(params.screen_no.is_defined())
    {
        html::screen_info si;
        if(!html::get_screen_info(params.screen_no,si))
            html::get_screen_info(0,si);
        displayrc = si.workarea;
    }
    
    
    if( html::document* pd = pview->doc() ){
        
        html::element* title = html::find_first( *pview, pd, WCHARS(":root>head>title") );
        if( title )
          caption = title->get_text(*pview);
        
        pview->required_frame_attributes(fatts);
        
        ft = (html::FRAME_TYPE) fatts.frame_type.val(ft);
        
        html::style* cs = pd->get_style(*pview);
        require_layered = ft == html::LAYERED || cs->is_transparent() || cs->opacity != 255 || cs->has_rounded_corners();
        gool::size ssz = parentrc.size();
        
        if(params.dim.empty()) {
          client_coordinates = true;
          size msz = wrc.size();
          msz.x = max(msz.x,pd->min_width(*pview, ssz.x));
          if (pd->dim().x != msz.x) pd->set_width(*pview, msz.x);
          msz.y = max(msz.y,pd->min_height(*pview, ssz.y));
          if (msz.x == 0) msz.x = 300;
          if (msz.y == 0) msz.y = 150;

          gool::point pt = params.pos;
          wrc = gool::rect(pt, msz);
            
        }
    }
    
    if( client_coordinates )
        wrc >>= pview->window_decoration();
    
    if( params.dim.x.is_undefined() && params.pos.x.is_defined())
    {
        wrc.pointOf((params.alignment ? params.alignment : 5),params.pos);
        wrc.inscribe(displayrc);
    }
    else if( params.dim.x.is_defined() && params.pos.x.is_defined())
    {
        //wrc = rect(params.pos,params.dim);
        //wrc.pointOf(params.alignment,params.pos);
    }
    
    if( params.pos.x.is_defined() )
      {;}
    else if( params.alignment < 0 && params.alignment >= -9)
    {
        if( params.parent ) {
            parentrc = params.parent->screen_place();
            drc = params.parent->screen_workarea();
        }
        wrc.pointOf(abs(params.alignment),parentrc.pointOf(abs(params.alignment)));
        wrc.inscribe(drc);
    }
    else if( params.alignment > 0 && params.alignment <= 9 )
    {
        wrc.pointOf(params.alignment,displayrc.pointOf(params.alignment));
        wrc.inscribe(displayrc);
    }
   
    gool::rect osxwrc = osx_screen_rect(wrc);
    //gool::rect osxwrc = wrc;
    
    frame.origin.x = osxwrc.start().x;
    frame.origin.y = osxwrc.start().y;
    frame.size.width = osxwrc.width();
    frame.size.height = osxwrc.height();
    
    [window setTitle: cvt(caption) ];
        
    pview->set_frame_type(ft);
    pview->set_resizeable(fatts.resizeable);

    if(!pview->window_was_moved)
      [window setFrame: frame display:NO ];

        
    [window invalidateShadow];
    
    return handle<html::view>(pview.ptr());
    
}
handle<html::view>  osx_create_dialog(const window_params& params)
{
    return osx_create_frame(params);
}
    
namespace osx {
    WINDOW_STATE view::get_window_state() const
    {
        SciterView* nv = (SciterView*) get_hwnd();
        NSWindow* pw = [nv window];
        if( !pw )
            return WINDOW_STATE_NA;
        if([pw isMiniaturized])
            return WINDOW_MINIMIZED;
        if(![pw isVisible] )
            return WINDOW_HIDDEN;
        
        //if([pw collectionBehavior] == NSWindowCollectionBehaviorFullScreenPrimary ||
        //   [pw collectionBehavior] == NSWindowCollectionBehaviorFullScreenAuxiliary)
        //    return WINDOW_FULL_SCREEN;
        if(([pw styleMask] & NSFullScreenWindowMask) == NSFullScreenWindowMask)
          return WINDOW_FULL_SCREEN;
        
        NSRect src = [[pw screen] visibleFrame];
        NSRect wrc = [pw frame];
        if( NSEqualRects(src,wrc) )
            return WINDOW_MAXIMIZED;
        return WINDOW_SHOWN;
    }
    bool         view::set_window_state( WINDOW_STATE ws )
    {
        SciterView* nv = (SciterView*) get_hwnd();
        NSWindow* pw = [nv window];
        if( !pw )
           return false;
        
        WINDOW_STATE pws = get_window_state();
        
        if( pws == ws)
            return false;
        
        //if(/*pws == WINDOW_MINIMIZED ||*/ pws == WINDOW_HIDDEN || pws == WINDOW_STATE_NA )
        //  this->release_backend();
        
        NSRect src = [[pw screen] visibleFrame];
        NSRect wrc = [pw frame];
        
        switch( ws ) {
            default:
            case WINDOW_SHOWN:
                if([pw isMiniaturized])
                  [pw deminiaturize:pw];
                else if(([pw styleMask] & NSFullScreenWindowMask) == NSFullScreenWindowMask)
                {
                    [pw setCollectionBehavior: NSWindowCollectionBehaviorFullScreenPrimary | [pw collectionBehavior]];
                    [pw toggleFullScreen: nil];
                    [pw setCollectionBehavior: ~NSWindowCollectionBehaviorFullScreenPrimary & [pw collectionBehavior]];
                } else {
                    
                  /*if( this->type == POPUP_WINDOW ) {
                    [pw setCollectionBehavior: [pw collectionBehavior] | NSWindowCollectionBehaviorCanJoinAllSpaces | NSWindowCollectionBehaviorFullScreenAuxiliary ];
                }*/
                    
                  if(NSEqualRects(src,wrc) && !_preminimized_window_place.empty()) {
                  [pw setFrame: tons(osx_screen_rect(_preminimized_window_place)) display:YES animate:YES];
                  }
                }
                [pw orderFront:pw];
                break;
            case WINDOW_MINIMIZED:
                [pw miniaturize:pw];
                break;
            case WINDOW_FULL_SCREEN:
                [pw setCollectionBehavior: NSWindowCollectionBehaviorFullScreenPrimary | [pw collectionBehavior]];
   
                [pw toggleFullScreen: nil];
                //[pw orderFront:pw];
                break;
            case WINDOW_MAXIMIZED:
                if( get_window_state() == WINDOW_SHOWN )
                    _preminimized_window_place = screen_place();
                [pw setFrame:[[pw screen] visibleFrame] display:YES animate:YES];
                [pw orderFront: pw];
                break;
            case WINDOW_HIDDEN:
                [pw orderOut:pw];
                this->release_backend();
                break;
        }
        
        //if(ws == WINDOW_MINIMIZED || ws == WINDOW_HIDDEN || ws == WINDOW_STATE_NA )
        //  this->release_backend();
        
        return true;
    }
   

}

/*void osx_do_animation_tick( osx::view* pv ) {
    
    SciterView* sv = (SciterView*) pv->get_hwnd();
    [sv performSelectorOnMainThread:@selector(doAnimationTick) withObject:nil waitUntilDone:NO];

}*/

