#ifndef __gtk_types_h__
#define __gtk_types_h__

#define PANGO_ENABLE_ENGINE

#include <gtk/gtk.h>
#include <cairo.h>
//#include "gool/gool.h"

namespace gtk
{


    template <class T> class ref
    {
    public:

        // Function Members ----------------------------------------------------------------------

        ref (T* cfo = nullptr) : fRef (cfo) { }
        // Construct a temporary reference for 'cfo' or NIL.
        // This constructor does NOT raise the use count of 'cfo'. It is assumed that
        // 'cfo' already carries a use count (which is the case when the object was
        // just created and returned by an AcquireXXX method).

        ref (const ref& src)
        // Copy constructor. This constructor DOES raise the use count.
        : fRef (src.fRef) {
            if (fRef)
                g_object_ref(fRef);
        }

        ~ref() {
            // Free the reference. To avoid any chance of recursive access to the
            // reference while it is being freed, the pointer is set to NIL first.
            T* cfo = fRef;
            fRef = nullptr;
            if (cfo)
                g_object_unref (cfo);
        }

        ref& operator= (T cfo) {
            // Assign a new reference to be owned and free the old reference.
            // Does NOT raise the use count of 'cfo', see comment for TempCFRef (T cfo)
            // above.
            T* old = fRef;
            fRef = cfo;
            if (old)
                g_object_unref (old);
            return *this;
        }

        ref& operator= (const ref& src) {
            // Assign the source reference to be owned and free the old reference.
            // This assignment operator DOES raise the use count.
            T* old = fRef;
            if ((fRef = src.fRef) != nullptr)
                g_object_ref (fRef);
            if (old)
                g_object_unref (old);
            return *this;
        }

        T* detach() {
            // Detach the reference from the owner, ownership is released.
            T* cfo = fRef;
            fRef = nullptr;
            return cfo;
        }

        T* acquire() {
            // Return the reference with an additional use count.
            if (fRef)
                g_object_ref (fRef);
            return fRef;
        }

        operator T*() const { return fRef; }
        // Return the pointer.

        T* ptr() const { return fRef; }
        // If the operator above it not triggered automatically, writing
        // "temp.Data()" is probably more readable than "(T) temp",
        // especially if another cast is present.

        T* operator&() {
            T* cfo = fRef;
            fRef = nullptr;
            if (cfo)
                g_object_unref (cfo);
            return &fRef;
        }

        operator bool() const { return fRef != 0; }

        // This operator is used to pass a pointer to the CF reference to some API
        // which will return a CF reference via pointer. To guarantee reference count
        // consistency, any old contents is release first.


        // Data Members --------------------------------------------------------------------------
    private:
        T *fRef;	// The object reference
    };

}

namespace cairo
{
    template <class T> struct ref_traits {
      static void addref(T*);
      static void release(T*);
    };

    template <> struct ref_traits<cairo_surface_t> {
      static void addref(cairo_surface_t* obj) { cairo_surface_reference(obj); }
      static void release(cairo_surface_t* obj) { cairo_surface_destroy(obj); }
    };

    template <class T> class ref
    {

    public:

        // Function Members ----------------------------------------------------------------------

        ref (T* cfo = nullptr) : fRef (cfo) { }
        // Construct a temporary reference for 'cfo' or NIL.
        // This constructor does NOT raise the use count of 'cfo'. It is assumed that
        // 'cfo' already carries a use count (which is the case when the object was
        // just created and returned by an AcquireXXX method).

        ref (const ref& src)
        // Copy constructor. This constructor DOES raise the use count.
        : fRef (src.fRef) {
            if (fRef)
                ref_traits<T>::addref(fRef);
        }

        ~ref() {
            // Free the reference. To avoid any chance of recursive access to the
            // reference while it is being freed, the pointer is set to NIL first.
            T* cfo = fRef;
            fRef = nullptr;
            if (cfo)
                ref_traits<T>::release(cfo);
        }

        ref& operator= (T cfo) {
            // Assign a new reference to be owned and free the old reference.
            // Does NOT raise the use count of 'cfo', see comment for TempCFRef (T cfo)
            // above.
            T* old = fRef;
            fRef = cfo;
            if (old)
                ref_traits<T>::release (old);
            return *this;
        }

        ref& operator= (const ref& src) {
            // Assign the source reference to be owned and free the old reference.
            // This assignment operator DOES raise the use count.
            T* old = fRef;
            if ((fRef = src.fRef) != nullptr)
                ref_traits<T>::addref(fRef);
            if (old)
                ref_traits<T>::release (old);
            return *this;
        }

        T* detach() {
            // Detach the reference from the owner, ownership is released.
            T* cfo = fRef;
            fRef = nullptr;
            return cfo;
        }

        T* acquire() {
            // Return the reference with an additional use count.
            if (fRef)
                ref_traits<T>::addref (fRef);
            return fRef;
        }

        operator T*() const { return fRef; }
        // Return the pointer.

        T* ptr() const { return fRef; }
        // If the operator above it not triggered automatically, writing
        // "temp.Data()" is probably more readable than "(T) temp",
        // especially if another cast is present.

        T* operator&() {
            T* cfo = fRef;
            fRef = nullptr;
            if (cfo)
                ref_traits<T>::release (cfo);
            return &fRef;
        }

        operator bool() const { return fRef != 0; }

        // This operator is used to pass a pointer to the CF reference to some API
        // which will return a CF reference via pointer. To guarantee reference count
        // consistency, any old contents is release first.


        // Data Members --------------------------------------------------------------------------
    private:
        T *fRef;	// The object reference
    };

 }




#endif


