#ifndef __gtk_graphics_h__
#define __gtk_graphics_h__

#include "gtk-types.h"
#include "tool/tool.h"
#include "gool/gool.h"

#include "pango/pango.h"

#define GRGBA(c) c.n_red(), c.n_green(), c.n_blue(), c.n_alpha()
#define GRGB(c) c.n_red(), c.n_green(), c.n_blue()
#define GRECT(r) r.left(), r.top(), r.width(), r.height()

inline gool::color cvt(const GdkRGBA& gc) {
  return gool::rgba(
    uint(gc.red * 255.0),
    uint(gc.green * 255.0),
    uint(gc.blue * 255.0),
    uint(gc.alpha * 255.0));
}

void gtk_init_sys_colors_table(GtkWidget* pw, bool force);

namespace gtk
{
  using namespace tool;
  using namespace gool;

  class graphics;

  struct font: public gool::font
  {
    typedef gool::font super;

    font():super(), _pg_font(nullptr) {}
    font(const font& f):super(f) { _pg_font = f._pg_font;  }
    virtual ~font() { }

    virtual bool has_glyph_for(uint ucodepoint) const override;
    virtual uint glyph_index(uint ucodepoint) const override;
    virtual bool glyph_metrics(::uint16 glyph_index, float em_size, float& width, float& height) const  override { return false; }
    //virtual void metrics(int& a,int& d,int& x,float sz = 0) const { a = ascent * ; d = descent; x = x_height; sz = size; }

    uint get_glyph_indices_and_advances( slice<wchar> text , function<void(uint16,float,float)> cb );

    mutable gtk::ref<PangoCairoFont> _pg_font;

    PangoCairoFont* pg_font() const { return _pg_font; }

  };

  extern cairo_surface_t* cairo_surface( gool::bitmap* bmp );

  struct brush : public resource {
    cairo_pattern_t* pat;

    brush():pat(nullptr) {}
    virtual ~brush() { if(pat) cairo_pattern_destroy(pat); }

    virtual bool is_solid() const { return false; }

    void fill(graphics* gfx);
    void stroke(graphics* gfx);

  };

  struct solid_brush : public brush {

    solid_brush(gool::argb c){
       pat = cairo_pattern_create_rgba ( GRGBA(c) );
    }
    virtual bool is_solid() const override { return true; }
  };

  struct gradient_brush : public brush
  {
    //affine_mtx_f mx;
    void init_stops(const gool::gradient_brush& bin);
  };

  struct linear_gradient_brush : public gradient_brush
  {
    linear_gradient_brush( const gool::linear_brush& bin  )
    {

      cairo_matrix_t mtx;

      auto m = bin.transform();

      pat = cairo_pattern_create_linear (bin.start.x, bin.start.y, bin.end.x, bin.end.y);
      init_stops(bin);

      if (m.is_identity())
        cairo_matrix_init_identity(&mtx);
      else {
        mtx.xx = m.sx;
        mtx.yx = m.shy;
        mtx.xy = m.shx;
        mtx.yy = m.sy;
        mtx.x0 = m.tx;
        mtx.y0 = m.ty;
      }
      cairo_pattern_set_matrix(pat,&mtx);
    }
  };

  struct radial_gradient_brush : public gradient_brush
  {
    radial_gradient_brush( const gool::radial_brush& bin  )
    {
      //center = bin.center; radius = bin.radius;

      cairo_matrix_t mtx;

      auto m = bin.transform();
      if (m.is_identity())
        cairo_matrix_init_identity(&mtx);
      else {
        mtx.xx = m.sx;
        mtx.yx = m.shy;
        mtx.xy = m.shx;
        mtx.yy = m.sy;
        mtx.x0 = m.tx;
        mtx.y0 = m.ty;
        cairo_matrix_invert(&mtx);
      }

      pat = cairo_pattern_create_radial( bin.center.x, bin.center.y, 0, bin.center.x, bin.center.y, max(bin.radius.x,bin.radius.y) );
      init_stops(bin);

      if( bin.radius.x != bin.radius.y ) {
         cairo_matrix_t matrix;
         cairo_matrix_init_translate(&matrix,bin.center.x, bin.center.y);
         cairo_matrix_scale(&matrix,1,bin.radius.x / bin.radius.y);
         cairo_matrix_translate(&matrix,-bin.center.x, -bin.center.y);
         cairo_matrix_multiply (&mtx, &mtx, &matrix);
      }
      cairo_pattern_set_matrix(pat,&mtx);
    }
  };

  struct image_brush : public brush
  {
    image_brush(const image* im)
    {
      handle<gool::bitmap> bmp = const_cast<image*>(im)->get_bitmap(nullptr,size());
      if(bmp)
        pat = cairo_pattern_create_for_surface(cairo_surface(bmp));
    }
  };

  struct state_registers
  {
    handle<brush>     stroke_brush;
    handle<brush>     fill_brush;

    state_registers(){}

    bool drawable() const { return stroke_drawable() || fill_drawable(); }
    bool stroke_drawable() const { return stroke_brush.is_defined(); }
    bool fill_drawable() const { return fill_brush.is_defined(); }
  };


  class graphics : public gool::graphics
  {
    friend class view;
    typedef gool::graphics super;

  protected:
    gool::point            _offset;
    cairo_t*               _target;
    state_registers        _registers;
    array<state_registers> _registers_stack;
    IMAGE_RENDERING        _image_rendering_mode;

    //graphics(bool transparent) :super(transparent), _target(nullptr), _image_rendering_mode(IMAGE_RENDERING::image_rendering_undefined) {}
  public:
    graphics(cairo_t* target, bool transparent) : super(transparent), _image_rendering_mode(IMAGE_RENDERING::image_rendering_undefined) {
       _target = cairo_reference(target);
    }
    virtual ~graphics() {
       cairo_destroy(_target);
    }

    cairo_t* target() const { return _target; }

    virtual void fill( argb c, const rect& dst) override;
    virtual void fill( argb c, const rect& dst, const size& radius) override;
    virtual void fill( argb c, const gool::path* dst) override;
    virtual void fill( const gool::brush* lb, const rect& dst) override;
    virtual void fill( image* img, const rect& dst, const rect& src,point offset = point(0,0), size celldim = size()) override;
    virtual void fill( image* img, const rect& dst, point offset = point(0,0)) { fill(img, dst,rect(img->dim()),offset); }
    virtual void fill( image* img, const gool::path* dst) override;

    //virtual image_renderer_f image_renderer(image* img) override;
    virtual path* create_path() const override;
    virtual gool::application*  app() const override { return gool::app(); }

    virtual void draw( const gool::path* dst, argb c, float width = 1.0f ) override;

    virtual void draw_line_v(const rect& rc, argb c, int stroke, int step, int_v off) override;
    virtual void draw_line_h(const rect& rc, argb c, int stroke, int step, int_v off) override;

    virtual void draw_line(pointf p1, pointf p2 , argb c, int width) override;

    //virtual void draw( gool::text_layout& tl, point  dst, argb c) override { draw( tl, pointf(dst),c); }
    //virtual void draw( gool::text_layout& tl, pointf dst, argb c) override;
    //virtual void draw( gool::text_layout& tl, pointf dst ) override; // using current fill and stroke

    virtual bool push_clip(const rect& r, clipper* cl) override { do_push_clip(r); super::push_clip(r,cl); return true;  }
    virtual void pop_clip() { super::pop_clip(); do_pop_clip(); }

            void do_push_clip(const rect& r);
            void do_pop_clip();

    virtual point offset(point noff) override;

    virtual bool set_antialiasing(bool onoff) override;

    virtual uint push_state() override;
    virtual bool pop_state() override;
    virtual void restore_state(uint level) override;

    //virtual const text_format& get_text_format();
    //virtual void               set_text_format(const text_format& tf);

    virtual void push_layer(const gool::rect& clip, byte opacity = 255, function<bool(filter_graph_builder*)> filters = nullptr) override;
    virtual void push_layer(const gool::path* clip, byte opacity = 255) override;
    virtual void push_layer(const bitmap* clip, bool draw1a, byte opacity = 255) override; // draw1a = true - pixels with alpha == 255 are visible, draw1a = false - pixels with alpha == 255 are clipped out.
    virtual void pop_layer() override;
    virtual uint n_layers() override { return uint(_layers_stack.length()); }


    struct layers_stack_item {
      byte opacity;
      handle<bitmap> mask;
    };
    array<layers_stack_item>  _layers_stack;

    virtual void transform(const affine_mtx_f& m) override;
    virtual affine_mtx_f transform() const override;
    virtual void reset_transform() override;

    virtual void            image_rendering_mode( IMAGE_RENDERING irm) override;
    virtual IMAGE_RENDERING image_rendering_mode() const  override;


    //|
    //| Graphin stuff
    //|

    virtual    CAP_STYLE cap_style() override;
    virtual         void cap_style(CAP_STYLE ncs) override;
    virtual    LINE_JOIN line_join() override;
    virtual         void line_join(LINE_JOIN ncs, float mitter_limit = 4) override;
    virtual   DASH_STYLE dash_style() override;
    virtual         void dash_style( DASH_STYLE st ) override;
    virtual         void custom_dash_style(slice<float> pattern, float offset) override;

    virtual void set_stroke( argb c = argb::no_color()) override ;
    virtual void set_stroke( const linear_brush& lb ) override;
    virtual void set_stroke( const radial_brush& rb ) override;
    virtual void set_stroke_width( float w ) override;


    virtual void set_fill( argb c = argb::no_color() ) override;
    virtual void set_fill( const linear_brush& lb ) override;
    virtual void set_fill( const radial_brush& rb ) override;
    virtual void set_fill( const image* ib ) override ;
    virtual void set_fill_even_odd( bool even_odd ) ;

    virtual void draw_ellipse(pointf center, sizef radius, bool stroke = true, bool fill = true)  override;
    virtual void draw_arc(pointf center, sizef radius, float angle_start, float angle_sweep, bool stroke = true, bool fill = true) override;
    virtual void draw_rectangle(pointf org, sizef dim, bool stroke = true, bool fill = true)  override ;
    //virtual void draw_rectangle(rectf r)  override;
    virtual void draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr, sizef rbr, sizef rbl, bool stroke = true, bool fill = true) override { super::draw_rectangle(org,dim, rtl, rtr,rbr,rbl); }
    virtual void translate(pointf pt) override;
    virtual void rotate(float radians, pointf center = pointf()) override;
    virtual void scale(sizef sz, pointf center = pointf()) override;
    virtual void skew(sizef sz, pointf center = pointf()) override;

    virtual pointf world_to_screen(pointf pt) const override;
    virtual pointf screen_to_world(pointf pt) const override;

    virtual void draw_path( const gool::path* dst, bool stroke = true, bool fill = true) override;
    virtual void draw_path( const gool::polygonf& p, bool stroke = true, bool fill = true) override;
    virtual void draw_line(pointf start, pointf end) override;
    virtual void draw_line(pointf origin, pointf end, line_end_style sorigin, line_end_style send) override { super::draw_line(origin, end, sorigin, send); }

    virtual void do_draw( image* img, const rectf&  dst, const rect& src, byte opacity = 255) override;
            void render_current_path(bool fill = true, bool stroke = true);

  };

  class path : public gool::path
  {
    public:
        path():_evenodd(true), _path(0),_path_graphics(0),_path_surface(0) {}
        virtual ~path() { reset(); }

        virtual bool is_inside(pointf pt);

        virtual void start(pointf start, bool filled) override;
        virtual void move_to(pointf pt, bool rel = false) override;
        virtual void line_to(pointf pt, bool rel = false) override;
        virtual void hline_to(float x, bool rel = false) override;
        virtual void vline_to(float y, bool rel = false) override;
        virtual void add_arc(pointf c, sizef r, float angle_start, float angle_swipe) override;
        virtual void quadratic_to(pointf pt, pointf cp, bool rel = false) override;
        virtual void cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel = false) override; // bezier
        virtual void close() override;
        virtual bool is_empty() const override;
        //virtual void arc_to(pointf to, sizef r, float rotation_angle, bool large_arc, bool clockwise, bool rel_to = false);

        virtual void reset() override;
        virtual void seal() override;

        virtual pointf lastp() override;

                bool even_odd() const { return _evenodd; }
        virtual void set_even_odd(bool yes) { _evenodd = yes; }

        virtual rectf bounds() const override { const_cast<path*>(this)->seal(); return _bounds; }

        cairo_path_t* gpath() { seal(); assert(_path); return _path; }

    protected:
        cairo_path_t*    _path;
        cairo_t*         _path_graphics;
        cairo_surface_t* _path_surface;
        rectf            _bounds;
        bool             _evenodd;
  };

  inline gool::path* graphics::create_path() const { return new gtk::path(); }

  class bitmap_graphics : public graphics
  {
     protected:
     typedef graphics super;
     handle<bitmap> pbmp;

     public:

       bitmap_graphics(bitmap* bmp, argb initc): super(nullptr,!initc.is_opaque()), pbmp(bmp)
       {
         _target = cairo_create( cairo_surface(bmp) );
         if( initc != argb::undefined() ) {
            cairo_set_source_rgba(_target, GRGBA(initc));
            cairo_set_operator(_target, CAIRO_OPERATOR_SOURCE);
            cairo_paint(_target);
         }
         cairo_set_operator(_target, CAIRO_OPERATOR_OVER);
         set_clip_rc( bmp->dim() );
       }
       virtual ~bitmap_graphics() {
       }

  };

  class bitmap_bits_graphics : public bitmap_graphics
  {
     typedef bitmap_graphics super;

     public:

       bitmap_bits_graphics(bitmap* bmp, argb initc):super(bmp,initc) {}
       virtual ~bitmap_bits_graphics() {
          slice<argb> src;
          src.start = (argb*)cairo_image_surface_get_data( cairo_surface(pbmp) );
          src.length = pbmp->dim().x * pbmp->dim().y;
          pbmp->set_bits(src);
       }

  };

#if 0
  class text_layout: public gool::text_layout
  {
    public:
        gtk::ref<PangoLayout>   _layout;
        TEXT_ALIGNMENT          _valign;
    public:
        typedef gool::text_layout super;

        //float       width_min;
        //float       width_max;
        //float       height;    // content height
        //float       ascent;    // content ascent
        //sizef       box;       // box dimensions

        text_layout() {}
        virtual ~text_layout() {}

        virtual void        set(resolution_provider& v, wchars text, const text_format& tf, gool::application* app) override;
        virtual void        set_width(float width);
        virtual void        set_height(float height);
        virtual sizef       get_dim() const { return super::get_dim(); }
        virtual sizef       get_box() const { return super::get_box(); }
        virtual uint        get_lines_count() const;
        virtual array<line> get_lines() const;

        void _measure();

        //void drop_layout() const { _frame = NULL; }

        PangoLayout*    layout() const { return _layout; }
  };

#endif

}

#endif
