
#include "xdom.h"
#include "xom.h"

namespace qjs 
{

  int64 context::request_animation_frame(hvalue fun) {
    if (!animator) {
      animator = new script_frame_animator();
      pview()->add_animation(pdoc(), animator);
    }
    else if (animator->requested_frames().contains(fun))
      return int64(fun.v);
    animator->requested_frames.push(fun);
    return int64(fun.v);
  }
  bool context::cancel_animation_frame(int64 id) {
    if (!animator)
      return false;
    for (int n = 0; n < animator->requested_frames.size(); ++n)
      if (animator->requested_frames[n].v == JSValue(id))
      {
        animator->requested_frames.remove(n);
        return true;
      }
    return false;
  }

  uint script_frame_animator::step(html::view &v, html::element *b, uint current_clock) {
    if (requested_frames.size() == 0)
      return 0;
    assert(processing_frames.size() == 0);
    processing_frames.swap(requested_frames);
    context& c = context::inst(b->doc());
    hvalue vclock = c.val(current_clock);
    for (hvalue& cbfun : processing_frames) {
      try {
        bool r;
        c.call(r, cbfun, c.global(), current_clock);
      }
      catch (qjs::exception) {
        c.report_exception();
      }
    }
    processing_frames.clear();
#ifdef WINDOWS
    return html::ANIMATION_TIMER_SPAN / 2;
#else
    return html::ANIMATION_TIMER_SPAN;
#endif
  }

  void script_frame_animator::stop(html::view &v, html::element *b) {
    requested_frames.clear(); processing_frames.clear(); 
    context& c = context::inst(b->doc());
    assert(!c.animator || c.animator == this);
    c.animator = nullptr;
  }
    
  struct bjson_stream : public resource 
  {
    array<byte> buffer;
    size_t      awaiting = 0;
    
    JSValue pack(JSContext* ctx, JSValueConst val) 
    {
      array<byte> buffer;
      
      int flags = JS_WRITE_OBJ_REFERENCE;
      size_t len;
      uint8_t *buf = JS_WriteObject(ctx, &len, val, flags);
      if (!buf)
        return JS_EXCEPTION;

      buffer.push(byte(len >> 24));
      buffer.push(byte(len >> 16));
      buffer.push(byte(len >> 8));
      buffer.push(byte(len >> 0));
      buffer.push(buf, len);

      JSValue array = JS_NewArrayBufferCopy(ctx, buffer.cbegin(), buffer.length());
      js_free(ctx, buf);
      return array;
    }
    
    bool unpack(JSContext* ctx, bytes chunk, JSValueConst cb) {
      auto read_length = [](bytes data) {
        uint sz = data[0];  sz <<= 8;
             sz |= data[1]; sz <<= 8;
             sz |= data[2]; sz <<= 8;
             sz |= data[3]; 
        return sz;
      };
      buffer.push(chunk);
      if (!awaiting) 
        awaiting = read_length(buffer());

      while (awaiting <= (buffer.length() - 4)) {
        JSValue obj = JS_ReadObject(ctx, buffer.cbegin() + 4, awaiting, 0);
        buffer.remove(0, awaiting + 4);
        JSValue rv = JS_Call(ctx, cb, JS_UNDEFINED, 1, &obj);
        JS_FreeValue(ctx, obj);
        if(JS_IsException(rv))
          throw exception();
        if (buffer.length() < 4) { 
          awaiting = 0; 
          break;
        }
        awaiting = read_length(buffer());
      }
      return true;
    }
  };

  JSClassID BJSON_class_id = 0;

  template <> struct conv<handle<bjson_stream>>
  {
    static handle<bjson_stream> unwrap(JSContext * ctx, JSValueConst v)
    {
      handle<bjson_stream> pr = (bjson_stream*)JS_GetOpaque(v, BJSON_class_id);
      return pr;
    }
  };
  
  static hvalue bjson_pack(xcontext& c, handle<bjson_stream> pbson, hvalue obj) { 
    hvalue r = pbson->pack(c,obj);
    if (c.is_exception(r))
      throw qjs::om::js_error(r);
    return r;
  }

  static bool bjson_unpack(xcontext& c, handle<bjson_stream> pbson, hvalue data, hvalue cb) {
    size_t size;
    byte* buf = JS_GetArrayBuffer(c, &size, data);
    if (!buf)
      throw qjs::om::type_error("ArrayBuffer is expected");
    return pbson->unpack(c, bytes(buf,size), cb);
  }
  
  JSOM_PASSPORT_BEGIN(BJSON_def, qjs::bjson_stream)
    JSOM_CONST_STR("[Symbol.toStringTag]", "BJSON", JS_PROP_CONFIGURABLE),
    JSOM_FUNC_DEF("pack", bjson_pack),
    JSOM_FUNC_DEF("unpack", bjson_unpack),
  JSOM_PASSPORT_END

  void init_BJSON_class(context& c)
  {
    JS_NewClassID(&BJSON_class_id); 

    static JSClassDef BJSON_class = {
      "BJSON",
      [](JSRuntime *rt, JSValue val)
      {
        bjson_stream* pe = (bjson_stream*)JS_GetOpaque(val,BJSON_class_id);  //object_of<html::element>(val);
        if (pe) pe->release();
      }
    };

    JS_NewClass(JS_GetRuntime(c), BJSON_class_id, &BJSON_class);
    JSValue BJSON_class_proto = JS_NewObject(c);

    auto list = BJSON_def();
    JS_SetPropertyFunctionList(c, BJSON_class_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      xcontext c(ctx);
      handle<bjson_stream> pbs = new bjson_stream();

      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto))
        goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, BJSON_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj))
        goto fail;
      JS_SetOpaque(obj, pbs.detach());
      return obj;
    fail:
      //js_free(ctx, s);
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue BJSON_class_obj = JS_NewCFunction2(c, ctor, "BJSON", 2, JS_CFUNC_constructor, 0);
    JS_DefinePropertyValueStr(c, c.global(), "BJSON", JS_DupValue(c, BJSON_class_obj), JS_PROP_CONFIGURABLE);

    JS_SetConstructor(c, BJSON_class_obj, BJSON_class_proto);
    JS_SetClassProto(c, BJSON_class_id, BJSON_class_proto);
  }

  JSClassID NativeFunctor_class_id = 0;

  JSOM_PASSPORT_BEGIN(NativeFunctor_def, native_functor_holder)
    JSOM_CONST_STR("[Symbol.toStringTag]", "NativeFunctor", JS_PROP_CONFIGURABLE),
    //JSOM_RO_PROP_DEF("pack", native_functor_argc),
  JSOM_PASSPORT_END

  JSValue JSNativeFunctorCall(JSContext *ctx, JSValueConst func_obj,
      JSValueConst this_val, int argc, JSValueConst *argv,
      int flags) {
    tool::native_functor_holder* pf = (tool::native_functor_holder*)JS_GetOpaque(func_obj, NativeFunctor_class_id);
    tool::buffer<tool::value, 8> args(argc);
    for (int n = 0; n < argc; ++n)
      args[n] = conv<tool::value>::unwrap(ctx, argv[n]);
    tool::value r = pf->call(uint(argc), args.cbegin());
    return conv<tool::value>::wrap(ctx, r);
  }

  void init_NativeFunctor_class(context& c)
  {
    JS_NewClassID(&NativeFunctor_class_id);

    static JSClassDef NativeFunctor_class = {
      "NativeFunctor",
      [](JSRuntime *rt, JSValue val)
      {
        native_functor_holder* pn = (native_functor_holder*)JS_GetOpaque(val,NativeFunctor_class_id);  //object_of<html::element>(val);
        if (pn) pn->release();
      }, 
      NULL,
      JSNativeFunctorCall
    };

    JS_NewClass(JS_GetRuntime(c), NativeFunctor_class_id, &NativeFunctor_class);
    JSValue NativeFunctor_class_proto = JS_NewObject(c);

    auto list = NativeFunctor_def();
    JS_SetPropertyFunctionList(c, NativeFunctor_class_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      return JS_EXCEPTION; // not constructible
    };

    hvalue NativeFunctor_class_obj = JS_NewCFunction2(c, ctor, "NativeFunctor", 2, JS_CFUNC_constructor, 0);
    JS_DefinePropertyValueStr(c, c.global(), "NativeFunctor", JS_DupValue(c, NativeFunctor_class_obj), JS_PROP_CONFIGURABLE);

    JS_SetConstructor(c, NativeFunctor_class_obj, NativeFunctor_class_proto);
    JS_SetClassProto(c, NativeFunctor_class_id, NativeFunctor_class_proto);
  }

  JSValue wrap_native_functor(JSContext* ctx, tool::native_functor_holder* pf) {
    JSValue v = JS_NewObjectClass(ctx, NativeFunctor_class_id);
    pf->add_ref();
    JS_SetOpaque(v, pf);
    return v;
  }

  ustring format(xcontext& c, ustring fmt, array<hvalue> args) {
    const wchar *pc, *fmtend;

    tool::ustring cfmt; // current format run
    fmtend = fmt.cend();

    uint argi = 0;
    
    array<wchar> out;

    auto get_arg = [&](uint n) -> hvalue { 
      return n < args.length() ? args[n] : hvalue();
    };

    for (pc = fmt; pc < fmtend && *pc; ++pc) {
      while (*pc && *pc != '%')
        out.push(*pc++);

      if (*pc == 0) break;

      if (*++pc == '%') out.push('%');

      cfmt = '%';
      for (; *pc; ++pc) {
        if (*pc == '*') {
          int n = c.get<int>(get_arg(argi));
          cfmt += tool::ustring::format(W("%d"), n);
          ++argi;
        }
        else {
          if (*pc == 's' || *pc == 'S') {
            cfmt += 's';
            ustring str = c.get<ustring>(get_arg(argi));
            if (*pc == 'S')
              out.push(ustring::format(cfmt,xml_escape(str).c_str())());
            else
              out.push(ustring::format(cfmt, str.c_str())());
            ++argi;
            break;
          }
          else if (*pc == 'c' || *pc == 'C') {
            cfmt += 'c';
            int n = c.get<int>(get_arg(argi));
            out.push(ustring::format(cfmt,n));
            ++argi;
            break;
          }
          else if (wcschr(L"dioxXbu", *pc)) {
            cfmt += *pc;
            int n = c.get<int>(get_arg(argi));
            out.push(ustring::format(cfmt, n));
            ++argi;
            break;
          }
          else if (wcschr(L"fgGeE", *pc)) {
            cfmt += *pc;
            double n = c.get<double>(get_arg(argi));
            out.push(ustring::format(cfmt, n));
            ++argi;
            break;
          }
          // print data as JSON literal - suitable for parsing later by
          // parseData().
          else if (*pc == 'v' || *pc == 'V') {
            cfmt += *pc;
            hvalue v = get_arg(argi);
            ustring s = c.toJSONw(v, *pc == 'V' ? 2 : 0);
            out.push(s());
            ++argi;
            break;
          }
          else if (*pc == 0 || *pc == '%')
            break;
          else if (isdigit(*pc) || *pc == '.')
            cfmt += *pc;
          else {
            cfmt += *pc;
            out.push(cfmt());
            break;
          }
        }
      }
    }
    return out();
  }

  array<hvalue> scan(xcontext &c, ustring fmt, ustring str)
  {
    struct istream : public scanf_input_stream {
      const wchar *pc;
      virtual bool get(int &c) {
        int t = *pc++;
        if (t == 0) return false;
        c = t;
        return true;
      }
    };

    struct ostream : public scanf_output_stream {
      xcontext& c;
      array<hvalue> list;
      ostream(xcontext &ctx) : c(ctx) { }
      virtual bool out(long i) { 
        list.push(c.val(int(i)));
        return true;
      }
      virtual bool out(double f) {
        list.push(c.val(f));
        return true;
      }
      virtual bool out(const char *str, unsigned str_len) { return false; }
      virtual bool out(const wchar *str, unsigned str_len) {
        list.push(c.val(wchars(str,str_len)));
        return true;
      }
    };

    istream pr;
    pr.pc = str.cbegin();
    ostream ov(c);
    do_w_scanf(&pr, &ov, fmt);
    return ov.list;
  }


}
