#include "xdom.h"
#include "xom.h"
#include "xurl.h"

namespace qjs {

  using namespace html;

  static string url_host(xcontext& c, handle<xurl> purl)
  {
    if (purl->dport && (purl->dport != purl->port))
      return string::format("%s:%d", purl->hostname.c_str(), purl->port);
    else 
      return purl->hostname;
  }
  static string url_hostname(xcontext& c, handle<xurl> purl)  {  return purl->hostname; }
  static string url_href(xcontext& c, handle<xurl> purl) {  return purl->src; }
  static string url_origin(xcontext& c, handle<xurl> purl) { return purl->compose_host(); }
  static string url_password(xcontext& c, handle<xurl> purl) { return purl->password; }
  static string url_pathname(xcontext& c, handle<xurl> purl) { return purl->filename.is_defined() ? (CHARS("/") + purl->filename) : string(); }
  static string url_dir(xcontext& c, handle<xurl> purl) { return purl->dir(); }
  static void   url_set_pathname(xcontext& c, handle<xurl> purl, string pathname) {
    if (pathname.is_empty())
      purl->filename.clear();
    else if (pathname().starts_with('/'))
      purl->filename = pathname(1);
    else
      purl->filename = pathname;
  }
  static string url_filename(xcontext& c, handle<xurl> purl) { return purl->name_ext(); }
  static string url_extension(xcontext& c, handle<xurl> purl) { return purl->ext(); }
  static int    url_port(xcontext& c, handle<xurl> purl) { return purl->port; }
  static string url_protocol(xcontext& c, handle<xurl> purl) { return purl->protocol.is_defined() ? (purl->protocol + CHARS(":")) : string(); }
  static void   url_set_protocol(xcontext& c, handle<xurl> purl, string protocol) {
    if (protocol.is_empty())
      purl->protocol.clear();
    else if (protocol().ends_with(':'))
      purl->protocol = protocol().sub(0,-1);
    else
      purl->protocol = protocol;
  }
  static string url_search(xcontext& c, handle<xurl> purl) { return purl->params.is_defined() ? (CHARS("?") + purl->params):string(); }
  static void url_set_search(xcontext& c, handle<xurl> purl, string params) {
    if (params.is_empty())
      purl->params.clear();
    else if (params().starts_with('?'))
      purl->params = params(1);
    else
      purl->params = params;
  }
  static string url_username(xcontext& c, handle<xurl> purl) { return purl->username; }
  static string url_tostring(xcontext& c, handle<xurl> purl) { return purl->compose(); }
  static string url_guess_mime_type(xcontext& c, handle<xurl> purl) { 
    if( purl->ext().is_defined() )
      return string(tool::mime_for_ext(purl->ext()));
    return string();
  }

  static string url_hash(xcontext& c, handle<xurl> purl) { return purl->anchor.is_empty() ? string() : CHARS("#") + purl->anchor; }
  static void url_set_hash(xcontext& c, handle<xurl> purl, string hash) { 
    if (hash.is_empty())
      purl->anchor.clear();
    else if (hash().starts_with('#'))
      purl->anchor = hash(1);
    else 
      purl->anchor = hash; 
  }

  JSOM_PASSPORT_BEGIN(URL_def, qjs::xurl)
    JSOM_RO_PROP_DEF("[Symbol.toStringTag]", url_tostring),
    JSOM_FUNC_DEF("toString", url_tostring),
    JSOM_RO_PROP_DEF("host", url_host),
    JSOM_RO_PROP_DEF("hostname", url_hostname),
    JSOM_RO_PROP_DEF("href", url_href),
    JSOM_RO_PROP_DEF("origin", url_origin),
    JSOM_PROP_DEF("hash", url_hash, url_set_hash),
    JSOM_RO_PROP_DEF("password", url_password),
    JSOM_PROP_DEF("pathname", url_pathname, url_set_pathname),
    JSOM_RO_PROP_DEF("filename", url_filename),
    JSOM_RO_PROP_DEF("dir", url_dir),
    JSOM_RO_PROP_DEF("extension", url_extension),
    JSOM_RO_PROP_DEF("port", url_port),
    JSOM_PROP_DEF("protocol", url_protocol, url_set_protocol),
    JSOM_PROP_DEF("search", url_search, url_set_search),
    JSOM_FUNC_DEF("guessMimeType", url_guess_mime_type),
  JSOM_PASSPORT_END

  static string url_from_file_path(xcontext& c, ustring path) { return url::path_to_file_url(path); }
  static string url_to_file_path(xcontext& c, ustring url) { return url::file_url_to_u8_path(url); }
      
  JSOM_PASSPORT_BEGIN(URL_static_def, qjs::xcontext)
    JSOM_GLOBAL_FUNC_DEF("fromPath", url_from_file_path),
    JSOM_GLOBAL_FUNC_DEF("toPath", url_to_file_path),
  JSOM_PASSPORT_END

  JSClassID URL_class_id;

  void init_URL_class(context& c)
  {
    JS_NewClassID(&URL_class_id);

    static JSClassDef URL_class = {
      "URL",
      [](JSRuntime *rt, JSValue val)
      {
        xurl* pe = (xurl*)JS_GetOpaque(val,URL_class_id);  //object_of<html::element>(val);
        if (pe) pe->release();
      }
    };

    JS_NewClass(JS_GetRuntime(c), URL_class_id, &URL_class);
    JSValue url_proto = JS_NewObject(c);

    auto list = URL_def();
    JS_SetPropertyFunctionList(c, url_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      xcontext c(ctx);
      handle<xurl> purl = new xurl();
      string turl, burl;

      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto))
        goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, URL_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj))
        goto fail;
       
      turl = argc > 0 ? c.get<string>(argv[0]) : string();
      /*if (turl.is_undefined()) {
        JS_ThrowTypeError(ctx, "string expected");
        goto fail;
      }*/
      burl = argc > 1 ? c.get<string>(argv[1]) : string();

      if (burl.is_defined())
        turl = abspath(burl, turl);

      purl->parse(turl);

      JS_SetOpaque(obj, purl.detach());
      return obj;
    fail:
      //js_free(ctx, s);
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue url_class = JS_NewCFunction2(c, ctor, "URL", 2, JS_CFUNC_constructor, 0);
    JS_DefinePropertyValueStr(c, c.global(), "URL", JS_DupValue(c, url_class), JS_PROP_CONFIGURABLE);

    auto URL_statics = URL_static_def();
    JS_SetPropertyFunctionList(c, url_class, URL_statics.start, URL_statics.length);

    JS_SetConstructor(c, url_class, url_proto);
    JS_SetClassProto(c, URL_class_id, url_proto);
  }

}
