#pragma once

#include <cassert>
#include <memory>
#include <cstddef>
#include <algorithm>
#include <tuple>
#include <functional>
#include <stdexcept>

#include "tool/tool.h"

struct JSRuntime;
struct JSContext;
typedef uint64_t JSValue;
typedef uint32_t JSAtom;

typedef JSAtom symbol_t;

namespace html {
  struct document;
  struct event;
  class view;
}

namespace qjs {

  //using namespace tool;
  //using namespace html;

  class xview;

  /** Exception type.
   * Indicates that exception has occured in JS context.
   */
  class exception {};
  
  struct known_atoms_list {
#define ATOM(aname) JSAtom aname;
#define ATOM2(aname,sname) JSAtom aname;
#include "xatoms.h"
#undef ATOM
#undef ATOM2
  };

  /** JSValue with smart handler semantics.
   * A wrapper over (JSValue v).
   * Calls JS_FreeValueRT(runtime::current(), v) on destruction. Can be copied and moved.
   * A JSValue can be released by either JSValue x = std::move(value); or JSValue x = value.release(), then the Value becomes invalid and FreeValue won't be called
   */

  class hvalue
  {
  public:
    JSValue v;
  public:

    hvalue();
    hvalue(NO_INIT) {}
    hvalue(const hvalue& rhs);
    hvalue(hvalue&& rhs);
    hvalue(JSValue v);
    hvalue(JSContext * ctx, JSValue jsv);

    hvalue &operator=(const hvalue& rhs);

    explicit operator bool() const;
    bool operator !() const;

    /** Returns true if 2 values are the same (equality for arithmetic types or point to the same object) */
    bool operator==(const hvalue& rhs) const { return v == rhs.v; }
    bool operator==(JSValue rhs) const { return v == rhs; }

    bool operator!=(const hvalue& rhs) const { return v != rhs.v; }
    bool operator!=(JSValue rhs) const { return v != rhs; }

    ~hvalue();

    bool is_defined() const;
    bool is_undefined() const { return !is_defined(); }
    bool is_nothing() const;
    bool is_null() const;

    void clear();

    //void got_owner();

    JSValue tearoff(); // dont call freevalue

    operator JSValue() const { return v; }

  };

  /** Thin wrapper over JSRuntime * rt
 * Calls JS_FreeRuntime on destruction. noncopyable.
 */
  class hruntime
  {
    uint ref_count = 0;
    mutable JSRuntime * rt = nullptr;

    void init();

  public:
    known_atoms_list known_atoms;

    hruntime() { init(); }

    // noncopyable
    hruntime(const hruntime&) = delete;

    ~hruntime();

    void add_ref();
    void release();

    tool::dictionary<uint64, tool::value> globals; // JSAtom -> value(asset)
    hvalue shared;

    operator JSRuntime *() const { return rt; }

    static hruntime& current();

    static hruntime& get(JSRuntime *rt);

    tool::hash_table<tool::string, symbol_t> symbols;

    tool::string symbol_name_a(symbol_t atom);
    tool::string symbol_name_a(som_atom_t atom) { return symbol_name_a(symbol_t(atom)); }
    tool::ustring symbol_name_w(symbol_t atom);
    symbol_t symbol_of(tool::chars s);
    symbol_t symbol_of(tool::wchars s);

  };


  class hatom {
    JSAtom v;

  public:
    hatom(const hatom& rhs);
    hatom(hatom&& rhs);
    
    hatom(JSAtom jsa = 0);
    hatom(JSContext * ctx, tool::chars s);

    hatom &operator=(const hatom& rhs);

    ~hatom() { clear(); }

    void clear();

    operator JSAtom() const { return v; }
  };

  //class context;

  class hcontext
  {
    JSContext* pctx;
  public:

    hcontext() { pctx = nullptr; }
    hcontext(NO_INIT) {}
    hcontext(const hcontext& rhs);
    hcontext(hcontext&& rhs);
    hcontext(JSContext* pc);
    
    ~hcontext() { clear(); }

    void clear();

    hcontext &operator=(const hcontext& rhs);
    hcontext &operator=(JSContext* rhs);

    explicit operator bool() const { return pctx != nullptr; }
    bool operator !() const { return pctx == nullptr; }

    /** Returns true if 2 values are the same (equality for arithmetic types or point to the same object) */
    bool operator==(const hcontext& rhs) const { return pctx == rhs.pctx; }
    bool operator==(const JSContext* rhs) const { return pctx == rhs; }

    bool operator!=(const hcontext& rhs) const { return pctx != rhs.pctx; }
    bool operator!=(const JSContext* rhs) const { return pctx != rhs; }
    
    operator JSContext*() const { return pctx; }


  };

  struct object_proxy : public tool::object_proxy
  {
    hcontext hc;
    hvalue   hv;
    object_proxy(JSContext *c, JSValue obj = 0);

    virtual ~object_proxy() {}

    virtual tool::ustring class_name() const;
    virtual uint size() const;
    virtual tool::value get_by_index(uint n) const;
    virtual bool        set_by_index(uint n, const tool::value &val);
    virtual tool::value get_by_key(const tool::value &key) const;
    virtual bool        set_by_key(const tool::value &key, const tool::value &val);

    virtual tool::value invoke(const tool::value &This, uint argc, const tool::value *argv);

    virtual bool get_user_data(void **ppv) const;
    virtual bool set_user_data(void *pv);

    virtual bool visit(const tool::kv_visitor &vis) const;
    virtual bool equal(const tool::object_proxy *pv) const;
    
    virtual bool isolate(tool::value &r) const;
  };

  bool call(xview* pxv, tool::chars path, int argc, const tool::value* argv, tool::value* retval);
  bool call(html::element* pel, tool::string name, int argc, const tool::value* argv, tool::value* retval);
  bool call(html::document* pd, tool::string name, int argc, const tool::value* argv, tool::value* retval);
  bool eval(xview* pxv, tool::chars script, tool::value* retval);


} // namespace qjs

#define NOTHING_VALUE (qjs::hvalue())
