
#include "xjs.h"
#include "xconv.h"
#include "xview.h"
#include "quickjs/quickjs.h"
#include "quickjs/quickjs-libc.h"

namespace qjs {

  hruntime& hruntime::current() {
    static thread_local hruntime _rt;
    return _rt;
  }
 
  hruntime::~hruntime()
  {
    if (rt) {
      shared.clear();
      for (auto atom : symbols.elements())
        JS_FreeAtomRT(rt, atom);
      symbols.clear();
#define ATOM(aname) JS_FreeAtomRT(rt, known_atoms.aname);
#define ATOM2(aname,sname) JS_FreeAtomRT(rt, known_atoms.aname);
#include "xatoms.h"
#undef ATOM
#undef ATOM2
      JS_FreeRuntime(rt);
      rt = nullptr;
    }
  }

  void hruntime::add_ref() {
    //---if (!rt) init(); // may happen if app uses multiple run()'s
    ++ref_count;
  }

  void hruntime::release() {
    if (!rt) return;
    if (ref_count == 0 || --ref_count == 0) {
      globals.clear();
      JS_CleanupRuntime(rt);
    } else
      JS_RunGC(rt);
  }
 
  tool::string hruntime::symbol_name_a(symbol_t atom) {
    char buf[64] = { 0 };
    tool::string name = JS_AtomGetStrRT(rt, buf, items_in(buf), atom);
    return name;
  }

  tool::ustring hruntime::symbol_name_w(symbol_t atom) {
    return tool::u8::cvt(symbol_name_a(atom));
  }

  symbol_t hruntime::symbol_of(tool::chars s) {
    bool created = false;
    symbol_t& r = symbols.get_ref(s, created);
    if(created) 
      r = JS_NewAtomLenRT(hruntime::current(), s.start, s.size());
    return r;
  }

  symbol_t hruntime::symbol_of(tool::wchars s) {
    return symbol_of(u8::cvt(s).chars());
  }

  void dump_sciter_object(int class_id, void* p) {
#ifdef _DEBUG
    if (class_id == qjs::Element_class_id) {
      ((html::element*)(html::node*)p)->dbg_report(" ");
    }
    else if (class_id == qjs::Document_class_id) {
      ((html::document*)(html::node*)p)->dbg_report(" ");
    }
#endif
  }


  char * module_url_resolver(JSContext *ctx, const char *module_base_name, const char *module_name, void *opaque)
  {
    if (module_name[0] == '@') // native module
      return js_strdup(ctx, module_name); // single nmtoken like "sciter", "sys", "os", etc.
//SCRIPT_NAME:
    tool::url base(module_base_name);
    tool::url that(module_name);
    that.absolute(base);
    if (that.ext().is_empty())
      that.filename += ".js";
    tool::string r = that.compose();
    return js_strndup(ctx, r.cbegin(), r.size());
  }

  JSModuleDef *module_loader(JSContext *ctx,
    const char *module_name, void *opaque)
  {
    if (module_name[0] == '@') // native module
    {
      JS_ThrowReferenceError(ctx, "Unknown built-in module '%s'", module_name);
      return nullptr;
    }

    JSModuleDef *m;

    html::handle<html::request> rq = new html::request(module_name, html::RESOURCE_DATA_TYPE::DATA_SCRIPT);

#ifdef DEBUG
    dbg_printf("loading %s\n", module_name);
#endif

    context& c = context::inst(ctx);
    if (auto pv = c.pview()) {
      rq->dst_view = pv;
      rq->dst = c.pdoc();
      if (!pv->load_data(rq, true)) {
        JS_ThrowReferenceError(ctx, "could not load module filename '%s'", module_name);
        return nullptr;
      }
    }
   
    JSValue func_val;

    /* compile the module */
    func_val = JS_Eval2(ctx, (char *)rq->data.cbegin(), rq->data.size(), module_name,
      JS_EVAL_TYPE_MODULE | JS_EVAL_FLAG_COMPILE_ONLY,1);
    if (JS_IsException(func_val)) {
      c.report_exception();
      return NULL;
    }

    /* the module is already referenced, so we must free it */
    m = (JSModuleDef *)JS_VALUE_GET_PTR(func_val);
    JS_FreeValue(ctx, func_val);
    
    return m;
  }

  void promise_rejection_tracker(JSContext *ctx, JSValueConst promise,
    JSValueConst reason,
    JS_BOOL is_handled, void *opaque)
  {
    if (!is_handled) {
      context& c = context::inst(ctx);
      c.report_promise_rejection(reason);
    }
  }

  void hruntime::init() {
    if (!rt) {
      rt = JS_NewRuntime();
      if (!rt)
        throw std::runtime_error{ "qjs: Cannot create runtime" };
      JS_SetModuleLoaderFunc(rt, module_url_resolver, module_loader, 0);
      JS_SetRuntimeOpaque(rt, this);
      JS_SetHostPromiseRejectionTracker(rt, promise_rejection_tracker,this);
      JS_SetDumpCustomObject(rt, dump_sciter_object);

      auto sym = [this](chars nm) -> JSAtom { 
        if (nm.starts_with(CHARS("Symbol.")))
          return JS_NewAtomSymbolLenRT(rt, nm.start, nm.length);
        else
          return JS_NewAtomLenRT(rt, nm.start, nm.length);  
      };
           
#define ATOM(aname) known_atoms.aname = sym(CHARS(#aname));
#define ATOM2(aname,sname) known_atoms.aname = sym(CHARS(sname));
#include "xatoms.h"
#undef ATOM
#undef ATOM2

    }
  }

  hruntime& hruntime::get(JSRuntime* prt) {
    return *static_cast<hruntime*>(JS_GetRuntimeOpaque(prt));
  }
  
  hvalue::hvalue() : v(JS_UNINITIALIZED) {}

  hvalue::hvalue(const hvalue& rhs)
  {
    v = JS_DupValueRT(hruntime::current(), rhs.v);
  }

  hvalue &hvalue::operator=(const hvalue& rhs) {
    if (rhs != *this && v != rhs.v) {
      clear();
      v = JS_DupValueRT(hruntime::current(), rhs.v);
    }
    return *this;
  }

  hvalue::hvalue(hvalue&& rhs)
  {
    v = rhs.v;
    rhs.v = JS_UNINITIALIZED;
  }

  hvalue::hvalue(JSValue jsv)
  {
    v = jsv; // note this value is addrefed already
  }

  hvalue::hvalue(JSContext * ctx, JSValue jsv)
  {
    v = JS_DupValue(ctx, jsv); // 
  }

  hvalue::operator bool() const { return is_defined(); }
  bool hvalue::operator !() const { return !is_defined(); }
  
  hvalue::~hvalue()
  {
    if (JS_VALUE_HAS_REF_COUNT(v)) {
      JSRuntime * rt = hruntime::current();
      assert(rt);
      if(rt) JS_FreeValueRT(rt, v);
    }
  }

  bool hvalue::is_defined() const {
    return v != JS_UNINITIALIZED && v != JS_UNDEFINED;
  }

  bool hvalue::is_null() const {
    return JS_IsNull(v);
  }
  
  bool hvalue::is_nothing() const {
    return v == JS_UNINITIALIZED;
  }

  //bool hvalue::is_string() const {
  //  return JS_IsString(v);
  //}

  void hvalue::clear()
  {
    if (JS_VALUE_HAS_REF_COUNT(v))
      JS_FreeValueRT(hruntime::current(), v);
    v = JS_UNINITIALIZED;
  }

  /*void hvalue::got_owner()
  {
    if (JS_VALUE_HAS_REF_COUNT(v))
      JS_DupValueRT(hruntime::current(), v);
  }*/


  JSValue hvalue::tearoff() // dont call freevalue
  {
    JSValue t = v;
    v = JS_UNINITIALIZED;
    return t;
  }

  hatom::hatom(JSAtom a) {
    v = a;
  }

  hatom::hatom(JSContext* ctx, tool::chars s) {
    v = JS_NewAtomLen(ctx, s.start, s.length);
  }

  hatom &hatom::operator=(const hatom& rhs) {
    if (rhs != *this && v != rhs.v) {
      clear();
      v = JS_DupAtomRT(hruntime::current(), rhs.v);
    }
    return *this;
  }

  void hatom::clear() {
    if (v) {
      JS_FreeAtomRT(hruntime::current(),v);
      v = JS_ATOM_NULL;
    }
  }

  hcontext::hcontext(const hcontext& rhs) {
    pctx = JS_DupContext(rhs.pctx);
  }
  hcontext::hcontext(hcontext&& rhs) {
    pctx = rhs.pctx;
    rhs.pctx = nullptr;
  }
  hcontext::hcontext(JSContext* pc) {
    pctx = JS_DupContext(pc);
  }
  void hcontext::clear() {
    if (pctx) {
      JS_FreeContext(pctx);
      pctx = nullptr;
    }
  }

  hcontext& hcontext::operator=(const hcontext& rhs) {
    if(pctx)
      JS_FreeContext(pctx);
    pctx = JS_DupContext(rhs.pctx);
    return *this;
  }
  hcontext& hcontext::operator=(JSContext* rhs) {
    if (pctx) {
      qjs::context& c = qjs::context::inst(pctx);
      c.purge_airborns();
      c.set_prop(c.known_atoms().document, c.global(), JS_UNINITIALIZED);
      c.set_prop(c.known_atoms().window, c.global(), JS_UNINITIALIZED);
      //c.set_prop(c.known_atoms().console, c.global(), JS_UNDEFINED);
      c.unhandled_exception_handler = JS_UNINITIALIZED;
      c.console_output_handler = JS_UNINITIALIZED;
      c.request_complete_handler = JS_UNINITIALIZED;
      c.sublimated_references.clear();
#ifdef CONFIG_DEBUGGER
      c.breakpoint_handler = JS_UNINITIALIZED;
      c.breakpoints.clear();
#endif
      if (c.pview()->debug_output == &c)
        c.pview()->set_debug_output(nullptr);
      //JS_RunGC(JS_GetRuntime(pctx));
      JS_FreeContext(pctx);
    }
    pctx = rhs;
    return *this;
  }

  void exec(const qjs::hcontext& hc, function<bool()> &&pc) {
    xcontext c(hc);
    if(c.pview())
      c.pview()->exec(std::move(pc));
  }

#define EXEC_START if (hc && hv) exec(hc,[this,&r]() -> bool {
#define EXEC_START_1(p) if (hc && hv) exec(hc,[this,&r,p]() -> bool {
#define EXEC_START_2(p1, p2) if (hc && hv) exec(hc,[this,&r,p1,p2]() -> bool {
#define EXEC_START_3(p1, p2, p3) if (hc && hv) exec(hc,[this,&r,p1,p2,p3]() -> bool {
#define EXEC_END                                                               \
  return true;                                                                 \
  });
   
  object_proxy::object_proxy(JSContext *c, JSValue obj) : hc(c), hv(obj) {
    JS_DupValue(c, hv);
  }

  tool::ustring object_proxy::class_name() const {
    tool::ustring r;
    EXEC_START
      hvalue s = JS_ToString(hc, hv);
      r = conv<tool::ustring>::unwrap(hc, s);
    EXEC_END
    return r;
  }
  uint object_proxy::size() const {
    uint r = 0;
    EXEC_START
      JSValue vlen = JS_GetPropertyStr(hc, hv, "length");
      r = conv<uint>::unwrap(hc, vlen);
    EXEC_END
    return r;
  }
  tool::value object_proxy::get_by_index(uint n) const {
    tool::value r;
    EXEC_START_1(n)
      JSValue v = JS_GetPropertyUint32(hc, hv, n);
      r = conv<value>::unwrap(hc, v);
    EXEC_END
    return r;
  }
  bool object_proxy::set_by_index(uint n, const tool::value &val)
  {
    bool r = false;
    EXEC_START_2(n,val)
      r = JS_SetPropertyUint32(hc, hv, n, conv<value>::wrap(hc, val)) > 0;
    EXEC_END
    return r;
  }
  tool::value object_proxy::get_by_key(const tool::value &key) const {
    tool::value r;
    EXEC_START_1(key)
      string sk = u8::cvt(key.to_string());
      JSValue v = JS_GetPropertyStr(hc, hv, sk);
      r = conv<value>::unwrap(hc, v);
    EXEC_END
    return r;
  }

  bool object_proxy::set_by_key(const tool::value &key, const tool::value &val)
  {
    bool r = false;
    EXEC_START_2(key, val)
      string sk = u8::cvt(key.to_string());
      r = JS_SetPropertyStr(hc, hv, sk, conv<value>::wrap(hc, val)) > 0;
    EXEC_END
    return r;
  }

  tool::value object_proxy::invoke(const tool::value &This, uint argc, const tool::value *argv)
  {
    tool::value r;
    EXEC_START_3(This, argc, argv)
      xcontext c(hc);
      if (JS_IsFunction(c, hv)) {
        buffer<hvalue, 8> args(argc);
        for (uint n = 0; n < argc; ++n)
          args[n] = c.val(argv[n]);
        hvalue rv = JS_Call(c, hv, c.val(This), int(argc), (JSValue*)args.begin());
        if (c.is_exception(rv)) {
          tool::ustring errs = c.report_exception();
          r = value::make_error(errs);
        }
        else {
          r = conv<value>::unwrap(hc, rv);
        }
      }
    EXEC_END
    return r;
  }

  bool object_proxy::get_user_data(void **ppv) const { return false; }
  bool object_proxy::set_user_data(void *pv) { return false; }

  bool object_proxy::visit(const tool::kv_visitor &vis) const
  {
    bool r = false;
    EXEC_START_1(vis)
      if (JS_IsArray(hc, hv)) {
        JSValue vlen = JS_GetPropertyStr(hc, hv, "length");
        auto len = conv<uint>::unwrap(hc, vlen);
        for (uint i = 0; i < len; i++) {
          hvalue el = JS_GetPropertyUint32(hc, hv, i);
          if (!vis(value(i), conv<value>::unwrap(hc, el)))
            break;
        }
        r = true;
      }
      else if (JS_IsObject(hv))
      {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(hc, &tab, &len, hv, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars name(hc,tab[n].atom);
          hvalue val = JS_GetProperty(hc, hv, tab[n].atom);
          if (!vis(value(name), conv<value>::unwrap(hc, val)))
            break;
        }
        js_free_prop_enum(hc, tab, len);
        r = true;
      }
    EXEC_END
    return r;
  }

  bool object_proxy::equal(const tool::object_proxy *pv) const
  {
    return hv == static_cast<const qjs::object_proxy*>(pv)->hv 
        && hc == static_cast<const qjs::object_proxy*>(pv)->hc;
  }

  bool object_proxy::isolate(tool::value &r) const 
  {
    if (!hc || !hv)
      return false;
    EXEC_START
      if (JS_IsArray(hc, hv)) {
        JSValue vlen = JS_GetPropertyStr(hc, hv, "length");
        auto len = conv<uint>::unwrap(hc, vlen);
        r = value::make_array();
        for (uint i = 0; i < len; i++) {
          hvalue el = JS_GetPropertyUint32(hc, hv, i);
          value t = conv<value>::unwrap(hc, el);
          t.isolate();
          r.set_element(i, t);
        }
      }
      else if (JS_IsObject(hv) || JS_IsFunction(hc,hv))
      {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(hc, &tab, &len, hv, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        r = value::make_map();
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars name(hc,tab[n].atom);
          hvalue val = JS_GetProperty(hc, hv, tab[n].atom);
          value t = conv<value>::unwrap(hc, val);
          t.isolate();
          r.set_item(name.start,t);
        }
        js_free_prop_enum(hc, tab, len);
      }
    EXEC_END
      return true;
  }

  bool call(xview* pxv, tool::chars path, int argc, const tool::value* argv, tool::value* retval)
  {
    if (!pxv->doc())
      return false;
    context& c = context::inst(pxv->doc());

    hvalue fun = c.eval_silent(path);
    if (!JS_IsFunction(c,fun))
      return false;

    array<hvalue> args(argc);
    for (int n = 0; n < argc; ++n)
      args[n] = c.val(argv[n]);

    hvalue r = JS_Call(c, fun, c.global(), args.size(), (JSValue*)args.cbegin());
    c.check_pending_job_status();
    if (retval)
      *retval = c.get<value>(r);
    return true;
  }

  bool call(html::element* pel, tool::string name, int argc, const tool::value* argv, tool::value* retval)
  {
    document* pdoc = pel->doc();
    if (!pdoc)
      return false;
    context& c = context::inst(pdoc);

    hvalue obj = c.val(pel);

    hvalue fun = c.get_prop<hvalue>(name, obj);
    if (!c.is_function(fun))
      return false;

    array<hvalue> args(argc);
    for (int n = 0; n < argc; ++n)
      args[n] = c.val(argv[n]);

    hvalue r = JS_Call(c, fun, obj, args.size(), (JSValue*)args.cbegin());
    c.check_pending_job_status();
    if (retval)
      *retval = c.get<value>(r);
    return true;
  }

  bool call(html::document* pdoc, tool::string path, int argc, const tool::value* argv, tool::value* retval)
  {
    context& c = context::inst(pdoc);

    hvalue fun = c.eval_silent(path);
    if (!JS_IsFunction(c, fun))
      return false;

    array<hvalue> args(argc);
    for (int n = 0; n < argc; ++n)
      args[n] = c.val(argv[n]);

    hvalue r = JS_Call(c, fun, c.global(), args.size(), (JSValue*)args.cbegin());
    c.check_pending_job_status();
    if (retval)
      *retval = c.get<value>(r);
    return true;
  }

  bool eval(xview* pxv, tool::chars script, tool::value* retval) {
    if (!pxv->doc())
      return false;
    context& c = context::inst(pxv->doc());
    hvalue r = c.eval_silent(script);
    if (retval)
      *retval = c.get<value>(r);
    return true;
  }
}

extern "C" uv_loop_t *tjs_get_loop(JSContext *ctx) {
  uv_loop_t* loop = async::dispatch::current()->uv_loop();
  return loop;
}
