#pragma once

#include "xconv.h"
#include "xcontext.h"
#include "xcolor.h"

#include "html/html.h"

namespace qjs {

  using namespace html;
  using namespace gool;

  extern JSClassID Graphics_class_id;

  struct xpath : public resource
  {
    handle<gool::path> cpath;
    xpath() {}
    xpath(const gool::path* pp): cpath(pp) {}
  };

  struct xtext : public resource {
    handle<text_layout> tl;
    xtext(xcontext& c, wchars t, wchars cls) {
      if (html::view* pv = c.pview())
      {
        tl = pv->create_text_layout(t);
        if (cls) tl->set_class(cls);
        tl->set_host(c.pdoc());
      }
    }
    void set_text(wchars t) {
      tl->set_text(t);
    }
  };

  extern JSClassID Graphics_Brush_class_id;

  typedef handle<gool::brush> hbrush;

  template <> struct conv<gool::brush*>
  {
    static gool::brush* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      gool::brush* pb = (gool::brush*)JS_GetOpaque(v, Graphics_Brush_class_id);
      return pb;
    }

    static JSValue wrap(JSContext * ctx, gool::brush* pb) noexcept
    {
      if (!pb)
        return JS_NULL;
      JSValue obj = JS_NewObjectClass(ctx, Graphics_Brush_class_id);
      pb->add_ref();
      JS_SetOpaque(obj, pb);
      return obj;
    }

    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Graphics_Brush_class_id) != NULL; }
  };

  template <> struct conv<hbrush>
  {
    static hbrush unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<gool::brush*>::unwrap(ctx, v);
    }
    static JSValue wrap(JSContext * ctx, hbrush pi) noexcept
    {
      return conv<gool::brush*>::wrap(ctx, pi);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<gool::brush*>::isa(ctx, v); }
  };


  struct xgraphics : public resource 
  {
    handle<gool::bitmap>        bmp;
    handle<gool::graphics>      gfx;
    bool                        is_image_gfx = false;
    weak_handle<html::view>     pv;
    weak_handle<html::element>  host;
    handle<gool::path>          cpath;
    handle<html::style>         style;
            
    xgraphics(html::element* phost, gool::size sz) {
      host = phost;
      pv = host->pview();
      style = new html::style();
      style->inherit(phost->get_style(*(pv.ptr())));
      bmp = new gool::bitmap(sz, true, false /*???*/);
    }

    xgraphics(html::view* pview, html::element* phost, gool::graphics* pg, bool for_mage) {
      host = phost;
      pv = pview;
      style = new html::style();
      style->inherit(phost->get_style(*(pv.ptr())));
      gfx = pg;
      is_image_gfx = for_mage;
    }

    ~xgraphics() {
      gfx.clear();
    }

    gool::graphics* get_gfx(xcontext& c) {
      if (!gfx && bmp) {
        //gool::graphics *pgfx = c.pview()->surface();
        //if (pgfx)
        //  WRONG for OFFSCRREN case: gfx = c.pview()->app->create_bitmap_graphics(pgfx, bmp, argb::no_color());
        //else
        gfx = c.pview()->app->create_bitmap_bits_graphics(bmp, argb::no_color());
        host->refresh(*pv);
        setup_gfx(c);
      }
      assert(gfx);
      return gfx;
    }

    void detach() {
      bmp.clear();
      gfx.clear();
      cpath.clear();
      style.clear();
    }

    void setup_gfx(xcontext& c) {
      if (!pv) return;
      gfx->scale(pv->pixels_per_dip(sizef(1, 1)));
      gfx->set_stroke(style->stroke_color.to_argb());
      gfx->cap_style(gool::CAP_STYLE(style->stroke_linecap.val()));
      gfx->line_join(gool::LINE_JOIN(style->stroke_linejoin.val()));
      gfx->set_stroke_width(html::dips(*(pv.ptr()),host,style->stroke_width).width_f());
      if (style->stroke_dasharray.is_array())
        gfx->custom_dash_style(style->stroke_dasharray.get_array<float>()(), dips(*(pv.ptr()), host, style->stroke_dashoffset).width_f());
      gfx->set_fill(style->fill_color.to_argb());
    }

    handle<gool::bitmap> get_bitmap_for(gool::graphics* master) {
      if (gfx)
        gfx = nullptr;
      return bmp;
    }

    void commit() {
      if (gfx)
        gfx->flush();
    }

    static xgraphics* begin_path(xcontext& hc, xgraphics* pg) { 
      pg->cpath = hc.pview()->app->create_path(); return pg; 
    }
    static xgraphics* close_path(xcontext& hc, xgraphics* pg) { 
      if (pg->cpath) pg->cpath->close(); return pg;
    }
    static xgraphics* move_to(xcontext& hc, xgraphics* pg, float x, float y) { 
      if (pg->cpath) pg->cpath->move_to(gool::pointf(x,y)); return pg;
    }
    static xgraphics* line_to(xcontext& hc, xgraphics* pg, float x, float y) { 
      if (pg->cpath) pg->cpath->line_to(gool::pointf(x, y)); return pg; }

    static xgraphics* bezier_curve_to(xcontext& hc, xgraphics* pg, float cp1x, float cp1y, float cp2x, float cp2y, float x, float y) {
      if (pg->cpath) pg->cpath->cubic_to(pointf(x, y), pointf(cp1x, cp1y), pointf(cp2x, cp2y)); 
      return pg; 
    }
    static xgraphics* quadratic_curve_to(xcontext& hc, xgraphics* pg, float cpx, float cpy, float x, float y) {
      if (pg->cpath) pg->cpath->quadratic_to(pointf(x, y), pointf(cpx, cpy));
      return pg;
    }

    static xgraphics* arc(xcontext& hc, xgraphics* pg, float x, float y, float radius, float startAngle, float endAngle, bool anticlockwise) {
      if (!pg->cpath)
        pg->cpath = hc.pview()->app->create_path();
      pg->cpath->arc(pointf(x, y), sizef(radius), startAngle, (anticlockwise ? -endAngle : endAngle) - startAngle);
      return pg;
    }

    static xgraphics* arc_to(xcontext& hc, xgraphics* pg, float x1, float y1, float x2, float y2, float radius) {
      if (pg->cpath)
        pg->cpath->circ_arc_to(pointf(x1, y1), pointf(x2, y2), radius);
      return pg;
    }

    static xgraphics* ellipse(xcontext& hc, xgraphics* pg, float x, float y, float radiusX, float radiusY, float rotation, float startAngle, float endAngle, bool anticlockwise) {
      auto gfx = pg->get_gfx(hc);
      if (gfx && pg->cpath)
      {
        pg->cpath->arc(pointf(x, y), sizef(radiusX, radiusY), startAngle, (anticlockwise ? -endAngle : endAngle) - startAngle);
      }
      return pg;
    }

    static xgraphics* rect(xcontext& hc, xgraphics* pg, float x, float y, float w, float h) {
      auto gfx = pg->get_gfx(hc);
      if (gfx && pg->cpath)
      {
        pg->cpath->add_rect(pointf(x, y), sizef(w, h));
      }
      return pg;
    }

    static xgraphics* stroke(xcontext& hc, xgraphics* pg, hvalue p1) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        handle<gool::path> pp = pg->cpath;
        if (hc.isa<xpath*>(p1))
          pp = hc.get<xpath*>(p1)->cpath;
        if(pp)
          gfx->draw_path(pp, true, false);
      }
      return pg; 
    }

    static xgraphics* fill(xcontext& hc, xgraphics* pg, hvalue p1, hvalue p2 ) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        string fill_rule;
        handle<gool::path> pp = pg->cpath;
        if (hc.isa<xpath*>(p1)) {
          pp = hc.get<xpath*>(p1)->cpath;
          fill_rule = hc.get<string>(p2);
        } else 
          fill_rule = hc.get<string>(p1);
        bool fill_even_odd = true;
        if (fill_rule == CHARS("nonzero"))
          fill_even_odd = false;
        gfx->set_fill_even_odd(fill_even_odd);
        if(pp)
          gfx->draw_path(pp, false, true);
      }
      return pg;
    }

    static xgraphics* clear_rect(xcontext& hc, xgraphics* pg, float x, float y, float w, float h) {
      if (auto gfx = pg->get_gfx(hc)) {
        gool::rect rc = gool::rect::make_xywh((int)roundf(x), (int)roundf(y), (int)roundf(w), (int)roundf(h));
        gfx->clear(rc);
      }
      return pg;
    }

    static xgraphics* fill_rect(xcontext& hc, xgraphics* pg, float x, float y, float w, float h) {
      if (auto gfx = pg->get_gfx(hc)) {
        gfx->draw_rectangle(gool::pointf(x, y), gool::sizef(w, h),false, true);
      }
      return pg;
    }

    static xgraphics* stroke_rect(xcontext& hc, xgraphics* pg, float x, float y, float w, float h) {
      if (auto gfx = pg->get_gfx(hc)) {
        gfx->draw_rectangle(gool::pointf(x, y), gool::sizef(w, h), true, false);
      }
      return pg;
    }

    static float get_stroke_width(xcontext& hc, xgraphics* pg) {
      auto gfx = pg->get_gfx(hc);
      if (!gfx) return 0;
      //return gfx->set_stroke_width
      return 0;
    }

    static void set_stroke_width(xcontext& hc, xgraphics* pg, float w) {
      auto gfx = pg->get_gfx(hc);
      if (!gfx) return;
      pg->style->stroke_width = size_v(w,size_v::unit_type::px);
      return gfx->set_stroke_width(w);
    }

    static tool::string get_stroke_style(xcontext& hc, xgraphics* pg) {
      return tool::string();
    }

    static void set_stroke_style(xcontext& hc, xgraphics* pg, qjs::hvalue sty) {
      auto gfx = pg->get_gfx(hc);
      if (!gfx)
        return;
      xcontext& c = static_cast<xcontext&>(hc);

      if (c.isa<gool::argb>(sty)) {
        gool::argb argb = c.get<gool::argb>(sty);
        pg->style->stroke_color = argb;
        gfx->set_stroke(argb);
      }
      /*else if (c.is_object(sty)) {
        float width = c.get_prop<float>(c.known_atoms().width, sty);
        pg->style->stroke_width = size_v(width, size_v::unit_type::dip);
        gfx->set_stroke_width(width);

        int type = 0;
        gool::argb argb;
        if (c.check_prop(c.known_atoms().color, sty, argb)) {
          pg->style->stroke_color = argb;
          gfx->set_stroke(argb);
          ++type;
        }
      }*/
      else if( c.is_string(sty)) {
        ustring st = c.get<ustring>(sty);
        /*html::document* pd = hc.pdoc();
        tool::string url = pd->doc_url();
        html::css_istream is(st(), url);
        tool::value val;
        if (!html::parse_value(pd, url, is, val) || !val.is_color())
          throw qjs::om::type_error("wrong style definition");
        color_v clr;
        html::color_value(clr, val,gfx->current_element ? gfx->current_element->get_style(*c.pview()): nullptr );
        pg->style->stroke_color = clr;*/

        style_prop_list list;
        parse_css_property_as(hc.pdoc(), cssa_stroke, st(), &list);
        list.apply_to(html::element_context(hc, pg->host), *pg->style, style::INHERIT_ALL, false);
        gfx->set_stroke(pg->style->stroke_color.to_argb());
      } 
      else if (c.isa<hbrush>(sty)) {
        hbrush hb = c.get<hbrush>(sty);
        hb->set_stroke_to(gfx);
      }
      else {
        pg->style->stroke_color.clear();
        gfx->set_stroke();
      }
    }

    static tool::string get_fill_style(xcontext& hc, xgraphics* pg) {
      return tool::string();
    }

    static void set_fill_style(xcontext& hc, xgraphics* pg, qjs::hvalue sty) {
      auto gfx = pg->get_gfx(hc);
      if (!gfx)
        return;
      xcontext& c = static_cast<xcontext&>(hc);
      if (c.isa<gool::argb>(sty)) {
        gool::argb argb = c.get<gool::argb>(sty);
        pg->style->fill_color = argb;
        gfx->set_fill(argb);
      }
      else if (c.is_string(sty)) {
        ustring st = c.get<ustring>(sty);
        style_prop_list list;
        parse_css_property_as(hc.pdoc(), cssa_fill, st(), &list);
        list.apply_to(html::element_context(hc, pg->host), *pg->style, style::INHERIT_ALL, false);
        gfx->set_fill(pg->style->fill_color.to_argb());
      }
      else if (c.isa<hbrush>(sty)) {
        hbrush hb = c.get<hbrush>(sty);
        hb->set_fill_to(gfx);
      }
      else {
        pg->style->fill_color.clear();
        pg->style->fill_id.clear();
        pg->style->fill_opacity.clear();
        gfx->set_fill();
      }
        
    }

    static void set_font(xcontext& hc, xgraphics* pg, ustring sty) {
      auto gfx = pg->get_gfx(hc);
      if (!gfx)
        return;
      style_prop_list list;
      parse_css_property_as(hc.pdoc(), cssa_font, sty(), &list);
      list.apply_to(html::element_context(hc,pg->host), *pg->style, style::INHERIT_CHAR, false);
    }

    static ustring get_font(xcontext& hc, xgraphics* pg) {
      return ustring();
    }
    
    static void set_line_cap(xcontext& hc, xgraphics* pg, html::stroke_linecap_ev ev)
    {
      auto gfx = pg->get_gfx(hc);
      if (!gfx) return;
      pg->style->stroke_linecap = ev;
      gfx->cap_style(gool::CAP_STYLE(pg->style->stroke_linecap.val()));
    }
    static html::stroke_linecap_ev get_line_cap(xcontext& hc, xgraphics* pg)
    {
      return pg->style->stroke_linecap;
    }

    static void set_line_join(xcontext& hc, xgraphics* pg, html::stroke_linejoin_ev ev)
    {
      auto gfx = pg->get_gfx(hc);
      if (!gfx) return;
      pg->style->stroke_linejoin = ev;
      gfx->line_join(gool::LINE_JOIN(pg->style->stroke_linejoin.val()));
    }
    static html::stroke_linejoin_ev get_line_join(xcontext& hc, xgraphics* pg)
    {
      return pg->style->stroke_linejoin;
    }

    static bool set_line_dash(xcontext& hc, xgraphics* pg, array<float> pattern)
    {
      auto gfx = pg->get_gfx(hc);
      if (!gfx) return false;
      if (pattern.length()) {
        pg->style->stroke_dasharray = value::make_array(pattern());
        gfx->custom_dash_style(pattern(), dips(*(pg->pv.ptr()),pg->host,pg->style->stroke_dashoffset).width_f() );
      }
      else {
        pg->style->stroke_dasharray.clear();
        gfx->dash_style(gool::DASH_STYLE_SOLID);
      }
      return true;
    }

    static xgraphics* save(xcontext& hc, xgraphics* pg) {
      auto gfx = pg->get_gfx(hc);
      if (gfx)
        gfx->push_state();
      return pg;
    }
    static xgraphics* restore(xcontext& hc, xgraphics* pg) {
      auto gfx = pg->get_gfx(hc);
      if (gfx)
        gfx->pop_state();
      return pg;
    }

    static xgraphics* rotate(xcontext& hc, xgraphics* pg, float angle, float_v x, float_v y) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        if(x.is_defined() && y.is_defined())
          gfx->rotate(angle,pointf(x,y));
        else
          gfx->rotate(angle);
      }
      return pg;
    }

    static xgraphics* scale(xcontext& hc, xgraphics* pg, float x, float y, float_v cx, float_v cy) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        if( cx.is_defined() && cy.is_defined() )
          gfx->scale(sizef(x, y),sizef(cx.val(),cy.val()));
        else 
          gfx->scale(sizef(x, y));
      }
      return pg;
    }

    static xgraphics* translate(xcontext& hc, xgraphics* pg, float x, float y) {
      auto gfx = pg->get_gfx(hc);
      if (gfx)
        gfx->translate(pointf(x, y));
      return pg;
    }

    static xgraphics* transform(xcontext& hc, xgraphics* pg, float a, float b, float c, float d, float e, float f) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        gool::affine_mtx_f m(a,b,c,d,e,f);
        gfx->transform(m);
      }
      return pg;
    }

    static xgraphics* reset_transform(xcontext& hc, xgraphics* pg, float a, float b, float c, float d, float e, float f) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        gfx->reset_transform();
        gool::affine_mtx_f m(a, b, c, d, e, f);
        gfx->transform(m);
      }
      return pg;
    }

    static xgraphics* fill_text(xcontext& hc, xgraphics* pg, ustring text, float x, float y, float maxWidth) {
      auto gfx = pg->get_gfx(hc);
      if (gfx) {
        handle<text_layout> tl = hc.pview()->create_text_layout(text());
        tl->set_style(pg->style);
        tl->set_host(pg->host);
        //if (maxWidth > 0)
        //  tl->set_width(maxWidth);
        //else
          tl->set_width(10000.0f);
        rectf rc(pointf(x,y), tl->get_box());
        gfx->draw(tl, rc.s, pg->style->fill_color.to_argb());
      }
      return pg;
    }

    static value parse_filter(xcontext& c, ustring f) {
      html::css_istream ss(f, string());
      tool::value val;
      //ss.parse_value(val);
      html::parse_value(c.pdoc(), string(), ss, val);
      return val;
    }

    static xgraphics* push_layer(xcontext& c, xgraphics* pg, array<hvalue> args) {
      auto gfx = pg->get_gfx(c);
      if (!gfx) return pg;

      byte opacity = 255;
      function<bool(gool::filter_graph_builder *)> filter_producer;

      /*sizef sc;
      if (!pg->is_image_gfx) 
      {
        sizef sc = c.pview()->pixels_per_dip(sizef(1, 1));
        gfx->scale(sizef(1, 1) / sc);
      }
      ON_SCOPE_EXIT(if(sc.x) gfx->scale(sc));*/

      string       clip_area;
      array<value> filter_list;

      auto is_filter = [&](hvalue val) -> bool {
        return c.is_array(val) || c.is_string(val) || c.is_tuple(val);
      };

      auto make_filter = [&](hvalue filters) -> function<bool(gool::filter_graph_builder *)> {

        html::element *pel = pg->host ? pg->host : c.pdoc();

        value vfilters = c.get<value>(filters);

        if (vfilters.is_function() || vfilters.is_string())
          filter_list.push(vfilters);
        else
          filter_list = vfilters.values();

        for (value& f : filter_list) {
          if (f.is_string())
            f = parse_filter(c,f.to_string());
        }

        view* pv = c.pview();

        return [pv, pel, &filter_list](gool::filter_graph_builder *target) -> bool
        {
          return html::produce_filter_graph(*pv, pel, filter_list(), target, false);
        };
      };

      int nargs = args.size();
      if (nargs == 6 || nargs == 5 || nargs == 4) // x,y,w,h[,opacity,filter]
      {
        pointf org;
        sizef dim;
        if (!c.is_float(args[0], org.x)) throw qjs::om::type_error("x shall be a number");
        if (!c.is_float(args[1], org.y)) throw qjs::om::type_error("y shall be a number");
        if (!c.is_float(args[2], dim.x)) throw qjs::om::type_error("width shall be a number");
        if (!c.is_float(args[3], dim.y)) throw qjs::om::type_error("height shall be a number");

        gool::rectf rc(org, dim);
        float t;
        if (nargs > 4) {
          if (c.is_float(args[4], t))
            opacity = byte(limit(0.0f,255.0f,255 * t));
          else if (is_filter(args[4]))
            filter_producer = make_filter(args[4]);
          if (nargs > 5) {
            if (c.is_float(args[5], t))
              opacity = byte(255 * t);
            else if (is_filter(args[5]))
              filter_producer = make_filter(args[5]);
          }
        }

        if (filter_producer) {
          gfx->push_layer(rc, opacity, filter_producer);
        }
        else {
          gfx->push_layer(rc, opacity);
        }
      }
      // path [,opacity]
      else if ((nargs == 2 || nargs == 1) && c.isa<xpath *>(args[0])) {
        xpath *pth = c.get<xpath *>(args[0]);
        if (!pth) return pg;
        if (nargs == 2) {
          float t;
          if (!c.is_float(args[1], t))
            throw qjs::om::type_error("opacity shall be a float");
          gfx->push_layer(pth->cpath, byte(limit(0.0f, 255.0f, 255 * t)));
        }
        else
          gfx->push_layer(pth->cpath);
      }
      // imageMask, draw1a [,opacity|filter]
      else if ((nargs == 3 || nargs == 2) && c.isa<gool::image*>(args[0])) {
        handle<gool::image> xim = c.get<gool::image*>(args[0]);

        if (!xim) return pg;

        bool draw1a = c.get<bool>(args[1]);

        if (nargs == 3) {
          float t;
          if (!c.is_float(args[2],t))
            throw qjs::om::type_error("opacity shall be a float");
          gfx->push_layer(xim->get_bitmap(gfx, xim->dim()), draw1a, byte(limit(0.0f, 255.0f, 255 * t)));
        }
        else
          gfx->push_layer(xim->get_bitmap(gfx, xim->dim()), draw1a);
      }
      else if (nargs == 1 && c.is_string(args[0], clip_area)) {
      ELEMENT_SHAPE:
        /*static value background_area = CsSymbolOf("background-area");
        static value client_box = CsSymbolOf("client-box");
        static value margin_box = CsSymbolOf("margin-box");
        static value border_box = CsSymbolOf("border-box");
        static value padding_box = CsSymbolOf("padding-box");
        static value inner_box = CsSymbolOf("inner-box");*/

        if (!pg->host || pg->gfx->is_bitmap_graphics())
          throw qjs::om::type_error("layer shape is applicable only to element's graphics");
        if (CHARS("background-area") == clip_area) {
          html::hstyle       cs = pg->host->get_style(*c.pview());
          gool::rect         rect = pg->host->border_box(*c.pview());
          handle<gool::path> path;
          cs->background_outline(*c.pview(), gfx, rect, rect, path, pg->host);

          if (filter_producer)
            gfx->push_layer(rect, opacity, filter_producer);
          else if (path)
            gfx->push_layer(path);
          else
            gfx->push_layer(rect);
          return pg;
        }
        gool::rect rect;
        if (clip_area == CHARS("border-box"))
          rect = pg->host->border_box(*c.pview());
        else if (clip_area == CHARS("padding-box"))
          rect = pg->host->padding_box(*c.pview());
        else if (clip_area == CHARS("margin-box"))
          rect = pg->host->margin_box(*c.pview());
        else if (clip_area == CHARS("client-box"))
          rect = pg->host->client_box(*c.pview());
        else if (clip_area == CHARS("content-box") || clip_area == CHARS("inner-box"))
          rect = pg->host->dim();
        else
          throw qjs::om::type_error("unknown box specifier");
        if (filter_producer)
          gfx->push_layer(rect, opacity, filter_producer);
        else
          gfx->push_layer(rect);

        return pg;
      }
      else if ((nargs == 2 || nargs == 3) && c.is_string(args[0], clip_area))
      {
        float t;
        if (c.is_float(args[1],t))
          opacity = byte(255 * t);
        if (nargs == 3 && c.is_float(args[2], t))
          opacity = byte(limit(0.0f, 255.0f, 255 * t));
        if (is_filter(args[1]))
          filter_producer = make_filter(args[1]);
        goto ELEMENT_SHAPE;
      }
      else
        throw qjs::om::type_error("unrecognized parameters");
                
      return pg;
    }

    static xgraphics* pop_layer(xcontext& c, xgraphics* pg)
    {
      auto gfx = pg->get_gfx(c);
      if (!gfx) return pg;
      gfx->pop_layer();
      return pg;
    }

    static xgraphics* draw(xcontext& c, xgraphics* pg, hvalue what, hvalue how);

    static hbrush create_linear_gradient(xcontext& c, xgraphics* pg, float x1, float y1, float x2, float y2)
    {
      handle<gool::linear_brush> lb = new gool::linear_brush(pointf(x1, y1), pointf(x2, y2));
      return lb;
    }

    static hbrush create_radial_gradient(xcontext& c, xgraphics* pg, float x0, float y0, float r0/*, float x1, float y1, float r1*/)
    {
      handle<gool::radial_brush> lb = new gool::radial_brush(pointf(x0, y0), sizef(r0, r0));
      return lb;
    }



  };

  template <> struct conv<xgraphics*>
  {
    static xgraphics* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      xgraphics* pn = (xgraphics*)JS_GetOpaque(v, Graphics_class_id);
      return pn;
    }

    static JSValue wrap(JSContext * ctx, xgraphics* pg) noexcept
    {
      if (!pg)
        return JS_NULL;
      JSValue obj = JS_NewObjectClass(ctx, Graphics_class_id);
      pg->add_ref();
      JS_SetOpaque(obj, pg);
      return obj;
    }
  };

  extern JSClassID Image_class_id;

  template <> struct conv<image*>
  {
    static image* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      image* pi = (image*)JS_GetOpaque(v, Image_class_id);
      return pi;
    }
    static JSValue wrap(JSContext * ctx, image* pi) noexcept
    {
      if (!pi)
        return JS_NULL;
      JSValue obj = JS_NewObjectClass(ctx, Image_class_id);
      pi->add_ref();
      JS_SetOpaque(obj, pi);
      return obj;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Image_class_id) != NULL; }
  };

  template <> struct conv<himage>
  {
    static himage unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<image*>::unwrap(ctx, v, argc);
    }
    static JSValue wrap(JSContext * ctx, himage pi) noexcept
    {
      return conv<image*>::wrap(ctx, pi);
    }
  };
   
  extern JSClassID Graphics_Path_class_id;

  template <> struct conv<xpath*>
  {
    static xpath* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      xpath* pi = (xpath*)JS_GetOpaque(v, Graphics_Path_class_id);
      return pi;
    }
    static JSValue wrap(JSContext * ctx, xpath* pi) noexcept
    {
      if (!pi)
        return JS_NULL;
      JSValue obj = JS_NewObjectClass(ctx, Graphics_Path_class_id);
      pi->add_ref();
      JS_SetOpaque(obj, pi);
      return obj;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Graphics_Path_class_id) != NULL; }
  };

  template <> struct conv<handle<xpath>>
  {
    static handle<xpath> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<xpath*>::unwrap(ctx, v);
    }
    static JSValue wrap(JSContext * ctx, handle<xpath> pi) noexcept
    {
      return conv<xpath*>::wrap(ctx, pi);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<xpath*>::isa(ctx,v); }
  };
    

  extern JSClassID Graphics_Text_class_id;

  template <> struct conv<xtext*>
  {
    static xtext* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      xtext* pi = (xtext*)JS_GetOpaque(v, Graphics_Text_class_id);
      return pi;
    }
    static JSValue wrap(JSContext * ctx, xtext* pi) noexcept
    {
      if (!pi)
        return JS_NULL;
      JSValue obj = JS_NewObjectClass(ctx, Graphics_Text_class_id);
      pi->add_ref();
      JS_SetOpaque(obj, pi);
      return obj;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Graphics_Text_class_id) != NULL; }
  };

  template <> struct conv<handle<xtext>>
  {
    static handle<xtext> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<xtext*>::unwrap(ctx, v);
    }
    static JSValue wrap(JSContext * ctx, handle<xtext> pi) noexcept
    {
      return conv<xtext*>::wrap(ctx, pi);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<xtext*>::isa(ctx, v); }
  };
  
 
   

}