#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "xcontext.h"
#include "xgraphics.h"

namespace qjs {

  using namespace html;
  using namespace gool;

  JSClassID Image_class_id = 0;

  //template <>
  //inline image* object_of<image>(JSValue val) {
  //  return (image*)JS_GetOpaque(val, Image_class_id);
  //}

  gool::image* image_ptr_of(JSValueConst obj) {
    image* pi = (image*)JS_GetOpaque(obj, Image_class_id);
    return pi;
  }

  //static bool image_complete(xcontext& hc, image* img) {
  //  return img->is_valid();
  //}

  static string image_src(xcontext& hc, image* img) {
    return img->get_url();
  }

  static uint image_width(xcontext& hc, image* img) {
    return uint(img->dimension().x);
  }
  static uint image_height(xcontext& hc, image* img) {
    return uint(img->dimension().y);
  }

  static string image_packaging(xcontext& hc, image* img) {
    return img->mime_type();
  }
    

  static array<byte> image_to_bytes(xcontext& hc, image* img, string packaging, int_v compression) {
    
    if (!img) return array<byte>();
   
    array<byte> data;
    gool::image::PACKAGING type = gool::image::PNG;

    if (packaging.length() == 0 || packaging == CHARS("png")) type = gool::image::PNG;
  #if defined(JPG_SUPPORT)
    else if (packaging == CHARS("jpeg")) type = gool::image::JPG;
  #endif
    else if (packaging == CHARS("webp")) type = gool::image::WEBP;
    else if (packaging == CHARS("raw")) type = gool::image::UNKNOWN;
    else if (packaging == CHARS("bgra")) type = gool::image::UNKNOWN;
    else throw qjs::om::type_error("unknown packaging");

    if (type == gool::image::UNKNOWN) {
      handle<gool::bitmap> bitmap = img->get_bitmap(nullptr, img->dim());
      if (bitmap)
        data = bitmap->pixels_as_bytes();
      else
        data = img->get_data();
    }
    else
      img->save(data, type, compression.val(75));
    return data;
  }

  static himage image_from_bytes(xcontext& hc, array<byte> data) {
    himage xim = gool::image::create(data(), string(), hc.pdoc());
    return xim;
  }

  static hvalue image_load(xcontext& c, string url) 
  {
    handle<html::request> hrq = new request(url, RESOURCE_DATA_TYPE::DATA_RAW_DATA);

    hrq->dst = c.pdoc();
    hrq->initiator = c.pdoc();
    hrq->dst_view = c.pview();

    JSValue resolving_callbacks[2] = { JS_UNINITIALIZED, JS_UNINITIALIZED };
    hvalue promise = JS_NewPromiseCapability(c, resolving_callbacks);

    hvalue   resolver = resolving_callbacks[0];
    hvalue   rejector = resolving_callbacks[1];
    hcontext hc = c.pdoc()->ns;

    hrq->add([resolver, rejector, hc](request* prq)->bool {
      xcontext c(hc);
      try {
        bool r;
        if (!prq->success_flag)
          c.call(r, rejector, JS_UNDEFINED, string::format("load failure %d",prq->status));
        else {
          himage xim = gool::image::create(prq->data, prq->content_url(), c.pdoc());
          if(xim)
            c.call(r, resolver, JS_UNDEFINED, xim);
          else 
            c.call(r, rejector, JS_UNDEFINED, string("unknown image format"));
        }
      }
      catch (qjs::exception) {
        c.report_exception();
      }
      return true; // consumed
    });

    c.pview()->load_data(hrq);

    return promise;
  }
 

  static himage image_ctor(xcontext& c, int argc, JSValueConst *argv) {
    handle<gool::image> xim;

    auto do_paint = [&](html::graphics *pg, hvalue func) {

      html::helement el = c.pdoc();
                  
      handle<xgraphics> xgfx = new xgraphics(c.pview(), el, pg, true);

      gool::aa_mode _0(pg);
      gool::state   _1(pg);
      tool::auto_state<html::graphics *> _3(c.pview()->drawing_surface, pg);

      xgfx->setup_gfx(c);

      bool handled = false;

      c.call(handled, func, el, xgfx.ptr());
    };

    hvalue         painter;
    color_v        initc;
    sizef          dim;
    int            dn = 0;
    sizef          dstdim;
    html::helement elem;

    //size           dipdim;

    for (int n = 0; n < argc; ++n) {
      JSValue v = argv[n];
      if(dn == 0 && c.is_float(v,dim.x)) { dstdim.x = dim.x; ++dn; }
      else if (dn == 1 && c.is_float(v,dim.y)) { dstdim.y = dim.y; ++dn; }
      else if (dn == 2 && c.is_float(v,dstdim.x)) { ++dn; }
      else if (dn == 3 && c.is_float(v,dstdim.y)) { ++dn; }
      else if (c.is_color(v, initc)) { ; }
      else if (c.is_function(v)) painter = JS_DupValue(c,v);
      else if (c.isa<element*>(v)) elem = element_ptr_of(c, v);
    }

    if (elem && (dim.x < 1 || dim.y < 1)) 
      dim = elem->border_box(*c.pview()).size();
    else
      dim = c.pview()->dip_to_ppx(dim);
    
    dstdim = c.pview()->dip_to_ppx(dstdim);

    if (dim.x < 1 || dim.y < 1)
      throw qjs::om::range_error("dimensions are less or equal to zero");

    gool::argb clr = initc.to_argb();

    if (elem) {

      rect brc = elem->border_box(*c.pview());

      size eldim = brc.size();
      size originaldim = eldim;

      elem->resolve_styles(*c.pview());
      elem->check_layout(*c.pview());

      if (eldim != size(dim)) 
      {
        if (elem->parent) {
          elem->measure_borders_x(*c.pview(), elem->parent->dim());
          elem->measure_borders_y(*c.pview(), elem->parent->dim());
        }
        elem->calc_intrinsic_widths(*c.pview());
        elem->set_border_width(*c.pview(), int(dim.x));
        elem->set_border_height(*c.pview(), int(dim.y));
        eldim = dim;
      }
      else
        elem->commit_measure(*c.pview());

      float scale = 1.0f;

      if ((dstdim != dim) && !dstdim.empty()) {
        sizef sc = sizef(dstdim) / sizef(dim);
        scale = min(sc.x, sc.y);
        dim = sizef(dim) * scale;
      }

      assert(dim.x > 10 && dim.y > 10);
      assert(eldim.x > 10 && eldim.y > 10);

      handle<gool::bitmap> bmp = new gool::bitmap(dim);

      assert(bmp);

      bmp->clear(clr);

      xim = bmp;

      point vpt = elem->view_pos(*c.pview());
      point rpt = vpt + elem->border_distance(*c.pview()).s;

      {
        handle<gool::graphics> bgr = c.pview()->app->create_bitmap_bits_graphics(bmp, clr);
        assert(bgr);
        if (bgr)
        {
          bgr->set_clip_rc(eldim);
          auto_state<graphics *> _1(c.pview()->drawing_surface, bgr);
          bgr->offset(-vpt);
          if (scale != 1.0f) bgr->scale(sizef(scale, scale), pointf(vpt));

          elem->do_draw(*c.pview(), bgr, rpt); // first draw the element

          if (painter) // call painter function if any
            do_paint(bgr, painter);

          if (originaldim != eldim) {
            elem->set_border_width(*c.pview(), originaldim.x);
            elem->set_border_height(*c.pview(), originaldim.y);
          }
          elem->drop_content_layout(c.pview());
        }
      }

    }
    else if (painter) {

      handle<gool::bitmap> bmp = new gool::bitmap(dim);
      bmp->clear(clr);

      xim = bmp;

      // new Image(painter); case
      handle<gool::graphics> bgr = c.pview()->app->create_bitmap_bits_graphics(bmp, clr);
      if (bgr) do_paint(bgr, painter);
    }
    
    if (!xim)
      throw qjs::om::type_error("no element and no painter callback provided");

    return xim;
  }

  static bool image_update(xcontext& c, himage img, hvalue painter, color_v initcolor)
  {
    if (!img) return false;
    if (!c.is_function(painter))
      throw qjs::om::type_error("painter is not a function");

    auto do_paint = [&](html::graphics *pg, hvalue func) {

      html::helement el = c.pdoc();

      handle<xgraphics> xgfx = new xgraphics(c.pview(), el, pg, true);

      gool::aa_mode _0(pg);
      gool::state   _1(pg);
      tool::auto_state<html::graphics *> _3(c.pview()->drawing_surface, pg);

      xgfx->setup_gfx(c);

      bool handled = false;
      c.call(handled, func, el, xgfx.ptr());
    };

    if (!img->is_bitmap())
      throw qjs::om::type_error("image is not a bitmap");

    handle<gool::bitmap> bmp = img.ptr_of<gool::bitmap>();

    argb initc = argb::undefined();
    if (initcolor.is_defined())
      initc = initcolor.to_argb();

    handle<gool::graphics> bgr = c.pview()->app->create_bitmap_bits_graphics(bmp, initc);
    if (bgr) do_paint(bgr, painter);

    return true;
  }

  static hvalue image_colorAt(xcontext& c, himage img, int x, int y)
  {
    if (!img) return hvalue();

    gool::hbitmap hbm = img->get_bitmap(nullptr, size());
    if (!hbm) return hvalue();

    return c.val(hbm->pixel(x, y));
  }

  himage image_compose(xcontext& c, himage dst, himage src, string cop, int_v dst_x, int_v dst_y, int_v src_x, int_v src_y, int_v src_w, int_v src_h) {

    point dstpos(-1, -1);

    if (!dst || !src)
      throw qjs::om::type_error("not an image");

    if (!dst->is_bitmap() || !src->is_bitmap())
      throw qjs::om::type_error("not a bitmap");


    handle<gool::bitmap> bmp_src = src.ptr_of<gool::bitmap>();
    handle<gool::bitmap> bmp_dst = dst.ptr_of<gool::bitmap>();

    rect src_area(bmp_src->dim());

    if (dst_x.is_undefined() || dst_y.is_undefined()) 
      dstpos = point(0, 0);
    else 
      dstpos = point(dst_x, dst_y);
    
    if (!src_h.is_undefined()) 
      src_area = rect::make_xywh(src_x,src_y,src_w,src_h);

    if (cop == CHARS("src-over"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::src_over);
    else if (cop == CHARS("dst-over"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_over);
    else if (cop == CHARS("src-in"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::src_in);
    else if (cop == CHARS("dst-in"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_in);
    else if (cop == CHARS("src-out"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::src_out);
    else if (cop == CHARS("dst-out"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_out);
    else if (cop == CHARS("src-atop"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::src_atop);
    else if (cop == CHARS("dst-atop"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_atop);
    else if (cop == CHARS("xor"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_xor_src);
    else if (cop == CHARS("copy"))
      bmp_dst->combine(dstpos, bmp_src, src_area, gool::combine::premultiplied::dst_copy_src);
    else
      throw qjs::om::type_error("unknown operation");
    return dst;
  }

  JSOM_PASSPORT_BEGIN(Image_def, image)
    JSOM_CONST_STR("[Symbol.toStringTag]", "Image", JS_PROP_CONFIGURABLE),

    //JSOM_RO_PROP_DEF("complete", image_complete),
    //JSOM_RO_PROP_DEF("currentSrc", image_src),
    JSOM_RO_PROP_DEF("src", image_src),

    JSOM_RO_PROP_DEF("height", image_height),
    JSOM_RO_PROP_DEF("width", image_width),

    // sciter
    JSOM_RO_PROP_DEF("packaging", image_packaging),

    JSOM_FUNC_DEF("colorAt", image_colorAt),
    JSOM_FUNC_DEF("update", image_update),
    JSOM_FUNC_DEF("compose", image_compose),
    JSOM_FUNC_DEF("toBytes", image_to_bytes),

  JSOM_PASSPORT_END

  JSOM_PASSPORT_BEGIN(Image_static_def, qjs::xcontext)
    JSOM_GLOBAL_FUNC_DEF("fromBytes", image_from_bytes),
    JSOM_GLOBAL_FUNC_DEF("load", image_load),
  JSOM_PASSPORT_END


  void init_Image_class(context& c, hvalue graphics)
  {
    JS_NewClassID(&Image_class_id);

    static JSClassDef Image_class = {
      "Image",
      [](JSRuntime *rt, JSValue val)
      {
        image* pi = image_ptr_of(val);
        if (pi)
          pi->release();
      }
    };

    JS_NewClass(JS_GetRuntime(c), Image_class_id, &Image_class);
    JSValue image_proto = JS_NewObject(c);

    auto list = Image_def();
    JS_SetPropertyFunctionList(c, image_proto, list.start, list.length);


    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue
    {
      xcontext c(ctx);
      himage img;
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto))
        goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, Image_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj))
        goto fail;
      try {
        img = image_ctor(c, argc, argv);
        if (img) {
          JS_SetOpaque(obj, img.ptr());
          img->add_ref();
        }
        return obj;
      } 
      catch (om::error& err) { 
        JS_FreeValue(ctx, obj);
        return err.raise_error(ctx); 
      }
    fail:
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue image_class = JS_NewCFunction2(c, ctor, "Image", 2, JS_CFUNC_constructor, 0);

    auto static_list = Image_static_def();
    JS_SetPropertyFunctionList(c, image_class, static_list.start, static_list.length);

    JS_SetConstructor(c, image_class, image_proto);
    JS_SetClassProto(c, Image_class_id, image_proto);

    c.set_prop<hvalue>("Image", graphics, image_class);

  }


}
