#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "xcontext.h"
#include "xfetch.h"

namespace qjs {

  using namespace tool;
  using namespace html;

  JSClassID Request_class_id  = 0;
  JSClassID Response_class_id = 0;

  string request_def(xcontext &c, request *rq) { return string::format("Request(%s)", rq->url.c_str()); }

  string response_def(xcontext &c, response *rq) { return string::format("Response(%s)", rq->as_request()->content_url.c_str()); }

  bool request_cache(xcontext &c, request *rq) { return !!rq->no_cache; }
  void set_request_cache(xcontext &c, request *rq, bool cache) { rq->no_cache = !cache; }

  string request_context(xcontext &c, request *rq) {
    switch (rq->data_type) {
      case DATA_HTML: return CHARS("html");
      case DATA_IMAGE: return CHARS("image");
      case DATA_STYLE: return CHARS("style");
      case DATA_CURSOR: return CHARS("cursor");
      case DATA_SCRIPT: return CHARS("script");
      case DATA_RAW_DATA: return CHARS("data");
      case DATA_FONT: return CHARS("font");
      case DATA_SOUND: return CHARS("audio");
    }
    return CHARS("unknown");
  }

  string request_destination(xcontext &c, request *rq) {
    switch (rq->data_type) {
      case DATA_HTML: return CHARS("document");
      case DATA_IMAGE: return CHARS("image");
      case DATA_STYLE: return CHARS("style");
      case DATA_CURSOR: return CHARS("cursor");
      case DATA_SCRIPT: return CHARS("script");
      case DATA_RAW_DATA: return CHARS("data");
      case DATA_FONT: return CHARS("font");
      case DATA_SOUND: return CHARS("audio");
    }
    return CHARS("");
  }

  dictionary<string, string> request_headers(xcontext &c, request *rq) { return rq->rq_headers; }
  dictionary<string, string> response_headers(xcontext &c, response *rq) { return rq->as_request()->rs_headers; }

  html::request *response_request(xcontext &c, response *rq) { return static_cast<html::request *>(rq); }

  string request_method(xcontext &c, request *rq) {
    switch (rq->rq_type) {
      case RQ_GET: return CHARS("GET");
      case RQ_POST: return CHARS("POST");
      case RQ_PUT: return CHARS("PUT");
      case RQ_DELETE: return CHARS("DELETE");
      case RQ_PUSH:
      default: return CHARS("");
    }
  }

  html::REQUEST_TYPE parse_request_type(const string &s) {
    if (s == CHARS("GET") || s == CHARS("get")) return RQ_GET;
    if (s == CHARS("POST") || s == CHARS("post")) return RQ_POST;
    if (s == CHARS("PUT") || s == CHARS("put")) return RQ_PUT;
    if (s == CHARS("DELETE") || s == CHARS("delete")) return RQ_DELETE;
    assert(false);
    return RQ_PUSH;
  }

  string request_url(xcontext &c, request *rq) { return rq->url; }

  string response_url(xcontext &c, response *rq) { return rq->as_request()->content_url; }

  string response_mime_type(xcontext &c, response *rq) { return rq->as_request()->response_mime_type(); }

  bool response_ok(xcontext &c, response *rq) { return rq->as_request()->success_flag; }

  bool response_redirected(xcontext &c, response *rq) { return rq->as_request()->content_url != rq->as_request()->url; }

  uint response_status(xcontext &c, response *rq) { return rq->as_request()->status; }

  typedef JSValue (*response_convertor_t)(xcontext &c, request *rq);

  hvalue response_data(xcontext &c, response *rq, response_convertor_t convertor) { return convertor(c, rq->as_request()); }

  JSValue convertor_array_buffer(xcontext &c, request *rq) {
    return JS_NewArrayBufferCopy(c, rq->data.cbegin(), rq->data.length()); // copy ???
  }

  JSValue convertor_blob(xcontext &c, request *rq) {
#pragma TODO("Blob anyone?")
    return JS_NULL;
  }

  JSValue convertor_form_data(xcontext &c, request *rq) {
#pragma TODO("FormData anyone?")
    return JS_NULL;
  }

  JSValue convertor_json(xcontext &c, request *rq) {
    rq->data.push(0);
    rq->data.pop();
    return JS_ParseJSON2(c, (const char *)rq->data.cbegin(), rq->data.length(), rq->content_url, 0);
  }

  JSValue convertor_text(xcontext &c, request *rq) {
    // rq->data.push(0); rq->data.pop();
    ustring str;
    if (rq->data_content_encoding.is_defined())
      tool::decode_bytes(rq->data(), str, rq->data_content_encoding);
    else
      str = u8::cvt(rq->data());
    return c.val(str);
  }

  hvalue response_array_buffer(xcontext &c, response *rq) { return response_data(c, rq, convertor_array_buffer); }
  hvalue response_blob(xcontext &c, response *rq) { return response_data(c, rq, convertor_blob); }
  hvalue response_form_data(xcontext &c, response *rq) { return response_data(c, rq, convertor_form_data); }
  hvalue response_json(xcontext &c, response *rq) { return response_data(c, rq, convertor_json); }
  hvalue response_text(xcontext &c, response *rq) { 
    return response_data(c, rq, convertor_text); 
  }

  JSOM_PASSPORT_BEGIN(Request_def, html::request)

      JSOM_RO_PROP_DEF("[Symbol.toStringTag]", request_def), 
      JSOM_PROP_DEF("cache", request_cache, set_request_cache), 
      JSOM_RO_PROP_DEF("context", request_context),
      JSOM_RO_PROP_DEF("destination", request_destination), 
      JSOM_RO_PROP_DEF("headers", request_headers), 
      JSOM_RO_PROP_DEF("method", request_method),
      JSOM_RO_PROP_DEF("url", request_url),

      // JSOM_RO_PROP_DEF("ok", request_ok),

      // JSOM_RO_PROP_DEF("redirected", request_redirected),
      // JSOM_RO_PROP_DEF("status", request_status),
      // JSOM_RO_PROP_DEF("statusText", request_status_text),
      // JSOM_RO_PROP_DEF("type", request_response_type), //  basic, cors

      // JSOM_RO_PROP_DEF("requestUrl", request_url),
      // JSOM_RO_PROP_DEF("requestHeaders", request_headers),
      // JSOM_RO_PROP_DEF("responseHeaders", response_headers),
      // JSOM_RO_PROP_DEF("responseUrl", response_url),
      // JSOM_RO_PROP_DEF("responseMimeType", response_mime_type),

      // JSOM_FUNC_DEF("arrayBuffer", response_array_buffer),
      // JSOM_FUNC_DEF("blob", response_blob),
      // JSOM_FUNC_DEF("formData", response_form_data),
      // JSOM_FUNC_DEF("json", response_json),
      // JSOM_FUNC_DEF("text", response_text)

      // useFinalURL ??
      JSOM_PASSPORT_END

      JSOM_PASSPORT_BEGIN(Response_def, html::request) 
        JSOM_RO_PROP_DEF("[Symbol.toStringTag]", response_def),
        JSOM_RO_PROP_DEF("request", response_request), 
        JSOM_RO_PROP_DEF("headers", response_headers), 
        JSOM_RO_PROP_DEF("url", response_url),
        JSOM_RO_PROP_DEF("ok", response_ok), 
        JSOM_RO_PROP_DEF("redirected", response_redirected), 
        JSOM_RO_PROP_DEF("status", response_status),
        JSOM_RO_PROP_DEF("mimeType", response_mime_type),
        JSOM_FUNC_DEF("arrayBuffer", response_array_buffer), 
        JSOM_FUNC_DEF("blob", response_blob), 
        JSOM_FUNC_DEF("formData", response_form_data),
        JSOM_FUNC_DEF("json", response_json),
        JSOM_FUNC_DEF("text", response_text)
      JSOM_PASSPORT_END

      typedef dictionary<string, string> headers_t;

  handle<html::request> request_ctor(JSContext *ctx, JSValue obj, JSValue input, JSValue init) {
    xcontext              c(ctx);
    handle<html::request> hrq;
    if (c.isa<string>(input))
      hrq = new request(combine_url(c.pdoc()->uri(),c.get<string>(input)), RESOURCE_DATA_TYPE::DATA_RAW_DATA);
    else if (c.isa<request *>(input))
      hrq = new request(c.get<request *>(input));
    else
      throw qjs::om::type_error("wrong input");
    if (JS_IsObject(init)) {
      hrq->rq_type    = parse_request_type(c.get_prop<string>("method", init, "GET"));
      hrq->rq_headers = c.get_prop<headers_t>("headers", init, headers_t());
      // hrq->params_map(c.get_prop<value>("body", init));
#pragma TODO("CORS handling please")
      string cors_mode = c.get_prop<string>("mode", init, string());
      // hrq->no_cache = !c.get_prop<bool>("cache", init, true);
      string cache = c.get_prop<string>("cache", init);
      if (cache == CHARS("no-cache") || cache == CHARS("reload")) // default, no-store, reload, no-cache, force-cache, and only-if-cached,
        hrq->no_cache = true;

      // credentials
    }
    if (JS_IsObject(obj)) {
      JS_SetOpaque(obj, hrq.ptr());
      hrq->add_ref();
      //  hrq->obj = obj;
    }
    return hrq;
  }

  void init_Request_class(context &ctx) {
    JS_NewClassID(&Request_class_id);

    static JSClassDef Request_class = {"Request", [](JSRuntime *rt, JSValue val) {
                                         html::request *pr = (html::request *)JS_GetOpaque(val, Request_class_id);
                                         if (pr) {
                                           // pr->obj.tearoff();
                                           pr->release();
                                         }
                                       }};

    JS_NewClass(JS_GetRuntime(ctx), Request_class_id, &Request_class);
    JSValue request_proto = JS_NewObject(ctx);

    auto list = Request_def();
    JS_SetPropertyFunctionList(ctx, request_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue {
      JSValue obj = JS_UNDEFINED;
      JSValue proto;
      /* using new_target to get the prototype is necessary when the class is extended. */
      proto = JS_GetPropertyStr(ctx, new_target, "prototype");
      if (JS_IsException(proto)) goto fail;
      obj = JS_NewObjectProtoClass(ctx, proto, Request_class_id);
      JS_FreeValue(ctx, proto);
      if (JS_IsException(obj)) goto fail;
      request_ctor(ctx, obj, argc > 0 ? argv[0] : JS_UNINITIALIZED, argc > 1 ? argv[1] : JS_UNINITIALIZED);
      return obj;
    fail:
      // js_free(ctx, s);
      JS_FreeValue(ctx, obj);
      return JS_EXCEPTION;
    };

    hvalue request_class = JS_NewCFunction2(ctx, ctor, "Request", 2, JS_CFUNC_constructor, 0);

    JS_DefinePropertyValueStr(ctx, ctx.global(), "Request", JS_DupValue(ctx, request_class), JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);

    JS_SetConstructor(ctx, request_class, request_proto);
    JS_SetClassProto(ctx, Request_class_id, request_proto);
  }

  void init_Response_class(context &ctx) {
    JS_NewClassID(&Response_class_id);

    static JSClassDef Response_class = {"Response", [](JSRuntime *rt, JSValue val) {
                                          html::request *pr = (html::request *)JS_GetOpaque(val, Response_class_id);
                                          if (pr) { pr->release(); }
                                        }};

    JS_NewClass(JS_GetRuntime(ctx), Response_class_id, &Response_class);
    JSValue response_proto = JS_NewObject(ctx);

    auto list = Response_def();
    JS_SetPropertyFunctionList(ctx, response_proto, list.start, list.length);

    auto ctor = [](JSContext *ctx, JSValueConst new_target, int argc, JSValueConst *argv) -> JSValue {
      return JS_EXCEPTION; // not constructable in script
    };

    hvalue response_class = JS_NewCFunction2(ctx, ctor, "Response", 2, JS_CFUNC_constructor, 0);

    JS_DefinePropertyValueStr(ctx, ctx.global(), "Response", JS_DupValue(ctx, response_class), JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE);

    JS_SetConstructor(ctx, response_class, response_proto);
    JS_SetClassProto(ctx, Response_class_id, response_proto);
  }

  void process_request_body(xcontext &c, handle<request> hrq, hvalue body) {

    if (body.is_undefined())
      return;

    if (c.is_object(body)) {

      bool multipart = false;

      c.each_prop(body, [&](chars name, hvalue val) {
        if (c.is_file(val))
          multipart = true;
        });

      if (multipart) {
        pump::multipart_composer mc(hrq);
        c.each_prop(body, [&](chars name, hvalue val) {
          if (c.is_undefined(val)) return;
          if (c.is_file(val)) {
            ustring path = c.file_path(val);
            tool::mm_file mf;
            if (mf.open(path)) {
              ustring fdrive, fdir, fname, fext;
              split_path(path, fdrive, fdir, fname, fext);
              mc.add(name, mf, u8::cvt(fname + fext), guess_mime_type(path, mf));
            }
          }
          else {
            mc.add(name, c.get<ustring>(val));
          }
          });
      }
      else {
        c.each_prop(body, [&](chars name, hvalue val) {
          if (c.is_undefined(val)) return;
          pump::param p;
          p.name = ustring(name);
          p.value = c.get<ustring>(val);
          hrq->rq_params.push(p);
          });
      }
    }
    else if (c.isa<string>(body)) {
      string text = c.get<string>(body);
      hrq->data.push(text.chars_as_bytes());
    }
    else if (c.isa<array<byte>>(body)) {
      array<byte> data = c.get<array<byte>>(body);
      hrq->data.push(data());
    }
    else
      throw qjs::om::type_error("unsupported request body type");

  }


  hvalue request_fetch(xcontext &c, JSValueConst resource, JSValueConst init) {
    handle<request> hrq;
    if (c.isa<string>(resource)) {
      hrq = request_ctor(c, JS_UNINITIALIZED, resource, init);
    } else if (c.isa<request *>(resource)) {
      hrq = c.get<request *>(resource);
    } else
      throw qjs::om::type_error("url or request expected");

    hrq->dst      = c.pdoc();
    hrq->dst_view = c.pview();

    bool sync_load = c.is_object(init) && c.get_prop<bool>("sync", init);

    process_request_body(c, hrq, c.get_prop<hvalue>("body", init));

    if (sync_load) {
      if (c.pview()->load_data(hrq, true))
        return c.val(hrq.ptr_of<html::response>());
      else
        return c.val(false);
    }

    JSValue resolving_callbacks[2] = {JS_UNINITIALIZED, JS_UNINITIALIZED};
    hvalue  promise                = JS_NewPromiseCapability(c, resolving_callbacks);

    hvalue   resolver = resolving_callbacks[0];
    hvalue   rejector = resolving_callbacks[1];
    hcontext hc       = c.pdoc()->ns;

    hrq->add([resolver, rejector, hc](request *prq) -> bool {
      xcontext c(hc);
      try {
        bool r;
        if (prq->success_flag)
          c.call(r, resolver, JS_UNDEFINED, (html::response *)prq);
        else if (prq->status < 100 || prq->status >= 600)
          c.call(r, rejector, JS_UNDEFINED, (html::response *)prq);
        else
          c.call(r, resolver, JS_UNDEFINED, (html::response *)prq);
      } catch (qjs::exception) { c.report_exception(); }
      return true; // consumed
    });

    c.pview()->load_data(hrq);

    return promise;
  }

  void xview::on_data_request_notify(html::element *self, pump::request *rq) {

    hvalue obj = self ? self->obj : this->obj;
    if (!obj) return;

    html::hdocument pd = self ? self->doc() : doc();
    if (!pd) return;

    xcontext c(pd);

    hvalue fcn = c.get_prop<hvalue>(c.known_atoms().onrequest, obj);

    if (!c.is_function(fcn)) return;

    try {
      hvalue rv;
      c.call(rv, fcn, obj, rq);
    } catch (qjs::exception) { c.report_exception(); }
  }

  void xview::on_data_arrived_notify(html::element *self, pump::request *rq) {
    hvalue obj = self ? self->obj : this->obj;
    if (!obj) return;

    html::hdocument pd = self ? self->doc() : doc();
    if (!pd) return;

    xcontext c(pd);

    hvalue fcn = c.get_prop<hvalue>(c.known_atoms().onrequestresponse, obj);

    if (!c.is_function(fcn)) return;

    try {
      hvalue rv;
      c.call(rv, fcn, obj, (html::response *)rq);
    } catch (qjs::exception) { c.report_exception(); }
  }

} // namespace qjs
