#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "sciter-revision.h"

#ifndef WINDOWS
#include <spawn.h>
#endif

namespace qjs {

  using namespace html;

  static bool system_launch(xcontext& c, ustring what)
  {
    if (what.is_undefined())
      return false;
    //if (c->features & FEATURE_SYSINFO)
    return tool::launch(what);
  }

  static int system_exec(xcontext& c, array<ustring> iargs)
  {
    //if ((c->features & FEATURE_SYSINFO) == 0) {
    //  CsThrowKnownError(c, CsErrNotAllowed, "SYS INFO");
    //  // return UNDEFINED_VALUE;
    //}

    if (iargs.length() == 0)
      throw om::type_error("no executable name");

#ifdef WINDOWS
    ustring        spath;
    array<ustring> args;
#else
    string        spath;
    array<string> args;
#endif

    auto wrap_spaced_string = [](ustring in) -> ustring {
      if (in().contains_one_of(WCHARS(" \r\n\t")) && in[0] != '"')
        return ustring::format(W("\"%s\""), in.c_str());
      return in;
    };

    for (int n = 0; n < iargs.size(); ++n) {
      args.push(wrap_spaced_string(iargs[n]));
    }

#ifdef WINDOWS
    spath = iargs[0];
#else
    spath = u8::cvt(iargs[0]);
#endif
    
    int r = 0;
#ifdef WINDOWS
    array<const wchar *> argv;
    for (int i = 0; i < args.size(); ++i) {
      const wchar *parg = args[i].c_str();
      argv.push(parg);
    }
    argv.push(nullptr);
    r = (int)_wspawnvp(_P_NOWAIT, spath.c_str(), argv.cbegin());
#else
    array<char *> argv;
    for (int i = 0; i < args.size(); ++i)
      argv.push(args[i].buffer());
    argv.push(nullptr);
    pid_t processID = 0;
    r = posix_spawn(&processID, spath.c_str(), nullptr, nullptr, argv.begin(),
      nullptr);
#endif
    if (r == -1)
      r = errno;
    else
      r = 0;
    return r;
  }

  static ustring system_language(xcontext& c)
  //f(c->features & FEATURE_SYSINFO) TODO
  {
    tool::ustring lang;
    tool::ustring country;
    tool::environment::used_lang_country(lang, country, true);
    return lang;
  }

  static ustring system_country(xcontext& c)
  //f(c->features & FEATURE_SYSINFO) TODO
  {
    tool::ustring lang;
    tool::ustring country;
    tool::environment::used_lang_country(lang, country, true);
    return country;
  }

  static ustring system_user(xcontext& c)
  //f(c->features & FEATURE_SYSINFO) TODO
  {
    return tool::environment::user_name();
  }

  static ustring system_machine_name(xcontext& c)
  //f(c->features & FEATURE_SYSINFO) TODO
  {
    return tool::environment::machine_name(false);
  }

  static ustring system_domain_name(xcontext& c)
    //f(c->features & FEATURE_SYSINFO) TODO
  {
    return tool::environment::machine_name(true);
  }

  static ustring system_home(xcontext& c, ustring relpath)
    //f(c->features & FEATURE_SYSINFO) TODO
  {
    tool::ustring p = tool::get_home_dir();
    wchars path = relpath;
    if (path) {
      if (*path == '\\' || *path == '/') path.prune(1);
      p += path;
    }
    p.replace_all('\\', '/');
    return normalize_path(p())();
  }

  static ustring system_home_url(xcontext& c, ustring relpath)
    //f(c->features & FEATURE_SYSINFO) TODO
  {
    ustring us = system_home(c, relpath);
    return url::path_to_file_url(us);
  }
  
  static ustring system_path(xcontext& c, string s, ustring relpath) {

    wchars path = relpath();

    auto sd = tool::parse_standard_dir(s);

    if(!sd) {
      tool::string msg = tool::string::format("#%s is not a valid name", s.c_str());
      throw qjs::om::type_error(msg);
    }

    tool::ustring p = tool::get_standard_dir(sd);

    if (path) {
      if (*path != '\\' && *path != '/') p += '/';
      p += path;
    }

    p.replace_all('\\', '/');

    if (relpath.is_defined())
      return normalize_path(p())();
    else
      return p;
  }


  static ustring system_path_url(xcontext& c, string s, ustring relpath)
  {
    ustring us = system_path(c, s, relpath);
    return url::path_to_file_url(us);
  }


  static ustring system_variable(xcontext& c, string s, hvalue val) {
    if (val.is_undefined()) {
      char   buffer[4096] = { 0 };
      size_t buffer_size = items_in(buffer);
      if (uv_os_getenv(s.c_str(), buffer, &buffer_size) == UV_ENOENT)
        return ustring();
      return u8::cvt(chars(buffer, buffer_size));
    }
    else if (val == JS_NULL) {
      uv_os_unsetenv(s.c_str());
    }
    else if (c.is_string(val)) {
      uv_os_setenv(s.c_str(), c.get<string>(val).c_str());
    }
    else
      throw qjs::om::type_error("string expected");
    return ustring();
  }
  

  
  JSOM_PASSPORT_BEGIN(Env_def, qjs::xcontext)
    JSOM_GLOBAL_FUNC_DEF("language", system_language),
    JSOM_GLOBAL_FUNC_DEF("country", system_country),
    JSOM_GLOBAL_FUNC_DEF("userName", system_user),
    JSOM_GLOBAL_FUNC_DEF("machineName", system_machine_name),
    JSOM_GLOBAL_FUNC_DEF("domainName", system_domain_name),
    JSOM_CONST_STR("OS", tool::environment::get_os_version_name(), JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
    JSOM_CONST_STR("PLATFORM", tool::environment::platform_name(), JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
#ifdef PLATFORM_HANDHELD
    JSOM_CONST_STR("DEVICE", "mobile", JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
#else
    JSOM_CONST_STR("DEVICE", "desktop", JS_PROP_CONFIGURABLE | JS_PROP_ENUMERABLE),
#endif
    JSOM_GLOBAL_FUNC_DEF("launch", system_launch),
    JSOM_GLOBAL_FUNC_DEF("exec", system_exec),
    JSOM_GLOBAL_FUNC_DEF("home", system_home),
    JSOM_GLOBAL_FUNC_DEF("homeURL", system_home),
    JSOM_GLOBAL_FUNC_DEF("path", system_path),
    JSOM_GLOBAL_FUNC_DEF("pathURL", system_path_url),
    JSOM_GLOBAL_FUNC_DEF("variable", system_variable),

  JSOM_PASSPORT_END

  static auto member_defs = Env_def();

  static int env_module_init(JSContext *ctx, JSModuleDef *m)
  {
    return JS_SetModuleExportList(ctx, m, member_defs.start, member_defs.length);
  }

  void init_env_module(context& c)
  {
    JSModuleDef *m = JS_NewCModule(c, "@env", env_module_init);
    if (!m) return;
    JS_AddModuleExportList(c, m, member_defs.start, member_defs.length);
  }

}
