#include "xjs.h"
#include "xview.h"
#include "xdom.h"
#include "xom.h"
#include "xview.h"
#include "xchildren.h"

namespace qjs {

  using namespace html;

  static hvalue set_ueh(xcontext& c, hvalue func)
  {
    hvalue p = context::inst(c).unhandled_exception_handler;
    context::inst(c).unhandled_exception_handler = func;
    return p;
  }

  static hvalue set_console_handler(xcontext& c, hvalue func)
  {
    hvalue p = context::inst(c).console_output_handler;
    context::inst(c).console_output_handler = func;
    c.pview()->set_debug_output(&context::inst(c));
    return p;
  }

  static bool set_resource_handler(xcontext& c, hvalue func)
  {
    context::inst(c).request_complete_handler = func;
    return true;
  }

  static bool set_breakpoint_handler(xcontext& c, hvalue func, string src_to_avoid) {
#ifdef CONFIG_DEBUGGER
    context::inst(c).set_breakpoint_handler(func, src_to_avoid);
    return true;
#else
    return false;
#endif
  }

#ifdef CONFIG_DEBUGGER

  void context::set_breakpoint_handler(hvalue handler, string src_to_avoid)
  {
    breakpoint_handler = handler;

    if (!JS_IsFunction(ctx, breakpoint_handler)) {
      JS_SetBreakpointHandler(ctx, nullptr);
      debugger_context.clear();
      breakpoints.clear();
      breakpoint_current.clear();
      return;
    }

    debugger_filename = hatom(ctx, src_to_avoid.is_empty() ? CHARS("sciter:debug-peer.js") : src_to_avoid());

    auto debugger_check_line_no = [](JSContext *ctx, JSAtom file_name, uint32_t line_no, const uint8_t *pc) -> JS_BOOL {
      context& c = context::inst(ctx);

      if (c.debugger_filename == file_name)
        return 0;

#ifdef DEBUG
      //if (line_no == 0)
      //  goto BREAK;
#endif

      for (auto& bp : c.breakpoints) {
        if (bp.filename == file_name && bp.lineno == line_no)
          goto BREAK;
      }

      switch (c.debugger_cmd)
      {
      case DEBUGGER_CONTINUE:
        break;
      case DEBUGGER_STEP_IN:
        if (c.breakpoint_current.filename != file_name || c.breakpoint_current.lineno != line_no)
          goto BREAK;
        break;
      case DEBUGGER_STEP_OUT:
        if (c.breakpoint_depth > (int)js_debugger_stack_depth(ctx))
          goto BREAK;
        break;
      case DEBUGGER_STEP_OVER:
        if (c.breakpoint_depth >= (int)js_debugger_stack_depth(ctx))
          goto BREAK;
        break;
      }
      return 0;
    BREAK:
      c.breakpoint_pc = pc;
      try {
        uint cmd;
        hvalue vfilename = JS_AtomToValue(ctx, file_name);
        hvalue stack_trace = js_debugger_build_backtrace(ctx, pc);
        c.call(cmd, c.breakpoint_handler, JS_UNDEFINED, vfilename, c.val(line_no), stack_trace);
        c.debugger_cmd = DEBUGGER_CMD(cmd);
        if (c.debugger_cmd) {
          // DEBUGGER_STEP_***
          c.breakpoint_current.filename = JS_DupAtom(ctx, file_name);
          c.breakpoint_current.lineno = line_no;
          c.breakpoint_depth = js_debugger_stack_depth(ctx);
        }
        else {
          c.breakpoint_current.filename = 0;
          c.breakpoint_current.lineno = 0;
          c.breakpoint_depth = 0;
        }
      }
      catch (qjs::exception) {
        c.report_exception();
      }
      return 1;
    };

    JS_SetBreakpointHandler(ctx, debugger_check_line_no);
  }
#endif


  static bool set_breakpoints(xcontext& c, hvalue breakpoints) {
#ifdef CONFIG_DEBUGGER
    array<context::breakpoint> list;

    c.each_item(breakpoints, [&](uint32_t idx, hvalue el) {
      context::breakpoint bp;
      string filename = c.get_prop<string>("filename", el);
      bp.filename = hatom(c, filename());
      bp.lineno = c.get_prop<uint>("lineno", el);
      list.push(bp);
    });

    context::inst(c).set_breakpoints(list);
    return true;
#else
    return false;
#endif
  }


  static helement debug_get_element_by_uid(xcontext& c, int uid)
  {
    return c.pdoc()->get_element_by_uid(uid);
  }

  static int debug_get_uid_of_element(xcontext& c, html::helement el)
  {
    return el->uid;
  }

  static child_list_provider* debug_get_element_nodes(xcontext& c, element* el)
  {
    return el;
  }

  element*  debug_get_node_parent(xcontext&, node* pn) {
    return pn->parent;
  }


  static bool debug_highlight_element(xcontext& c, html::helement el) {
    if(auto pv = c.pview())
      return pv->set_highlighted(el);
    return false;
  }

  static html::helement debug_highlighted_element(xcontext& c) {
    if (auto pv = c.pview())
      return pv->get_highlighted();
    return nullptr;
  }


  static tool::value debug_get_style_rules(xcontext& c, html::helement el) {
    hash_table<ustring, value> subst;
    tool::value map = tool::value::make_map();
    {
      tool::value retval;
      el->applied_style_rules_report(*c.pview(), retval);
      retval = retval.clone(subst);
      retval.mutate([](value& v)->bool {
        if (v.is_color() || v.is_length())
          v = v.to_string();
        return true;
      });
      map.set_prop("appliedStyleRules", retval);
    }
    {
      tool::value retval;
      el->used_style_props_report(*c.pview(), retval);
      element_context ectx(c, el);
      retval = retval.clone(subst);
      retval.mutate([&ectx](value& v)->bool {
        if (v.is_color())
          v = ectx.color_value(v).to_string();
        else if (v.is_length())
          v = ectx.length_value(v).to_string();
        return true;
      });
      map.set_prop("usedStyleProperties", retval);
    }
    return map;
  }

  static int64 debug_container_id(xcontext& c) {
    void* p = c.pdoc()->parent;
    if (!p) p = c.pview();
    return reinterpret_cast<int64>(p);
  }

  static string debug_object_kind(xcontext& c, hvalue any)
  {
    JSClassID cid = JS_GetClassID(any, nullptr);
    hvalue cn = JS_GetClassName(c, cid);
    return c.get<string>(cn);
  }

  // used for passing values from debugee to debugger for inspection
  static hvalue debug_sublimated_value(xcontext& c, hvalue val, bool expanded)
  {
    auto JS_IsInteger = [](JSValueConst v) -> bool
    {
      int tag = JS_VALUE_GET_TAG(v);
      return tag == JS_TAG_INT || tag == JS_TAG_BIG_INT;
    };

    bool as_is = JS_IsString(val)
      || JS_IsInteger(val)
      || JS_IsNumber(val)
      || JS_IsBigFloat(val)
      || JS_IsBool(val)
      || JS_IsNull(val)
      || JS_IsUndefined(val);

    if (as_is) return val;

    if (JS_IsArray(c, val)) {
      hvalue sub = JS_NewObject(c);
      c.set_prop("type", sub, CHARS("Array"));
      c.set_prop("length", sub, c.get_length(val));
      int64 rn = qjs::context::inst(c).add_reference(val);
      c.set_prop("reference", sub, rn);
      if (expanded) {
        array<hvalue> elements;
        c.each_item(val, [&](uint i, hvalue iv) {elements.push(debug_sublimated_value(c,iv,false));});
        hvalue ve = JS_NewFastArray(c, elements.size(), (JSValue*)elements.begin());
        c.set_prop("elements", sub, ve );
      }
      return sub;
    }
    else {
      hvalue sub = JS_NewObject(c);
      JSClassID cid = JS_GetClassID(val, nullptr);
      hvalue kind = JS_GetClassName(c, cid);
      c.set_prop("type", sub, kind);
      int64 rn = qjs::context::inst(c).add_reference(val);
      c.set_prop("reference", sub, rn);
      string caption = c.get<string>(val);
      if( caption().starts_with(CHARS("[object ")))
        caption = caption(8,caption.size()-1);
      c.set_prop("caption", sub, caption);
      int n = 0;
      if (expanded) {
        hvalue props = JS_NewObject(c);
        c.each_prop(val, [&](chars k, hvalue kv) { c.set_prop(k.start, props, debug_sublimated_value(c, kv, false)); ++n; });
        c.set_prop("properties", sub, props);
      }
      else {
        c.each_prop(val, [&](chars k, hvalue kv) { ++n; });
      }
      c.set_prop("length", sub, n);
      return sub;
    }
  }

  static hvalue debug_sublimated_value_elements(xcontext& c, int64 ref) {
    hvalue val = qjs::context::inst(c).get_reference(ref);
    if (JS_IsArray(c, val)) {
      array<hvalue> elements;
      c.each_item(val, [&](uint i, hvalue iv) {elements.push(debug_sublimated_value(c, iv, false)); });
      hvalue ve = JS_NewFastArray(c, elements.size(), (JSValue*)elements.begin());
      return ve;
    }
    else if(JS_IsObject(val)) {
      hvalue props = JS_NewObject(c);
      c.each_prop(val, [&](chars k, hvalue kv) { c.set_prop(k.start, props, debug_sublimated_value(c, kv, false)); });
      return props;
    }
    return hvalue();
  }

  static hvalue debug_frame_variables(xcontext& c, int frameId) {
#ifdef CONFIG_DEBUGGER
    hvalue local = js_debugger_local_variables(c, frameId);
    hvalue closure = js_debugger_closure_variables(c, frameId);
    hvalue def = JS_NewObject(c);
    c.set_prop("local", def, debug_sublimated_value(c, local, true));
    c.set_prop("closure", def, debug_sublimated_value(c, closure, true));
    c.set_prop("global", def, debug_sublimated_value(c, c.global(), false));
    return def;
#else
    return hvalue();
#endif
  }

  static hvalue debug_call_stack(xcontext& c, uint level) {
    JSCallStackItem itm = { 0 };
    if (!JS_GetCallStackItem(c, level+1, &itm))
      return hvalue();
    hvalue rv = JS_NewObject(c);
    c.set_prop("isNative", rv, bool(itm.isNative));
    c.set_prop("functionName", rv, hvalue(itm.functionName));
    c.set_prop("functionLineNo", rv, itm.functionLineNo);
    c.set_prop("fileName", rv, hvalue(itm.fileName));
    c.set_prop("lineNo", rv, int(itm.lineNo));
    return rv;
  }

  JSOM_PASSPORT_BEGIN(Debug_def, qjs::xcontext)
    JSOM_GLOBAL_FUNC_DEF("setUnhandledExeceptionHandler", set_ueh),
    JSOM_GLOBAL_FUNC_DEF("setConsoleOutputHandler", set_console_handler),
    JSOM_GLOBAL_FUNC_DEF("setResourceArrivalHandler", set_resource_handler),
    JSOM_GLOBAL_FUNC_DEF("setBreakpointHandler", set_breakpoint_handler),
    JSOM_GLOBAL_FUNC_DEF("setBreakpoints", set_breakpoints),
    JSOM_GLOBAL_FUNC_DEF("getElementByUID", debug_get_element_by_uid),
    JSOM_GLOBAL_FUNC_DEF("elementNodes", debug_get_element_nodes),
    JSOM_GLOBAL_FUNC_DEF("nodeParent", debug_get_node_parent),
    JSOM_GLOBAL_FUNC_DEF("getUIDofElement", debug_get_uid_of_element),
    JSOM_GLOBAL_FUNC_DEF("highlightElement", debug_highlight_element),
    JSOM_GLOBAL_FUNC_DEF("highlightedElement", debug_highlighted_element),
    JSOM_GLOBAL_FUNC_DEF("getStyleRulesOfElement", debug_get_style_rules),
    JSOM_GLOBAL_FUNC_DEF("containerId", debug_container_id),
    JSOM_GLOBAL_FUNC_DEF("objectKind", debug_object_kind),
    JSOM_GLOBAL_FUNC_DEF("sublimatedValue", debug_sublimated_value),
    JSOM_GLOBAL_FUNC_DEF("sublimatedValueElements", debug_sublimated_value_elements),
    JSOM_GLOBAL_FUNC_DEF("frameVariables", debug_frame_variables),
    JSOM_GLOBAL_FUNC_DEF("callStackAt", debug_call_stack),
  JSOM_PASSPORT_END

  static auto member_defs = Debug_def();

  static int sciter_module_init(JSContext *ctx, JSModuleDef *m)
  {
    return JS_SetModuleExportList(ctx, m, member_defs.start, member_defs.length);
  }

  void init_debug_module(context& c)
  {
    JSModuleDef *m = JS_NewCModule(c, "@debug", sciter_module_init);
    if (!m) return;
    JS_AddModuleExportList(c, m, member_defs.start, member_defs.length);
  }

}
