#pragma once

#ifndef __xdomjs_conv_h__
#define __xdomjs_conv_h__

#include "tool/tool.h"
#include "html/html.h"

#include "quickjs/quickjs.h"
#include "quickjs/quickjs-libc.h"

namespace qjs {
  using namespace tool;
  using namespace gool;

  class  xview;
  struct xcontext;

  typedef JSAtom symbol_t;

  namespace om {

    struct error : public tool::error {
      typedef tool::error super;
      error(const char* msg) : super(msg) {}
      virtual JSValue raise_error(JSContext* ctx) {
        return JS_ThrowInternalError(ctx, "%s", what());
      }
    };

    struct type_error : public error {
      typedef error super;
      type_error(const char* msg) : super(msg) {}
      virtual JSValue raise_error(JSContext* ctx) override { return JS_ThrowTypeError(ctx, "%s", what()); }
    };
    struct range_error : public error {
      typedef error super;
      range_error(const char* msg) : super(msg) {}
      virtual JSValue raise_error(JSContext* ctx) override { return JS_ThrowRangeError(ctx, "%s", what()); }
    };

    struct js_error : public error {
      typedef error super;
      hvalue err;
      js_error(hvalue he) : super(""), err(he) {}
      virtual JSValue raise_error(JSContext* ctx) override { return err.tearoff(); }
    };


  }


  extern JSClassID Document_class_id;
  extern JSClassID Element_class_id;
  extern JSClassID Style_class_id;
  extern JSClassID ElementState_class_id;
  extern JSClassID Node_class_id;
  extern JSClassID Text_class_id;
  extern JSClassID Comment_class_id;
  extern JSClassID Event_class_id;
  extern JSClassID Asset_class_id;
  
  html::node*     node_ptr_of(JSContext * ctx, JSValueConst obj);
  html::element*  element_ptr_of(JSContext * ctx,JSValueConst obj);
  bool can_get_element_from(JSContext * ctx, JSValueConst obj);
  html::text*     text_ptr_of(JSContext * ctx, JSValueConst obj);
  html::comment*  comment_ptr_of(JSContext * ctx, JSValueConst obj);
  html::document* document_ptr_of(JSContext * ctx, JSValueConst obj);
  gool::image*    image_ptr_of(JSValueConst obj);
  html::request*  request_ptr_of(JSContext * ctx, JSValueConst obj);
  som_asset_t*    asset_ptr_of(JSContext * ctx, JSValueConst obj);
      
  extern html::document* document_of(JSContext * ctx);
  extern qjs::xview* xview_of(JSContext * ctx);
  extern html::view* view_of(JSContext * ctx);
  extern html::timer_id next_timer_id_of(JSContext * ctx);
  
  //symbol_t symbol_of(const wchars s);
  //symbol_t symbol_of(const chars s);
  //tool::ustring symbol_name_w(symbol_t sym);
  //tool::string  symbol_name_a(symbol_t sym);

  struct atom_chars : public chars
  {
    char buf[64];
    atom_chars(JSContext* ctx, JSAtom atom) {
      this->start = JS_AtomGetStr(ctx, buf, items_in(buf), atom);
      this->length = str_len(this->start);
    }
  };

  typedef JSValue(*getter_t)(JSContext *ctx, JSValueConst this_val);
  typedef JSValue(*setter_t)(JSContext *ctx, JSValueConst this_val, JSValueConst val);
  typedef JSValue(*getter_magic_t)(JSContext *ctx, JSValueConst this_val, int magic);
  typedef JSValue(*setter_magic_t)(JSContext *ctx, JSValueConst this_val, JSValueConst val, int magic);
  typedef JSValue(*function_t)(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv);
  typedef JSValue(*function_magic_t)(JSContext *ctx, JSValueConst this_val, int argc, JSValueConst *argv, int magic);


  inline JSCFunctionListEntry prop_def(const char* name, getter_t g, setter_t s = nullptr, int flags = JS_PROP_C_W_E) {
    //#define JS_CGETSET_DEF(name, fgetter, fsetter) { name, JS_PROP_CONFIGURABLE, JS_DEF_CGETSET, 0, .u = { .getset = { .get = { .getter = fgetter }, .set = { .setter = fsetter } } } }
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = flags;
    def.def_type = JS_DEF_CGETSET;
    def.magic = 0;
    def.u.getset.get.getter = g;
    def.u.getset.set.setter = s;
    return def;
  }

  inline JSCFunctionListEntry prop_def_magic(const char* name, getter_magic_t g, setter_magic_t s, int magic) {
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = JS_PROP_CONFIGURABLE;
    def.def_type = JS_DEF_CGETSET_MAGIC;
    def.magic = magic;
    def.u.getset.get.getter_magic = g;
    def.u.getset.set.setter_magic = s;
    return def;
  }


  inline JSCFunctionListEntry func_def(const char* name, byte minargs, function_t pf)
  {
    //#define JS_CFUNC_DEF(name, length, func1) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, 0, 
             //.u = { .func = { length, JS_CFUNC_generic, { .generic = func1 } } } }
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE;
    def.def_type = JS_DEF_CFUNC;
    def.magic = 0;
    def.u.func.length = minargs;
    def.u.func.cfunc.generic = pf;
    def.u.func.cproto = JS_CFUNC_generic;
    return def;
  }

  inline JSCFunctionListEntry func_def_magic(const char* name, function_magic_t pf, int magic)
  {
    //#define JS_CFUNC_DEF(name, length, func1) { name, JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE, JS_DEF_CFUNC, 0, 
             //.u = { .func = { length, JS_CFUNC_generic, { .generic = func1 } } } }
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = JS_PROP_WRITABLE | JS_PROP_CONFIGURABLE;
    def.def_type = JS_DEF_CFUNC;
    def.magic = magic;
    def.u.func.length = 0;
    def.u.func.cfunc.generic_magic = pf;
    def.u.func.cproto = JS_CFUNC_generic_magic;
    return def;
  }

  inline JSCFunctionListEntry cstring_def(const char* name, const char* val, uint8_t prop_flags)
  {
    //#define JS_PROP_STRING_DEF(name, cstr, prop_flags) { name, prop_flags, JS_DEF_PROP_STRING, 0, .u = { .str = cstr } }
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = prop_flags;
    def.def_type = JS_DEF_PROP_STRING;
    def.magic = 0;
    def.u.str = val;
    return def;
  }

  inline JSCFunctionListEntry int_cdef(const char* name, int val, uint8_t prop_flags = JS_PROP_ENUMERABLE) {
    //JS_PROP_INT32_DEF(name, val, prop_flags) { name, prop_flags, JS_DEF_PROP_INT32, 0, .u = { .i32 = val } }
    JSCFunctionListEntry def = { 0 };
    def.name = name;
    def.prop_flags = prop_flags;
    def.def_type = JS_DEF_PROP_INT32;
    def.u.i32 = val;
    return def;
  }

  // convertors
  template <typename R>
  struct conv
  {
    /** Create an object of C++ type R given JSValue v and JSContext.
     * This function is intentionally not implemented. User should implement this function for their own type.
     * @param v This value is passed as JSValueConst so it should be freed by the caller.
     * @throws exception in case of conversion error
     */
    static R unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0);
    /** Create JSValue from an object of type R and JSContext.
     * This function is intentionally not implemented. User should implement this function for their own type.
     * @return Returns JSValue which should be freed by the caller or JS_EXCEPTION in case of error.
     */
    static JSValue wrap(JSContext * ctx, R value);

    static bool isa(JSContext * ctx, R value);
  };

  /** Conversion traits for boolean.
     */
  template <>  struct conv<bool>
  {
    static bool unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0) noexcept
    {
      return JS_ToBool(ctx, v);
    }
    static JSValue wrap(JSContext * ctx, bool i) noexcept
    {
      return JS_NewBool(ctx, i);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return v == JS_TRUE || v == JS_FALSE; }
  };

  /** Conversion traits for nullable boolean.
     */
  template <>  struct conv<tristate_v>
  {
    static tristate_v unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0) noexcept
    {
      if (v == JS_UNDEFINED || v == JS_UNINITIALIZED)
        return tristate_v();
      return JS_ToBool(ctx, v) ? 1 : 0;
    }
    static JSValue wrap(JSContext * ctx, tristate_v i) noexcept
    {
      if (i.is_defined())
        return JS_NewBool(ctx, i.val(0));
      else
        return JS_UNDEFINED;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return v == JS_TRUE || v == JS_FALSE || v == JS_UNDEFINED; }
  };


  /** Conversion trait for void.
     */
  template <> struct conv<void>
  {
    /// @throws exception if jsvalue is neither undefined nor null
    static void unwrap(JSContext * ctx, JSValueConst value)
    {
      if (JS_IsException(value))
        throw exception{};
    }

    //static JSValue wrap(JSContext * ctx, void) noexcept
    //{
    //  return JS_UNDEFINED;
    //}


  };

  /** Conversion traits for integers.
   */
  template <> struct conv<int>
  {
    /// @throws exception
    static int unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      int r;
      if (JS_ToInt32(ctx, &r, v))
        throw exception{};
      return r;
    }

    static JSValue wrap(JSContext * ctx, int i) noexcept
    {
      return JS_NewInt32(ctx, i);
    }

    static bool isa(JSContext * ctx, JSValueConst v) { 
      return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_INT;
    }

  };

  template <> struct conv<uint>
  {
    /// @throws exception
    static uint unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0) { return (uint)conv<int>::unwrap(ctx, v, argc); }
    static JSValue wrap(JSContext * ctx, uint i) noexcept {  return conv<int>::wrap(ctx, (int)i); }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<int>::isa(ctx, v); }
  };

  template <> struct conv<uint_v>
  {
    static uint_v unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      int r;
      if (JS_IsUndefined(v) || JS_IsUninitialized(v))
        return uint_v();
      if (JS_ToInt32(ctx, &r, v))
        return uint_v();
      return uint_v(r);
    }
    static JSValue wrap(JSContext * ctx, uint_v i) noexcept
    {
      if (i.is_undefined())
        return JS_UNDEFINED;
      return JS_NewInt32(ctx, i.val(0));
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<int>::isa(ctx, v); }
  };

  template <> struct conv<int_v>
  {
    static int_v unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      int r;
      if (JS_IsUndefined(v) || JS_IsUninitialized(v))
        return int_v();
      if (JS_ToInt32(ctx, &r, v))
        return int_v();
      return int_v(r);
    }
    static JSValue wrap(JSContext * ctx, int_v i) noexcept
    {
      if (i.is_undefined())
        return JS_UNDEFINED;
      return JS_NewInt32(ctx, i.val(0));
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<int>::isa(ctx, v); }
  };
  
  template <> struct conv<int64>
  {
    /// @throws exception
    static int64 unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      int64 r;
      if (JS_ToInt64(ctx, &r, v))
        throw exception{};
      return r;
    }

    static JSValue wrap(JSContext * ctx, int64 i) noexcept
    {
      return JS_NewInt64(ctx, i);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<int>::isa(ctx, v); }

  };

  template <> struct conv<JSValue> // note will conflict with JSValue
  {
    /// @throws exception
    static JSValue unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return v;
    }

    static JSValue wrap(JSContext * ctx, JSValueConst v) noexcept
    {
      return v;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return true; }
  };

  /** Conversion traits for float64/double.
   */
  template <> struct conv<double>
  {
    /// @throws exception
    static double unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      double r;
      if (JS_ToFloat64(ctx, &r, v))
        throw exception{};
      return r;
    }

    static JSValue wrap(JSContext * ctx, double i) noexcept
    {
      return JS_NewFloat64(ctx, i);
    }

    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_FLOAT64; }

  };

  /** Conversion traits for float64/double.
   */
  template <> struct conv<float>
  {
    /// @throws exception
    static float unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return (float)conv<double>::unwrap(ctx,v,argc);
    }

    static JSValue wrap(JSContext * ctx, float i) noexcept
    {
      return conv<double>::wrap(ctx,i);
    }

    static bool isa(JSContext * ctx, JSValueConst v) { return conv<double>::isa(ctx,v); }

  };

  template <> struct conv<double_v>
  {
    /// @throws exception
    static double_v unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      double r;
      if (JS_ToFloat64(ctx, &r, v))
        return double_v();
      return double_v(r);
    }

    static JSValue wrap(JSContext * ctx, double_v i) noexcept
    {
      if(i.is_defined())
        return JS_NewFloat64(ctx, i.val(0.0));
      return JS_NAN;
    }

    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_FLOAT64; }
  };

  template <> struct conv<float_v>
  {
    /// @throws exception
    static float_v unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      double r = 0;
      if (!JS_IsNumber(v))
        return float_v();
      if (JS_ToFloat64(ctx, &r, v))
        return float_v();
      return float(r);
    }

    static JSValue wrap(JSContext * ctx, float_v i) noexcept
    {
      if (i.is_defined())
        return JS_NewFloat64(ctx, i.val(0.0f));
      return JS_NAN;
    }

    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_FLOAT64; }
  };


  /** Conversion traits for tool::chars */
  template <> struct conv<tool::chars>
  {
    static tool::string unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      size_t plen;
      const char * ptr = JS_ToCStringLen(ctx, &plen, v);
      if (!ptr) {
        ptr = JS_ToCStringLen(ctx, &plen, v);
        return CHARS("[object]");
      }
        //throw exception{};
      tool::string rv{ ptr,plen };
      JS_FreeCString(ctx, ptr);
      return rv;
    }
    static JSValue wrap(JSContext * ctx, tool::chars str) noexcept
    {
      return JS_NewStringLen(ctx, str.start, str.length);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_STRING; }
  };

  /** Conversion traits for tool::chars */
  template <> struct conv<tool::wchars>
  {
    /*static tool::ustring unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      size_t plen;
      const char * ptr = JS_ToCStringLen(ctx, &plen, v);
      if (!ptr)
        throw exception{};
      tool::ustring rv = tool::u8::cvt(tool::chars(ptr, plen));
      JS_FreeCString(ctx, ptr);
      return rv;
    }*/
    static JSValue wrap(JSContext * ctx, tool::wchars s) noexcept
    {
      tool::string str = tool::u8::cvt(s);
      return JS_NewStringLen(ctx, str.cbegin(), str.length());
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_STRING; }
  };

  /** Conversion traits for tool::string */
  template <> struct conv<tool::string>
  {
    static tool::string unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<tool::chars>::unwrap(ctx, v);
    }

    static JSValue wrap(JSContext * ctx, const tool::string& str) noexcept
    {
      return JS_NewStringLen(ctx, str.cbegin(), str.length());
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_STRING; }
  };

  /** Conversion traits for tool::string */
  template <> struct conv<tool::ustring>
  {
    static tool::ustring unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      //if (!JS_IsString(v))
      //  return tool::ustring();
      size_t plen;
      const char * ptr = JS_ToCStringLen(ctx, &plen, v);
      if (!ptr)
        return WCHARS("[object]");
      tool::ustring rv = tool::u8::cvt(tool::chars(ptr, plen));
      JS_FreeCString(ctx, ptr);
      return rv;
    }

    static JSValue wrap(JSContext * ctx, const tool::ustring& s) noexcept
    {
      tool::string str = tool::u8::cvt(s);
      return JS_NewStringLen(ctx, str.cbegin(), str.length());
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_VALUE_GET_NORM_TAG(v) == JS_TAG_STRING; }
  };

  hvalue dom_rect(JSContext * ctx, gool::rectf rc);
  hvalue dom_rect(JSContext * ctx, gool::rect rc);

  template <> struct conv<gool::rectf>
  {
    static JSValue wrap(JSContext * ctx, gool::rectf rc) noexcept
    {
      hvalue obj = dom_rect(ctx, rc);
      return obj.tearoff();
    }
  };

  template <> struct conv<gool::rect>
  {
    static JSValue wrap(JSContext * ctx, gool::rect rc) noexcept
    {
      hvalue obj = dom_rect(ctx, rc);
      return obj.tearoff();
    }
  };

  /** Conversion traits for html::document */
  template <> struct conv<html::document*>
  {
    static html::document* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (JS_IsNull(v))
        return nullptr;
      html::document* pd = document_ptr_of(ctx,v);
      //html::node* p = (html::node*)JS_GetOpaque(v, Document_class_id);
      if (!pd) 
        throw qjs::om::type_error("Document was deleted");
      return pd;
    }

    static JSValue wrap(JSContext * ctx, html::document* pd) noexcept
    {
      if (!pd)
        return JS_NULL;
      if (!pd->obj && !pd->flags.strayed && pd->pview()) {
        pd->obj.v = JS_NewObjectClass(ctx, Document_class_id);
        pd->add_ref();
        JS_SetOpaque(pd->obj.v, static_cast<html::node*>(pd));
      }
      return JS_DupValue(ctx, pd->obj.v);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Document_class_id) != NULL; }
  };

  template <> struct conv<html::hdocument>
  {
    static html::hdocument unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<html::document*>::unwrap(ctx, v, argc);
    }
    static JSValue wrap(JSContext * ctx, html::hdocument pel) noexcept
    {
      return conv<html::document*>::wrap(ctx, pel);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Document_class_id) != NULL; }
  };



  /** Conversion traits for html::element */
  template <> struct conv<html::element*>
  {
    static html::element* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (JS_IsNull(v))
        return nullptr;
      html::element* pe = element_ptr_of(ctx,v);
      if (!pe) {
        if (can_get_element_from(ctx, v))
          throw qjs::om::type_error("Element was deleted");
        else
          throw qjs::om::type_error("Element expected");
      }
      return pe;
    }

    static JSValue wrap(JSContext * ctx, html::element* pel) noexcept
    {
      if (!pel)
        return JS_NULL;
      if (pel->is_document())
        return conv<html::document*>::wrap(ctx, static_cast<html::document*>(pel));
      if (!pel->obj && !pel->flags.strayed) {
        pel->obj.v = JS_NewObjectClass(ctx, Element_class_id);
        pel->add_ref();
        JS_SetOpaque(pel->obj.v, (html::node*)pel);
        pel->get_style(*view_of(ctx));
        if (pel->flags.is_synthetic)
          return pel->obj.tearoff(); // these live in the air
      }
      return JS_DupValue(ctx, pel->obj.v);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { 
      html::element* pe = element_ptr_of(ctx, v);
      return pe != nullptr; 
    }
  };

  template <> struct conv<html::helement>
  {
    static html::helement unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<html::element*>::unwrap(ctx, v, argc);
    }
    static JSValue wrap(JSContext * ctx, html::helement pel) noexcept
    {
      return conv<html::element*>::wrap(ctx, pel);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Element_class_id) != NULL; }
  };

  template <> struct conv<tool::array<html::helement>>
  {
    static tool::array<html::helement> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      assert(false);
      return tool::array<html::helement>();
    }

    static JSValue wrap(JSContext * ctx, tool::array<html::helement> pel) noexcept
    {
      JSValue arr = JS_NewArray(ctx);
      for (int n = 0; n < pel.size(); ++n)
        JS_DefinePropertyValueUint32(ctx, arr, n, conv<html::helement>::wrap(ctx,pel[n]), JS_PROP_C_W_E);
      return arr;
    }
    //static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Document_class_id) != NULL; }
  };

  /** Conversion traits for html::text */
  template <> struct conv<html::text*>
  {
    static html::text* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (JS_IsNull(v))
        return nullptr;
      html::text* pe = text_ptr_of(ctx, v);
      if (!pe)
        throw qjs::om::type_error("Element was deleted");
      return pe;
    }

    static JSValue wrap(JSContext * ctx, html::text* pel) noexcept
    {
      if (!pel)
        return JS_NULL;
      if (!pel->obj) {
        pel->obj.v = JS_NewObjectClass(ctx, Text_class_id);
        pel->add_ref();
        JS_SetOpaque(pel->obj.v, (html::node*)pel);
      }
      return JS_DupValue(ctx, pel->obj.v);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Text_class_id) != NULL; }
  };

  template <> struct conv<handle<html::text>>
  {
    static handle<html::text> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<html::text*>::unwrap(ctx, v, argc);
    }
    static JSValue wrap(JSContext * ctx, handle<html::text> pel) noexcept
    {
      return conv<html::text*>::wrap(ctx, pel);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Text_class_id) != NULL; }
  };

  /** Conversion traits for html::comment */
  template <> struct conv<html::comment*>
  {
    static html::comment* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (JS_IsNull(v))
        return nullptr;
      html::comment* pe = comment_ptr_of(ctx, v);
      if (!pe)
        throw qjs::om::type_error("Element was deleted");
      return pe;
    }

    static JSValue wrap(JSContext * ctx, html::comment* pel) noexcept
    {
      if (!pel)
        return JS_NULL;
      if (!pel->obj) {
        pel->obj.v = JS_NewObjectClass(ctx, Comment_class_id);
        pel->add_ref();
        JS_SetOpaque(pel->obj.v, (html::node*)pel);
      }
      return JS_DupValue(ctx, pel->obj.v);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Comment_class_id) != NULL; }
  };

  template <> struct conv<handle<html::comment>>
  {
    static handle<html::comment> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<html::comment*>::unwrap(ctx, v, argc);
    }
    static JSValue wrap(JSContext * ctx, handle<html::comment> pel) noexcept
    {
      return conv<html::comment*>::wrap(ctx, pel);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Comment_class_id) != NULL; }
  };

  /** Conversion traits for html::node */
  template <> struct conv<html::node*>
  {
    static html::node* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (JS_IsNull(v))
        return nullptr;
      html::node* pn = node_ptr_of(ctx,v);
      if (!pn)
        throw qjs::om::type_error("Node was deleted");
      return pn;
    }

    static JSValue wrap(JSContext * ctx, html::node* pn) noexcept
    {
      if (!pn)
        return JS_NULL;
      if (!pn->obj) {
        if (pn->is_element())
          pn->obj.v = JS_NewObjectClass(ctx, Element_class_id);
        else if (pn->is_text())
          pn->obj.v = JS_NewObjectClass(ctx, Text_class_id);
        else if (pn->is_comment())
          pn->obj.v = JS_NewObjectClass(ctx, Comment_class_id);
        else if (pn->is_document())
          pn->obj.v = JS_NewObjectClass(ctx, Document_class_id);
        else
          assert(false);
        pn->add_ref();
        JS_SetOpaque(pn->obj.v, (html::node*)pn);
      }
      return JS_DupValue(ctx, pn->obj.v);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Node_class_id) != NULL; }
  };

  template <> struct conv<html::hnode>
  {
    static html::hnode unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      return conv<html::node*>::unwrap(ctx,v,argc);
    }

    static JSValue wrap(JSContext * ctx, html::hnode pn) noexcept
    {
      return conv<html::node*>::wrap(ctx, pn);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return conv<html::node*>::isa(ctx,v); }
  };

  /** Conversion traits for html::node::NODE_TYPE */
  template <> struct conv<html::node::NODE_TYPE>
  {
    static html::node::NODE_TYPE unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0) { return (html::node::NODE_TYPE)conv<int>::unwrap(ctx, v);}
    static JSValue wrap(JSContext * ctx, html::node::NODE_TYPE v) noexcept { return conv<int>::wrap(ctx, (int)v); }
  };

  /** Conversion traits for html::subscription::flags */
  template <> struct conv<html::subscription::flags>
  {
    static html::subscription::flags unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0);
    static JSValue wrap(JSContext * ctx, html::subscription::flags v) noexcept {
      assert(false);
      return JS_UNDEFINED;
      //return conv<int>::unwrap(ctx, (int)v); 
    }
  };

  template <> struct conv<hvalue>
  {
    static hvalue unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      //return hvalue(v);
      return hvalue(ctx, v);
    }

    static JSValue wrap(JSContext * ctx, hvalue v) noexcept
    {
      return JS_DupValue(ctx, v); // ????
      //return v.v;
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return true; }
  };

  template <> struct conv<qjs::xcontext*>
  {
    static qjs::xcontext* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0) noexcept;
  };

  template <typename T>
  struct conv<tool::array<T>>
  {
    static JSValue wrap(JSContext * ctx, const tool::array<T>& arr) noexcept
    {
      hvalue vec = JS_NewArray(ctx);
      for (uint n = 0; n < arr.length(); ++n) {
        JSValue el = conv<T>::wrap(ctx, arr[n]);
        JS_SetPropertyUint32(ctx, vec, n, el);
      }
      return vec.tearoff();
      //return JS_UNDEFINED;
    }

    static tool::array<T> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      if (argc == 0 || JS_IsArray(ctx,v)) {
        JSValue vlen = JS_GetPropertyStr(ctx, v, "length");
        auto len = conv<uint>::unwrap(ctx, vlen);
        array<T> arr(len);
        for (uint32_t i = 0; i < (uint32_t)len; i++) {
          hvalue el = JS_GetPropertyUint32(ctx, v, i);
          arr[i] = conv<T>::unwrap(ctx, el);
        }
        return arr;
      }
      else {
        argc = max(0, argc);
        tool::array<T> arr(argc);
        const JSValue* pv = &v;
        for (int i = 0; i < argc; ++i)
          arr[i] = conv<T>::unwrap(ctx, pv[i]);
        return arr;
      }
    }

  };

  template <typename TK, typename TV>
  struct conv<tool::dictionary<TK,TV>>
  {
    static JSValue wrap(JSContext * ctx, const tool::dictionary<TK, TV>& dict) noexcept
    {
      JSValue obj = JS_NewObject(ctx);
      for (int n = 0; n < dict.size(); ++n) {
        string k = u8::cvt(dict.key(n).to_string());
        JSAtom ka = JS_NewAtomLen(ctx, k.cbegin(), k.length());
        JSValue v = conv<TV>::wrap(ctx, dict.value(n));
        JS_SetProperty(ctx, obj, ka, v);
        JS_FreeAtom(ctx, ka);
      }
      return obj;
      //return JS_UNDEFINED;
    }

    static tool::dictionary<TK, TV> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      tool::dictionary<TK, TV> dict;
      if (JS_IsObject(v)) {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(ctx, &tab, &len, v, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars pname(ctx,tab[n].atom);
          hvalue val = JS_GetProperty(ctx, v, tab[n].atom);
          dict[TK(pname)] = conv<TV>::unwrap(ctx, val);
        }
        js_free_prop_enum(ctx, tab, len);
      }
      else if (v != JS_NULL)
        throw qjs::om::type_error("not an object");
      return dict;
    }
  };

  template <typename TV>
  struct conv<tool::dictionary<string, TV>>
  {
    static JSValue wrap(JSContext * ctx, const tool::dictionary<string, TV>& dict) noexcept
    {
      JSValue obj = JS_NewObject(ctx);
      for (int n = 0; n < dict.size(); ++n) {
        string k = dict.key(n);
        JSAtom ka = JS_NewAtomLen(ctx, k.cbegin(), k.length());
        JSValue v = conv<TV>::wrap(ctx, dict.value(n));
        JS_SetProperty(ctx, obj, ka, v);
        JS_FreeAtom(ctx, ka);
      }
      return obj;
      //return JS_UNDEFINED;
    }

    static tool::dictionary<string, TV> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      tool::dictionary<string, TV> dict;
      if (JS_IsObject(v)) {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(ctx, &tab, &len, v, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars pname(ctx, tab[n].atom);
          hvalue val = JS_GetProperty(ctx, v, tab[n].atom);
          dict[string(pname)] = conv<TV>::unwrap(ctx, val);
        }
        js_free_prop_enum(ctx, tab, len);
      }
      else if (v != JS_NULL)
        throw qjs::om::type_error("not an object");
      return dict;
    }
  };
  

  template <typename T>
  struct conv<tool::slice<T>>
  {
    static JSValue wrap(JSContext * ctx, const tool::slice<T>& arr) noexcept
    {
      hvalue vec = JS_NewArray(ctx);
      for (uint n = 0; n < arr.length; ++n) {
        JSValue el = conv<T>::wrap(ctx, arr[n]);
        JS_SetPropertyUint32(ctx, vec, n, el);
      }
      return vec.tearoff();
    }

    /*static tool::array<T> unwrap(JSContext * ctx, const JSValueConst& v, int argc)
    {
      tool::array<T> arr;
      if (JS_IsArray(ctx, v)) {
        JSValue vlen = JS_GetPropertyStr(ctx, v, "length");
        auto len = conv<uint>::unwrap(ctx, vlen);
        for (uint i = 0; i < len; i++) {
          hvalue el = JS_GetPropertyUint32(ctx, v, i);
          arr.push(conv<T>::unwrap(ctx, el));
        }
      }
      return arr;
    }*/
  };

  template <>
  struct conv<tool::slice<JSValue>>
  {
    static JSValue wrap(JSContext * ctx, tool::slice<JSValue> seq) noexcept
    {
      return JS_UNDEFINED;
    }

    static tool::slice<JSValue> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      argc = max(0, argc);
      return tool::slice<JSValue>(&v,argc);
    }
  };

  template <typename TE>
  struct conv<html::enumv<TE>>
  {
    static JSValue wrap(JSContext * ctx, const html::enumv<TE>& en) noexcept
    {
      return conv<ustring>::wrap(ctx,en.to_string());
    }

    static html::enumv<TE> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      ustring sv = conv<ustring>::unwrap(ctx,v);
      html::enumv<TE> rv;
      rv.set(sv);
      return rv;
    }
  };

  JSValue wrap_asset(JSContext* ctx, som_asset_t* pass);

  /** Conversion traits for som_asset_t */
  template <> struct conv<som_asset_t*>
  {
    static som_asset_t* unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      som_asset_t* pass = (som_asset_t*)JS_GetOpaque(v, Asset_class_id);
      if (!pass)
        throw qjs::om::type_error("The asset is gone");
      return pass;
    }

    static JSValue wrap(JSContext * ctx, som_asset_t* pass) noexcept
    {
      if (!pass)
        return JS_NULL;
      return wrap_asset(ctx, pass);
    }
    static bool isa(JSContext * ctx, JSValueConst v) { return JS_GetOpaque(v, Asset_class_id) != NULL; }
  };

  /** Conversion traits for tool::string */
  template <> struct conv<tool::value>
  {
    static tool::value unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0);
    static JSValue wrap(JSContext * ctx, tool::value v) noexcept;

    static bool isa(JSContext * ctx, JSValueConst v) { return true; /*check if it can be converted*/ }

  };

  template <> struct conv<array<byte>>
  {
    static array<byte> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      size_t size = 0;
      uint8_t * ptr = JS_GetArrayBuffer(ctx, &size, v);
      //if(!ptr)
      //  throw exception{};
      return bytes(ptr, size);
    }
    static JSValue wrap(JSContext * ctx, array<byte> v) noexcept
    {
      return JS_NewArrayBufferCopy(ctx, v.cbegin(), v.length());
    }
    static bool isa(JSContext * ctx, JSValueConst v) {
      size_t size;
      uint8_t * ptr = JS_GetArrayBuffer(ctx, &size, v);
      return ptr != nullptr;
    }
  };

  template <> struct conv<bytes>
  {
    /*static array<byte> unwrap(JSContext * ctx, const JSValueConst& v, int argc = 0)
    {
      size_t size = 0;
      uint8_t * ptr = JS_GetArrayBuffer(ctx, &size, v);
      //if(!ptr)
      //  throw exception{};
      return bytes(ptr, size);
    }*/
    static JSValue wrap(JSContext * ctx, bytes v) noexcept
    {
      return JS_NewArrayBufferCopy(ctx, v.start, v.length);
    }
    static bool isa(JSContext * ctx, JSValueConst v) {
      size_t size;
      uint8_t * ptr = JS_GetArrayBuffer(ctx, &size, v);
      return ptr != nullptr;
    }
  };


  JSValue wrap_native_functor(JSContext* ctx, tool::native_functor_holder* pf);

}

#include "xfetch.h"

#endif
