#pragma once

#ifndef __xdomjs_xutils_h__
#define __xdomjs_xutils_h__

#include <type_traits>
#include "xconv.h"
#include "xcolor.h"
#include "xevent.h"

namespace qjs {
  using namespace tool;
  using namespace gool;

  enum {
    __JS_ATOM_NULL = JS_ATOM_NULL,
#define DEF(name, str) JS_ATOM_ ## name,
#include "quickjs/quickjs-atom.h"
#undef DEF
    JS_ATOM_END,
  };


  struct xcontext : public html::context
  {
    JSContext *ctx;

    xcontext(JSContext *c = nullptr): ctx(c) {}
    xcontext(html::document *pd);
    xcontext(const xcontext& ctx) : xcontext(ctx.pdoc()) {}

    const known_atoms_list& known_atoms();

    virtual ~xcontext() {}

    virtual html::document*   pdoc() const override;
    virtual html::view*       pview() const override;
            xview*            pxview() const;

    virtual uint64            next_timer_id();

    virtual void add_airborn_node(html::node* pn);

    operator JSContext*() const { return ctx; }

    template<typename T> T get(JSValueConst v) {
      return conv<std::decay_t<T>>::unwrap(ctx, v);
    }

    template<typename T> T get(JSValueConst v, T df) {
      if (v == JS_UNDEFINED) return df;
      return conv<std::decay_t<T>>::unwrap(ctx, v);
    }

    template<typename T> JSValue val(T v) {
      return conv<T>::wrap(ctx, v);
    }

    template<typename T> JSValue retval(T v) {
      if (std::is_void<T>::value)
        return JS_UNDEFINED;
      else
        return conv<T>::wrap(ctx, v);
    }

    template<typename T> T get_last(JSValueConst& v, int rest) {
      return conv<std::decay_t<T>>::unwrap(ctx, v, rest);
    }

    template<typename T> bool isa(JSValueConst v) {
      return conv<std::decay_t<T>>::isa(ctx, v);
    }

    bool is_event_handler(JSValueConst v) {
      if (JS_IsFunction(ctx, v))
        return true;
      if (JS_IsObject(v) && JS_HasProperty(ctx,v,known_atoms().handleEvent))
        return true;
      return false;
    }


// PROPERTY
    // get
    template<typename T> T get_prop(const char* name, JSValueConst v) {
      hvalue pv = JS_GetPropertyStr(ctx, v, name);
      return get<T>(pv);
    }
    template<typename T> T get_prop(JSAtom name, JSValueConst v) {
      hvalue pv = JS_GetProperty(ctx, v, name);
      return get<T>(pv);
    }

    template<typename T, typename DT> T get_prop(const char* name, JSValueConst v, DT defv) {
      hvalue pv = JS_GetPropertyStr(ctx, v, name);
      if (JS_IsUndefined(pv)) return T(defv);
      return get<T>(pv);
    }

    template<typename T, typename DT> T get_prop(JSAtom name, JSValueConst v, DT defv) {
      hvalue pv = JS_GetProperty(ctx, v, name);
      if (JS_IsUndefined(pv)) return T(defv);
      return get<T>(pv);
    }

    template<typename T> bool check_prop(const char* name, JSValueConst obj, T& v) {
      hvalue pv = JS_GetPropertyStr(ctx, obj, name);
      if (JS_IsUndefined(pv)) return false;
      v = get<T>(pv);
      return true;
    }

    template<typename T> bool check_prop(JSAtom name, JSValueConst obj, T& v) {
      hvalue pv = JS_GetProperty(ctx, obj, name);
      if (JS_IsUndefined(pv)) return false;
      v = get<T>(pv);
      return true;
    }



    template<typename T> T get_item(uint index, JSValueConst v) {
      hvalue pv = JS_GetPropertyUint32(ctx, v, index);
      return get<T>(pv);
    }

    template<typename T> void set_item(uint index, JSValueConst v, T iv) {
      JS_SetPropertyUint32(ctx, v, index, val(iv));
    }

    uint get_length(JSValueConst v) {
      return get_prop<uint>(known_atoms().length, v);
    }

    template<typename T> void set_prop(JSAtom name, JSValueConst obj, T pval) {
      JS_SetProperty(ctx, obj, name, val(pval));
    }

    template<typename T> void set_prop(const char* name, JSValueConst obj, T pval) {
      JS_SetPropertyStr(ctx, obj, name, val(pval));
    }

    bool is_undefined(JSValue v) { return JS_IsUndefined(v); }
    bool is_null(JSValue v) { return JS_IsNull(v); }

    bool is_function(JSValue v) { return JS_IsFunction(ctx, v); }
    bool is_function_of_this_realm(JSValue v) { return JS_IsFunctionOfThisRealm(ctx, v); }

    bool is_array(JSValue v) { return JS_IsArray(ctx, v); }
    bool is_string(JSValue v) { return JS_IsString(v); }
    bool is_string(JSValue v, string& s);
    bool is_object(JSValue v) { return JS_IsObject(v); }
    bool is_color(JSValue v, color_v& cv);
    bool is_int(JSValue v, int& cv);
    bool is_float(JSValue v, float& cv);
    bool is_tuple(JSValue v);
    bool is_length(JSValue v, html::size_v& sv);
    bool is_number(JSValue v);
    bool is_bool(JSValue v);
    string value_type(JSValue v);

    bool        is_file(JSValue v);
    ustring     file_path(JSValue v);
   
    hvalue global() { return hvalue(JS_GetGlobalObject(ctx)); }

    hvalue eval(tool::chars buffer, tool::chars filename = CHARS("<eval>"), unsigned eval_flags = 0, int line_no = 1)
    {
      assert(buffer.start[buffer.length] == '\0' && "eval buffer is not null-terminated"); // JS_Eval requirement
      assert(filename.start[filename.length] == '\0' && "filename is not null-terminated"); // JS_Eval requirement
      hvalue rv = hvalue(JS_Eval2(ctx, buffer.start, buffer.length, filename.start, eval_flags | JS_EVAL_FLAG_BACKTRACE_BARRIER, line_no));
      check_pending_job_status();
      if (JS_IsException(rv)) {
        throw exception();
      }
      return rv;
    }

    hvalue eval_silent(tool::chars buffer, tool::chars filename = CHARS("<eval>"), int line_no = 1)
    {
      assert(buffer.start[buffer.length] == '\0' && "eval buffer is not null-terminated"); // JS_Eval requirement
      assert(filename.start[filename.length] == '\0' && "filename is not null-terminated"); // JS_Eval requirement
      hvalue rv = hvalue(JS_Eval2(ctx, buffer.start, buffer.length, filename.start, JS_EVAL_TYPE_GLOBAL | JS_EVAL_FLAG_BACKTRACE_BARRIER, line_no));
      check_pending_job_status();
      return rv;
    }

    hvalue eval_this(hvalue _this,tool::chars buffer, tool::chars filename = CHARS("<eval>"), unsigned eval_flags = 0, int line_no = 1)
    {
      assert(buffer.start[buffer.length] == '\0' && "eval buffer is not null-terminated"); // JS_Eval requirement
      assert(filename.start[filename.length] == '\0' && "filename is not null-terminated"); // JS_Eval requirement
      hvalue rv = hvalue(JS_EvalThis(ctx, _this,buffer.start, buffer.length, filename.start, JS_EVAL_TYPE_GLOBAL | JS_EVAL_FLAG_BACKTRACE_BARRIER));
      check_pending_job_status();
      if (JS_IsException(rv)) {
        throw exception();
      }
      return rv;
    }


    void check_pending_job_status();

    template<typename R, typename T>
    void call(R& rv, hvalue fun, T to) {
      hvalue self = conv<T>::wrap(ctx, to);
      hvalue r = JS_Call(ctx, fun, self, 0, nullptr);
      check_pending_job_status();
      if (JS_IsException(r))
        throw exception();
      else
        rv = conv<R>::unwrap(ctx, r);
    }

    template<typename R, typename T, typename P1>
    void call(R& rv, hvalue fun, T to, P1 p1) {
      hvalue self = conv<T>::wrap(ctx, to);
      hvalue args[] = { conv<P1>::wrap(ctx, p1) };
      hvalue r = JS_Call(ctx, fun, self, items_in(args), (JSValue*)args);
      check_pending_job_status();
      if (JS_IsException(r))
        throw exception();
      else
        rv = conv<R>::unwrap(ctx, r);
    }
    template<typename R, typename T, typename P1, typename P2>
    void call(R& rv, hvalue fun, T to, P1 p1, P2 p2) {
      hvalue self = conv<T>::wrap(ctx, to);
      hvalue args[] = { conv<P1>::wrap(ctx, p1),
                        conv<P2>::wrap(ctx, p2) };
      hvalue r = JS_Call(ctx, fun, self, items_in(args), (JSValue*)args);
      check_pending_job_status();
      if (JS_IsException(r))
        throw exception();
      else
        rv = conv<R>::unwrap(ctx, r);
    }

    template<typename R, typename T, typename P1, typename P2, typename P3>
    void call(R& rv, hvalue fun, T to, P1 p1, P2 p2, P3 p3) {
      hvalue self = conv<T>::wrap(ctx, to);
      hvalue args[] = { conv<P1>::wrap(ctx, p1),
                        conv<P2>::wrap(ctx, p2),
                        conv<P3>::wrap(ctx, p3) };
      hvalue r = JS_Call(ctx, fun, self, items_in(args), (JSValue*)args);
      check_pending_job_status();
      if (JS_IsException(r))
        throw exception();
      else
        rv = conv<R>::unwrap(ctx, r);
    }
    
    template<typename F>
    bool each_prop(JSValue v, F cb) {
      if (JS_IsObject(v)) {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(ctx, &tab, &len, v, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          atom_chars ac(*this, tab[n].atom);
          hvalue val = JS_GetProperty(ctx, v, tab[n].atom);
          cb(ac,val);
        }
        js_free_prop_enum(ctx, tab, len);
        return true;
      }
      return false;
    }

    template<typename F>
    bool each_prop_atom(JSValue v, F cb) {
      if (JS_IsObject(v)) {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(ctx, &tab, &len, v, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          hvalue val = JS_GetProperty(ctx, v, tab[n].atom);
          cb(tab[n].atom, val);
        }
        js_free_prop_enum(ctx, tab, len);
        return true;
      }
      return false;
    }



    template<typename F>
    bool each_key(JSValue v, F cb) {
      if (JS_IsObject(v)) {
        JSPropertyEnum* tab = nullptr;
        uint32_t len = 0;
        JS_GetOwnPropertyNames(ctx, &tab, &len, v, JS_GPN_STRING_MASK | JS_GPN_SYMBOL_MASK | JS_GPN_ENUM_ONLY);
        for (uint32_t n = 0; n < len; ++n) {
          hvalue val = JS_GetProperty(ctx, v, tab[n].atom);
          cb(tab[n].atom, val);
        }
        js_free_prop_enum(ctx, tab, len);
        return true;
      }
      return false;
    }

    template<typename F>
    bool each_item(JSValue v, F cb) {
      if (JS_IsArray(ctx,v)) {
        uint32_t len = get_length(v);
        for (uint32_t i = 0; i < (uint32_t)len; i++) {
          hvalue el = JS_GetPropertyUint32(ctx, v, i);
          cb(i,el);
        }
        return true;
      }
      return false;
    }

    array<hvalue> items(JSValue v) {
      array<hvalue> out;
      if (JS_IsArray(ctx, v)) {
        uint32_t len = get_length(v);
        for (uint32_t i = 0; i < (uint32_t)len; i++) {
          hvalue el = JS_GetPropertyUint32(ctx, v, i);
          out.push(el);
        }
      }
      return out;
    }


    hvalue clone(hvalue src) {
      if (is_array(src)) {
        array<hvalue> cpy;
        each_item(src, [&](uint i, hvalue el){ cpy.push(el); });
        return JS_NewFastArray(ctx,cpy.size(),(JSValue*)cpy.begin());
      }
      else if (is_object(src)) {
        hvalue r = JS_NewObject(ctx);
        each_prop_atom(src, [&](JSAtom key, hvalue val) {
          set_prop(key, r, val);
        });
        return r;
      }
      return src;
    }

    hvalue parseJSON(chars buffer, const char * filename = "<fromJSON>")
    {
      assert(buffer.start[buffer.length] == '\0' && "fromJSON buffer is not null-terminated"); // JS_ParseJSON requirement
      JSValue v = JS_ParseJSON(ctx, buffer.start, buffer.length, filename);
      return hvalue(v);
    }

    tool::string toJSON(JSValueConst v, const int spaces = 0)
    {
      assert(ctx);
      hvalue sv = JS_JSONStringify(ctx, v, JS_UNDEFINED, val(spaces));
      return get<tool::string>(sv);
    }

    tool::ustring toJSONw(JSValueConst v, const int spaces = 0)
    {
      assert(ctx);
      hvalue sv = JS_JSONStringify(ctx, v, JS_UNDEFINED, val(spaces));
      return get<tool::ustring>(sv);
    }

    /** returns current exception associated with hxcontext, and resets it. Should be called when qjs::exception is caught */
    hvalue get_exception() { return hvalue(JS_GetException(ctx)); }

    bool   is_exception(JSValueConst v ) { return JS_IsException(v); }

    ustring report_exception(JSValue iex = JS_UNINITIALIZED);
    void report_promise_rejection(JSValue reason);

    bool log(html::OUTPUT_SEVERITY os, slice<hvalue> argv);

    bool println(html::OUTPUT_SEVERITY os, chars s)
    {
      return println(os,u8::cvt(s)());
    }

    bool println(html::OUTPUT_SEVERITY os, wchars s)
    {
      tool::array<wchar> str = s;
      str.push(WCHARS("\n"));
      if(pview())
        pview()->debug_print(html::OT_TIS, os, str());
      return true;
    }

    bool printfln(html::OUTPUT_SEVERITY os, const char* fmt, ...) {
      va_list args;
      va_start(args, fmt);
      tool::string rv = tool::string::format_args(fmt, args);
      va_end(args);
      return println(os, rv());
    }

    string format(chars format, slice<JSValueConst> args);

    enum box_part_units {
      AS_PX,  // logical units : css::px
      AS_PPX, // physical pixels
    };

    hvalue box_part(chars part, gool::rect rc_ppx, box_part_units as = AS_PX);

    // JS facing functions:
    //int64  set_timeout(xcontext&, hvalue fun, int ms);
    //bool   clear_timeout(xcontext&, int64 tid);
    //int64  set_interval(xcontext&, hvalue fun, int ms);
    //bool   clear_interval(xcontext&, int64 iid);
    //int64  request_animation_frame(xcontext&, hvalue fun);
    //bool   cancel_animation_frame(xcontext&, int64 id);
    //bool   println(xcontext&, array<ustring> argv) { return print_ln(html::OS_INFO, argv()); }
    //int    device_pixels(xcontext&, JSValue of_what, string units, ustring axis);

    // Element.func - static functions

    //tristate_v js_element_get_state(xcontext&, html::element* p, tool::string name);
    //   bool    js_element_set_state(xcontext&, html::element* p, tool::string to_set, tool::string to_clear);

    // Window.func - static props and functions
    //xview* js_window_this(xcontext& c);
    //hvalue js_window_screenbox(xcontext& c, JSValue screenNo, JSValue area, JSValue part);

  };

  struct script_frame_animator : public html::animation {
    DEFINE_TYPE_ID_DERIVED(script_animator, animation)

    tool::array<hvalue> requested_frames;
    tool::array<hvalue> processing_frames;

    script_frame_animator() {}
    virtual ~script_frame_animator() {}

    virtual const char *name() { return "script-frame-animator"; }

    virtual bool is_finite(html::view &, html::element *) { return requested_frames.size() == 0; }

    virtual uint start(html::view &v, html::element *b, const html::style *nst, const html::style *ost) { return html::ANIMATION_TIMER_SPAN; }
    virtual uint step(html::view &v, html::element *b, uint current_clock);
    virtual void stop(html::view &v, html::element *b);
  };

  template<>
    inline hvalue xcontext::get_prop<hvalue>(const char* name, JSValueConst v) {
      hvalue pv = JS_GetPropertyStr(ctx, v, name);
      return pv;
    }

  template<>
    inline hvalue xcontext::get_prop<hvalue>(JSAtom name, JSValueConst v) {
      hvalue pv = JS_GetProperty(ctx, v, name);
      return pv;
    }

 template<>
    inline hvalue xcontext::get_item<hvalue>(uint index, JSValueConst v) {
      hvalue pv = JS_GetPropertyUint32(ctx, v, index);
      return pv;
    }


  template<>
    JSValue xcontext::get_prop<JSValue>(const char* name, JSValueConst v); // deliberately left undefined


  // represents physical context instance associated with document
  class context : public xcontext,
                  public html::view_debug_output
  {
    friend struct script_frame_animator;
    weak_handle<html::document> _pd;
    weak_handle<html::view>     _pv;
    //hvalue            cevent; // current event proxy object
    uint64                      last_timer_id = 0;
    handle<html::document_fragment> airborns;

    struct script_def : resource {
      JSModuleDef *module = nullptr;   // if module
      handle<html::pump::request> prq; // is set for modules that are is still loading
    };
    tool::hash_table<string, handle<script_def>> scripts;

    handle<script_frame_animator> animator;

    context(html::view* pv, html::document* pd);
  public:

    enum _ {
      SUBLIMATED_REFERENCES_CAP = 200
    };

    // noncopyable
    context(const context&) = delete;

    virtual ~context();

    virtual void finalize() override {} // view_debug_output

    virtual void add_airborn_node(html::node* pn) override
    {
      if (!airborns) {
        airborns = new html::document_fragment();
        airborns->parent = pdoc();
      }
      airborns->append(pn);
    }
    void purge_airborns() {
      if (airborns)
        airborns->stray(*pview());
      airborns = nullptr;
    }


    bool is_valid() const { return ctx != nullptr; }

    virtual html::document*   pdoc() const override { return _pd.ptr(); }
    virtual html::view*       pview() const override { return _pv.ptr(); }
    virtual uint64            next_timer_id() { return ++last_timer_id; }

    void init_locale_functions();

    void         load_script_element(html::helement b);
    JSModuleDef* load_script_module(string url);

    /** Get qjs::context from JSContext opaque pointer */
    static context& inst(JSContext * ctx);
    static context& create(html::view* pv, html::document* pd);
    static context& inst(html::document* pd);
        
    int64 request_animation_frame(hvalue fun);
    bool  cancel_animation_frame(int64 id);

    hvalue unhandled_exception_handler; // also unhandled_promise_rejection_handler;
    hvalue console_output_handler;
    hvalue request_complete_handler;

#ifdef CONFIG_DEBUGGER
    struct breakpoint {
      hatom filename;
      uint  lineno;

      bool  is_set() const { return filename && lineno; }
      void  clear() { filename.clear(); lineno = 0; }
    };

    enum DEBUGGER_CMD {
      DEBUGGER_CONTINUE,
      DEBUGGER_STEP_IN,
      DEBUGGER_STEP_OUT,
      DEBUGGER_STEP_OVER,
    };

    array<breakpoint> breakpoints;
    breakpoint        breakpoint_current;
    const byte*       breakpoint_pc = 0;
    hvalue            breakpoint_handler;
    hvalue            debugger_context; // API interface debugger.eval(), etc.
    int               breakpoint_depth = 0;
    DEBUGGER_CMD      debugger_cmd = DEBUGGER_CONTINUE;
    hatom             debugger_filename;
        
    void set_breakpoints(array<breakpoint> list) {
      breakpoints = list;
    }

    void set_breakpoint_handler(hvalue handler, string src_to_avoid);

#endif // CONFIG_DEBUGGER

    struct sublimated_reference {
      int64  id;
      hvalue val;
    };
    array<sublimated_reference> sublimated_references;

    int64 add_reference(hvalue v)
    {
      int64 id = (int64)(uint_ptr)js_debugger_get_object_id(v);
      for (auto& r : sublimated_references)
        if (r.id == id) return id;

      if (sublimated_references.length() >= SUBLIMATED_REFERENCES_CAP)
        sublimated_references.remove(0);
      
      sublimated_reference sr;
      sr.val = v;
      sr.id = id;
      sublimated_references.push(sr); 
      return id;
    }

    hvalue get_reference(int64 id)
    {
      for (auto& r : sublimated_references)
        if (r.id == id) return r.val;
      return hvalue();
    }
    
    //view_debug_output
    virtual void print(uint subs, uint sev, wchars msg) override;

    hvalue attr_translator;
    hvalue text_translator;
    hvalue vnod_translator;

  };

  inline qjs::xcontext* conv<qjs::xcontext*>::unwrap(JSContext * ctx, const JSValueConst& v, int argc) noexcept
  {
    return &context::inst(ctx);
  }


}
#endif
