
#include "xcontext.h"
#include "xview.h"
#include "xcolor.h"
#include "quickjs/quickjs.h"

static uint ver[4] = { SCITER_VERSION };

uint module_version(bool major) {
  if (major)
    return ver[0] << 16 | ver[1];
  else
    return ver[2] << 16 | ver[3];
}

extern "C" JS_BOOL tjs_is_file(JSContext *ctx, JSValue val);
extern "C" JSValue tjs_get_file_path(JSContext *ctx, JSValue val);


namespace qjs {

  /** Wrapper over JSContext * ctx with smart handler semantics
   * Calls JS_SetContextOpaque(ctx, this); on construction and JS_FreeContext on destruction
   */
  
  context::~context() {
    if (ctx) {
      JS_FreeContext(ctx);
      ctx = nullptr;
    }
  }

  /** Get qjs::context from JSContext opaque pointer */
  context& context::inst(JSContext * ctx)
  {
    void * ptr = JS_GetContextOpaque(ctx);
    assert(ptr);
    return *reinterpret_cast<context *>(ptr);
  }

  context& context::inst(html::document * pd)
  {
    assert(pd);
    if (!pd->ns)
      assert(pd->ns);
    return context::inst(pd->ns);
  }

  context& context::create(html::view* pv, html::document* pd)
  {
    assert(!pd->ns);
    if (!pd->ns) {
      context* pctx = new context(pv, pd);
      pd->ns = *pctx;
    }
    return context::inst(pd->ns);
  }

  void context::load_script_element(html::helement b)
  {
    xview *pv = pxview();
    string src = b->atts[html::attr::a_src];
    ustring media = b->atts[html::attr::a_media];
    bool is_module = b->atts[html::attr::a_type] == WCHARS("module");
    if (media.is_defined()) {
      if (!pv->match_media_type(media))
        return;
    }
    if (!src.is_empty()) {
      src = html::combine_url(b->doc()->uri(), src);
      bool created = false;
      handle<script_def>& sd = scripts.get_ref(src, created);
      if (created) {
        sd = new script_def();
        sd->prq = new html::pump::request(src, html::DATA_SCRIPT);
        sd->prq->dst = pdoc();
        if (pv->load_data(sd->prq, true/*sync loading, sic!*/) && sd->prq->success_flag) {
          pv->run(pdoc(), chars_of(sd->prq->data()), src(), is_module);
          sd->prq = nullptr;
        }
        else {
          //this->printfln(html::OS_WARNING, "cannot load script <%s>", src.c_str());
          //ustring errs = ustring::format(W("cannot load script <%S>"), src.c_str());
          //pv->do_debug_print(html::OT_TIS, html::OS_WARNING, errs);
        }
      }
      return;
    }

    ustring text = b->get_text(*pv);

    if (trim(text()).length) {
      ustring id = b->attr_id();
      string src = b->doc()->uri().src;
      string utf8 = u8::cvt(text);
      // include_handler ih(pv);
      pv->run(pdoc(), utf8(), src(), is_module, b->line_no());
    }
  }

  JSModuleDef* context::load_script_module(tool::string url) {
    url = html::combine_url(pdoc()->uri(), url);
    bool created = false;
    handle<script_def>& sd = scripts.get_ref(url, created);
    if (created) {
      sd = new script_def();
      sd->module = JS_RunModule(ctx, pdoc()->uri().src, url);
    }
    return sd->module;
  }

  // view debug log
  void context::print(uint subs, uint sev, wchars msg) {
    if (is_function(console_output_handler))
    {
      try {
        bool r;
        call(r, console_output_handler, JS_UNDEFINED, subs, sev, ustring(msg));
        if (r) return;
      }
      catch (qjs::exception) {
        report_exception();
      }
    }
    if (pview())
      pview()->debug_print(subs, sev, msg);
    else
      html::view::get_current()->debug_print(subs, sev, msg);
  }

  // script log
  bool xcontext::log(html::OUTPUT_SEVERITY os, slice<hvalue> argv)
  {
    tool::ustring str;
    for (int i = 0; i < argv.size(); ++i) {
      if (i) str += WCHARS(",");
      str += get<ustring>(argv[i]);
    }
    str += WCHARS("\n");
    if (pview())
      pview()->debug_print(html::OT_TIS, os, str());
    else
      html::view::get_current()->debug_print(html::OT_TIS, os, str());
    return true;
  }



  html::document* document_of(JSContext * ctx) {
    return context::inst(ctx).pdoc();
  }
  qjs::xview* xview_of(JSContext * ctx) {
    return static_cast<qjs::xview*>(context::inst(ctx).pview());
  }
  html::view* view_of(JSContext * ctx) {
    return context::inst(ctx).pview();
  }

  html::timer_id next_timer_id_of(JSContext * ctx) {
    return context::inst(ctx).next_timer_id();
  }

  xcontext::xcontext(html::document *pd) :ctx(pd->ns) {}

  html::document*   xcontext::pdoc() const { return qjs::context::inst(ctx).pdoc(); }
  html::view*       xcontext::pview() const { return qjs::context::inst(ctx).pview(); }
  uint64            xcontext::next_timer_id() { return qjs::context::inst(ctx).next_timer_id(); }
  void              xcontext::add_airborn_node(html::node* pn) { qjs::context::inst(ctx).add_airborn_node(pn); }

  //hvalue            xcontext::get_current_event() { return context::inst(ctx).get_current_event(); }
  //void              xcontext::set_current_event(html::event* pevt) { context::inst(ctx).set_current_event(pevt); }

  //string            xcontext::risen_exception() { return context::inst(ctx).risen_exception(); }
  //void              xcontext::rise_exception(string msg) { context::inst(ctx).rise_exception(msg); }

  const qjs::known_atoms_list& xcontext::known_atoms() {
    return hruntime::get(JS_GetRuntime(ctx)).known_atoms;
  }

  xview* xcontext::pxview() const {
    qjs::xview* pv = static_cast<qjs::xview*>(pview());
    return pv;
  }

  void xcontext::check_pending_job_status() {
    if (qjs::xview* pv = pxview()) {
      if (pv->has_pending_script_jobs())
        pv->request_idle();
    }
  }

  ustring xcontext::report_exception(JSValue iex) {
    hvalue ex = iex == JS_UNINITIALIZED ? get_exception() : hvalue(iex);
    qjs::context& c = qjs::context::inst(ctx);
    auto_state<hvalue> ueh(c.unhandled_exception_handler, hvalue());
    if (is_function(ueh.value)) {
      try {
        bool r = false;
        call(r, ueh.value, JS_UNDEFINED, ex);
        if (r) return get<tool::ustring>(ex);
      }
      catch (exception) {
        ex = get_exception();
        //report_exception();
      }
    }
    tool::ustring errs = get<tool::ustring>(ex);
    if (get_prop<bool>(known_atoms().stack, ex)) {
      errs += WCHARS("\n");
      errs += get_prop<tool::ustring>(known_atoms().stack, ex)();
    }

    c.pview()->do_debug_print(html::OT_TIS, html::OS_ERROR, errs);
    //if (get_prop<bool>(known_atoms().stack, ex))
    //  c.pview()->do_debug_print(html::OT_TIS, html::OS_ERROR, get_prop<tool::ustring>(known_atoms().stack, ex)());

    return errs;
    /*
    tool::string errs = get<tool::string>(ex);
    println(html::OS_ERROR, errs());
    if (get_prop<bool>(known_atoms().stack, ex))
      println(html::OS_ERROR, get_prop<tool::string>(known_atoms().stack, ex)());
      */
  }

  void xcontext::report_promise_rejection(JSValue reason) {
    qjs::context& c = qjs::context::inst(ctx);
    auto_state<hvalue> ueh(c.unhandled_exception_handler, hvalue());
    if (is_function(ueh.value)) {
      bool r;
      try {
        hvalue re = JS_DupValue(ctx,reason); // this is the Error so we shall not deref it here more. 
        call(r, ueh.value, JS_UNDEFINED, re, JS_TRUE);
        if (r) return;
      }
      catch (exception) {
        report_exception();
      }
    }
    tool::string errs = CHARS("Unhandled rejection:") + get<tool::string>(reason);
    println(html::OS_ERROR, errs());
    if (get_prop<bool>(known_atoms().stack, reason))
      println(html::OS_ERROR, get_prop<tool::string>(known_atoms().stack, reason)());
  }


  string xcontext::format(chars fmt, slice<JSValueConst> args) {
    const char *pc = fmt.cbegin();
    const char *fmtend = fmt.cend();

    tool::string cfmt; // current format run

    tool::array<char> out; out.reserve(fmt.length);

    auto next_arg = [&]() -> hvalue {
      if (args) return ++args;
      return hvalue();
    };

    for (; pc < fmtend && *pc; ++pc) {
      while (*pc && *pc != '%')
        out.push(*pc++);

      if (*pc == 0) break;

      if (*++pc == '%') out.push('%');

      cfmt = '%';
      for (; *pc; ++pc) {
        if (*pc == '*') {
          int n = max(1, get<int>(next_arg()));
          cfmt += tool::string::format("%d", n);
        }
        else {
          if (*pc == 's' || *pc == 'S') {
            cfmt += "s";
            string str = get<string>(next_arg());
            if (*pc == 'S')
              str = tool::xml_escape_seq(str())();
            out.push(str());
            break;
          }
          else if (CHARS("dioxXbucC").contains(*pc)) {
            cfmt += *pc;
            int_v n = get<int_v>(next_arg());
            if (n.is_defined())
              out.push(tool::string::format(cfmt, n.val(0))());
            else
              out.push(CHARS("<NaN>"));
            break;
          }
          else if (CHARS("fgGeE").contains(*pc)) {
            cfmt += *pc;
            double_v n = get<double_v>(next_arg());
            if (n.is_defined())
              out.push(tool::string::format(cfmt, n.val(0.0))());
            else
              out.push(CHARS("<NaN>"));
          }
          // print data as JSON literal - suitable for parsing later by
          // parseData().
          else if (*pc == 'v' || *pc == 'V') {
            cfmt += *pc;
            hvalue v = next_arg();
            out.push(toJSON(v)());
            break;
          }
          else if (*pc == 0 || *pc == '%')
            break;
          else if (isdigit(*pc) || *pc == '.')
            cfmt += *pc;
          else {
            cfmt += *pc;
            out.push(cfmt());
            break;
          }
        }
      }
    }
    return out();
  }

  hvalue xcontext::box_part(chars part, gool::rect ppx_rc, box_part_units as) {

    rectf rc = ppx_rc;

    if (as == box_part_units::AS_PX)
      rc = pview()->ppx_to_dip(rc);

    if (part == CHARS("rectw") || part == CHARS("ltwh") || part == CHARS("xywh")) {
      float rw[] = { rc.left(), rc.top(), rc.width(), rc.height() };
      return val(items_of(rw));
    }
    if (part == CHARS("position")) {
      float rw[] = { rc.left(), rc.top() };
      return val(items_of(rw));
    }
    if (part == CHARS("dimension"))
    {
      float rw[] = { rc.width(), rc.height() };
      return val(items_of(rw));
    }
    if (part == CHARS("left")) { return val(rc.left()); }
    if (part == CHARS("right")) { return val(rc.right()); }
    if (part == CHARS("top")) { return val(rc.top()); }
    if (part == CHARS("bottom")) { return val(rc.bottom()); }
    if (part == CHARS("width")) { return val(rc.width()); }
    if (part == CHARS("height")) { return val(rc.height()); }
    // part == "rect" or "ltrb"
    float rw[] = { rc.left(), rc.top(), rc.right(), rc.bottom() };
    return val(items_of(rw));
  }

  bool xcontext::is_color(JSValue v, color_v& cv) {
    color_v t = get<color_v>(v);
    if (t.is_undefined()) return false;
    cv = t;
    return true;
  }

  bool xcontext::is_int(JSValue v, int& cv) {
    if (isa<int>(v)) {
      cv = get<int>(v);
      return true;
    }
    return false;
  }

  bool xcontext::is_float(JSValue v, float& cv) {
    if (JS_IsNumber(v)) {
      cv = get<float>(v);
      return true;
    }
    return false;
  }
  bool xcontext::is_number(JSValue v) {
    return JS_IsNumber(v);
  }

  bool xcontext::is_bool(JSValue v) {
    return JS_IsBool(v);
  }

  string xcontext::value_type(JSValue v) {
    switch (JS_VALUE_GET_TAG(v)) {
    case JS_TAG_UNINITIALIZED: return "void";
    case JS_TAG_INT : return "integer";
    case JS_TAG_BOOL: return "bool";
    case JS_TAG_NULL: return "null";
    case JS_TAG_UNDEFINED: return "undefined";
    case JS_TAG_CATCH_OFFSET: return "internal";
    case JS_TAG_EXCEPTION: return "exception";
    case JS_TAG_FLOAT64: return "float";

    case JS_TAG_OBJECT: {
    JSClassID cid = JS_GetClassID(v, nullptr);
    hvalue vs = JS_GetClassName(ctx, cid);
    return get<string>(vs);
  }
    case JS_TAG_FUNCTION_BYTECODE: return "bytecode";
    case JS_TAG_MODULE: return "module";
    case JS_TAG_STRING: return "string";
    case JS_TAG_SYMBOL: return "symbol";
    case JS_TAG_BIG_FLOAT: return "big float";
    case JS_TAG_BIG_INT: return "big int";
    case JS_TAG_BIG_DECIMAL: return "big decimal";
    }
    return string();
    
  }

  bool xcontext::is_string(JSValue v, tool::string& s) {
    if (JS_IsString(v)) {
      s = get<tool::string>(v);
      return true;
    }
    return false;
  }

  bool xcontext::is_tuple(JSValue v) {
    return !!JS_IsTuple(ctx, v);
  }

  bool xcontext::is_length(JSValue v, html::size_v& sv) {
    struct def { chars token; size_v::unit_type ut; };

    def defs[] = {
      { CHARS("em"), size_v::unit_type::em },
      { CHARS("ex"), size_v::unit_type::ex },
      { CHARS("pr"), size_v::unit_type::pr },
      { CHARS("sp"), size_v::unit_type::sp },
      { CHARS("fr"), size_v::unit_type::sp },
      { CHARS("fx"), size_v::unit_type::sp },
      { CHARS("px"), size_v::unit_type::px },
      { CHARS("in"), size_v::unit_type::in },
      { CHARS("cm"), size_v::unit_type::cm },
      { CHARS("mm"), size_v::unit_type::mm },
      { CHARS("pt"), size_v::unit_type::pt },
      { CHARS("pc"), size_v::unit_type::pc },
      { CHARS("dip"), size_v::unit_type::dip },
      { CHARS("width"), size_v::unit_type::pr_width }, // width(n%)
      { CHARS("height"), size_v::unit_type::pr_height }, // height(n%)
      { CHARS("vw"), size_v::unit_type::pr_view_width },
      { CHARS("vh"), size_v::unit_type::pr_view_height },
      { CHARS("vmin"), size_v::unit_type::pr_view_min },
      { CHARS("vmax"), size_v::unit_type::pr_view_max },
      { CHARS("rem"), size_v::unit_type::rem },
      { CHARS("ppx"), size_v::unit_type::ppx },
      { CHARS("ch"), size_v::unit_type::ch },
    };
    //size_v::unit_type
    //size_v::unit_type::ch
    return false;
  }

  bool xcontext::is_file(JSValue v) {
    return (bool)tjs_is_file(ctx, v);
  }

  ustring xcontext::file_path(JSValue v) {
    hvalue hn = tjs_get_file_path(ctx, v);
    if (is_string(hn))
      return get<ustring>(hn);
    return ustring();
  }

  tool::value conv<tool::value>::unwrap(JSContext * ctx, const JSValueConst& v, int argc)
  {
#pragma TODO("TODO-TODO-TODO...")
    int tag = JS_VALUE_GET_TAG(v);
    if (tag == JS_TAG_STRING)
      return tool::value(conv<ustring>::unwrap(ctx, v, argc));
    if (tag == JS_TAG_INT)
      return tool::value(conv<int>::unwrap(ctx, v, argc));
    if (tag == JS_TAG_UNDEFINED || tag == JS_TAG_UNINITIALIZED)
      return tool::value();
    if (tag == JS_TAG_NULL)
      return tool::value::null_val();
    if (tag == JS_TAG_BOOL)
      return tool::value(conv<bool>::unwrap(ctx, v, argc));
    if (tag == JS_TAG_BIG_INT)
      return tool::value(conv<int64>::unwrap(ctx, v, argc));
    if (JS_IsNumber(v))
      return tool::value(conv<double>::unwrap(ctx, v, argc));
    if (JS_IsException(v)) {
      tool::ustring errs = conv<tool::ustring>::unwrap(ctx, v, argc);
      return tool::value::make_error(errs);
    }
    if (JS_IsTuple(ctx, v))
    {
      string name = conv<string>::unwrap(ctx, JS_GetTupleTag(ctx, v));
      array<tool::value> elements = conv<array<tool::value>>::unwrap(ctx, v, 0);

      if (name == "url" && elements.size() == 1 && elements[0].is_string())
        return tool::value::make_url(elements[0].get_string());

      handle<function_value> pfv = new function_value();
      pfv->name = name;
      for (auto& e : elements)
        pfv->params.push(e);
      return tool::value::make_function(pfv);
    }
    if (JS_IsArray(ctx, v))
    {
#if 1 // proxify it
      return tool::value::make_proxy(new object_proxy(ctx, v), value::UT_OBJECT_ARRAY);
#else
      array_value* arrv = new array_value();
      arrv->elements = conv<array<tool::value>>::unwrap(ctx, v, 0);
      return tool::value::make_array(arrv);
#endif
    }
    if (JS_IsFunction(ctx, v)) {
      return tool::value::make_proxy(new object_proxy(ctx, v), value::UT_OBJECT_FUNCTION);
    }

    if (JS_IsError(ctx, v)) {
      xcontext c(ctx);
      tool::ustring errs = c.get_prop<tool::ustring>(c.known_atoms().message,v);
      if (c.get_prop<bool>(c.known_atoms().stack, v))
        errs += WCHARS("\n") + c.get_prop<tool::ustring>(c.known_atoms().stack, v);
      return tool::value::make_error(errs);
    }

    if (JS_IsObject(v)) {
#if 1
      double ms_since_1970;
      if (JS_IsDate(ctx, v, &ms_since_1970)) {
        tool::date_time dt = tool::date_time::from_absolute_millis_since_1970(ms_since_1970);
        return value(dt);
      }

      tool::bytes by;
      if (JS_IsArrayBuffer(ctx, v,&by.start,&by.length)) {
        return tool::value::make_bytes(by);
      }

      void* ptr = nullptr;
      JSClassID cid = JS_GetClassID(v, &ptr);

      if( cid == Color_class_id)
        return tool::value::make_packed_color((uint)(uintptr_t)(ptr));
      if (!ptr)
        return tool::value::make_proxy(new object_proxy(ctx, v), value::UT_OBJECT_OBJECT);
      if (auto pn = node_ptr_of(ctx, v))
        return tool::value::make_resource(pn);
      else if (auto pi = image_ptr_of(v))
        return tool::value::make_resource(pi);
      else if (auto pr = request_ptr_of(ctx, v))
        return tool::value::make_resource(pr);
      else if (auto pa = asset_ptr_of(ctx, v))
        return tool::value::wrap_asset(pa);

#else
      map_value *mapv = new map_value();
      mapv->params = conv<dictionary<tool::value, tool::value>>::unwrap(ctx, v, 0);
      return tool::value::make_map(mapv);
#endif
    }
    return tool::value(conv<ustring>::unwrap(ctx, v, argc));
  }


  /** Conversion traits for tool::string */
  JSValue conv<tool::value>::wrap(JSContext * ctx, tool::value v) noexcept
  {
    switch (v.type()) {
    case tool::value::t_undefined: return JS_UNDEFINED;
    case tool::value::t_null:   return JS_NULL;
    case tool::value::t_bool:   return conv<bool>::wrap(ctx, v.get_bool());
    case tool::value::t_int:    return conv<int>::wrap(ctx, v.get_int());
    case tool::value::t_double: return conv<double>::wrap(ctx, v.get_double());
    case tool::value::t_string: return conv<ustring>::wrap(ctx, v.get_string());
    case tool::value::t_date: {
      tool::date_time dt = v.get_date();
      return JS_NewDate(ctx, dt.seconds_since_1970() * 1000.0);
    }
    case tool::value::t_currency:
      return conv<int64>::wrap(ctx, v.get_int64());
    case tool::value::t_array:
    {
      auto parr = v.get_array();
      return conv<tool::array<value>>::wrap(ctx, parr->elements);
    }
    case tool::value::t_map:
    {
      auto pmap = v.get_map();
      return conv<tool::dictionary<value, value>>::wrap(ctx, pmap->params);
    }
    case tool::value::t_bytes: {
      bytes bs = v.get_bytes();
      return JS_NewArrayBufferCopy(ctx, bs.start, bs.length);
    }
    case tool::value::t_object_proxy: {
      return JS_DupValue(ctx, static_cast<qjs::object_proxy*>(v.get_proxy())->hv);
    }

    case tool::value::t_asset: {
      som_asset_t* pass = v.get_asset();
      return conv<som_asset_t*>::wrap(ctx, pass);
    }

    case tool::value::t_resource: {
      tool::resource* pr = v.get_resource();
      if (html::element* pel = v.get_resource<html::element>())
      {
        return conv<html::element*>::wrap(ctx, pel);
      }
      if (gool::image *pimg = v.get_resource<gool::image>()) {
        return conv<gool::image*>::wrap(ctx, pimg);
      }
      else if (gool::path *ppath = v.get_resource<gool::path>()) {
        handle<xpath> xp = new xpath(ppath);
        return conv<handle<xpath>>::wrap(ctx, xp);
      }
      else if (gool::brush* pel = v.get_resource<gool::brush>())
      {
        return conv<gool::brush*>::wrap(ctx, pel);
      }
      else if (auto sei = pr->get_interface<html::script_expando_interface>()) {
        xcontext c(ctx);
        hvalue val;
        bool r = sei->get_expando(c, val);
        assert(r); r = r;
        return val.tearoff(); //???????
      }
      else if (native_functor_holder* pf = v.get_resource<native_functor_holder>()) {
        return wrap_native_functor(ctx, pf);
      }
      // fall trough
    }

    //case tool::value::t_range,
    case tool::value::t_color: 
    {
      gool::argb c = (color)v.get_packed_color();
      return conv<gool::argb>::wrap(ctx, c);
    }

    case tool::value::t_duration:
    case tool::value::t_angle:
    case tool::value::t_enum:
    case tool::value::t_function:
    case tool::value::t_object:
#pragma TODO("TODO-TODO-TODO...")
    default:
    {
      ustring us = v.to_string();
      return conv<ustring>::wrap(ctx, us);
    }
    }
    return JS_UNINITIALIZED;
  }

  hvalue dom_rect(JSContext * ctx, gool::rectf rc)
  {
    xcontext c(ctx);
    hvalue obj = JS_NewObject(ctx);
    c.set_prop("x", obj, rc.left());
    c.set_prop("y", obj, rc.top());
    c.set_prop("width", obj, rc.width());
    c.set_prop("height", obj, rc.height());
    c.set_prop("left", obj, rc.left());
    c.set_prop("top", obj, rc.top());
    c.set_prop("right", obj, rc.left() + rc.width());
    c.set_prop("bottom", obj, rc.top() + rc.height());
    return obj;
  }
  hvalue dom_rect(JSContext * ctx, gool::rect rc) {
    xcontext c(ctx);
    hvalue obj = JS_NewObject(ctx);
    c.set_prop("x", obj, rc.left());
    c.set_prop("y", obj, rc.top());
    c.set_prop("width", obj, rc.width());
    c.set_prop("height", obj, rc.height());
    c.set_prop("left", obj, rc.left());
    c.set_prop("top", obj, rc.top());
    c.set_prop("right", obj, rc.left() + rc.width());
    c.set_prop("bottom", obj, rc.top() + rc.height());
    return obj;
  }

}

extern "C" void js_sciter_dump_error(JSContext *ctx, JSValue err) {
  qjs::xcontext c(ctx);
  c.report_exception(err);
}
