#include "xsciter.h"
#include "xview.h"

namespace tis {
  typedef tool::markup::scanner<wchar> markup_scanner;

  html::bookmark value2bookmark(xvm *c, value obm);
  value          bookmark2value(xvm *c, html::bookmark bm);

#define EOC wchar(-1)

  struct xtok_stream : public resource, public tool::markup::instream<wchar> {
    typedef tool::markup::instream<wchar> super;
    xtok_stream(ustring url) : super(url), chunk_src(nullptr), got_eof(false) {}

    virtual wchar get_char() {
      if (!chunk) {
        buffer.clear();
        if (got_eof || !get_chunk(buffer, chunk_src)) {
          got_eof = true;
          return 0;
        }
        chunk = buffer();
        ++line_no;
        return '\n';
      }

      if (until.length() && chunk.starts_with(until())) return wchar(-1);

      wchar t = chunk++;
      if (t == '\n') ++line_no;
      return t;
    }
    uint get_pos() const { return uint(chunk.start - buffer.cbegin()); }

    virtual bool get_chunk(array<wchar> &ch, void *&ch_src) = 0;

    array<wchar> buffer;
    wchars       chunk;
    void *       chunk_src;
    ustring      until;
    bool         got_eof;
  };

  struct element_xtok_stream : public xtok_stream {

    typedef xtok_stream super;

    html::each_node gen;

    element_xtok_stream(html::element *pel) : super(string()), gen(pel) {}

    virtual bool get_chunk(array<wchar> &ch, void *&ch_src) {
      html::hnode pn;
      while (gen(pn)) {
        if (pn->is_text()) {
          html::text *pt = pn.ptr_of<html::text>();
          ch             = pt->chars();
          ch_src         = pn;

          if (pt->marks.length()) {
            html::element *pel = pn->nearest_text_box();
            if (pel) {
              pt->marks.clear();
              pel->drop_layout();
            }
          }
          return true;
        }
      }
      return false;
    }
  };

  struct xtokenizer : public resource {
    handle<xtok_stream> in;
    handle<xtokenizer>  next;
    html::bookmark      start;
    html::bookmark      end;
    ustring             saved_tail;

    xtokenizer(xtok_stream *ps, xtokenizer *n = nullptr) : in(ps), next(n) {}

    virtual value get_token()       = 0;
    virtual value get_value(xvm *c) = 0;
    virtual value get_attr(xvm *c) { return NULL_VALUE; }
    virtual value get_tag(xvm *c) { return NULL_VALUE; }

    /*    void scan(VM* c)
        {
          if(handler)
            handler = CsCopyValue(c,handler);
          if( next )
            next->scan(c);
        } */

    virtual bool is_markup() const { return false; }
    virtual bool is_source() const { return false; }
  };

  struct xtokenizer_markup : public xtokenizer {
    typedef xtokenizer super;
    markup_scanner     scan;
    xtokenizer_markup(xtok_stream *ps, xtokenizer *n = nullptr)
        : super(ps, n), scan(*ps) {
      scan.token_start_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->start    = html::bookmark(pn, ps->get_pos() + off, false);
      };
      scan.token_end_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->end      = html::bookmark(pn, ps->get_pos() + off, false);
      };
    }

    virtual bool is_markup() const { return true; }

    value get_token() {
      switch (scan.get_token(false)) {
      case markup_scanner::TT_ERROR: {
        static value sym = CsSymbolOf(WCHARS("ERROR"));
        return sym;
      }
      case markup_scanner::TT_EOF:
        return NULL_VALUE; //{ static value sym = CsSymbolOf(WCHARS("EOF"));
                           // return sym; }
      case markup_scanner::TT_TAG_START: {
        static value sym = CsSymbolOf(WCHARS("TAG-START"));
        return sym;
      }
      case markup_scanner::TT_TAG_END: {
        static value sym = CsSymbolOf(WCHARS("TAG-END"));
        return sym;
      }
      case markup_scanner::TT_TAG_HEAD_END: {
        static value sym = CsSymbolOf(WCHARS("TAG-HEAD-END"));
        return sym;
      }
      case markup_scanner::TT_EMPTY_TAG_END: {
        static value sym = CsSymbolOf(WCHARS("TAG-EMPTY-END"));
        return sym;
      }
      case markup_scanner::TT_ATTR: {
        static value sym = CsSymbolOf(WCHARS("TAG-ATTR"));
        return sym;
      }
      case markup_scanner::TT_TEXT: {
        static value sym = CsSymbolOf(WCHARS("TEXT"));
        return sym;
      }
      case markup_scanner::TT_COMMENT: {
        static value sym = CsSymbolOf(WCHARS("COMMENT"));
        return sym;
      }
      case markup_scanner::TT_CDATA: {
        static value sym = CsSymbolOf(WCHARS("CDATA"));
        return sym;
      }
      case markup_scanner::TT_PI: {
        static value sym = CsSymbolOf(WCHARS("PI"));
        return sym;
      }
      // case markup_scanner::TT_WORD,        // in details mode these will be
      // generated  case markup_scanner::TT_SPACE,       // instead of TT_TEXT
      // above
      case markup_scanner::TT_DOCTYPE: {
        static value sym = CsSymbolOf(WCHARS("DOCTYPE"));
        return sym;
      }
      case markup_scanner::TT_ENTITIY: {
        static value sym = CsSymbolOf(WCHARS("ENTITIY"));
        return sym;
      }
      default: return UNDEFINED_VALUE;
      }
    }

    virtual value get_value(xvm *c) {
      return CsMakeString(c, scan.get_value());
    }

    virtual value get_attr(xvm *c) {
      return CsMakeString(c, scan.get_attr_name());
    }
    virtual value get_tag(xvm *c) {
      return CsMakeString(c, scan.get_tag_name());
    }
  };

  struct xtokenizer_source : public xtokenizer {
    typedef xtokenizer super;
    source_scanner     scan;

    xtokenizer_source(xtok_stream *ps, xtokenizer *n = nullptr,
                      bool css = false)
        : super(ps, n), scan(ps, css) {
      scan.token_start_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->start    = html::bookmark(pn, ps->get_pos() + off, false);
      };
      scan.token_end_cb = [this, ps](int off) {
        html::node *pn = static_cast<html::node *>(ps->chunk_src);
        this->end      = html::bookmark(pn, ps->get_pos() + off, false);
      };
    }

    value get_token() {
      switch (scan.get_token()) {
      case source_scanner::T_EOF:
        return NULL_VALUE; //{ static value sym = CsSymbolOf(WCHARS("EOF"));
                           // return sym; }
      case source_scanner::T_NUMBER: {
        static value sym = CsSymbolOf(WCHARS("NUMBER"));
        return sym;
      }
      case source_scanner::T_NUMBER_UNIT: {
        static value sym = CsSymbolOf(WCHARS("NUMBER-UNIT"));
        return sym;
      }
      case source_scanner::T_COLOR: {
        static value sym = CsSymbolOf(WCHARS("COLOR"));
        return sym;
      }
      case source_scanner::T_STRING: {
        static value sym = CsSymbolOf(WCHARS("STRING"));
        return sym;
      }
      case source_scanner::T_NAME: {
        static value sym = CsSymbolOf(WCHARS("NAME"));
        return sym;
      }
      case source_scanner::T_COMMENT: {
        static value sym = CsSymbolOf(WCHARS("COMMENT"));
        return sym;
      }
      case source_scanner::T_OPERATOR: {
        static value sym = CsSymbolOf(WCHARS("OPERATOR"));
        return sym;
      }
      case source_scanner::T_OPAREN: {
        static value sym = CsSymbolOf(WCHARS("O-PAREN"));
        return sym;
      }
      case source_scanner::T_CPAREN: {
        static value sym = CsSymbolOf(WCHARS("C-PAREN"));
        return sym;
      }
      case source_scanner::T_END_OF_ISLAND: {
        static value sym = CsSymbolOf(WCHARS("END-OF-ISLAND"));
        return sym;
      }
      case source_scanner::T_ERROR: {
        static value sym = CsSymbolOf(WCHARS("ERROR"));
        return sym;
      }
      default: {
        static value sym = CsSymbolOf(WCHARS("UNKNOWN"));
        return sym;
      }
      }
    }

    virtual value get_value(xvm *c) {
      return CsMakeString(c, scan.get_value());
    }
  };

  inline xtokenizer *xtokenizer_ptr(xvm *c, value obj) {
    assert(CsGetDispatch(obj) == c->tokenizerDispatch);
    if (!CsIsBaseType(obj, c->tokenizerDispatch)) return 0;
    return static_cast<xtokenizer *>(CsCObjectValue(obj));
  }

  /*inline value xtokenizer_object(xvm* c, xtokenizer* ptok)
  {
    ptok->add_ref();
    return CsMakeCPtrObject(c,c->tokenizerDispatch,ptok);
  }*/

  static value sym_source() {
    static value sym = CsSymbolOf("source");
    return sym;
  }
  static value sym_markup() {
    static value sym = CsSymbolOf("markup");
    return sym;
  }

  static value CSF_ctor(xvm *c) {
    value obj, el, sym, subType = NULL_VALUE;
    CsParseArguments(c, "V=*V=V=|V=", &obj, c->tokenizerDispatch, &el,
                     c->elementDispatch, &sym, &CsSymbolDispatch, &subType,
                     &CsSymbolDispatch);

    xtok_stream *ps = new element_xtok_stream(element_ptr(c, el));

    xtokenizer *pt;
    if (sym == sym_markup())
      pt = new xtokenizer_markup(ps);
    else
      pt = new xtokenizer_source(ps, nullptr,
                                 subType == CsSymbolOf("style") ||
                                     subType == CsSymbolOf("css"));
    pt->add_ref();
    CsSetCObjectValue(obj, pt);
    CsCtorRes(c) = obj;
    return obj;
  }

  // extern html::block* block_ptr(xvm* c, value obj);

  static value CSF_destroy(xvm *c) {
    /*    value obj;

        CsParseArguments(c,"V=*", &obj, c->imageDispatch);
        ximage* xim = image_ptr(c, obj);
        if( !xim )
          return UNDEFINED_VALUE;
        CsSetCObjectValue(obj,0);
        xim->release();*/
    return UNDEFINED_VALUE;
  }

  static value CSF_push(xvm *c) {
    value  obj, type, subType = NULL_VALUE;
    wchars until;

    CsParseArguments(c, "V=*V=|S#|V=", &obj, c->tokenizerDispatch, &type,
                     &CsSymbolDispatch, &until.start, &until.length, &subType,
                     &CsSymbolDispatch);
    handle<xtokenizer> xtkn = xtokenizer_ptr(c, obj);
    if (!xtkn) return UNDEFINED_VALUE;

    // static value s_markup = CsSymbolOf("markup");
    // static value s_source = CsSymbolOf("source");

    handle<xtokenizer> nxtkn;

    if (type == sym_markup()) {
      if (xtkn->is_markup()) return FALSE_VALUE;
      nxtkn = new xtokenizer_markup(xtkn->in, xtkn);
    } else if (type == sym_source()) {
      if (xtkn->is_source()) return FALSE_VALUE;
      nxtkn = new xtokenizer_source(xtkn->in, xtkn,
                                    subType == CsSymbolOf("style") ||
                                        subType == CsSymbolOf("css"));
    } else
      return NULL_VALUE;

    nxtkn->saved_tail = nxtkn->in->until;
    nxtkn->in->until  = until;

    xtkn->release();
    nxtkn->add_ref();
    CsSetCObjectValue(obj, nxtkn.ptr());
    return obj;
  }

  static value CSF_pop(xvm *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, c->tokenizerDispatch);
    handle<xtokenizer> xtkn = xtokenizer_ptr(c, obj);
    if (!xtkn) return UNDEFINED_VALUE;

    handle<xtokenizer> nxtkn = xtkn->next;
    if (!nxtkn) return obj;

    nxtkn->in->until = nxtkn->saved_tail;

    xtkn->next = nullptr;
    xtkn->release();
    nxtkn->add_ref();
    CsSetCObjectValue(obj, nxtkn.ptr());
    return obj;
  }

  static value CSF_token(xvm *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, c->tokenizerDispatch);
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;

    return xtok->get_token();
    // return NULL_VALUE;
  }

  static value CSF_get_tokenStart(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    return bookmark2value(c, xtok->start);
  }
  static value CSF_get_tokenEnd(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    if (xtok->start == xtok->end) {
      xtok->end.after_it = true;
      return bookmark2value(c, xtok->end);
    }
    return bookmark2value(c, xtok->end);
  }
  static value CSF_get_tag(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    return xtok->get_tag(c);
  }
  static value CSF_get_attr(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    return xtok->get_attr(c);
  }
  static value CSF_get_value(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    return xtok->get_value(c);
  }

  static value CSF_get_type(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    if (xtok->is_source()) {
      static value sym = CsSymbolOf("source");
      return sym;
    }
    if (xtok->is_markup()) {
      static value sym = CsSymbolOf("markup");
      return sym;
    }
    return xtok->get_value(c);
  }

  static value CSF_get_element(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    if (!xtok) return UNDEFINED_VALUE;
    html::node *pn = static_cast<html::node *>(xtok->in->chunk_src);
    if (pn) return element_object(c, pn->get_element());
    return NULL_VALUE;
  }

  static value CSF_elementType(xvm *c) {
    wchars tag;
    CsParseArguments(c, "**S#", &tag.start, &tag.length);

    html::tag::symbol_t t = html::tag::symbol(string(tag), false);
    if (!t) CS_RETURN3(c, NULL_VALUE, NULL_VALUE, NULL_VALUE);

    auto tt = html::tag::type(t);
    auto cm = html::tag::content_model(t);
    auto pm = html::tag::parsing_model(t);

    CS_RETURN3(c, CsMakeInteger(tt), CsMakeInteger(cm), CsMakeInteger(pm));
  }

  /* constants */
  static constant constants[] = {CONSTANT_ENTRY_X(0, 0)};

  /* methods */
  static c_method methods[] = {C_METHOD_ENTRY_X("this", CSF_ctor),
                               // C_METHOD_ENTRY_X( "destroy",    CSF_destroy ),
                               C_METHOD_ENTRY_X("push", CSF_push),
                               C_METHOD_ENTRY_X("pop", CSF_pop),
                               C_METHOD_ENTRY_X("token", CSF_token),

                               C_METHOD_ENTRY_X("elementType", CSF_elementType),

                               C_METHOD_ENTRY_X(0, 0)};

  /* properties */
  static vp_method properties[] = {
      // VP_METHOD_ENTRY_X( "lineNo",        CSF_get_lineNo,      0),
      // VP_METHOD_ENTRY_X( "url",           CSF_get_url,         0),
      VP_METHOD_ENTRY_X("tokenStart", CSF_get_tokenStart, 0),
      VP_METHOD_ENTRY_X("tokenEnd", CSF_get_tokenEnd, 0),
      VP_METHOD_ENTRY_X("tag", CSF_get_tag, 0),
      VP_METHOD_ENTRY_X("attr", CSF_get_attr, 0),
      VP_METHOD_ENTRY_X("value", CSF_get_value, 0),
      VP_METHOD_ENTRY_X("type", CSF_get_type, 0),
      VP_METHOD_ENTRY_X("element", CSF_get_element, 0),
      VP_METHOD_ENTRY_X(0, 0, 0)};

  void destroy_tokenizer(xvm *c, value obj) {
    xtokenizer *xtok = xtokenizer_ptr(c, obj);
    CsSetCObjectValue(obj, 0);
    xtok->release();
  }

  void TokenizerScan(VM *c, value obj) {
    CsCObjectScan(c, obj);
    // xtokenizer* xtok = xtokenizer_ptr(static_cast<xvm *>(c), obj);
    // xtok->scan(c);
  }

  void xvm::init_tokenizer_class() {
    dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(this), "Tokenizer", methods, properties, constants);

    /* create the 'image' type */
    if (!pd) CsInsufficientMemory(this);

    pd->destroy       = (destructor_t)destroy_tokenizer;
    pd->scan          = TokenizerScan;
    tokenizerDispatch = pd;
  }

} // namespace tis
