#include "xsciter.h"

namespace tis {

  static html::element *state_element_ptr(xvm *c, value obj) {
    assert(CsGetDispatch(obj) == c->elementInterfaces[STATE_IFACE]);
    if (CsGetDispatch(obj) != c->elementInterfaces[STATE_IFACE]) return 0;
    html::node *n = static_cast<html::node *>(CsCObjectValue(obj));
    assert(n && n->is_element());
    return n->cast<html::element>();
    // return static_cast<html::element*>(ptr<html::element>(obj));
  }

  static value CSF_value(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    html::view *pv = self->pview();
    if (!pv) return UNDEFINED_VALUE;
    tool::value v;
    if (pv->get_element_native_value(self, v,false)) return value_to_value(c, v);
    return UNDEFINED_VALUE;
  }

  static void CSF_set_value(xvm *c, value obj, value val) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return;
    html::view *pv = self->pview();
    if (!pv) return;
    tool::value v = value_to_value(c, val);
    pv->set_element_native_value(self, v,false);
  }

  static value CSF_allowReconciliation(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    return self->allow_reconciliation() ? TRUE_VALUE : FALSE_VALUE;
  }

  static void CSF_set_allowReconciliation(xvm *c, value obj, value val) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return;
    tristate_v flag;
    if (val == TRUE_VALUE) flag = TRUE;
    else if (val == FALSE_VALUE) flag = FALSE;
    self->allow_reconciliation(flag);
  }

  static value CSF_delayedLayout(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    return self->flags.need_delayed_measurement ? TRUE_VALUE : FALSE_VALUE;
  }

  static void CSF_set_delayedLayout(xvm *c, value obj, value val) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return;
    self->flags.need_delayed_measurement = val == TRUE_VALUE ? 1 : 0;
  }
  

  static value CSF_flowType(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    html::view *pv = self->pview();
    if (!pv) return UNDEFINED_VALUE;

    switch (self->check_layout(*pv)) {
    case html::flow_default: {
      static value sym = CsSymbolOf("default");
      return sym;
    }
    case html::flow_vertical: {
      static value sym = CsSymbolOf("vertical");
      return sym;
    }
    case html::flow_horizontal: {
      static value sym = CsSymbolOf("horizontal");
      return sym;
    }
    case html::flow_h_flow: {
      static value sym = CsSymbolOf("horizontal-flow");
      return sym;
    }
    case html::flow_v_flow: {
      static value sym = CsSymbolOf("vertical-flow");
      return sym;
    }
    case html::flow_grid: {
      static value sym = CsSymbolOf("grid");
      return sym;
    }
    case html::flow_table: {
      static value sym = CsSymbolOf("table");
      return sym;
    }
    case html::flow_table_fixed: {
      static value sym = CsSymbolOf("table-fixed");
      return sym;
    }
    case html::flow_stack: {
      static value sym = CsSymbolOf("stack");
      return sym;
    }
    case html::flow_text: {
      static value sym = CsSymbolOf("text");
      return sym;
    }
    case html::flow_table_row: {
      static value sym = CsSymbolOf("table-row");
      return sym;
    }
    case html::flow_table_body: {
      static value sym = CsSymbolOf("table-body");
      return sym;
    }
    case html::flow_columns: {
      static value sym = CsSymbolOf("columns");
      return sym;
    }
    case html::flow_null: {
      static value sym = CsSymbolOf("null");
      return sym;
    }
    case html::flow_image: {
      static value sym = CsSymbolOf("image");
      return sym;
    }
    case html::flow_svg: {
      static value sym = CsSymbolOf("svg");
      return sym;
    }
    case html::flow_svg_element: {
      static value sym = CsSymbolOf("svg-child");
      return sym;
    }
    default: { return UNDEFINED_VALUE; }
    }
  }

  static value CSF_get(xvm *c) {
    value obj;
    uint  stateMask = 0xffffffff;
    CsParseArguments(c, "V=*|i", &obj, c->elementInterfaces[STATE_IFACE],
                     &stateMask);

    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    uint r = uint((self->get_state(true).data & stateMask) &
                  html::STATE_ONOFF_BITMASK);
    return CsMakeInteger(int(r));
  }

  static value CSF_set(xvm *c) {
    value obj;
    uint  stateMask = 0;
    CsParseArguments(c, "V=*i", &obj, c->elementInterfaces[STATE_IFACE],
                     &stateMask);

    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    html::view *pv = self->pview();

    html::ui_state st;
    st.data = (stateMask & html::STATE_ONOFF_BITMASK);
    if (self->state.is_clear(st)) { self->set_state(st, pv); }
    return UNDEFINED_VALUE;
  }

  static value CSF_clear(xvm *c) {
    value obj;
    uint  stateMask = 0;
    CsParseArguments(c, "V=*i", &obj, c->elementInterfaces[STATE_IFACE],
                     &stateMask);

    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;
    html::view *pv = self->pview();

    html::ui_state st;
    st.data = (stateMask & html::STATE_ONOFF_BITMASK);
    if (self->state.is_set(st)) { self->reset_state(st, pv); }
    return UNDEFINED_VALUE;
  }

  static value CSF_state(xvm *c, value obj, uint64 st) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    st = st & html::STATE_ONOFF_BITMASK;

    bool r = (self->get_state(true).data & st) == st;
    return r ? TRUE_VALUE : FALSE_VALUE;
  }

  static void CSF_set_state(xvm *c, value obj, value val, uint64 st) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return;
    html::view *pv = self->pview();

    bool bval = CsToBoolean(c, val) == TRUE_VALUE;

    st = st & html::STATE_ONOFF_BITMASK;

    if (bval) {
      if (self->state.is_clear(st)) self->set_state(st, pv);
    } else {
      if (self->state.is_set(st)) self->reset_state(st, pv);
    }
  }

  static value CSF_focusable(xvm *c, value obj) {
    return CSF_state(c, obj, html::S_FOCUSABLE);
  }
  static void CSF_set_focusable(xvm *c, value obj, value val) {
    if (val == TRUE_VALUE)
      CSF_set_state(c, obj, TRUE_VALUE, html::S_WANTS_FOCUS);
    else if (val == FALSE_VALUE)
      CSF_set_state(c, obj, TRUE_VALUE, html::S_WANTS_NO_FOCUS);
    else
      CSF_set_state(c, obj, FALSE_VALUE,
                    html::S_WANTS_NO_FOCUS | html::S_WANTS_FOCUS);
  }

  static value CSF_expanded(xvm *c, value obj) {
    return CSF_state(c, obj, html::S_EXPANDED);
  }
  static void CSF_set_expanded(xvm *c, value obj, value val) {
    if (val == TRUE_VALUE)
      CSF_set_state(c, obj, TRUE_VALUE, html::S_EXPANDED);
    else if (val == FALSE_VALUE)
      CSF_set_state(c, obj, FALSE_VALUE, html::S_EXPANDED);
    else
      CSF_set_state(c, obj, FALSE_VALUE, html::S_EXPANDED | html::S_COLLAPSED);
  }
  static value CSF_collapsed(xvm *c, value obj) {
    return CSF_state(c, obj, html::S_COLLAPSED);
  }
  static void CSF_set_collapsed(xvm *c, value obj, value val) {
    if (val == TRUE_VALUE)
      CSF_set_state(c, obj, TRUE_VALUE, html::S_COLLAPSED);
    else if (val == FALSE_VALUE)
      CSF_set_state(c, obj, FALSE_VALUE, html::S_COLLAPSED);
    else
      CSF_set_state(c, obj, FALSE_VALUE, html::S_EXPANDED | html::S_COLLAPSED);
  }

  static value CSF_screen(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    html::view *pv = self->pview();
    if (!pv) return UNDEFINED_VALUE;

    html::iwindow *pw = self->get_window(*pv, true);
    if (!pw) return UNDEFINED_VALUE;

    return CsMakeInteger(screen_of(pw));
  }
  static value CSF_awaitsDraw(xvm *c, value obj) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    return self->flags.was_drawn ? FALSE_VALUE : TRUE_VALUE;
  }

  static void CSF_set_awaitsDraw(xvm *c, value obj, value val) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return;
    self->flags.was_drawn = CsToBoolean(c, val) == FALSE_VALUE ? 1 : 0;
  }

#define STATE(nn, NN)                                                          \
  static value CSF_##nn(xvm *c, value obj) {                                   \
    return CSF_state(c, obj, html::S_##NN);                                    \
  }                                                                            \
  static void CSF_set_##nn(xvm *c, value obj, value val) {                     \
    CSF_set_state(c, obj, val, html::S_##NN);                                  \
  }

  STATE(link, LINK)
  STATE(hover, HOVER)
  STATE(active, ACTIVE)
  STATE(focus, FOCUS)
  STATE(ownsfocus, OWNS_FOCUS)
  STATE(visited, VISITED)
  STATE(current, CURRENT)
  STATE(checked, CHECKED)
  STATE(selected, SELECTED)
  STATE(disabled, DISABLED)
  STATE(readonly, READONLY)
  // STATE(expanded       ,EXPANDED   )
  // STATE(collapsed      ,COLLAPSED  )
  STATE(incomplete, INCOMPLETE)
  STATE(invalid, INVALID)
  STATE(animating, ANIMATING)
  // STATE(focusable      ,FOCUSABLE  )
  STATE(anchor, ANCHOR)
  STATE(synthetic, SYNTHETIC)
  STATE(ownspopup, OWNS_POPUP)
  STATE(tabfocus, TABFOCUS)
  STATE(empty, EMPTY)
  STATE(busy, BUSY)
  STATE(dragover, DRAG_OVER)
  STATE(droptarget, DROP_TARGET)
  STATE(moving, MOVING)
  STATE(copying, COPYING)
  STATE(dragsource, DRAG_SOURCE)
  STATE(pressed, PRESSED)
  STATE(popup, POPUP)
  STATE(isltr, IS_LTR)
  STATE(isrtl, IS_RTL)
  STATE(ready, READY)
  STATE(unchecked, UNCHECKED)

  static value CSF_hoverLeft(xvm *c, value obj) { return CSF_state(c, obj, html::S_HOVER_LEFT); }
  static value CSF_hoverRight(xvm *c, value obj) { return CSF_state(c, obj, html::S_HOVER_RIGHT); }
  static value CSF_hoverTop(xvm *c, value obj) { return CSF_state(c, obj, html::S_HOVER_TOP); }
  static value CSF_hoverBottom(xvm *c, value obj) { return CSF_state(c, obj, html::S_HOVER_BOTTOM); }
  //static void CSF_set_##nn(xvm *c, value obj, value val) { CSF_set_state(c, obj, val, html::S_##NN); }


  /* element methods */
  static c_method methods[] = {
      C_METHOD_ENTRY_X("get", CSF_get), C_METHOD_ENTRY_X("set", CSF_set),
      C_METHOD_ENTRY_X("clear", CSF_clear), C_METHOD_ENTRY_X(0, 0)};

  /* String properties */
  static vp_method properties[] = {

      VP_METHOD_ENTRY_X("link", CSF_link, CSF_set_link),
      VP_METHOD_ENTRY_X("hover", CSF_hover, CSF_set_hover),
      VP_METHOD_ENTRY_X("active", CSF_active, CSF_set_active),
      VP_METHOD_ENTRY_X("focus", CSF_focus, CSF_set_focus),
      VP_METHOD_ENTRY_X("ownsfocus", CSF_ownsfocus, 0),
      VP_METHOD_ENTRY_X("visited", CSF_visited, CSF_set_visited),
      VP_METHOD_ENTRY_X("current", CSF_current, CSF_set_current),
      VP_METHOD_ENTRY_X("checked", CSF_checked, CSF_set_checked),
      VP_METHOD_ENTRY_X("unchecked", CSF_unchecked, CSF_set_unchecked),
      VP_METHOD_ENTRY_X("selected", CSF_selected, CSF_set_selected),
      VP_METHOD_ENTRY_X("disabled", CSF_disabled, CSF_set_disabled),
      VP_METHOD_ENTRY_X("readonly", CSF_readonly, CSF_set_readonly),
      VP_METHOD_ENTRY_X("expanded", CSF_expanded, CSF_set_expanded),
      VP_METHOD_ENTRY_X("collapsed", CSF_collapsed, CSF_set_collapsed),
      VP_METHOD_ENTRY_X("incomplete", CSF_incomplete, CSF_set_incomplete),
      VP_METHOD_ENTRY_X("invalid", CSF_invalid, CSF_set_invalid),
      VP_METHOD_ENTRY_X("animating", CSF_animating, 0),
      VP_METHOD_ENTRY_X("focusable", CSF_focusable, CSF_set_focusable),
      VP_METHOD_ENTRY_X("anchor", CSF_anchor, CSF_set_anchor),
      VP_METHOD_ENTRY_X("synthetic", CSF_synthetic, CSF_set_synthetic),
      VP_METHOD_ENTRY_X("ownspopup", CSF_ownspopup, 0),
      VP_METHOD_ENTRY_X("tabfocus", CSF_tabfocus, CSF_set_tabfocus),
      VP_METHOD_ENTRY_X("empty", CSF_empty, 0),
      VP_METHOD_ENTRY_X("busy", CSF_busy, CSF_set_busy),
      VP_METHOD_ENTRY_X("dragover", CSF_dragover, CSF_set_dragover),
      VP_METHOD_ENTRY_X("droptarget", CSF_droptarget, CSF_set_droptarget),
      VP_METHOD_ENTRY_X("moving", CSF_moving, CSF_set_moving),
      VP_METHOD_ENTRY_X("copying", CSF_copying, CSF_set_copying),
      VP_METHOD_ENTRY_X("dragsource", CSF_dragsource, CSF_set_dragsource),
      VP_METHOD_ENTRY_X("pressed", CSF_pressed, CSF_set_pressed),
      VP_METHOD_ENTRY_X("popup", CSF_popup, CSF_set_popup),
      VP_METHOD_ENTRY_X("ready", CSF_ready, CSF_set_ready),
      VP_METHOD_ENTRY_X("isltr", CSF_isltr, 0),
      VP_METHOD_ENTRY_X("isrtl", CSF_isrtl, 0),

      VP_METHOD_ENTRY_X("hoverLeft", CSF_hoverLeft, 0),
      VP_METHOD_ENTRY_X("hoverRight", CSF_hoverRight, 0),
      VP_METHOD_ENTRY_X("hoverTop", CSF_hoverTop, 0),
      VP_METHOD_ENTRY_X("hoverBottom", CSF_hoverBottom, 0),

      VP_METHOD_ENTRY_X("value", CSF_value, CSF_set_value),
      VP_METHOD_ENTRY_X("screen", CSF_screen, 0),
      VP_METHOD_ENTRY_X("awaitsDraw", CSF_awaitsDraw, CSF_set_awaitsDraw),
      VP_METHOD_ENTRY_X("delayedLayout", CSF_delayedLayout, CSF_set_delayedLayout),
      VP_METHOD_ENTRY_X("allowReconciliation", CSF_allowReconciliation, CSF_set_allowReconciliation),
            
      VP_METHOD_ENTRY_X("flowType", CSF_flowType, 0),

      VP_METHOD_ENTRY_X(0, 0, 0)};

  void destroy_state(xvm *c, value obj) { ; }

  extern long ElementSize(value obj);
  extern void ElementScan(VM *c, value obj);
  // extern value ElementCopy(VM *c,value obj);
  extern int_t ElementHash(value obj);

  /* GetElementProperty - mimics GetObjectProperty */
  static bool GetProperty(xvm *c, value &obj, value tag, value *pValue) {
    value     p;
    dispatch *d = c->elementInterfaces[STATE_IFACE];
    if ((p = CsFindProperty(c, d->obj, tag, nullptr, nullptr)) != NIL_VALUE) {
      value propValue = CsPropertyValue(p);
      if (CsVPMethodP(propValue)) {
        vp_method *method = ptr<vp_method>(propValue);
        if (method->get(c, obj, *pValue))
          return true;
        else
          CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
      } else {
        *pValue = propValue;
        return true;
      }
    }
    return false;
  }

  /* SetProperty - mimics SetObjectProperty  */
  static bool SetProperty(xvm *c, value obj, value tag, value val) {
    value     p;
    dispatch *d = c->elementInterfaces[STATE_IFACE];
    if ((p = CsFindProperty(c, d->obj, tag, nullptr, nullptr)) != NIL_VALUE) {
      value propValue = CsPropertyValue(p);
      if (CsVPMethodP(propValue)) {
        vp_method *method = ptr<vp_method>(propValue);
        return method->set(c, tag, obj, val);
      }
    }
    return false;
  }

  static value GetItem(xvm *c, value obj, value tag) {
    html::element *self = state_element_ptr(c, obj);
    if (!self) return UNDEFINED_VALUE;

    html::ui_state st;

    if (CsStringP(tag)) {
      string s = value_to_string(tag);
      if (!html::parse_state_flag(s, st)) goto ERR;
    } else if (CsSymbolP(tag)) {
      tool::string s = CsSymbolName(tag);
      if (!html::parse_state_flag(s, st)) goto ERR;
    } else if (CsIntegerP(tag)) {
      st.data = CsIntegerValue(tag);
    } else {
    ERR:
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag,
                        "valid state identifier");
    }
    return (self->get_state(true).data & st.data) != 0 ? TRUE_VALUE
                                                       : FALSE_VALUE;
  }
  static void SetItem(xvm *c, value obj, value tag, value value) {

    html::element *self = state_element_ptr(c, obj);
    if (!self) return;

    bool           v = CsToBoolean(c, value) == TRUE_VALUE;
    html::ui_state st;

    if (CsStringP(tag)) {
      string s = value_to_string(tag);
      if (!html::parse_state_flag(s, st)) goto ERR;
    } else if (CsSymbolP(tag)) {
      string s = CsSymbolName(tag);
      if (!html::parse_state_flag(s, st)) goto ERR;
    } else if (CsIntegerP(tag)) {
      st.data = CsIntegerValue(tag);
    } else {
    ERR:
      CsThrowKnownError(c, CsErrUnexpectedTypeError, tag,
                        "bad state identifier");
    }
    html::view *pv = self->pview();
    if (v)
      self->set_state(st, pv);
    else
      self->reset_state(st, pv);
  }

  void xvm::init_element_state_class() {
    dispatch *pd = CsEnterCPtrObjectType(CsGlobalScope(this), "States", methods, properties);

    /* create the 'State' type */
    if (!pd) CsInsufficientMemory(this);

    /* setup alternate handlers */

    pd->getProperty = (get_property_t)GetProperty;
    pd->setProperty = (set_property_t)SetProperty;

    pd->scan = ElementScan;
    // pd->size = ElementSize;
    // pd->copy = ElementCopy;
    pd->hash = ElementHash;

    pd->getItem = (get_item_t)GetItem;
    pd->setItem = (set_item_t)SetItem;

    pd->baseType = &CsCObjectDispatch;
    // pd->getNextElement = (get_next_element_t)GetNextElement;

    pd->destroy = (destructor_t)destroy_state;

    elementInterfaces[STATE_IFACE] = pd;
  }

} // namespace tis
