#include "win.h"
#include "win-layered.h"
#include "win-accessible.h"
#include "win-application.h"
#include "win-delayload.h"

void screen2client(HWND hwnd, gool::point &pt, bool standard_frame);
void client2screen(HWND hwnd, gool::point &pt, bool standard_frame);

namespace html {

  void iwindow::set_enabled(bool on_off) 
  {
    ::EnableWindow(get_hwnd(), BOOL(on_off));
  }
  bool iwindow::is_enabled() const { 
    return ::IsWindowEnabled(get_hwnd()) != FALSE; 
  }

  point iwindow::screen_pos() {
    rect r;
    ::GetWindowRect(get_hwnd(), PRECT(r));
    return r.s;
  }

  size iwindow::client_dim() {
    rect r;
    ::GetClientRect(get_hwnd(), PRECT(r));
    return r.size();
  }

  size iwindow::window_dim() {
    rect r;
    ::GetWindowRect(get_hwnd(), PRECT(r));
    return r.size();
  }

  point iwindow::client_screen_pos() {
    point pt(0, 0);
    client2screen(get_hwnd(), pt, get_frame_type() == STANDARD);
    return pt;
  }

  point iwindow::cursor_pos() {
    point pt;
    ::GetCursorPos(PPOINT(pt));
    screen2client(get_hwnd(), pt, get_frame_type() == STANDARD);
    return pt;
  }

  void iwindow::invalidate(const rect &area) {
    rect r = area;
    ::InvalidateRect(get_hwnd(), PRECT(r), FALSE);
  }
  void iwindow::update() { ::UpdateWindow(get_hwnd()); }

  void iwindow::update_window_frame() {
    RECT rc;
    ::GetWindowRect(get_hwnd(), &rc);
    ::SetWindowPos(get_hwnd(), NULL, rc.left, rc.top, rc.right - rc.left, rc.bottom - rc.top,
      SWP_FRAMECHANGED | SWP_NOACTIVATE | SWP_NOOWNERZORDER | SWP_NOZORDER);
  }

  bool iwindow::show(uint animate_window_flags, uint ms) {
    return ::AnimateWindow(get_hwnd(), ms, animate_window_flags) != FALSE;
  }

  bool iwindow::is_window_visible() {
    if (::IsWindowVisible(get_hwnd()) == FALSE) return false;
    WINDOWPLACEMENT wp;
    wp.length = sizeof(WINDOWPLACEMENT);
    GetWindowPlacement(get_hwnd(), &wp);
    if (wp.showCmd == SW_SHOWMINIMIZED || wp.showCmd == SW_MINIMIZE)
      return false;
    return true;
  }


} // namespace html

namespace mswin {
  popup *popup::ptr(HWND hwnd) {
    LONG_PTR lp = ::GetWindowLongPtr(hwnd, GWLP_USERDATA);
    return reinterpret_cast<popup *>(lp);
  }

  void popup::init(bool startup) {
    if (!startup) return;
    static ATOM wcls             = 0;
    static ATOM wcls_transparent = 0;
    if (wcls) return;

    WNDCLASS wcx = {0};
    wcx.style    = CS_VREDRAW | CS_HREDRAW | CS_DBLCLKS;

    if (environment::get_os_version() >= environment::WIN_XP)
      wcx.style |= CS_DROPSHADOW;

    wcx.lpfnWndProc   = proc;                // points to window procedure
    wcx.cbClsExtra    = 0;                   // no extra class memory
    wcx.cbWndExtra    = 0;                   // no extra window memory
    wcx.hInstance     = HINST_THISCOMPONENT; // handle to instance
    wcx.hIcon         = 0;                   // predefined app. icon
    wcx.hCursor       = LoadCursor(NULL, IDC_ARROW);         // predefined arrow
    wcx.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH); //
    wcx.lpszMenuName  = 0;                // name of menu resource
    wcx.lpszClassName = POPUP_CLASS_NAME; // name of window class

    wcls = RegisterClass(&wcx);
    assert(wcls);

    wcx.lpszClassName = POPUP_CLASS_NAME_TRANSPARENT; // name of window class
    wcx.style         = CS_OWNDC | CS_VREDRAW | CS_HREDRAW | CS_DBLCLKS;
    wcx.hbrBackground = (HBRUSH)GetStockObject(NULL_BRUSH);

    wcls_transparent = RegisterClass(&wcx);
    assert(wcls_transparent);

    // if( !SciterPopupClass )
    //  ::MessageBox(NULL,TEXT("Unable to register window
    //  class"),TEXT("Error"),MB_OK);
  }

  popup::~popup() { _hwnd = _hwnd; }

  HWND hwnd_foreground = 0;

  iwindow * window::create_window(element *b, element *anchor, WINDOW_TYPE wt,
                        function<rect(view &, element *, element *)> place,
                        ELEMENT_WINDOW_MODE                          mode) {
    hstyle cs             = b->get_style(*this);

    bool   create_layered = cs->is_transparent() || cs->has_rounded_corners() || cs->has_outline() || cs->box_shadow;
    DWORD  dwStyle        = WS_POPUP;
    DWORD  dwExStyle      = WS_EX_NOACTIVATE;

    if (is_direct())
      dwExStyle |= WS_EX_NOREDIRECTIONBITMAP;

    //(create_layered? WS_EX_LAYERED : 0) | - note this is moved to
    //puw->set_layered(true); below
    ////WS_EX_TOOLWINDOW | // it doesn't show up in taskbar
    ////WS_EX_TOPMOST |    // WRONG: it must *not* be topmost, otherwise it will
    ///be on top of all windows on screen.
    // WS_EX_NOACTIVATE;  // it should be non active from the very beginning

    if (wt == POPUP_WINDOW /*|| wt == TOOLTIP_WINDOW*/ || mode == ELEMENT_WINDOW_DETACHED_TOPMOST)
      dwExStyle |= WS_EX_TOPMOST;

    // if (wt == TOOLTIP_WINDOW)
    //  wt = TOOLTIP_WINDOW;

    HWND parent_hwnd = get_hwnd();

    if (anchor) {
      iwindow *parent_window = anchor->get_window(*this, true);
      if (parent_window) parent_hwnd = parent_window->get_hwnd();
    }

    //  Now find a proper home for the window that won't go off
    //  the screen or straddle two monitors.
    LPCTSTR class_name =
        create_layered ? POPUP_CLASS_NAME_TRANSPARENT : POPUP_CLASS_NAME;

    hwnd_foreground = GetForegroundWindow();

    HWND hwnd = CreateWindowEx(dwExStyle, class_name, L"", dwStyle, 0, 0, 0, 0,
                               parent_hwnd, NULL,
                               // HINST_THISCOMPONENT,
                               mswin::app()->h_instance(), NULL);
    
    if (!hwnd) {
      view::debug_printf(html::OT_DOM, html::OS_ERROR,
                         "unable to create popup window.\n");
      //::MessageBox(NULL,TEXT("Unable to create popup
      //window"),TEXT("Error"),MB_OK);
      return 0;
    }

    SetTimer(hwnd, 0xAF, 20, NULL); // poll timer

    popup *puw = create_popup();
    if (!puw) return nullptr;
    puw->add_ref();

    puw->set_hwnd(hwnd);

    puw->windowing_mode = mode;

    puw->root(b);
    puw->anchor(anchor);
    puw->type    = wt;

    windows.push(puw);

    rect wrc = place(*this, b, anchor);

    puw->_dim    = wrc.size();
    puw->invalid = rect(puw->_dim);

    if (create_layered) 
      puw->set_layered(true);

    wrc = b->outline_box(*this, element::TO_SCREEN);

#ifdef DEBUG
    dbg_printf("create popup %d %d %d %d\n", wrc.left(), wrc.top(), wrc.width(),wrc.height());
#endif
    ::MoveWindow(hwnd, wrc.left(), wrc.top(), wrc.width(), wrc.height(), FALSE);

    ::SetWindowLongPtr(hwnd, GWLP_USERDATA, LONG_PTR(puw));
    setup_hook();

    ::ShowWindow(hwnd, SW_SHOWNOACTIVATE); // must be here (FCC)
    if (create_layered)
      ::InvalidateRect(hwnd, NULL, TRUE);
    ::UpdateWindow(hwnd);
    return puw;
  }

  static bool inside_system_menu_loop = false;

  LRESULT config_GESTURE(window *pw, WPARAM wParam, LPARAM lParam, HWND targetHwnd, BOOL &handled);
  LRESULT handle_GESTURE(window *pw, WPARAM wParam, LPARAM lParam, HWND targetHwnd, BOOL &handled);

  LRESULT CALLBACK popup::proc(HWND hwnd, UINT msg, WPARAM wParam,
                               LPARAM lParam) {
    // dbg_printf("msg=%x\n",msg);
    handle<popup> self = ptr(hwnd);
    if (!self || !self->_root) return DefWindowProc(hwnd, msg, wParam, lParam);

    switch (msg) {
    case WM_MOUSEACTIVATE: return MA_NOACTIVATE;

    case WM_ACTIVATE: return 0;

    case WM_ENTERMENULOOP: inside_system_menu_loop = true; return 0;

    case WM_EXITMENULOOP: inside_system_menu_loop = false; return 0;

    case WM_TIMER: {
        HWND hf = GetForegroundWindow();
        if ((wParam == 0xAF) && (hwnd_foreground != hf))
        {
          html::element *b = self->root();
          html::view *pv;
          if(!b)
            goto CLOSE_ALL;
          pv = b->pview();
          if(!pv || (hf != pv->get_hwnd()) )
            goto CLOSE_ALL; // some other window was activated
        }
      }
      break;
    
    case WM_PAINT: {
      // auto_state<HWND> _(drawing_hwnd,hwnd);
      self->do_paint();
      // dbg_printf("popup::WM_PAINT\n");
    }
      return 0;
    case WM_PRINTCLIENT: {
      HDC  hdc = (HDC)wParam;
      RECT rc;
      GetClientRect(hwnd, &rc);
      self->print(hdc, fromRECT(rc));
    }
      return 0;

    case WM_ERASEBKGND: return TRUE;

    case WM_SIZE: {
      size sz;
      sz.x       = LOWORD(lParam);
      sz.y       = HIWORD(lParam);
      self->_dim = sz;
      self->refresh();
      // dbg_printf("popup::WM_SIZE\n");
      self->update();
    } break;

    // case WM_WINDOWPOSCHANGING:
    //  dbg_printf("popup::WM_WINDOWPOSCHANGING\n");
    //  break;

    // case WM_NCPAINT:
    //  dbg_printf("popup::WM_NCPAINT\n");
    //  break;

#if defined(ACCESSIBLE)
    case WM_GETOBJECT: {
      html::element *pb = self->_root;
      if (pb) {
        LONG what = (LONG)(DWORD)lParam;
#ifdef USE_UIAUTOMATION
        if (what == UiaRootObjectId && _UiaReturnRawElementProvider) {
          IRawElementProviderSimple *provider =
              html::accessible_root_provider::make(hwnd, pb);
          LRESULT lr =
              _UiaReturnRawElementProvider()(hwnd, wParam, lParam, provider);
          provider->Release();
          return lr;
        }
#endif
        if (what == OBJID_CLIENT) {
          html::accessible *pa = new html::accessible(pb);
          pa->is_popup         = true;
          LRESULT lr =  LresultFromObject(IID_IAccessible, wParam, (IAccessible *)pa);
          pa->Release();
          return lr;
        }
      }
    } break;
#endif

    case WM_CREATE: break;

    case WM_DESTROY:
      /*{
        RECT rc; GetWindowRect(hwnd,&rc);
        ::InvalidateRect(NULL,&rc,TRUE);
      }*/
      self->destroy();
      return 0;

    case WM_ACTIVATEAPP:
      if (wParam == FALSE) {
CLOSE_ALL:
        html::element *b = self->_root;
        if (b && !self->is_tool()) {
          html::view *pv = b->pview();
          if (pv) pv->close_popup(b, false);
        }
      }
      break;

      /*case WM_LBUTTONDOWN:
        break;
      case WM_MOUSEMOVE:
        break;
      case WM_NCHITTEST:
        break;*/

    case WM_MOUSEWHEEL: {
      // dbg_printf("WM_MOUSEWHEEL\n");
      HWND owner = GetParent(hwnd);
      return ::SendMessage(owner, msg, wParam, lParam);
    }
    case WM_SETCURSOR: {
      html::element *b = self->_root;
      if (b) {
        html::view *pv = b->pview();
        if (pv) pv->check_mouse(true);
      }
      return TRUE;
    } break;

#ifdef USE_TOUCH
    case WM_GESTURE: 
    case WM_GESTURENOTIFY: 
    {
	    html::element *b = self->_root;
	    if (b) {
		    mswin::window *pw = static_cast<mswin::window *>(b->pview());
		    BOOL handled = false;
		    if(msg == WM_GESTURENOTIFY)
		      return config_GESTURE(pw, wParam, lParam, hwnd, handled);
		    else
		      return handle_GESTURE(pw, wParam, lParam, hwnd, handled);
	    }
      // dbg_printf("WM_GESTURE\n");
      //HWND owner = GetParent(hwnd);
      //return ::SendMessage(owner, msg, wParam, lParam);
	  } break;
#endif

    default:
      if (msg == WM_GETWINDOWBLOCK) {
        html::element **ppblock = (html::element **)lParam;
        if (ppblock) *ppblock = self->_root;
        return 0xAF;
      } 
      //else if (msg == WM_UPDATELAYERED) {
      //  self->render_layered();
      //  return 0;
      //}
      break;
    }
    return ::DefWindowProc(hwnd, msg, wParam, lParam);
  }

  void destroy_window(HWND _hwnd) {
#ifdef THEMES_SUPPORT
    if (theme::current()->is_classic_theme()) {
      HWND parent = GetParent(_hwnd);
      ::DestroyWindow(_hwnd);
      // seems like a bug: W7 in Classic mode workaround, for some reason popup
      // window with CS_DROPSHADOW leaves artifacts in nonclient area.
      ::RedrawWindow(parent, 0, 0, RDW_FRAME | RDW_INVALIDATE | RDW_UPDATENOW);
    } else
#endif
      ::DestroyWindow(_hwnd);
  }

  void popup::dismiss() { destroy_window(_hwnd); }

  void popup::destroy() {
    HWND  hwnd = _hwnd;
    view *pv   = _root->pview();
    if (pv) {
      critical_section _(pv->guard);
      pv->forget_window(this);
    }
    ::SetWindowLongPtr(hwnd, GWLP_USERDATA, 0);
    _root = 0;
    // delete this;
    release();
  }

  iwindow *is_popup_window_of(window *pw, HWND hwnd) {
    for (int n = 0; n < pw->windows.size(); ++n) {
      auto tw = pw->windows[n];
      if (tw->get_hwnd() == hwnd)
        return tw->type == CHILD_WINDOW ? nullptr : tw;
    }
    return nullptr;
  }

  bool translate_mouse_message(window *pw, MSG &msg) {
    if (msg.hwnd != pw->get_hwnd()) {
      point pt;
      POINTSTOPOINT(pt, msg.lParam);
      client2screen(msg.hwnd, pt,pw->get_frame_type() == STANDARD);
      screen2client(pw->get_hwnd(), pt,pw->get_frame_type() == STANDARD);
      //MapWindowPoints(msg.hwnd, pw->get_hwnd(), &pt, 1);
      msg.lParam = POINTTOPOINTS(pt);
      msg.hwnd   = pw->get_hwnd();
      return true;
    }
    return false;
  }

  bool popup::translate_event(window *pwin, MSG &msg) {
    HWND whwnd = pwin->get_hwnd();

    handle<window> pw = pwin;

    bool is_active_process =
        ::GetWindowThreadProcessId(whwnd, NULL) == GetCurrentThreadId();
    //  return false;

    auto any_popup = pw->windows().find_if(
        [](const handle<iwindow> &iw) { return iw->is_airborn(); });

    if (!any_popup) return false;

    if (!whwnd || !::IsWindow(whwnd)) goto CLOSE_ALL_FALSE;

    //  All mouse messages are remunged and directed at our
    //  popup menu.  If the mouse message arrives as client
    //  coordinates, then we have to convert it from the
    //  client coordinates of the original target to the
    //  client coordinates of the new target.

    switch (msg.message) {
    // Windows 10 requires this for WM_MOUSEWHEEL
    case WM_MOUSEWHEEL: {
      handle<iwindow> pup = is_popup_window_of(pw, msg.hwnd);
      if (pup) {
        msg.hwnd = whwnd;
        return true;
      }
    } break;

      //  These mouse messages arrive in client coordinates,
      //  so in addition to stealing the message, we also
      //  need to convert the coordinates.

    case WM_LBUTTONDOWN:
    case WM_MBUTTONDOWN:
    case WM_RBUTTONDOWN:
    case WM_XBUTTONDOWN: {
      // dbg_printf("doPopupEvent: WM_BUTTONDOWN\n");
      // dispatch mouse messages
      point pt; POINTSTOPOINT(pt, msg.lParam);
      //pt.x = (short)LOWORD(msg.lParam);
      //pt.y = (short)HIWORD(msg.lParam);
      client2screen(msg.hwnd, pt, pw->get_frame_type() == STANDARD);
      HWND            target = WindowFromPoint(*PPOINT(pt));
      handle<iwindow> pup    = is_popup_window_of(pw, target);
      if (!pup)
        // hit on non popup window, could be another our window in dialog
        // if(popup_hview && GetCapture() == popup_hview)
        //   ReleaseCapture();
        goto CLOSE_ALL_FALSE;

      if (pup->type == TOOL_WINDOW && (!is_active_process || !pw->is_active()))
        SetForegroundWindow(pw->get_hwnd());

      pw->close_owned_popups(pup->root());

      if ((pup->type == TOOL_WINDOW || pup->type == POPUP_WINDOW) &&
          (pup->windowing_mode >= ELEMENT_WINDOW_DETACHED)) {
        pw->windows.remove_by_value(pup);
        pw->windows.push(pup);
        SetWindowPos(pup->get_hwnd(), HWND_TOP, 0, 0, 0, 0,
                     SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOOWNERZORDER |
                         SWP_NOSIZE);
      }

      return translate_mouse_message(pw, msg);
    } break;
#ifndef PLATFORM_WINCE
    case WM_NCLBUTTONDOWN:
    case WM_NCMBUTTONDOWN:
    case WM_NCRBUTTONDOWN:
    case WM_NCXBUTTONDOWN: {
      // dbg_printf("doPopupEvent: WM_NCBUTTONDOWN\n");
      // dispatch mouse messages
      POINT pt; POINTSTOPOINT(pt, msg.lParam);
      //pt.x        = (short)LOWORD(msg.lParam);
      //pt.y        = (short)HIWORD(msg.lParam);
      HWND target = WindowFromPoint(pt);
      if (!is_popup_window_of(pw, target)) {
        // hit on non popup window, could be another our window in dialog
        goto CLOSE_ALL_FALSE;
      }
    } break;
#endif
    case WM_LBUTTONUP:
    case WM_MBUTTONUP:
    case WM_RBUTTONUP:
    case WM_XBUTTONUP:
    case WM_MOUSEMOVE: {
      // dbg_printf("MOUSE_MOVE\n");
      //POINT pt;
      //pt.x = (short)LOWORD(msg.lParam);
      //pt.y = (short)HIWORD(msg.lParam);
      // ClientToScreen(msg.hwnd, &pt);
      // HWND target = WindowFromPoint(pt);
      if (is_popup_window_of(pw, msg.hwnd))
        return translate_mouse_message(pw, msg);
    } break;
    case WM_LBUTTONDBLCLK:
    case WM_MBUTTONDBLCLK:
    case WM_RBUTTONDBLCLK: {
      // dbg_printf("MOUSE_MOVE\n");
      //POINT pt;
      //pt.x = (short)LOWORD(msg.lParam);
      //pt.y = (short)HIWORD(msg.lParam);
      // ClientToScreen(msg.hwnd, &pt);
      // HWND target = WindowFromPoint(pt);
      if (is_popup_window_of(pw, msg.hwnd))
        return translate_mouse_message(pw, msg);
    } break;
    default: return false;
    }
    return false;
  CLOSE_ALL_FALSE:
    // WRONG: pw->post_close_all_popups(); - MOUSE_DOWN may cause popup to
    // appear and posted processing will kill it.
    //       Check sdk\samples\ideas\tray-notifications
    pw->close_all_popups();
    return false;
  }

  void popup::refresh(const rect &area) {
    invalid |= area;
    rect r = area;
    ::InvalidateRect(_hwnd, PRECT(r), FALSE);
  }
  void popup::update() {
    //if (is_layered()) {
    //  render_layered();
    //} else
    ::UpdateWindow(_hwnd);
  }

  bool window::close_popup(element *b, bool set_auto_focus) {
    if (inside_system_menu_loop) return false;
    handle<iwindow> hw = b->window(*this);
    if (!hw) return false;
    HWND hpopup = hw->get_hwnd();
    super::close_popup(b, set_auto_focus);
    if (hpopup && ::IsWindow(hpopup)) {
      // HWND p = ::GetParent(hpopup);
      //::DestroyWindow(hpopup);
      destroy_window(hpopup);
      check_mouse(true);
      // update(); -- many things happen after
      // if(::IsWindow(p))
      //  ::UpdateWindow(p);
      return true;
    }
    return false;
  }

} // namespace mswin
