
#include "win.h"

// Copyright (c) 2006-2008 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
#include "win-ime.h"
#include "win-delayload.h"

#define USE_IME

// "imm32.lib" is required by IMM32 APIs used in this file.
#if defined(USE_IME)
#pragma comment(lib, "imm32.lib")
#endif

using namespace tool;

///////////////////////////////////////////////////////////////////////////////
// ImeInput

ImeInput::ImeInput()
    : ime_status_(false), input_language_id_(LANG_USER_DEFAULT),
      is_composing_(false), system_caret_(false) {}

ImeInput::~ImeInput() {}

bool ImeInput::SetInputLanguage() {
  // Retrieve the current keyboard layout from Windows and determine whether
  // or not the current input context has IMEs.
  // Also save its input language for language-specific operations required
  // while composing a text.
  HKL keyboard_layout = ::GetKeyboardLayout(0);
  input_language_id_  = LOWORD(keyboard_layout);
#if defined(USE_IME)
  ime_status_ = (::ImmIsIME(keyboard_layout) == TRUE) ? true : false;
#else
  ime_status_ = false;
#endif
  return ime_status_;
}

void ImeInput::CreateImeWindow(HWND window_handle) {
  // When a user disables TSF (Text Service Framework) and CUAS (Cicero
  // Unaware Application Support), Chinese IMEs somehow ignore function calls
  // to ::ImmSetCandidateWindow(), i.e. they do not move their candidate
  // window to the position given as its parameters, and use the position
  // of the current system caret instead, i.e. it uses ::GetCaretPos() to
  // retrieve the position of their IME candidate window.
  // Therefore, we create a temporary system caret for Chinese IMEs and use
  // it during this input context.
  // Since some third-party Japanese IME also uses ::GetCaretPos() to determine
  // their window position, we also create a caret for Japanese IMEs.
  if (PRIMARYLANGID(input_language_id_) == LANG_CHINESE ||
      PRIMARYLANGID(input_language_id_) == LANG_JAPANESE) {
    if (!system_caret_) {
      if (::CreateCaret(window_handle, NULL, 1, 1)) { system_caret_ = true; }
    }
  }
  // Restore the positions of the IME windows.
  UpdateImeWindow(window_handle);
}

void ImeInput::SetImeWindowStyle(HWND window_handle, UINT message,
                                 WPARAM wparam, LPARAM lparam, BOOL *handled) {
  // To prevent the IMM (Input Method Manager) from displaying the IME
  // composition window, Update the styles of the IME windows and EXPLICITLY
  // call ::DefWindowProc() here.
  // NOTE(hbono): We can NEVER let WTL call ::DefWindowProc() when we update
  // the styles of IME windows because the 'lparam' variable is a local one
  // and all its updates disappear in returning from this function, i.e. WTL
  // does not call ::DefWindowProc() with our updated 'lparam' value but call
  // the function with its original value and over-writes our window styles.
  *handled = TRUE;
  lparam &= ~ISC_SHOWUICOMPOSITIONWINDOW;
  ::DefWindowProc(window_handle, message, wparam, lparam);
}

void ImeInput::DestroyImeWindow(HWND window_handle) {
  // Destroy the system caret if we have created for this IME input context.
  if (system_caret_) {
    ::DestroyCaret();
    system_caret_ = false;
  }
}

void ImeInput::MoveImeWindow(HWND window_handle, HIMC imm_context) {

#if defined(USE_IME)
  int       x            = caret_rect_.s.x;
  int       y            = caret_rect_.s.y;
  const int kCaretMargin = 1;
  // As written in a comment in ImeInput::CreateImeWindow(),
  // Chinese IMEs ignore function calls to ::ImmSetCandidateWindow()
  // when a user disables TSF (Text Service Framework) and CUAS (Cicero
  // Unaware Application Support).
  // On the other hand, when a user enables TSF and CUAS, Chinese IMEs
  // ignore the position of the current system caret and uses the
  // parameters given to ::ImmSetCandidateWindow() with its 'dwStyle'
  // parameter CFS_CANDIDATEPOS.
  // Therefore, we do not only call ::ImmSetCandidateWindow() but also
  // set the positions of the temporary system caret if it exists.
  CANDIDATEFORM candidate_position = {
      0, CFS_CANDIDATEPOS, {x, y}, {0, 0, 0, 0}};
  ::ImmSetCandidateWindow(imm_context, &candidate_position);
  if (system_caret_) {
    switch (PRIMARYLANGID(input_language_id_)) {
    case LANG_JAPANESE: ::SetCaretPos(x, y + caret_rect_.height()); break;
    default: ::SetCaretPos(x, y); break;
    }
  }
  if (PRIMARYLANGID(input_language_id_) == LANG_KOREAN) {
    // Chinese IMEs and Japanese IMEs require the upper-left corner of
    // the caret to move the position of their candidate windows.
    // On the other hand, Korean IMEs require the lower-left corner of the
    // caret to move their candidate windows.
    y += kCaretMargin;
  }
  // Japanese IMEs and Korean IMEs also use the rectangle given to
  // ::ImmSetCandidateWindow() with its 'dwStyle' parameter CFS_EXCLUDE
  // to move their candidate windows when a user disables TSF and CUAS.
  // Therefore, we also set this parameter here.
  CANDIDATEFORM exclude_rectangle = {
      0,
      CFS_EXCLUDE,
      {x, y},
      {x, y, x + caret_rect_.width(), y + caret_rect_.height()}};
  ::ImmSetCandidateWindow(imm_context, &exclude_rectangle);
#endif
}

void ImeInput::UpdateImeWindow(HWND window_handle) {
#if defined(USE_IME)
  // Just move the IME window attached to the given window.
  if (!caret_rect_.empty()) {
    HIMC imm_context = ::ImmGetContext(window_handle);
    if (imm_context) {
      MoveImeWindow(window_handle, imm_context);
      ::ImmReleaseContext(window_handle, imm_context);
    }
  }
#endif
}

void ImeInput::CleanupComposition(HWND window_handle) {
#if defined(USE_IME)
  // Notify the IMM attached to the given window to complete the ongoing
  // composition, (this case happens when the given window is de-activated
  // while composing a text and re-activated), and reset the omposition status.
  if (is_composing_) {
    HIMC imm_context = ::ImmGetContext(window_handle);
    if (imm_context) {
      ::ImmNotifyIME(imm_context, NI_COMPOSITIONSTR, CPS_COMPLETE, 0);
      ::ImmReleaseContext(window_handle, imm_context);
    }
    ResetComposition(window_handle);
  }
#endif
}

void ImeInput::ResetComposition(HWND window_handle) {
  // Currently, just reset the composition status.
  is_composing_ = false;
}

void ImeInput::CompleteComposition(HWND window_handle, HIMC imm_context) {
#if defined(USE_IME)
  // We have to confirm there is an ongoing composition before completing it.
  // This is for preventing some IMEs from getting confused while completing an
  // ongoing composition even if they do not have any ongoing compositions.)
  if (is_composing_) {
    ::ImmNotifyIME(imm_context, NI_COMPOSITIONSTR, CPS_COMPLETE, 0);
    ResetComposition(window_handle);
  }
#endif
}

void ImeInput::GetCaret(HIMC imm_context, LPARAM lparam,
                        ImeComposition *composition) {

#if defined(USE_IME)
  // This operation is optional and language-dependent because the caret
  // style is depended on the language, e.g.:
  //   * Korean IMEs: the caret is a blinking block,
  //     (It contains only one hangul character);
  //   * Chinese IMEs: the caret is a blinking line,
  //     (i.e. they do not need to retrieve the target selection);
  //   * Japanese IMEs: the caret is a selection (or underlined) block,
  //     (which can contain one or more Japanese characters).
  int target_start = -1;
  int target_end   = -1;
  switch (PRIMARYLANGID(input_language_id_)) {
  case LANG_KOREAN:
    if (lparam & CS_NOMOVECARET) {
      target_start = 0;
      target_end   = 1;
    }
    break;
  case LANG_CHINESE: break;
  case LANG_JAPANESE:
    // For Japanese IMEs, the robustest way to retrieve the caret
    // is scanning the attribute of the latest composition string and
    // retrieving the begining and the end of the target clause, i.e.
    // a clause being converted.
    if (lparam & GCS_COMPATTR) {
      int attribute_size =
          ::ImmGetCompositionStringW(imm_context, GCS_COMPATTR, NULL, 0);
      if (attribute_size > 0) {
        array<char> attribute_data(attribute_size);
        if (attribute_data.head()) {
          ::ImmGetCompositionStringW(imm_context, GCS_COMPATTR,
                                     attribute_data.head(),
                                     attribute_data.size());
          for (target_start = 0; target_start < attribute_size;
               ++target_start) {
            if (IsTargetAttribute(attribute_data[target_start])) break;
          }
          for (target_end = target_start; target_end < attribute_size;
               ++target_end) {
            if (!IsTargetAttribute(attribute_data[target_end])) break;
          }
          if (target_start == attribute_size) {
            // This composition clause does not contain any target clauses,
            // i.e. this clauses is an input clause.
            // We treat whole this clause as a target clause.
            target_end   = target_start;
            target_start = 0;
          }
        }
      }
    }
    break;
  }
  composition->target_start = target_start;
  composition->target_end   = target_end;
#endif
}

bool ImeInput::GetString(HIMC imm_context, WPARAM lparam, int type,
                         ImeComposition *composition) {
#if defined(USE_IME)
  bool result = false;
  if (lparam & type) {
    int string_size = ::ImmGetCompositionStringW(imm_context, type, NULL, 0);
    if (string_size > 0) {
      int          string_length = string_size / sizeof(wchar);
      array<wchar> arr(string_length + 1);
      arr.size(string_length);
      // Fill the given ImeComposition object.
      ::ImmGetCompositionStringW(imm_context, type, arr.head(), string_size);
      composition->ime_string  = arr();
      composition->string_type = type;
      result                   = true;
    }
  }
  return result;
#else
  return false;
#endif
}

bool ImeInput::GetResult(HWND window_handle, LPARAM lparam,
                         ImeComposition *composition) {
#if defined(USE_IME)
  bool result      = false;
  HIMC imm_context = ::ImmGetContext(window_handle);
  if (imm_context) {
    // Copy the result string to the ImeComposition object.
    result = GetString(imm_context, lparam, GCS_RESULTSTR, composition);
    // Reset all the other parameters because a result string does not
    // have composition attributes.
    composition->cursor_position = -1;
    composition->target_start    = -1;
    composition->target_end      = -1;
    ::ImmReleaseContext(window_handle, imm_context);
  }
  return result;
#else
  return false;
#endif
}

bool ImeInput::GetComposition(HWND window_handle, LPARAM lparam,
                              ImeComposition *composition) {
  bool result = false;
#if defined(USE_IME)
  HIMC imm_context = ::ImmGetContext(window_handle);
  if (imm_context) {
    // Copy the composition string to the ImeComposition object.
    result = GetString(imm_context, lparam, GCS_COMPSTR, composition);

    // Retrieve the cursor position in the IME composition.
    int cursor_position =
        ::ImmGetCompositionStringW(imm_context, GCS_CURSORPOS, NULL, 0);
    composition->cursor_position = cursor_position;
    composition->target_start    = -1;
    composition->target_end      = -1;

    // Retrieve the target selection and Update the ImeComposition
    // object.
    GetCaret(imm_context, lparam, composition);

    // Mark that there is an ongoing composition.
    is_composing_ = true;

    ::ImmReleaseContext(window_handle, imm_context);
  }
#endif
  return result;
}

void ImeInput::DisableIME(HWND window_handle) {
#if defined(USE_IME)
  // A renderer process have moved its input focus to a password input
  // when there is an ongoing composition, e.g. a user has clicked a
  // mouse button and selected a password input while composing a text.
  // For this case, we have to complete the ongoing composition and
  // clean up the resources attached to this object BEFORE DISABLING THE IME.
  CleanupComposition(window_handle);
  ::ImmAssociateContextEx(window_handle, NULL, 0);
#endif
}

void ImeInput::EnableIME(HWND window_handle, const gool::rect &caret_rect,
                         bool complete) {
#if defined(USE_IME)
  // Load the default IME context.
  // NOTE(hbono)
  //   IMM ignores this call if the IME context is loaded. Therefore, we do
  //   not have to check whether or not the IME context is loaded.
  ::ImmAssociateContextEx(window_handle, NULL, IACE_DEFAULT);
  // Complete the ongoing composition and move the IME windows.
  HIMC imm_context = ::ImmGetContext(window_handle);
  if (imm_context) {
    if (complete) {
      // A renderer process have moved its input focus to another edit
      // control when there is an ongoing composition, e.g. a user has
      // clicked a mouse button and selected another edit control while
      // composing a text.
      // For this case, we have to complete the ongoing composition and
      // hide the IME windows BEFORE MOVING THEM.
      CompleteComposition(window_handle, imm_context);
    }
    // Save the caret position, and Update the position of the IME window.
    // This update is used for moving an IME window when a renderer process
    // resize/moves the input caret.
    if (!caret_rect.empty()) {
      caret_rect_ = caret_rect;
      MoveImeWindow(window_handle, imm_context);
    }
    ::ImmReleaseContext(window_handle, imm_context);
  }
#endif
}

void ImeInput::SetCaretRect(HWND window_handle, gool::rect caret_rect) {
#if defined(USE_IME)
  // Load the default IME context.
  //   IMM ignores this call if the IME context is loaded. Therefore, we do
  //   not have to check whether or not the IME context is loaded.
  ::ImmAssociateContextEx(window_handle, NULL, IACE_DEFAULT);
  // move the IME windows.
  HIMC imm_context = ::ImmGetContext(window_handle);
  if (imm_context && !caret_rect.empty()) {
    caret_rect_ = caret_rect;
    MoveImeWindow(window_handle, imm_context);
  }
  ::ImmReleaseContext(window_handle, imm_context);
#endif
}

#if defined(CAN_WORK_AS_TABLET)

  // VIRTUAL KEYBOARD support on touch devices

#include <aclapi.h>
#include <cfgmgr32.h>
#include <lm.h>
#include <powrprof.h>
#include <shellapi.h>
#include <shlobj.h>
#include <shobjidl.h> // Must be before propkey.
#include <initguid.h>
#include <propkey.h>
#include <propvarutil.h>
#include <sddl.h>
#include <setupapi.h>

DLOADV(_PowerDeterminePlatformRole, PowerDeterminePlatformRole, PowrProf.dll,
       POWER_PLATFORM_ROLE(WINAPI *)(void));

DLOADV(_SetupDiGetClassDevs, SetupDiGetClassDevsW, SetupAPI.dll,
       HDEVINFO(WINAPI *)(const GUID *ClassGuid, PCWSTR Enumerator,
                          HWND hwndParent, DWORD Flags));

DLOADV(_SetupDiEnumDeviceInfo, SetupDiEnumDeviceInfo, SetupAPI.dll,
       HDEVINFO(WINAPI *)(HDEVINFO DeviceInfoSet, DWORD MemberIndex,
                          PSP_DEVINFO_DATA DeviceInfoData));

DLOADV(_CM_Get_Device_IDW, CM_Get_Device_IDW, SetupAPI.dll,
       CONFIGRET(WINAPI *)(DEVINST dnDevInst, PWSTR Buffer, ULONG BufferLen,
                           ULONG ulFlags));

//#pragma comment (lib, "SetupAPI.lib")

namespace virtual_keyboard {
  // Returns true if a physical keyboard is detected on Windows 8 and up.
  // Uses the Setup APIs to enumerate the attached keyboards and returns true
  // if the keyboard count is 1 or more.. While this will work in most cases
  // it won't work if there are devices which expose keyboard interfaces which
  // are attached to the machine.
  bool IsKeyboardPresentOnSlate() {

    bool result = false;
    // This function is only supported for Windows 8 and up.
    // This function should be only invoked for machines with touch screens.
    if ((GetSystemMetrics(SM_DIGITIZER) & NID_INTEGRATED_TOUCH) !=
        NID_INTEGRATED_TOUCH) {
      return true;
    }

    if (!_SetupDiGetClassDevs()) return false;
    if (!_SetupDiEnumDeviceInfo()) return false;

#if defined(TOUCH_KEYBOARD_DETECTION_BY_ROLE)

    // If the device is docked, the user is treating the device as a PC.
    if (GetSystemMetrics(SM_SYSTEMDOCKED) != 0) { return true; }
    // To determine whether a keyboard is present on the device, we do the
    // following:-
    // 1. Check whether the device supports auto rotation. If it does then
    //    it possibly supports flipping from laptop to slate mode. If it
    //    does not support auto rotation, then we assume it is a desktop
    //    or a normal laptop and assume that there is a keyboard.
    // 2. If the device supports auto rotation, then we get its platform role
    //    and check the system metric SM_CONVERTIBLESLATEMODE to see if it is
    //    being used in slate mode. If yes then we return false here to ensure
    //    that the OSK is displayed.
    // 3. If step 1 and 2 fail then we check attached keyboards and return true
    //    if we find ACPI\* or HID\VID* keyboards.
    typedef BOOL(WINAPI * GetAutoRotationState)(PAR_STATE state);
    GetAutoRotationState get_rotation_state =
        reinterpret_cast<GetAutoRotationState>(::GetProcAddress(
            GetModuleHandle(L"user32.dll"), "GetAutoRotationState"));
    if (get_rotation_state) {
      AR_STATE auto_rotation_state = AR_ENABLED;
      get_rotation_state(&auto_rotation_state);
      if ((auto_rotation_state & AR_NOSENSOR) ||
          (auto_rotation_state & AR_NOT_SUPPORTED)) {
        // If there is no auto rotation sensor or rotation is not supported in
        // the current configuration, then we can assume that this is a desktop
        // or a traditional laptop.
        return true;
      }
    }

    if (!_PowerDeterminePlatformRole) return true;
    // Check if the device is being used as a laptop or a tablet. This can be
    // checked by first checking the role of the device and then the
    // corresponding system metric (SM_CONVERTIBLESLATEMODE). If it is being
    // used as a tablet then we want the OSK to show up.
    POWER_PLATFORM_ROLE role = _PowerDeterminePlatformRole()();
    if (((role == PlatformRoleMobile) || (role == PlatformRoleSlate)) &&
        (GetSystemMetrics(SM_CONVERTIBLESLATEMODE) == 0)) {
      return false;
    }
#endif
    const GUID KEYBOARD_CLASS_GUID = {
        0x4D36E96B,
        0xE325,
        0x11CE,
        {0xBF, 0xC1, 0x08, 0x00, 0x2B, 0xE1, 0x03, 0x18}};
    // Query for all the keyboard devices.

    HDEVINFO device_info =
        _SetupDiGetClassDevs()(&KEYBOARD_CLASS_GUID, NULL, NULL, DIGCF_PRESENT);
    if (device_info == INVALID_HANDLE_VALUE) { return result; }
    // Enumerate all keyboards and look for ACPI\PNP and HID\VID devices. If
    // the count is more than 1 we assume that a keyboard is present. This is
    // under the assumption that there will always be one keyboard device.

    for (DWORD i = 0;; ++i) {
      SP_DEVINFO_DATA device_info_data = {0};
      device_info_data.cbSize          = sizeof(device_info_data);
      if (!_SetupDiEnumDeviceInfo()(device_info, i, &device_info_data)) break;
      // Get the device ID.
      wchar_t   device_id[MAX_DEVICE_ID_LEN];
      CONFIGRET status = _CM_Get_Device_IDW()(device_info_data.DevInst,
                                              device_id, MAX_DEVICE_ID_LEN, 0);
      if (status == CR_SUCCESS) {
        // To reduce the scope of the hack we only look for ACPI and HID\\VID
        // prefixes in the keyboard device ids.

        wchars device_id_chars = tool::chars_of(device_id);

        if (lexical::ci::eq(device_id_chars(0, 4), WCHARS("ACPI")) ||
            lexical::ci::eq(device_id_chars(0, 7), WCHARS("HID\\VID"))) {

          // The heuristic we are using is to check the count of keyboards and
          // return true if the API's report one or more keyboards. Please note
          // that this will break for non keyboard devices which expose a
          // keyboard PDO.
          result = true;
        }
      }
    }
    return result;
  }

  ustring get_osk_path() {
    const wchar_t kWindows8OSKRegPath[] = L"Software\\Classes\\CLSID\\{"
                                          L"054AAE20-4BEA-4347-8A35-"
                                          L"64A533254A9D}\\LocalServer32";

    HKEY hkey;
    LONG lRes = RegOpenKeyExW(HKEY_LOCAL_MACHINE, kWindows8OSKRegPath, 0,
                              KEY_READ | KEY_WOW64_64KEY, &hkey);
    if (lRes != ERROR_SUCCESS) return ustring();

    WCHAR szBuffer[MAX_PATH] = {0};
    DWORD dwBufferSize       = sizeof(szBuffer);
    lRes =
        RegQueryValueExW(hkey, NULL, 0, NULL, (LPBYTE)szBuffer, &dwBufferSize);
    if (lRes != ERROR_SUCCESS) return ustring();

    return ustring(szBuffer);
  }

  bool show_if_needed() {
    if (environment::get_os_version() < environment::WIN_8) return false;
    if (IsKeyboardPresentOnSlate()) return false;

    static ustring osk_path;
    if (osk_path.is_undefined()) {
      // We need to launch TabTip.exe from the location specified under the
      // LocalServer32 key for the {{054AAE20-4BEA-4347-8A35-64A533254A9D}}
      // CLSID.
      // TabTip.exe is typically found at
      // c:\program files\common files\microsoft shared\ink on English Windows.
      // We don't want to launch TabTip.exe from
      // c:\program files (x86)\common files\microsoft shared\ink. This path is
      // normally found on 64 bit Windows.

      osk_path = get_osk_path();

      int common_program_files_offset = osk_path().index_of(WCHARS("%CommonProgramFiles%"));
      // Typically the path to TabTip.exe read from the registry will start with
      // %CommonProgramFiles% which needs to be replaced with the corrsponding
      // expanded string.
      // If the path does not begin with %CommonProgramFiles% we use it as is.
      if (common_program_files_offset >= 0) {
        // Preserve the beginning quote in the path.
        osk_path.cut(common_program_files_offset, WCHARS("%CommonProgramFiles%").size());

        // The path read from the registry contains the %CommonProgramFiles%
        // environment variable prefix. On 64 bit Windows the
        // SHGetKnownFolderPath function returns the common program files path
        // with the X86 suffix for the FOLDERID_ProgramFilesCommon value. To get
        // the correct path to TabTip.exe we first read the environment variable
        // CommonProgramW6432 which points to the desired common files path.
        // Failing that we fallback to the SHGetKnownFolderPath API. We then
        // replace the %CommonProgramFiles% value with the actual common files
        // path found in the process.
        wchar   common_program_files_path[MAX_PATH] = {0};
        ustring common_program_files;

        GetEnvironmentVariable(L"CommonProgramW6432", common_program_files_path,
                               items_in(common_program_files_path));

        if (common_program_files_path[0] != 0) {
          common_program_files = common_program_files_path;
        } else {
          common_program_files = get_standard_dir(APPLICATIONS_DIR);
        }
        osk_path.insert(common_program_files, common_program_files_offset);
      }
    }

    HINSTANCE ret =
        ::ShellExecuteW(NULL, L"", osk_path.c_str(), NULL, NULL, SW_SHOW);
    return reinterpret_cast<intptr_t>(ret) > 32;
  }

  void dismiss_if_shown() {
    if (environment::get_os_version() < environment::WIN_8) return;
    // We dismiss the virtual keyboard by generating the ESC keystroke
    // programmatically.
    const wchar_t kOSKClassName[] = L"IPTip_Main_Window";
    HWND          osk             = ::FindWindow(kOSKClassName, NULL);
    if (::IsWindow(osk) && ::IsWindowEnabled(osk))
      PostMessage(osk, WM_SYSCOMMAND, SC_CLOSE, 0);
  }

} // namespace virtual_keyboard

namespace mswin {
    bool window::activate_keyboard_for(element *b, uint type) {
      if (super::activate_keyboard_for(b, type))
        return true;
#if defined(CAN_WORK_AS_TABLET)
      if (type) {
        virtual_keyboard::show_if_needed();
        return true;
      }
      else {
        virtual_keyboard::dismiss_if_shown();
        return true;
      }
#endif
      return false;
    }
}

#endif
