﻿#include "win.h"

#include <windowsx.h>
#include <shlobj.h>
#include <tpcshrd.h>
#include <commdlg.h>
#include <winspool.h>
#include <shellscalingapi.h>


#include <mlang.h>

EXTERN_DLOADV(_IsCompositionActive, IsCompositionActive, uxtheme.dll,
  BOOL(WINAPI *)());

EXTERN_DLOADV(_GetDpiForMonitor, GetDpiForMonitor, shcore.dll,
  HRESULT(WINAPI *)(HMONITOR         hmonitor,
    MONITOR_DPI_TYPE dpiType,
    UINT             *dpiX,
    UINT             *dpiY
));

namespace d2d {
  application *app_factory();
}

#if defined(USE_GDI)
namespace gdi {
  application *app_factory();
}
#endif

#if defined(USE_SKIA)
namespace xgl {
  application *app_factory();
}
#endif

DLOADV(_EnumPrinters, EnumPrinters, winspool.drv,
       BOOL(WINAPI *)(DWORD Flags, LPTSTR Name, DWORD Level,
                      LPBYTE pPrinterEnum, DWORD cbBuf, LPDWORD pcbNeeded,
                      LPDWORD pcReturned));

namespace mswin {
  using namespace tool;

  application::application() {

    html::init();
    window::init(true);
    popup::init(true);
    behavior::init(true);
//#ifdef WIC_SUPPORT
    //wic_factory(); // why ????? - force wic factory to be created in this gui thread
//#endif
  }

  void application::final_release() {
    critical_section _(guard);
    instance = 0;

    super::final_release();
    popup::init(false);
    window::init(false);
    behavior::init(false);
#ifdef THEMES_SUPPORT
    theme::current(theme::DESTROY);
    themes::shutdown();
#endif
    // document::destroy_stock_styles();
    if (!normal_shutdown)
      // means destruction of static object (sigleton) at DLL unload
      p_WIC_factory.detach(); // COM can be dead at that moment so we are not
                              // calling Release on it.
  }

  application::~application() {}

  void application::init() {
    int     nargs   = 0;
    LPWSTR *arglist = CommandLineToArgvW(GetCommandLineW(), &nargs);
    for (int n = 0; n < nargs; ++n) {
      wchars arg = trim(chars_of(arglist[n]));
#if defined(USE_GDI)
      if (arg == WCHARS("sciter-gfx=gdi"))
        requested_gfx_layer =
            SOFTWARE_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_GDI);
      else
#endif
          if (arg == WCHARS("sciter-gfx=d2d-warp"))
        requested_gfx_layer =
            WARP_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_D2D_WARP);
      else if (arg == WCHARS("sciter-gfx=d2d"))
        requested_gfx_layer =
            HARDWARE_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_D2D);
      else if (arg == WCHARS("sciter-gfx=d2d"))
        requested_gfx_layer =
            HARDWARE_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_D2D);
      else if (arg == WCHARS("sciter-gfx=skia"))
        requested_gfx_layer =
            SKIA_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_D2D);
      else if (arg == WCHARS("sciter-gfx=skia-opengl"))
        requested_gfx_layer =
            SKIA_OPENGL_GRAPHICS; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_SKIA_CPU);
      else if (arg == WCHARS("ux-theming"))
        html::use_platform_theming =
            false; // SciterSetOption(NULL,SCITER_SET_GFX_LAYER,GFX_LAYER_SKIA_OPENGL);
    }
    LocalFree(arglist);
  }

#ifdef WIC_SUPPORT

#define DEFINE_GUID_HERE(name, l, w1, w2, b1, b2, b3, b4, b5, b6, b7, b8)      \
  EXTERN_C const GUID DECLSPEC_SELECTANY name = {                              \
      l, w1, w2, {b1, b2, b3, b4, b5, b6, b7, b8}}

  DEFINE_GUID_HERE(CLSID_WICImagingFactory1, 0xcacaf262, 0x9370, 0x4615, 0xa1,
                   0x3b, 0x9f, 0x55, 0x39, 0xda, 0x4c, 0xa);

#if 0 // Buggy, must not call CoInitialize(NULL); 
  IWICImagingFactory *application::wic_factory() const {
    if (!p_WIC_factory) {
      HRESULT hr = 0;
      // Create WIC factory
      for (int n = 0; n < 2; ++n) {
        hr = CoCreateInstance(
            CLSID_WICImagingFactory1, NULL, CLSCTX_INPROC_SERVER,
            IID_IWICImagingFactory,
            reinterpret_cast<void **>(
                const_cast<application *>(this)->p_WIC_factory.target()));
        if (hr == CO_E_NOTINITIALIZED)
          CoInitialize(NULL);
        else
          break;
      }
      assert(SUCCEEDED(hr));
      // if (FAILED(hr)) break;
    }
    return p_WIC_factory;
  }
#endif 
  IWICImagingFactory *application::wic_factory() const {
    if (!p_WIC_factory) 
    {
      // Create WIC factory
      HRESULT hr = CoCreateInstance(
        CLSID_WICImagingFactory1, NULL, CLSCTX_INPROC_SERVER, IID_IWICImagingFactory,
        reinterpret_cast<void **>(const_cast<application *>(this)->p_WIC_factory.target()));
      assert(SUCCEEDED(hr));
      (void)hr;
    }
    return p_WIC_factory;
  }


#endif

  void application::get_system_font(/*inout*/ ustring &name, /*out*/ int &size,
                                    /*out*/ uint &weight,
                                    /*out*/ bool &italic) {
    LPLOGFONT lplf = 0;

    NONCLIENTMETRICSW ncm;
    memset(&ncm, 0, sizeof(NONCLIENTMETRICS));
    ncm.cbSize = sizeof(NONCLIENTMETRICS);
    if (environment::get_os_version() < environment::WIN_VISTA)
      ncm.cbSize -= sizeof(int);
    SystemParametersInfoW(SPI_GETNONCLIENTMETRICS, sizeof(NONCLIENTMETRICS),
                          &ncm, 0);
    wchars cname = name;
    if (cname == WCHARS("system"))
      lplf = &ncm.lfMessageFont;
    else if (cname == WCHARS("system-caption"))
      lplf = &ncm.lfCaptionFont;
    else if (cname == WCHARS("system-menu"))
      lplf = &ncm.lfMenuFont;
    else if (cname == WCHARS("system-status"))
      lplf = &ncm.lfStatusFont;
    else // if(strcmp(fname,".message"))
      lplf = &ncm.lfMessageFont;
    // if(lplf)
    {
      name = lplf->lfFaceName;
      // alert(ustring("sys font:") + name);
      size = abs(lplf->lfHeight);

      gool::size pxin = resolution_provider::desktop().pixels_per_inch();

      size *= 96;
      size /= pxin.y; // to dips

      // alert(ustring::format(L"%s %d %d",lplf->lfFaceName, lplf->lfHeight,
      // size));
      weight = lplf->lfWeight;
      italic = lplf->lfItalic != 0;
    }
  }

  application *application::instance;

  application *app() {
    critical_section _(application::guard);

    while (!application::instance) {
      mswin::application::init();
      if ((mswin::application::requested_gfx_layer > SOFTWARE_GRAPHICS &&
           mswin::application::requested_gfx_layer <= HARDWARE_GRAPHICS)) {
        application::instance = d2d::app_factory();
        if (application::instance) break;
      }
#if defined(USE_SKIA)
      if (mswin::application::requested_gfx_layer == SKIA_GRAPHICS ||
          mswin::application::requested_gfx_layer == SKIA_OPENGL_GRAPHICS) {
        application::instance = xgl::app_factory();
        break;
      }
#endif

#if defined(USE_GDI)
      if (!application::instance) application::instance = gdi::app_factory();
#endif
#if defined(USE_SKIA)
      if (!application::instance) application::instance = xgl::app_factory();
#endif
      break;
    }
    return application::instance;
  }

  void application::shutdown() {
    if (application::instance) {
      application *pa       = application::instance;
      application::instance = 0;
      pa->normal_shutdown   = false;
      pa->final_release();
      delete pa;
    }
  }

  handle<print_target> application::select_printer(html::view *ui_parent,
                                                   ustring     printer_id) {
    const DEVMODE *     pdm = nullptr;
    array<printer_info> pilist;

    handle<print_target> target = create_print_target();
    if (!target) return target;

    HDC hDC = NULL;

    if (printer_id.is_undefined()) {
      PRINTDLGW pd;
      memzero(pd);
      pd.lStructSize = sizeof(pd);
      pd.nFromPage = pd.nMinPage = 1;
      pd.nToPage = pd.nMaxPage = 1; // max(pd.nFromPage,total_pages.val(100));

      if (ui_parent) {
        pd.hwndOwner = ui_parent->get_hwnd();
        pd.Flags     = PD_RETURNIC | PD_PRINTSETUP | PD_ALLPAGES
            /*| PD_USEDEVMODECOPIESANDCOLLATE*/
            ; //  | PD_NOSELECTION | PD_USEDEVMODECOPIESANDCOLLATE
      } else {
        pd.Flags = PD_RETURNIC | PD_RETURNDEFAULT;
      }

      //}
      if (!::PrintDlgW(&pd)) {
        DWORD dwError = CommDlgExtendedError();
        dwError       = dwError;
        return nullptr;
      }

      bytes devmode;

      devmode.start  = (byte *)::GlobalLock(pd.hDevMode);
      devmode.length = GlobalSize(pd.hDevMode);
      // target.device_mode.destroy();
      target->device_mode = devmode;
      ::GlobalUnlock(pd.hDevMode);

      hDC = pd.hDC;
      pdm = (const DEVMODE *)target->device_mode.cbegin();
      // target->device_name = pdm->dmDeviceName;
      const DEVNAMES *devnames = (const DEVNAMES *)::GlobalLock(pd.hDevNames);
      target->device_name = (const wchar *)devnames + devnames->wDeviceOffset;
      ::GlobalUnlock(pd.hDevNames);
    } else {
      pilist = get_printer_list();
      for (auto &pidef : pilist) {
        if (pidef.id == printer_id) {
          target->device_mode = pidef.device_mode;
          target->device_name = pidef.name;
          pdm                 = (const DEVMODE *)pidef.device_mode.cbegin();
          hDC                 = CreateDC(NULL, pidef.name, NULL, pdm);
          break;
        }
      }
    }

    if (!pdm) {
      if (hDC) ::DeleteDC(hDC);
      return nullptr;
    }

    target->page_paper_size.x = pdm->dmPaperWidth;
    target->page_paper_size.y = pdm->dmPaperLength;

    target->page_device_size = size(::GetDeviceCaps(hDC, PHYSICALWIDTH),
                                    ::GetDeviceCaps(hDC, PHYSICALHEIGHT));
    target->page_device_rect = rect(-::GetDeviceCaps(hDC, PHYSICALOFFSETX),
                                    -::GetDeviceCaps(hDC, PHYSICALOFFSETY),
                                    ::GetDeviceCaps(hDC, PHYSICALWIDTH) -
                                        ::GetDeviceCaps(hDC, PHYSICALOFFSETX),
                                    ::GetDeviceCaps(hDC, PHYSICALHEIGHT) -
                                        ::GetDeviceCaps(hDC, PHYSICALOFFSETY));

    target->device_dpi = size(::GetDeviceCaps(hDC, LOGPIXELSX),
                              ::GetDeviceCaps(hDC, LOGPIXELSY));

    // if (hDC) ::DeleteDC(hDC);

    return target;
  }

  array<printer_info> application::get_printer_list() {

    array<printer_info> out;

#ifdef WINDOWS

    if (!_EnumPrinters()) return out;

    DWORD dwSizeNeeded = 0;
    DWORD dwNumItems   = 0;
    DWORD dwItem       = 0;

    wchar default_printer_name[1024] = {0};
    DWORD dwSize                     = items_in(default_printer_name);
    GetDefaultPrinter(default_printer_name, &dwSize);

    LPPRINTER_INFO_2 lpInfo = NULL;

    // Get buffer size
    _EnumPrinters()(PRINTER_ENUM_LOCAL | PRINTER_ENUM_CONNECTIONS, NULL, 2,
                    NULL, 0, &dwSizeNeeded, &dwNumItems);

    // allocate memory
    lpInfo = (LPPRINTER_INFO_2)HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY,
                                         dwSizeNeeded);
    if (!lpInfo) {
      // puts ( "Not enough memory\n" );
      return out;
    }

    if (_EnumPrinters()(
            PRINTER_ENUM_LOCAL | PRINTER_ENUM_CONNECTIONS, // what to enumerate
            NULL,           // printer name (NULL for all)
            2,              // level
            (LPBYTE)lpInfo, // buffer
            dwSizeNeeded,   // size of buffer
            &dwSizeNeeded,  // returns size
            &dwNumItems     // return num. items
            ) == 0) {
      // printf ( "EnumPrinters() Failed with error: %d\n", GetLastError () );
      HeapFree(GetProcessHeap(), 0, lpInfo);
      return out;
    }

    // display printers
    for (dwItem = 0; dwItem < dwNumItems; dwItem++) {
      printer_info pi;
      pi.name = lpInfo[dwItem].pPrinterName;
      if (lpInfo[dwItem].Attributes & PRINTER_ATTRIBUTE_SHARED)
        pi.share_name = lpInfo[dwItem].pShareName;
      // printf ( "Ports: %s\n" , lpInfo[dwItem].pPortName );
      // printf ( "Driver Name: %s\n" , lpInfo[dwItem].pDriverName );
      if (lpInfo[dwItem].pComment) pi.comment = lpInfo[dwItem].pComment;
      if (lpInfo[dwItem].pLocation) pi.location = lpInfo[dwItem].pLocation;
      pi.device_mode = bytes((const byte *)lpInfo[dwItem].pDevMode,
                             lpInfo[dwItem].pDevMode->dmSize +
                                 lpInfo[dwItem].pDevMode->dmDriverExtra);
      // auto pdm = lpInfo[dwItem].pDevMode;
      pi.id         = lpInfo[dwItem].pDevMode->dmDeviceName;
      pi.is_default = pi.name == chars_of(default_printer_name);
      // pi.is_default = (lpInfo[dwItem].Attributes & PRINTER_ATTRIBUTE_DEFAULT)
      // == PRINTER_ATTRIBUTE_DEFAULT;
      if (pi.is_default)
        out.insert(0, pi);
      else
        out.push(pi);
    }

    // free memory
    HeapFree(GetProcessHeap(), 0, lpInfo);
#endif
    return out;
  }

#if defined(USE_GDI)

  handle<print_target> application::create_print_target() {
    return handle<print_target>(new hdc_print_target());
  }

  bool hdc_print_target::print(
      const ustring &                                 doc_name,
      function<bool(graphics *page_gfx, int page_no)> renderer) {
    HDC hDC = CreateDC(NULL, device_name, NULL,
                       (const DEVMODE *)device_mode.cbegin());

    DOCINFO di;
    di.cbSize      = sizeof(DOCINFO);
    di.lpszDocName = doc_name.c_str();
    di.lpszOutput  = NULL;
    ::StartDoc(hDC, &di);

    rect rc(-::GetDeviceCaps(hDC, PHYSICALOFFSETX),
            -::GetDeviceCaps(hDC, PHYSICALOFFSETY),
            ::GetDeviceCaps(hDC, PHYSICALWIDTH) -
                ::GetDeviceCaps(hDC, PHYSICALOFFSETX),
            ::GetDeviceCaps(hDC, PHYSICALHEIGHT) -
                ::GetDeviceCaps(hDC, PHYSICALOFFSETY));

    ::SetMapMode(hDC, MM_TEXT);

    for (uint page_no = 1;; ++page_no) {
      ::StartPage(hDC);
      handle<gdi::graphics> gfx = new gdi::graphics(hDC);
      gfx->translate(rc.s);
      // this->print_page(v, self, gfx, page_no);
      bool has_more_pages = renderer(gfx, page_no);
      ::EndPage(hDC);
      if (!has_more_pages) break;
    }

    ::EndDoc(hDC);

    ::DeleteDC(hDC);

    return true;
  }
#else
  handle<print_target> application::create_print_target() { return nullptr; }
#endif

  static UINT GetAnsiCodePageForLocale(LCID lcid)
  {
    UINT acp = CP_ACP;
    int sizeInChars = sizeof(acp) / sizeof(TCHAR);
    GetLocaleInfo(lcid, LOCALE_IDEFAULTANSICODEPAGE | LOCALE_RETURN_NUMBER, reinterpret_cast<LPTSTR>(&acp), sizeInChars);
    return acp;
  }
  
  bool application::font_families_for_char_code(tool::wchars lang_id, WRITING_SCRIPT script, uint char_code, array<ustring>& out)
  {
    assert(char_code);

    static IMLangFontLink2* font_link = 0;

    if (!font_link) {
      com::asset<IMultiLanguage> multi_language;
      if (::CoCreateInstance(CLSID_CMultiLanguage, 0, CLSCTX_ALL,
        IID_IMultiLanguage,
        reinterpret_cast<void**>(multi_language.target())) != S_OK)
        return false;
      if (multi_language->QueryInterface(&font_link) != S_OK)
        return false;
    }

    DWORD code_pages[2] = { 0,0 };
    long  n_code_pages = 0;

    wchar w2[3] = { 0,0,0};

    uint n = u16::putc(char_code, w2);

    HRESULT hr;

    UINT  dcp = GetAnsiCodePageForLocale(locale_name_2_LCID(lang_id));
    DWORD dwACP = 0;

    // give priority to lang_id
    hr = font_link->CodePageToCodePages(dcp, &dwACP);
    if (FAILED(hr)) return false;
    hr = font_link->GetStrCodePages(w2, n, dwACP, code_pages, &n_code_pages);
    if (FAILED(hr)) return false;

    SCRIPT_ID sid = sidDefault;

    hr = font_link->CodePageToScriptID(code_pages[0], &sid);
    //if (FAILED(hr)) return false;
    
    SCRIPTFONTINFO buffer[64] = {0};
    DWORD          dwFlags = SCRIPTCONTF_PROPORTIONAL_FONT;
    UINT           nfams = items_in(buffer);
    
    hr = font_link->GetScriptFontInfo(sid, dwFlags, &nfams, buffer);
    if (FAILED(hr)) return false;

    for (uint i = 0; i < nfams; ++i)
      out.push(buffer[i].wszFont);

    return true;
  }


} // namespace mswin

namespace gool {
  application *app() { return mswin::app(); }
} // namespace gool

namespace html {
  static BOOL CALLBACK MonitorEnumProc(_In_ HMONITOR hMonitor,
                                       _In_ HDC hdcMonitor,
                                       _In_ LPRECT lprcMonitor,
                                       _In_ LPARAM dwData) {
    array<HMONITOR> *parr = reinterpret_cast<array<HMONITOR> *>(dwData);
    parr->push(hMonitor);
    return TRUE;
  }

  static array<HMONITOR> get_monitors() {
    array<HMONITOR> arr;
    EnumDisplayMonitors(NULL, NULL, &MonitorEnumProc, LPARAM(&arr));
    return arr;
  }

  int number_of_screens() { return get_monitors().size(); }
  int screen_of(html::iwindow *hw) {
    // MONITORINFO minf;
    HMONITOR hmon = MonitorFromWindow(hw->get_hwnd(), MONITOR_DEFAULTTONEAREST);
    return get_monitors().get_index(hmon);
  }

  ustring get_monitor_name(int nDeviceIndex) {
    DISPLAY_DEVICE dd;
    wchar          device_name[64] = {0};

    memzero(dd);
    dd.cb = sizeof(DISPLAY_DEVICE);

    // After first call to EnumDisplayDevices dd.DeviceString
    // contains graphic card name
    if (EnumDisplayDevices(NULL, nDeviceIndex, &dd, 0)) {
      target(device_name).copy(chars_of(dd.DeviceName));
      // after second call DispDev.DeviceString contains monitor's name
      EnumDisplayDevices(device_name, 0, &dd, 0);
      return dd.DeviceString;
    } else {
      return ustring::format(L"monitor %d", nDeviceIndex);
    }
  }

  bool get_screen_info(int n, screen_info &si) {

    array<HMONITOR> mons = get_monitors();

    if (n < 0 || n >= mons.size()) return false;

    MONITORINFO mi;
    memzero(mi);
    mi.cbSize = sizeof(mi);

    if (!GetMonitorInfoW(mons[n], &mi)) return false;

    si.monitor     = mi.rcMonitor;
    si.workarea    = mi.rcWork;
    si.is_primary  = (mi.dwFlags & MONITORINFOF_PRIMARY) != 0;
    si.device_name = get_monitor_name(n);
    
    if (_GetDpiForMonitor) {
      UINT x, y;
      if (SUCCEEDED(_GetDpiForMonitor()(mons[n], MDT_EFFECTIVE_DPI, &x, &y))) {
        si.dpi.x = int(x);
        si.dpi.y = int(y);
      }
    }

    return true;
  }

#ifdef SCREENSHOT_SUPPORT
  gool::bitmap *get_screen_shot(int n) {
    screen_info si;
    if (!get_screen_info(n, si)) return nullptr;

    HDC hdc_screen = GetDC(NULL);
    if (!hdc_screen) return nullptr;
    ON_SCOPE_EXIT(ReleaseDC(nullptr, hdc_screen));

    dib32 db(si.monitor.size());

    // copy the bits from the screen into dib
    if (!BitBlt(db.DC(),                                  // destination
                0, 0,                                     // upper left corner
                si.monitor.width(), si.monitor.height(),  // width and height
                hdc_screen,                               // source
                si.monitor.s.x, si.monitor.s.y,           // upper left corner
                SRCCOPY)) {
      return nullptr;
    }
    return bitmap::create(db, false);
  }
#endif
  
  bool event::get_key_state(uint vk_code) {
    auto ks = GetKeyState(int(vk_code));
    switch (vk_code) {
      case VK_CAPITAL:
      case VK_SCROLL:
      case VK_NUMLOCK:
        return (ks & 0x8001) != 0;
      default:
        return (ks & 0x8000) != 0;
    }
  }

} // namespace html
