#include "config.h"

#include "tool.h"

#include <stdlib.h>

#include "zlib.h"

#ifndef MONOLITHIC 
  #pragma comment(lib, "zlib")
#endif

namespace tool {

  bool gzip_uncompress(bytes compressedBytes, array<byte> &uncompressedBytes) {
    if (compressedBytes.size() == 0) {
      uncompressedBytes.clear();
      return true;
    }

    uncompressedBytes.clear();

    size_t full_length = compressedBytes.length;
    size_t half_length = compressedBytes.length / 2;

    size_t uncompLength = full_length;
    byte * uncomp       = (byte *)calloc(sizeof(byte), uncompLength);

    z_stream strm;
    strm.next_in   = (Bytef *)compressedBytes.start;
    strm.avail_in  = uInt(compressedBytes.length);
    strm.total_out = 0;
    strm.zalloc    = Z_NULL;
    strm.zfree     = Z_NULL;

    bool done = false;

    if (inflateInit2(&strm, (16 + MAX_WBITS)) != Z_OK) {
      free(uncomp);
      return false;
    }

    while (!done) {
      // If our output buffer is too small
      if (strm.total_out >= uncompLength) {
        // Increase size of output buffer
        byte *uncomp2 = (byte *)calloc(sizeof(byte), uncompLength + half_length);
        memcpy(uncomp2, uncomp, uncompLength);
        uncompLength += half_length;
        free(uncomp);
        uncomp = uncomp2;
      }

      strm.next_out  = (Bytef *)(uncomp + strm.total_out);
      strm.avail_out = uInt(uncompLength - strm.total_out);

      // Inflate another chunk.
      int err = inflate(&strm, Z_SYNC_FLUSH);
      if (err == Z_STREAM_END)
        done = true;
      else if (err != Z_OK) {
        break;
      }
    }

    if (inflateEnd(&strm) != Z_OK) {
      free(uncomp);
      return false;
    }

    uncompressedBytes.push(uncomp, strm.total_out);

    free(uncomp);
    return true;
  }

  bool gzip(bool compress, bytes src, array<byte>& dst, bool raw_format) 
  {
    const uint kBufferSize = 1024;

    uint8_t outputBuffer[kBufferSize];
    z_stream flateData;
    flateData.zalloc = Z_NULL;
    flateData.zfree = Z_NULL;
    flateData.opaque = NULL;
    flateData.next_in = NULL;
    flateData.avail_in = 0;
    flateData.next_out = outputBuffer;
    flateData.avail_out = kBufferSize;
    int rc;
    if (compress)
    { 
      if (!raw_format)
        rc = deflateInit2(&flateData, Z_DEFAULT_COMPRESSION,Z_DEFLATED, 16 + MAX_WBITS, 8, Z_DEFAULT_STRATEGY);
      else
        rc = deflateInit(&flateData, Z_DEFAULT_COMPRESSION);
    }
    else {
      if (!raw_format)
        rc = inflateInit2(&flateData, 16 + MAX_WBITS);
      else
        rc = inflateInit(&flateData);
    }
    if (rc != Z_OK)
      return false;

    uint8_t* input = (uint8_t*)src.start;
    size_t inputLength = src.length;
    if (input == NULL || inputLength == 0) {
      input = NULL;
      flateData.next_in = input;
      flateData.avail_in = 0;
    }
    else {
      flateData.next_in = input;
      flateData.avail_in = uint(inputLength);
    }

    rc = Z_OK;
    while (true) {
      if (flateData.avail_out < kBufferSize) {
        dst.push(outputBuffer, kBufferSize - flateData.avail_out);
        flateData.next_out = outputBuffer;
        flateData.avail_out = kBufferSize;
      }
      if (rc != Z_OK)
        break;
      if (flateData.avail_in == 0) {
        if (input != NULL)
          break;
        bytes read = src.read(kBufferSize);
        if (read.length == 0)
          break;
        flateData.next_in = (uint8_t*)read.start;
        flateData.avail_in = (uint)read.length;
      }
      if (compress)
        rc = deflate(&flateData, Z_NO_FLUSH);
      else
        rc = inflate(&flateData, Z_NO_FLUSH);
    }
    while (rc == Z_OK) {
      if (compress)
        rc = deflate(&flateData, Z_FINISH);
      else
        rc = inflate(&flateData, Z_FINISH);
      if (flateData.avail_out < kBufferSize) {
        dst.push(outputBuffer, kBufferSize - flateData.avail_out);
        flateData.next_out = outputBuffer;
        flateData.avail_out = kBufferSize;
      }
    }

    if (compress)
      deflateEnd(&flateData);
    else
      inflateEnd(&flateData);
    if (rc == Z_STREAM_END)
      return true;
    return false;
  }


} // namespace tool

#if defined(ZIP_SUPPORT)

#include "ioapi.h"
#include "unzip.h"

/*#ifndef INVALID_HANDLE_VALUE
#define INVALID_HANDLE_VALUE (0xFFFFFFFF)
#endif

#ifndef INVALID_SET_FILE_POINTER
#define INVALID_SET_FILE_POINTER ((DWORD)-1)
#endif*/

namespace tool {

static voidpf ZCALLBACK open_file_func OF((voidpf opaque, const char *filename,
                                           int mode));

static uLong ZCALLBACK read_file_func OF((voidpf opaque, voidpf stream,
                                          void *buf, uLong size));

static uLong ZCALLBACK write_file_func OF((voidpf opaque, voidpf stream,
                                           const void *buf, uLong size));

static long ZCALLBACK tell_file_func OF((voidpf opaque, voidpf stream));

static long ZCALLBACK seek_file_func OF((voidpf opaque, voidpf stream,
                                         uLong offset, int origin));

static int ZCALLBACK close_file_func OF((voidpf opaque, voidpf stream));

static int ZCALLBACK error_file_func OF((voidpf opaque, voidpf stream));

struct zip_io_ctl {
  handle<cabinet::item> item;
  int                   pos;
  int                   error;
  cabinet*              pcab;
  zip_io_ctl(cabinet::item *pi) : pos(0), error(0), item(pi) {}
};

#define CAB (((zip_io_ctl *)opaque)->pcab)
#define CTL ((zip_io_ctl *)stream)

voidpf ZCALLBACK open_file_func(voidpf opaque, const char *filename, int mode) {
  /*bool create = false;

  if (mode & ZLIB_FILEFUNC_MODE_CREATE) {
    create = true;
  }*/
  return opaque;
}

uLong ZCALLBACK read_file_func(voidpf /*opaque*/, voidpf stream, void *buf,
                               uLong size) {
  array<byte> &data = CTL->item->data;

  if (CTL->pos + size > uLong(data.size())) {
    long sz = (long)data.size() - CTL->pos;
    if (sz <= 0)
      return 0;
    size = (uLong)sz;
  }

  // mem cpy( buf, data.head() + CTL->pos, size );
  target(buf, size).copy(data.head() + CTL->pos);
  CTL->pos += size;

  return size;
}

uLong ZCALLBACK write_file_func(voidpf /*opaque*/, voidpf stream,
                                const void *buf, uLong size) {
  array<byte> &data = CTL->item->data;
  data.push((const byte *)buf, size);
  return size;
}

long ZCALLBACK tell_file_func(voidpf /*opaque*/, voidpf stream) {
  return CTL->pos;
}

long ZCALLBACK seek_file_func(voidpf /*opaque*/, voidpf stream, uLong offset,
                              int origin) {
  if (!stream)
    return -1;

  long pos = CTL->pos;

  switch (origin) {
  case ZLIB_FILEFUNC_SEEK_CUR:
    pos += offset;
    break;
  case ZLIB_FILEFUNC_SEEK_END:
    pos = long(CTL->item->data.size() + offset);
    break;
  case ZLIB_FILEFUNC_SEEK_SET:
    pos = offset;
    break;
  default:
    return -1;
  }

  if (CTL->pos >= 0 && CTL->pos <= CTL->item->data.size()) {
    CTL->pos = pos;
    return 0;
  }
  return -1;
}

int ZCALLBACK close_file_func(voidpf /*opaque*/, voidpf stream) {
  delete CTL;
  return 0;
}

int ZCALLBACK error_file_func(voidpf /*opaque*/, voidpf stream) {
  if (stream)
    return CTL->error;
  return -1;
}

void fill_filefunc(zlib_filefunc_def *pzlib_filefunc_def) {
  pzlib_filefunc_def->zopen_file  = open_file_func;
  pzlib_filefunc_def->zread_file  = read_file_func;
  pzlib_filefunc_def->zwrite_file = write_file_func;
  pzlib_filefunc_def->ztell_file  = tell_file_func;
  pzlib_filefunc_def->zseek_file  = seek_file_func;
  pzlib_filefunc_def->zclose_file = close_file_func;
  pzlib_filefunc_def->zerror_file = error_file_func;
  pzlib_filefunc_def->opaque      = NULL;
}

#define WRITEBUFFERSIZE (8192)

static int do_extract_currentfile(unzFile uf, cabinet *cab,
                                  const string &zip_url, const char *password) {
  char  filename_inzip[MAX_PATH];
  char *filename_withoutpath;
  char *p;
  int   err = UNZ_OK;
  uint  size_buf;
  // int   overwrite = 0;

  unz_file_info file_info;
  // uLong ratio=0;
  err = unzGetCurrentFileInfo(uf, &file_info, filename_inzip,
                              sizeof(filename_inzip), NULL, 0, NULL, 0);

  if (cab->exist(zip_url + filename_inzip))
    return UNZ_OK;

  if (err != UNZ_OK) {
    assert(false);
    // printf("error %d with zipfile in unzGetCurrentFileInfo\n",err);
    return err;
  }

  size_buf = WRITEBUFFERSIZE;
  byte buf[WRITEBUFFERSIZE];

  p = filename_withoutpath = filename_inzip;
  while (*p != '\0') {
    if (((*p) == '/') || ((*p) == '\\'))
      filename_withoutpath = p + 1;
    p++;
  }

  if ((*filename_withoutpath) == '\0') {
    // printf("creating directory: %s\n",filename_inzip);
    array<byte> data;
    cab->move(zip_url + filename_inzip, data, true);
  } else {
    const char *write_filename;
    /*int skip=0;*/

    write_filename = filename_inzip;

    err = unzOpenCurrentFilePassword(uf, password);
    if (err != UNZ_OK) {
      // printf("error %d with zipfile in unzOpenCurrentFilePassword\n",err);
      assert(false);
    }

    /*        if ( (overwrite==0) && (err==UNZ_OK) )
            {
                char rep=0;
                FILE* ftestexist;
                ftestexist = fopen(write_filename,"rb");
                if (ftestexist!=NULL)
                {
                    fclose(ftestexist);
                    do
                    {
                        char answer[128];
                        int ret;

                        printf("The file %s exists. Overwrite ? [y]es, [n]o,
       [A]ll: ",write_filename); ret = scanf("%1s",answer); if (ret != 1)
                        {
                           exit(EXIT_FAILURE);
                        }
                        rep = answer[0] ;
                        if ((rep>='a') && (rep<='z'))
                            rep -= 0x20;
                    }
                    while ((rep!='Y') && (rep!='N') && (rep!='A'));
                }

                if (rep == 'N')
                    skip = 1;

                if (rep == 'A')
                    *popt_overwrite=1;
            }
            if ((skip==0) && (err==UNZ_OK))
    */

    /*
            if ( err == UNZ_OK )
            {
                fout=fopen(write_filename,"wb");

                // some zipfile don't contain directory alone before file
                if ((fout==NULL) && ((*popt_extract_without_path)==0) &&
                                    (filename_withoutpath!=(char*)filename_inzip))
                {
                    char c=*(filename_withoutpath-1);
                    *(filename_withoutpath-1)='\0';
                    makedir(write_filename);
                    *(filename_withoutpath-1)=c;
                    fout=fopen(write_filename,"wb");
                }

                if (fout==NULL)
                {
                    printf("error opening %s\n",write_filename);
                }
            }
    */
    if (err == UNZ_OK) {
      dbg_printf(" extracting: %s\n", write_filename);
      array<byte> data;
      do {
        err = unzReadCurrentFile(uf, buf, size_buf);
        if (err < 0) {
          // printf("error %d with zipfile in unzReadCurrentFile\n",err);
          assert(false);
          break;
        }
        if (err > 0)
          data.push(buf, err);
      } while (err > 0);
      cab->move(zip_url + filename_inzip, data, false);
    }

    if (err == UNZ_OK) {
      err = unzCloseCurrentFile(uf);
      if (err != UNZ_OK) {
        dbg_printf("error %d with zipfile in unzCloseCurrentFile\n", err);
      }
    } else
      unzCloseCurrentFile(uf); /* don't lose the error */
  }

  return err;
}

bool cabinet::is_zip_data(bytes data) {
  return data.length > 10 && data[0] == 0x50 && data[1] == 0x4B &&
         data[2] == 0x03 && data[3] == 0x04;
}

int cabinet::unzip(array<byte> &data, const string &zip_url,
                   const char *password) {
  handle<cabinet::item> pi = new cabinet::item;
  pi->data.swap(data);
  zip_io_ctl *stream = new zip_io_ctl(pi);
  stream->pcab = this;

  zlib_filefunc_def filefunc_def;
  fill_filefunc(&filefunc_def);
  filefunc_def.opaque = stream;

  unzFile uf = ::unzOpen2("", &filefunc_def);

  unz_global_info gi;
  int             err = unzGetGlobalInfo(uf, &gi);
  assert(err == UNZ_OK);

  for (uint i = 0; i < gi.number_entry; i++) {
    if (do_extract_currentfile(uf, this, zip_url, password) != UNZ_OK)
      break;

    if ((i + 1) < gi.number_entry) {
      err = unzGoToNextFile(uf);
      if (err != UNZ_OK) {
        // printf("error %d with zipfile in unzGoToNextFile\n",err);
        assert(false);
        break;
      }
    }
  }
  return (int)items.size();
}
//     int zip(array<byte>& data);

} // namespace tool

#endif
