#include "config.h"
#include "tl_wregexp.h"
#include "tool.h"
#include <locale>

namespace tool {

bool wregexp::compile(const ustring &psz_pattern, bool ignorecase, bool global,
                      bool multiline) {
  m_pattern    = psz_pattern;
  m_ignorecase = ignorecase;
  m_multiline  = multiline;
  m_global     = global;
  m_next_index = 0;
  m_test.clear();
  m_matches.clear();

  int flags = 0;
  if (m_multiline)
    flags |= REX_FLAG_M;
  if (m_ignorecase)
    flags |= REX_FLAG_I;
  // if (m_global)
  //  flags |= REX_FLAG_G;
  int r = rex_compile(m_pattern.c_str(), m_pattern.length(), flags, &m_compiled,
                      true);
  if (r != REX_OK) {
    m_error    = ustring(rex_errmsg(r));
    m_compiled = nullptr;
    return false;
  }

  m_has_only_group = rex_get_captures(m_compiled) == 1
    && psz_pattern().starts_with('(')
    && psz_pattern().ends_with(')');

  return true;
}

tool::ustring wregexp::get_error_string() const { return m_error; }

wregexp::~wregexp() {
  if (m_compiled)
    rex_free(m_compiled);
}

bool wregexp::exec(wchars text) {
  // tool::ustring tst = m_test;

  if (m_test == text || text.start == NEXT_CHUNK().start)
    m_index = m_global ? m_next_index : 0;
  else {
    m_index = m_next_index = 0;
    m_test                 = text;
  }

  m_matches.clear();

  if (m_index >= m_test.length()) {
    if (m_global)
      m_index = m_next_index = uint(m_test.length());
    else
      m_index = m_next_index = 0;
    return false;
  }

  wchars input = m_test();
  input.prune(m_index);

  struct rex_loot matches;
  memzero(matches);

  int r = rex_exec(m_compiled, false, input.cbegin(), input.cend(), &matches);

  if (r != REX_OK || matches.num_captures == 0) {
    if (m_global)
      m_index = m_next_index = uint(m_test.length());
    else
      m_index = m_next_index = 0;
    return false;
  }

  m_next_index = m_index + uint(matches.caps[0].end - input.cbegin());

  for (int n = 0; n < matches.num_captures; ++n) {
    regmatch rm;
    rm.begin = m_index + uint(matches.caps[n].start - input.cbegin());
    rm.end   = m_index + uint(matches.caps[n].end - input.cbegin());
    m_matches.push(rm);
  }
  return m_matches.size() > 0;
}

bool wregexp::exec_all(wchars text) {
  m_test       = text;
  m_next_index = 0;
  m_matches.clear();

  while (true) {
    m_index = m_next_index;
    if (m_index >= m_test.length())
      break;

    struct rex_loot matches;
    memzero(matches);

    wchars input = m_test();
    input.prune(m_index);

    int r = rex_exec(m_compiled, false, input.cbegin(), input.cend(), &matches);

    if (r != REX_OK || matches.num_captures == 0) {
      m_next_index = 0;
      break;
    }
    regmatch rm;
    rm.begin = m_index + uint(matches.caps[0].start - input.cbegin());
    rm.end   = m_index + uint(matches.caps[0].end - input.cbegin());
    if (rm.begin == rm.end) {
      m_next_index = rm.begin = rm.end = rm.begin + 1;
      if (m_next_index >= m_test.length())
        break;
    } else
      m_next_index = rm.end;

    m_matches.push(rm);
  }
  m_next_index = 0;
  return m_matches.size() > 0;
}

bool wregexp::is_matched(int n_sub_exp) const {
  return n_sub_exp < m_matches.size();
}

int wregexp::get_match_start(int matchNo) const {
  return m_matches[matchNo].begin;
}

int wregexp::get_match_end(int matchNo) const { return m_matches[matchNo].end; }

tool::wchars wregexp::get_match(int matchNo) const {
  if (matchNo >= m_matches.size())
    return tool::wchars();

  regmatch rm = m_matches[matchNo];

  return tool::wchars(m_test.c_str() + rm.begin, rm.end - rm.begin);
}

int wregexp::get_number_of_matches() const { return (int)m_matches.size(); }

/* substitute string using the matches from the last regexec() */
ustring wregexp::substitute(wchars text) {
  // const wchar *ssp;
  int          i = 0;
  array<wchar> out;
  const wchar *sp    = text.start;
  const wchar *spend = text.end();

  while (sp != spend) {
    if (*sp != '$') {
      out.push(*sp);
      ++sp;
      continue;
    }
    if (++sp == spend)
      break;
    switch (*sp) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      i = *sp - '0';
      if (i < get_number_of_matches())
        out.push(get_match(i));
      break;
    case '$':
      out.push('$');
      break;
    case '&':
      if (get_number_of_matches() > 0)
        out.push(get_match(0));
      break;
    default:
      out.push(*sp);
      break;
    }
    ++sp;
  }
  return out();
}

} // namespace tool
