//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| aux stuff
//|
//|

#include "config.h"
#include "tool.h"

#if defined(WINDOWS)
#include <mlang.h>
#include <shlobj.h>
#include <urlmon.h>
#include <shellapi.h>
#include <knownfolders.h>
#include "tl_com.h"
#include "win/win-delayload.h"
#include "win/win-version.h"

#ifndef MONOLITHIC
  #pragma comment( lib, "uv" )
  #if defined(TLS_SUPPORT)
    #pragma comment( lib, "uv-tls" )
  #endif
#endif

#pragma comment( lib, "ws2_32" )
#pragma comment(lib,  "iphlpapi.lib")
#pragma comment(lib,  "psapi.lib")
#pragma comment(lib,  "userenv.lib")

DLOADV(_SHGetKnownFolderPath, SHGetKnownFolderPath, shell32.dll,
  HRESULT(WINAPI *)(REFKNOWNFOLDERID rfid,
    DWORD            dwFlags,
    HANDLE           hToken,
    PWSTR            *ppszPath));

#elif defined(OSX)
#include <errno.h>
#include <libproc.h>
#include <monetary.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#elif defined(ANDROID)
#include <dlfcn.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#elif defined(LINUX)
#include <dlfcn.h>
#include <errno.h>
#include <monetary.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#endif

#include <limits.h>

#ifdef OSX
tool::ustring osx_get_exe_path();
#endif

namespace tool {

  mutex lock;

void split_path(const char *path, string &drive, string &dir, string &name,
                string &ext) {
#if defined(_MAX_DRIVE) && !defined(PLATFORM_WINCE)
  char cdrive[_MAX_DRIVE];
  char cdir[_MAX_DIR];
  char cname[_MAX_FNAME];
  char cext[_MAX_EXT];
  cdrive[0] = 0;
  cdir[0]   = 0;
  cname[0]  = 0;
  cext[0]   = 0;
  _splitpath(path, cdrive, cdir, cname, cext);
  drive = cdrive;
  dir   = cdir;
  name  = cname;
  ext   = cext;
#else
  assert(false);
#endif
}

void split_path(const wchar *path, ustring &drive, ustring &dir, ustring &name,
                ustring &ext) {
#if defined(_MAX_DRIVE) && !defined(PLATFORM_WINCE)
  wchar cdrive[_MAX_DRIVE];
  wchar cdir[_MAX_DIR];
  wchar cname[_MAX_FNAME];
  wchar cext[_MAX_EXT];
  cdrive[0] = 0;
  cdir[0]   = 0;
  cname[0]  = 0;
  cext[0]   = 0;
  _wsplitpath(path, cdrive, cdir, cname, cext);
  drive = cdrive;
  dir   = cdir;
  name  = cname;
  ext   = cext;
#else
  wchars wc(path, str_len(path));
  int    fnp = wc.last_index_of('/');
  if (fnp < 0)
    fnp = wc.last_index_of('\\');
  drive.clear();
  if (fnp >= 0) {
    ++fnp;
    dir      = ustring(wc.start, fnp);
    int extp = wc.last_index_of('.');
    if (extp < 0)
      name = ustring(wc.start + fnp, wc.length - fnp);
    else {
      name = ustring(wc.start + fnp, extp - fnp);
      ext  = ustring(wc.start + extp + 1, wc.length - extp - 1);
    }
  }

#endif
}

ustring real_path(ustring maybe_local) {
  ustring out;
#ifdef WINDOWS
  WCHAR full_path[MAX_PATH];
  if (GetFullPathName(maybe_local.c_str(), MAX_PATH, full_path, NULL))
    out = full_path;
  else
    out = maybe_local;
  out.replace_all('\\', '/');
#else
  char full_path[PATH_MAX] = {0};
  if(realpath(string(maybe_local).c_str(), full_path))
    out = u8::cvt(chars_of(full_path));
  else
    out = maybe_local;
#endif
  return out;
}

/*tstring get_home_dir(void* hinst)
{
#ifdef WIN32
  TCHAR buffer[2048]; buffer[0] = 0;
  GetModuleFileName(HINSTANCE(hinst),buffer, sizeof(buffer));

  tstring drive, dir, name, ext;
  split_path(buffer, drive, dir, name, ext);

  return drive + dir;
#else
#pragma TODO("Not supported yet!")
  return "";
#endif
}*/

ustring get_home_dir(const wchar *relpath, void *hinst) {
#if defined(WINDOWS)
  TCHAR buffer[2048];
  buffer[0] = 0;
  GetModuleFileNameW(HINSTANCE(hinst), buffer, items_in(buffer));

  tstring drive, dir, name, ext;
  split_path(buffer, drive, dir, name, ext);

  if (relpath)
    return tstring::format(TEXT("%s%s%s"), drive.c_str(), dir.c_str(), relpath);
  return drive + dir;
#elif defined(OSX)
  ustring us  = osx_get_exe_path();
  wchars  usc = us().r_head('/');
  ++usc.length;
  ustring rs = usc;
  if (relpath)
    rs += relpath;
  return rs;
#else
  char exe_path[1024] = {0};
  int  len      = readlink("/proc/self/exe", exe_path, items_in(exe_path));
  exe_path[len] = 0;
  char *p       = strrchr(exe_path, '/');
  if (p)
    *(p + 1) = '\0';
  ustring rs = u8::cvt(chars_of(exe_path));
  if (relpath)
    rs += relpath;
  return rs;
#endif
}

ustring get_self_path() {
#if defined(WINDOWS)
  TCHAR buffer[2048];
  buffer[0] = 0;
  GetModuleFileNameW(HINSTANCE(NULL), buffer, items_in(buffer));
  return buffer;
#elif defined(OSX)
  ustring us  = osx_get_exe_path();
  return us;
#else
  char exe_path[1024];
  int  len = readlink("/proc/self/exe", exe_path, items_in(exe_path));
  exe_path[len] = 0;
  ustring rs = u8::cvt(chars_of(exe_path));
  return rs;
#endif
}


unsigned int get_ticks() { return timer::get_ticks(); }

STANDARD_DIR parse_standard_dir(chars s) {
  if (s == CHARS("home") || s == CHARS("USER_HOME"))
    return USER_HOME_DIR;
  else if (s == CHARS("appdata") || s == CHARS("appData") || s == CHARS("USER_APPDATA"))
    return USER_APPDATA_DIR;
  else if (s == CHARS("desktop"))
    return USER_DESKTOP_DIR;
  else if (s == CHARS("applications"))
    return APPLICATIONS_DIR;
  else if (s == CHARS("downloads") || s == CHARS("DOWNLOADS"))
    return USER_DOWNLOADS_DIR;
  else if (s == CHARS("documents") || s == CHARS("USER_DOCUMENTS"))
    return USER_DOCUMENTS_DIR;
  else if (s == CHARS("music"))
    return USER_MUSIC_DIR;
  else if (s == CHARS("videos"))
    return USER_VIDEO_DIR;
  else if (s == CHARS("pictures"))
    return USER_PICTURES_DIR;
  else if (s == CHARS("root"))
    return USER_ROOT_DIR;
  return UNKNOWN_DIR;
}

#ifdef WINDOWS
ustring get_standard_dir(STANDARD_DIR sd) {

  if (sd == USER_DOWNLOADS_DIR) {
    ustring rv;
    if (_SHGetKnownFolderPath)
    {
      PWSTR path = nullptr;
      _SHGetKnownFolderPath()(FOLDERID_Downloads, 0, NULL, &path);
      if (path) {
        rv = path;
        CoTaskMemFree(path);
      }
    }
    return rv;
  }

  static int sysids[] = {
    0, // UNKNOWN_DIR,
    CSIDL_PROGRAM_FILES,//APPLICATIONS_DIR,
    CSIDL_APPDATA, //USER_APPDATA_DIR,
    CSIDL_PROFILE, //USER_HOME_DIR,
    CSIDL_DESKTOPDIRECTORY, //USER_DESKTOP_DIR,
    0, //USER_DOWNLOADS_DIR,
    CSIDL_MYDOCUMENTS, // USER_DOCUMENTS_DIR,
    CSIDL_MYMUSIC, // USER_MUSIC_DIR,
    CSIDL_MYVIDEO, // USER_VIDEO_DIR,
    CSIDL_MYPICTURES, //USER_PICTURES_DIR,
    CSIDL_DRIVES, //USER_ROOT_DIR,
  };

  TCHAR path[MAX_PATH];
  path[0] = 0;

  BOOL r = SHGetSpecialFolderPath(NULL, path, sysids[sd], FALSE);
  assert(r);
  r = r;
  return ustring(path);

}
#endif

/*#ifdef _WINDOWS
// http://blogs.msdn.com/oldnewthing/archive/2004/10/25/247180.aspx
EXTERN_C IMAGE_DOS_HEADER __ImageBase;
#define HINSTANCE_THISCOMPONENT ((HINSTANCE)&__ImageBase)
#endif*/

static char* ext2mimes[] = {
  "323","text/h323",
  "3dm","x-world/x-3dmf",
  "3dmf","x-world/x-3dmf",
  "a","application/octet-stream",
  "aab","application/x-authorware-bin",
  "aam","application/x-authorware-map",
  "aas","application/x-authorware-seg",
  "abc","text/vndabc",
  "acgi","text/html",
  "acx","application/internet-property-stream",
  "afl","video/animaflex",
  "ai","application/postscript",
  "aif","audio/x-aiff,audio/aiff",
  "aifc","audio/x-aiff,audio/aiff",
  "aiff","audio/x-aiff,audio/aiff",
  "aim","application/x-aim",
  "aip","text/x-audiosoft-intra",
  "ani","application/x-navi-animation",
  "aos","application/x-nokia-9000-communicator-add-on-software",
  "aps","application/mime",
  "arc","application/octet-stream",
  "arj","application/arj,application/octet-stream",
  "art","image/x-jg",
  "asc","text/plain",
  "asf","video/x-ms-asf",
  "asm","text/x-asm",
  "asp","text/asp",
  "asr","video/x-ms-asf",
  "asx","application/x-mplayer2,video/x-ms-asf,video/x-ms-asf-plugin",
  "au","audio/basic,audio/x-au",
  "avi","video/x-msvideo,application/x-troff-msvideo,video/avi,video/msvideo",
  "avs","video/avs-video",
  "axs","application/olescript",
  "bas","text/plain",
  "bat","application/bat,application/x-bat,application/x-msdos-program,application/textedit,application/octet-stream",
  "bcpio","application/x-bcpio",
  "bin","application/octet-stream,application/mac-binary,application/macbinary,application/x-binary,application/x-macbinary",
  "bm","image/bmp",
  "bmp","image/bmp,image/x-windows-bmp",
  "boo","application/book",
  "book","application/book",
  "boz","application/x-bzip2",
  "bsh","application/x-bsh",
  "bz","application/x-bzip",
  "bz2","application/x-bzip2",
  "c","text/plain,text/x-c",
  "C","application/x-cplusplus,text/x-c++src,text/plain",
  "c++","application/x-cplusplus,text/x-c++src,text/plain",
  "cab","application/vnd.ms-cab-compressed,application/cab,application/x-compress,application/x-compressed,zz-application/zz-winassoc-cab",
  "cat","application/vndms-pkiseccat",
  "cc","text/plain,text/x-c",
  "ccad","application/clariscad",
  "cco","application/x-cocoa",
  "cdf","application/x-netcdf,application/cdf,application/x-cdf",
  "cer","application/pkix-cert,application/x-x509-ca-cert",
  "cha","application/x-chat",
  "chat","application/x-chat",
  "class","application/octet-stream,application/java,application/java-byte-code,application/x-java-class",
  "clp","application/x-msclip",
  "cmx","image/x-cmx",
  "cod","image/cis-cod",
  "com","application/octet-stream,text/plain",
  "conf","text/plain",
  "cpio","application/x-cpio",
  "cpp","text/x-c",
  "cpt","application/mac-compactpro,application/x-compactpro,application/x-cpt",
  "crd","application/x-mscardfile",
  "crl","application/pkcs-crl,application/pkix-crl",
  "crt","application/pkix-cert,application/x-x509-ca-cert,application/x-x509-user-cert",
  "csh","application/x-csh,text/x-scriptcsh",
  "css","text/css,application/x-pointplus",
  "cxx","text/plain",
  "dcr","application/x-director",
  "deepv","application/x-deepv",
  "def","text/plain",
  "der","application/x-x509-ca-cert",
  "dif","video/x-dv",
  "dir","application/x-director",
  "dl","video/dl,video/x-dl",
  "dll","application/x-msdownload",
  "dms","application/octet-stream",
  "doc","application/msword",
  "docx","application/vnd.openxmlformats-officedocument.wordprocessingml.document",
  "dot","application/msword",
  "dp","application/commonground",
  "drw","application/drafting",
  "dump","application/octet-stream",
  "dv","video/x-dv",
  "dvi","application/x-dvi",
  "dwf","drawing/x-dwf,model/vnddwf",
  "dwg","application/acad,image/vnddwg,image/x-dwg",
  "dxf","application/dxf,image/vnddwg,image/x-dwg",
  "dxr","application/x-director",
  "el","text/x-scriptelisp",
  "elc","application/x-bytecodeelisp,application/x-elc",
  "env","application/x-envoy",
  "eps","application/postscript",
  "es","application/x-esrehber",
  "etx","text/x-setext",
  "evy","application/envoy,application/x-envoy",
  "exe","application/octet-stream",
  "ez","application/andrew-inset",
  "f","text/plain,text/x-fortran",
  "f77","text/x-fortran",
  "f90","text/plain,text/x-fortran",
  "fdf","application/vndfdf",
  "fif","application/fractals,image/fif",
  "fli","video/x-fli,video/fli",
  "flo","image/florian",
  "flr","x-world/x-vrml",
  "flx","text/vndfmiflexstor",
  "fmf","video/x-atomic3d-feature",
  "for","text/plain,text/x-fortran",
  "fpx","image/vndfpx,image/vndnet-fpx",
  "frl","application/freeloader",
  "funk","audio/make",
  "g","text/plain",
  "g3","image/g3fax",
  "gif","image/gif",
  "gl","video/gl,video/x-gl",
  "gsd","audio/x-gsm",
  "gsm","audio/x-gsm",
  "gsp","application/x-gsp",
  "gss","application/x-gss",
  "gtar","application/x-gtar",
  "gz","application/x-gzip,application/x-compressed",
  "gzip","application/x-gzip,multipart/x-gzip",
  "h","text/plain,text/x-h",
  "h++","application/x-cplusplus,text/x-c++src,text/plain",
  "H","application/x-cplusplus,text/x-c++src,text/plain",
  "hdf","application/x-hdf",
  "help","application/x-helpfile",
  "hgl","application/vndhp-hpgl",
  "hh","text/plain,text/x-h",
  "hlb","text/x-script",
  "hlp","application/hlp,application/x-helpfile,application/x-winhelp,application/winhlp",
  "hpg","application/vndhp-hpgl",
  "hpgl","application/vndhp-hpgl",
  "hpp","application/x-cplusplus,text/x-c++src,text/plain",
  "hqx","application/mac-binhex40,application/binhex,application/binhex4,application/mac-binhex,application/x-binhex40,application/x-mac-binhex40",
  "hta","application/hta",
  "htc","text/x-component",
  "htm","text/html",
  "html","text/html",
  "htmls","text/html",
  "htt","text/webviewhtml",
  "htx","text/html",
  "ice","x-conference/x-cooltalk",
  "ico","image/x-icon",
  "idc","text/plain",
  "ief","image/ief",
  "iefs","image/ief",
  "iges","model/iges,application/iges",
  "igs","model/iges,application/iges",
  "iii","application/x-iphone",
  "ima","application/x-ima",
  "imap","application/x-httpd-imap",
  "ini","text/plain,application/textedit,zz-application/zz-winassoc-ini",
  "inf","application/inf",
  "ins","application/x-internett-signup,application/x-internet-signup",
  "ip","application/x-ip2",
  "ips","application/x-ipscript",
  "ipx","application/x-ipix",
  "isp","application/x-internet-signup",
  "isu","video/x-isvideo",
  "it","audio/it",
  "iv","application/x-inventor",
  "ivr","i-world/i-vrml",
  "ivy","application/x-livescreen",
  "jam","audio/x-jam",
  "jar","application/java-archive",
  "jav","text/plain,text/x-java-source",
  "java","text/plain,text/x-java-source",
  "jcm","application/x-java-commerce",
  "jfif","image/jpeg,image/pjpeg,image/pipeg",
  "jfif-tbnl","image/jpeg",
  "jpe","image/jpeg,image/pjpeg",
  "jpeg","image/jpeg,image/pjpeg",
  "jpg","image/jpeg,image/pjpeg",
  "jps","image/x-jps",
  "js","application/x-javascript",
  "json", "application/json",
  "jsp","text/html",
  "jut","image/jutvision",
  "kar","audio/midi,music/x-karaoke",
  "ksh","application/x-ksh,text/x-scriptksh",
  "la","audio/nspaudio,audio/x-nspaudio",
  "lam","audio/x-liveaudio",
  "latex","application/x-latex",
  "lha","application/octet-stream,application/lha,application/x-lha",
  "lhx","application/octet-stream",
  "list","text/plain",
  "lma","audio/nspaudio,audio/x-nspaudio",
  "log","text/plain",
  "lnk","application/x-ms-shortcut",
  "lsf","video/x-la-asf",
  "lsp","application/x-lisp,text/x-scriptlisp",
  "lst","text/plain",
  "lsx","text/x-la-asf,video/x-la-asf",
  "ltx","application/x-latex",
  "lzh","application/octet-stream,application/x-lzh",
  "lzx","application/lzx,application/octet-stream,application/x-lzx",
  "m","text/plain,text/x-m",
  "m13","application/x-msmediaview",
  "m14","application/x-msmediaview",
  "m1v","video/mpeg",
  "m2a","audio/mpeg",
  "m2v","video/mpeg",
  "m3u","audio/x-mpequrl,audio/x-mpegurl",
  "man","application/x-troff-man",
  "map","application/x-navimap",
  "mar","text/plain",
  "mbd","application/mbedlet",
  "mc$","application/x-magic-cap-package-10",
  "mcd","application/mcad,application/x-mathcad",
  "mcf","image/vasa,text/mcf",
  "mcp","application/netmc",
  "mdb","application/x-msaccess",
  "me","application/x-troff-me",
  "mesh","model/mesh",
  "mht","message/rfc822",
  "mhtml","message/rfc822",
  "mid","audio/midi,application/x-midi,audio/x-mid,audio/x-midi,music/crescendo,x-music/x-midi,audio/mid",
  "midi","audio/midi,application/x-midi,audio/x-mid,audio/x-midi,music/crescendo,x-music/x-midi",
  "mif","application/vndmif,application/x-frame,application/x-mif",
  "mime","www/mime,message/rfc822",
  "mjf","audio/x-vndaudioexplosionmjuicemediafile",
  "mjpg","video/x-motion-jpeg",
  "mm","application/base64,application/x-meme",
  "mme","application/base64",
  "mny","application/x-msmoney",
  "mod","audio/mod,audio/x-mod",
  "moov","video/quicktime",
  "mov","video/quicktime",
  "movie","video/x-sgi-movie",
  "mp2","audio/mpeg,audio/x-mpeg,video/mpeg,video/x-mpeg,video/x-mpeq2a",
  "mp3","audio/mpeg,audio/mpeg3,audio/x-mpeg-3,video/mpeg,video/x-mpeg",
  "mpa","audio/mpeg,video/mpeg",
  "mpc","application/x-project",
  "mpe","video/mpeg",
  "mpeg","video/mpeg",
  "mpg","video/mpeg,audio/mpeg",
  "mpga","audio/mpeg",
  "mpp","application/vndms-project",
  "mpt","application/x-project",
  "mpv","application/x-project",
  "mpv2","video/mpeg",
  "mpx","application/x-project",
  "mrc","application/marc",
  "ms","application/x-troff-ms",
  "msh","model/mesh",
  "mv","video/x-sgi-movie",
  "mvb","application/x-msmediaview",
  "my","audio/make",
  "mzz","application/x-vndaudioexplosionmzz",
  "nap","image/naplps",
  "naplps","image/naplps",
  "nc","application/x-netcdf",
  "ncm","application/vndnokiaconfiguration-message",
  "nif","image/x-niff",
  "niff","image/x-niff",
  "nix","application/x-mix-transfer",
  "nsc","application/x-conference",
  "nvd","application/x-navidoc",
  "nws","message/rfc822",
  "o","application/octet-stream",
  "oda","application/oda",
  "odb","application/vnd.oasis.opendocument.database",
  "odc","application/vnd.oasis.opendocument.chart",
  "odf","application/vnd.oasis.opendocument.formula",
  "odg","application/vnd.oasis.opendocument.graphics",
  "odi","application/vnd.oasis.opendocument.image",
  "odm","application/vnd.oasis.opendocument.text-master",
  "odp","application/vnd.oasis.opendocument.presentation",
  "ods","application/vnd.oasis.opendocument.spreadsheet",
  "odt","application/vnd.oasis.opendocument.text",
  "omc","application/x-omc",
  "omcd","application/x-omcdatamaker",
  "omcr","application/x-omcregerator",
  "otg","application/vnd.oasis.opendocument.graphics-template",
  "oth","application/vnd.oasis.opendocument.text-web",
  "otp","application/vnd.oasis.opendocument.presentation-template",
  "ots","application/vnd.oasis.opendocument.spreadsheet-template",
  "ott","application/vnd.oasis.opendocument.text-template",
  "oxt","application/vnd.openofficeorg.extension",
  "p","text/x-pascal",
  "p10","application/pkcs10,application/x-pkcs10",
  "p12","application/pkcs-12,application/x-pkcs12",
  "p7a","application/x-pkcs7-signature",
  "p7b","application/x-pkcs7-certificates",
  "p7c","application/pkcs7-mime,application/x-pkcs7-mime",
  "p7m","application/pkcs7-mime,application/x-pkcs7-mime",
  "p7r","application/x-pkcs7-certreqresp",
  "p7s","application/pkcs7-signature,application/x-pkcs7-signature",
  "part","application/pro_eng",
  "pas","text/pascal",
  "pbm","image/x-portable-bitmap",
  "pcl","application/vndhp-pcl,application/x-pcl",
  "pct","image/x-pict",
  "pcx","image/x-pcx",
  "pdb","chemical/x-pdb",
  "pdf","application/pdf",
  "pfunk","audio/make,audio/makemyfunk",
  "pfx","application/x-pkcs12",
  "pgm","image/x-portable-graymap,image/x-portable-greymap",
  "pgn","application/x-chess-pgn",
  "pic","image/pict",
  "pict","image/pict",
  "pkg","application/x-newton-compatible-pkg",
  "pko","application/vndms-pkipko,application/yndms-pkipko",
  "pl","text/plain,text/x-scriptperl",
  "plx","application/x-pixclscript",
  "pm","image/x-xpixmap,text/x-scriptperl-module",
  "pm4","application/x-pagemaker",
  "pm5","application/x-pagemaker",
  "pma","application/x-perfmon",
  "pmc","application/x-perfmon",
  "pml","application/x-perfmon",
  "pmr","application/x-perfmon",
  "pmw","application/x-perfmon",
  "png","image/png",
  "pnm","image/x-portable-anymap,application/x-portable-anymap",
  "pot","application/mspowerpoint,application/vndms-powerpoint",
  "pov","model/x-pov",
  "ppa","application/vndms-powerpoint",
  "ppm","image/x-portable-pixmap",
  "pps","application/mspowerpoint,application/vndms-powerpoint",
  "ppt","application/mspowerpoint,application/powerpoint,application/vndms-powerpoint,application/x-mspowerpoint",
  "ppz","application/mspowerpoint",
  "pre","application/x-freelance",
  "prf","application/pics-rules",
  "properties","text/plain",
  "prt","application/pro_eng",
  "ps","application/postscript",
  "psd","application/octet-stream",
  "pub","application/x-mspublisher",
  "pvu","paleovu/x-pv",
  "pwz","application/vndms-powerpoint",
  "py","text/x-scriptphyton",
  "pyc","applicaiton/x-bytecodepython",
  "qcp","audio/vndqcelp",
  "qd3","x-world/x-3dmf",
  "qd3d","x-world/x-3dmf",
  "qif","image/x-quicktime",
  "qt","video/quicktime",
  "qtc","video/x-qtc",
  "qti","image/x-quicktime",
  "qtif","image/x-quicktime",
  "ra","audio/x-realaudio,audio/x-pn-realaudio,audio/x-pn-realaudio-plugin",
  "ram","audio/x-pn-realaudio",
  "ras","image/cmu-raster,application/x-cmu-raster,image/x-cmu-raster",
  "rast","image/cmu-raster",
  "rexx","text/x-scriptrexx",
  "rf","image/vndrn-realflash",
  "rgb","image/x-rgb",
  "rm","audio/x-pn-realaudio,application/vndrn-realmedia",
  "rmi","audio/mid",
  "rmm","audio/x-pn-realaudio",
  "rmp","audio/x-pn-realaudio,audio/x-pn-realaudio-plugin",
  "rng","application/ringing-tones,application/vndnokiaringing-tone",
  "rnx","application/vndrn-realplayer",
  "roff","application/x-troff",
  "rp","image/vndrn-realpix",
  "rpm","audio/x-pn-realaudio-plugin",
  "rt","text/richtext,text/vndrn-realtext",
  "rtf","text/rtf,application/rtf,application/x-rtf,text/richtext",
  "rtx","text/richtext,application/rtf",
  "rv","video/vndrn-realvideo",
  "s","text/x-asm",
  "s3m","audio/s3m",
  "saveme","application/octet-stream",
  "sbk","application/x-tbook",
  "scd","application/x-msschedule",
  "scm","application/x-lotusscreencam,text/x-scriptguile,text/x-scriptscheme,video/x-scm",
  "sct","text/scriptlet",
  "sda","application/vnd.stardivision.draw,application/x-stardraw",
  "sdc","application/vnd.stardivision.calc,application/x-starcalc",
  "sdd","application/vnd.stardivision.impress,application/x-starimpress",
  "sdm","application/vnd.stardivision.mail",
  "sdml","text/plain",
  "sdp","application/vnd.stardivision.impress-packed,application/sdp,application/x-sdp",
  "sdr","application/sounder",
  "sds","application/vnd.stardivision.chart,application/x-starchart",
  "sdw","application/vnd.stardivision.writer,application/x-starwriter",
  "sea","application/sea,application/x-sea",
  "set","application/set",
  "setpay","application/set-payment-initiation",
  "setreg","application/set-registration-initiation",
  "sgl","application/vnd.stardivision.writer-global",
  "sgm","text/sgml,text/x-sgml",
  "sgml","text/sgml,text/x-sgml",
  "sh","application/x-sh,application/x-bsh,application/x-shar,text/x-scriptsh",
  "shar","application/x-shar,application/x-bsh",
  "shtml","text/html,text/x-server-parsed-html",
  "sid","audio/x-psid",
  "silo","model/mesh",
  "sit","application/x-stuffit,application/x-sit",
  "skd","application/x-koan",
  "skm","application/x-koan",
  "skp","application/x-koan",
  "skt","application/x-koan",
  "sl","application/x-seelogo",
  "smf","application/vnd.stardivision.math,application/x-starmath",
  "smi","application/smil",
  "smil","application/smil",
  "snd","audio/basic,audio/x-adpcm",
  "sol","application/solids",
  "spc","application/x-pkcs7-certificates,text/x-speech",
  "spl","application/x-futuresplash,application/futuresplash",
  "spr","application/x-sprite",
  "sprite","application/x-sprite",
  "src","application/x-wais-source",
  "ssi","text/x-server-parsed-html",
  "ssm","application/streamingmedia",
  "sst","application/vndms-pkicertstore",
  "stc","application/vnd.sun.xml.calc.template",
  "std","application/vnd.sun.xml.draw.template",
  "step","application/STEP,application/step",
  "sti","application/vnd.sun.xml.impress.template",
  "stl","application/SLA,application/sla,application/vndms-pkistl,application/x-navistyle",
  "stm","text/html",
  "stp","application/STEP,application/step",
  "stw","application/vnd.sun.xml.writer.template",
  "sv4cpio","application/x-sv4cpio",
  "sv4crc","application/x-sv4crc",
  "svf","image/vnddwg,image/x-dwg",
  "svg","image/svg+xml",
  "svr","application/x-world,x-world/x-svr",
  "swf","application/x-shockwave-flash",
  "sxc","application/vnd.sun.xml.calc",
  "sxd","application/vnd.sun.xml.draw",
  "sxg","application/vnd.sun.xml.writer.global",
  "sxi","application/vnd.sun.xml.impress",
  "sxm","application/vnd.sun.xml.math",
  "sxw","application/vnd.sun.xml.writer",
  "sys","video/x-mpeg-system",
  "t","application/x-troff",
  "talk","text/x-speech",
  "tar","application/x-tar",
  "tar.gz","application/x-gzip",
  "tbk","application/toolbook,application/x-tbook",
  "tcl","application/x-tcl,text/x-scripttcl",
  "tcsh","text/x-scripttcsh",
  "tex","application/x-tex",
  "texi","application/x-texinfo",
  "texinfo","application/x-texinfo",
  "text","application/plain,text/plain",
  "tgz","application/gnutar,application/x-compressed",
  "tif","image/tiff,image/x-tiff",
  "tiff","image/tiff,image/x-tiff",
  "tis", "text/tiscript",
  "tr","application/x-troff",
  "trm","application/x-msterminal",
  "tsi","audio/TSP-audio,audio/tsp-audio",
  "tsp","application/dsptype,audio/tsplayer",
  "tsv","text/tab-separated-values",
  "turbot","image/florian",
  "txt","text/plain",
  "uil","text/x-uil",
  "uls","text/iuls",
  "uni","text/uri-list",
  "unis","text/uri-list",
  "unv","application/i-deas",
  "uri","text/uri-list",
  "url","wwwserver/redirection,application/internet-shortcut,application/x-url,message/external-body,text/url,text/x-url",
  "uris","text/uri-list",
  "ustar","application/x-ustar,multipart/x-ustar",
  "uu","application/octet-stream,text/x-uuencode",
  "uue","text/x-uuencode",
  "vcd","application/x-cdlink",
  "vcf","text/x-vcard",
  "vcs","text/x-vcalendar",
  "vda","application/vda",
  "vdo","video/vdo",
  "vew","application/groupwise",
  "viv","video/vndvivo,video/vivo",
  "vivo","video/vndvivo,video/vivo",
  "vmd","application/vocaltec-media-desc",
  "vmf","application/vocaltec-media-file",
  "voc","audio/voc,audio/x-voc",
  "vos","video/vosaic",
  "vox","audio/voxware",
  "vqe","audio/x-twinvq-plugin",
  "vqf","audio/x-twinvq",
  "vql","audio/x-twinvq-plugin",
  "vrml","model/vrml,application/x-vrml,x-world/x-vrml",
  "vrt","x-world/x-vrt",
  "vsd","application/x-visio",
  "vst","application/x-visio",
  "vsw","application/x-visio",
  "w60","application/wordperfect60",
  "w61","application/wordperfect61",
  "w6w","application/msword",
  "wav","audio/x-wav,audio/wav",
  "wax","audio/x-ms-wax",
  "wb1","application/x-qpro",
  "wbmp","image/vndwapwbmp",
  "wcm","application/vndms-works",
  "wdb","application/vndms-works",
  "web","application/vndxara",
  "wiz","application/msword",
  "wk1","application/x-123",
  "wks","application/vndms-works",
  "wm","video/x-ms-wm",
  "wmf","windows/metafile,application/x-msmetafile",
  "wma","audio/x-ms-wma",
  "wmd","application/x-ms-wmd",
  "wml","text/vndwapwml",
  "wmlc","application/vndwapwmlc",
  "wmls","text/vndwapwmlscript",
  "wmlsc","application/vndwapwmlscriptc",
  "wmv","audio/x-ms-wmv",
  "wmx","video/x-ms-wmx",
  "wmz","application/x-ms-wmz",
  "word","application/msword",
  "wp","application/wordperfect",
  "wp5","application/wordperfect,application/wordperfect60",
  "wp6","application/wordperfect",
  "wpd","application/wordperfect,application/x-wpwin",
  "wps","application/vndms-works",
  "wq1","application/x-lotus",
  "wri","application/mswrite,application/x-wri,application/x-mswrite",
  "wrl","model/vrml,application/x-world,x-world/x-vrml",
  "wrz","model/vrml,x-world/x-vrml",
  "wsc","text/scriplet",
  "wsrc","application/x-wais-source",
  "wtk","application/x-wintalk",
  "wvx","video/x-ms-wvx",
  "x-png","image/png",
  "xaf","x-world/x-vrml",
  "xbm","image/x-xbitmap,image/x-xbm,image/xbm",
  "xdr","video/x-amt-demorun",
  "xgz","xgl/drawing",
  "xhtml","application/xhtml+xml,text/html",
  "xif","image/vndxiff",
  "xl","application/excel",
  "xla","application/excel,application/x-excel,application/x-msexcel,application/vndms-excel",
  "xlb","application/excel,application/vndms-excel,application/x-excel",
  "xlc","application/vndms-excel,application/excel,application/x-excel",
  "xld","application/excel,application/x-excel",
  "xlk","application/excel,application/x-excel",
  "xll","application/vndms-excel,application/excel,application/x-excel",
  "xlm","application/vndms-excel,application/excel,application/x-excel",
  "xls","application/vndms-excel,application/excel,application/x-excel,application/x-msexcel",
  "xlsx","application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
  "xlt","application/excel,application/x-excel,application/vndms-excel",
  "xlv","application/excel,application/x-excel",
  "xlw","application/vndms-excel,application/excel,application/x-excel,application/x-msexcel",
  "xm","audio/xm",
  "xml","text/xml,application/xml",
  "xmz","xgl/movie",
  "xof","x-world/x-vrml",
  "xpix","application/x-vndls-xpix",
  "xpm","image/x-xpixmap,image/xpm",
  "xsd","text/xml",
  "xsl","text/xml",
  "xsr","video/x-amt-showrun",
  "xwd","image/x-xwindowdump,image/x-xwd",
  "xyz","chemical/x-pdb",
  "z","application/x-compress,application/x-compressed",
  "zip","application/zip,application/x-compressed,application/x-zip-compressed,multipart/x-zip",
  "zoo","application/octet-stream",
  "zsh","text/x-scriptzsh",
 };

 array<chars> ext_for_mime(chars mimetype)
 {
   string mt = mimetype;
   array<chars> exts;
   for( int n = 0; n < items_in(ext2mimes); n += 2 )
   {
     const char* ext = ext2mimes[n];
     chars mimes = chars_of(ext2mimes[n+1]);
     chars mime;
     atokens toks(mimes,CHARS(","));
     while(toks.next(mime))
       if( mime.like(mt) )
         exts.push( chars_of(ext) );
   }
   return exts;
 }

 chars mime_for_ext(chars ext)
 {
   for (int n = 0; n < items_in(ext2mimes); n += 2)
   {
     chars t_ext = chars_of(ext2mimes[n]);
     if (t_ext != ext) continue;
     chars mimes = chars_of(ext2mimes[n + 1]);
     return mimes.head(',');
   }
   return chars();
 }


#if defined(WINDOWS)
#pragma comment(lib, "Urlmon.lib")
#endif

string guess_mime_type(string filename, bytes data) {
#if defined(WINDOWS) && false
  if (data.length > 256)
    data.length = 256;
  array<byte> test = data;
  wchar *     mime = 0;
  ustring proposed = ustring(mime_for_ext(filename().r_tail('.')));
  HRESULT hr = FindMimeFromData(0, ustring(filename), test.head(), (DWORD)test.length(),
                                proposed.c_str() /*pwzMimeProposed*/, 0 /*dwMimeFlags*/,
                                &mime /*ppwzMimeOut*/, 0);
  return (SUCCEEDED(hr) && mime) ? string(mime) : string();
#else
  return mime_for_ext(filename().r_tail('.'));
#endif
}

#include "html_encodings_ph.h"

int get_lang_id(const string &name) {
  if (!name.length())
    return 0;

  string lcasename = name;
  lcasename.to_lower();
  int id = 0;

  const html_encoding_def *hed =
      html_encodings::find_def(lcasename, unsigned(lcasename.length()));
  if (hed != 0)
    id = hed->value;
  else
    assert(false);

  return id;
}

bool decode_bytes(bytes bs, ustring &str, const string &encoding) {
#ifdef WINDOWS
  int cp = get_lang_id(encoding);
  if (cp == 0)
    return false;
  if (cp == 65001)
    str = u8::cvt(bs);
  else {
    int nu = MultiByteToWideChar(cp, 0, (LPCSTR)bs.start, int(bs.length), 0, 0);
    ustring r(' ', nu);
    MultiByteToWideChar(cp, 0, (LPCSTR)bs.start, int(bs.length), r.buffer(),
                        r.size());
    str = r;
  }
  return true;
#else
  str = u8::cvt(bs);
  return true;
#endif
}

bool encode_bytes(wchars str, array<byte> &data, const string &encoding) {
#ifdef WINDOWS
  int cp = get_lang_id(encoding);
  if (cp == 0)
    return false;
  if (cp == 65001)
    u8::from_utf16(str, data);
  else {
    int n = WideCharToMultiByte(cp, 0, str.start, str.size(), 0, 0, 0, 0);
    data.size(n);
    WideCharToMultiByte(cp, 0, str.start, str.size(), (LPSTR)data.begin(),
                        data.size(), 0, 0);
  }
  return true;
#else
  u8::from_utf16(str, data);
  return true;
#endif
}

#ifdef WINDOWS

#ifndef LOCALE_NAME_USER_DEFAULT
#define LOCALE_NAME_USER_DEFAULT NULL
#endif

DLOADV(_LocaleNameToLCID, LocaleNameToLCID, kernel32.dll,
       LCID(WINAPI *)(LPWSTR lpLocaleNam, int cchLocaleName));

DWORD locale_name_2_LCID(const ustring &name) {
  if (_LocaleNameToLCID)
    return _LocaleNameToLCID()((LPWSTR)name.c_str(), LOCALE_NAME_USER_DEFAULT);

  LCID lcid = LOCALE_USER_DEFAULT;

  com::asset<IMultiLanguage> multi_lang;

  HRESULT hr = ::CoCreateInstance(
      CLSID_CMultiLanguage, 0, CLSCTX_ALL, IID_IMultiLanguage,
      reinterpret_cast<void **>(multi_lang.target()));

  if (SUCCEEDED(hr)) {
    BSTR bstr_name = SysAllocStringLen(name.c_str(), (UINT)name.length());
    multi_lang->GetLcidFromRfc1766(&lcid, bstr_name);
    SysFreeString(bstr_name);
  }

  return lcid;
}

int compare_strings(wchars s1, wchars s2, bool case_insensitive, wchars lang) {
  ustring locale = lang;
  LCID    lcid   = locale_name_2_LCID(
      locale.length() ? locale.c_str() : LPCWSTR(LOCALE_NAME_USER_DEFAULT));
  DWORD flags = case_insensitive ? NORM_IGNORECASE : 0;
  return CompareStringW(lcid, flags, s1.start, int(s1.length), s2.start,
                        int(s2.length)) -
         2;
}

int compare_strings(chars s1, chars s2, bool case_insensitive, chars lang) {
  ustring locale = lang;
  LCID    lcid   = locale_name_2_LCID(
      locale.length() ? locale.c_str() : LPCWSTR(LOCALE_NAME_USER_DEFAULT));
  DWORD flags = case_insensitive ? NORM_IGNORECASE : 0;
  return CompareStringA(lcid, flags, s1.start, int(s1.length), s2.start,
                        int(s2.length)) -
         2;
}

#else

int compare_strings(wchars s1, wchars s2, bool case_insensitive, wchars lang) {
  if (case_insensitive)
    for (size_t i = 0; i < min(s1.length, s2.length); ++i) {
      int c1 = (int)uctolower(s1[i]);
      int c2 = (int)uctolower(s2[i]);
      if (c1 != c2)
        return c1 - c2;
    }
  else
    for (size_t i = 0; i < min(s1.length, s2.length); ++i) {
      int c1 = (int)(s1[i]);
      int c2 = (int)(s2[i]);
      if (c1 != c2)
        return c1 - c2;
    }
  return s1.size() - s2.size();
}

int compare_strings(chars s1, chars s2, bool case_insensitive, chars lang) {
  if (case_insensitive)
    for (size_t i = 0; i < min(s1.length, s2.length); ++i) {
      int c1 = (int)uctolower(s1[i]);
      int c2 = (int)uctolower(s2[i]);
      if (c1 != c2)
        return c1 - c2;
    }
  else
    for (size_t i = 0; i < min(s1.length, s2.length); ++i) {
      int c1 = (int)(s1[i]);
      int c2 = (int)(s2[i]);
      if (c1 != c2)
        return c1 - c2;
    }
  return s1.size() - s2.size();
}

#endif

ustring format_number_str(const ustring &locale, const ustring &str,
                          const number_format_def *pf) {
#ifdef WINDOWS // for win32 and winCE
  wchar buf[256] = {0};
  uint  length   = 0;

  if (!pf) {
    length = GetNumberFormat(
        locale_name_2_LCID(locale.length()
                               ? locale.c_str()
                               : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
        LOCALE_NOUSEROVERRIDE,                                       // options
        str,          // input number string
        NULL,         // formatting information
        buf,          // output buffer
        items_in(buf) // size of output buffer
    );
  } else {
    NUMBERFMTW nf;
    nf.Grouping    = pf->grouping.val(3);
    nf.NumDigits   = pf->num_digits.val(2);
    nf.LeadingZero = pf->leading_zero.val(0);
    nf.lpDecimalSep =
        pf->decimal_sep.is_defined() ? pf->decimal_sep.c_str() : L".";
    nf.lpThousandSep =
        pf->group_sep.is_defined() ? pf->group_sep.c_str() : L",";
    nf.NegativeOrder = pf->negative_order.val(1);

    length = GetNumberFormat(
        locale_name_2_LCID(locale.length()
                               ? locale.c_str()
                               : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
        0,                                                           // options
        str,          // input number string
        &nf,          // formatting information
        buf,          // output buffer
        items_in(buf) // size of output buffer
    );
  }
  if (length)
    return ustring(buf, length - 1);
#else
#pragma TODO("implement this!")
#endif
  return str;
}

ustring format_currency_str(const ustring &locale, const ustring &str,
                            const number_format_def *pf) {

#if defined(WINDOWS) // for win32 and winCE

  wchar buf[256] = {0};
  uint  length   = 0;

  if (!pf) {
    length = GetCurrencyFormat(
        locale_name_2_LCID(locale.length()
                               ? locale.c_str()
                               : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
        LOCALE_NOUSEROVERRIDE,                                       // options
        str,          // input number string
        NULL,         // formatting information
        buf,          // output buffer
        items_in(buf) // size of output buffer
    );
  } else {
    CURRENCYFMT nf;
    nf.Grouping    = pf->grouping.val(3);
    nf.NumDigits   = pf->num_digits.val(2);
    nf.LeadingZero = pf->leading_zero.val(1);
    nf.lpDecimalSep =
        pf->decimal_sep.is_defined() ? pf->decimal_sep.c_str() : L".";
    nf.lpThousandSep =
        pf->group_sep.is_defined() ? pf->group_sep.c_str() : L",";
    nf.NegativeOrder    = pf->negative_order.val(0);
    nf.PositiveOrder    = pf->positive_order.val(0);
    nf.lpCurrencySymbol = (LPWSTR)pf->currency_symbol.c_str();

    length = GetCurrencyFormat(
        locale_name_2_LCID(locale.length()
                               ? locale.c_str()
                               : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
        0,                                                           // options
        str,          // input number string
        &nf,          // formatting information
        buf,          // output buffer
        items_in(buf) // size of output buffer
    );
  }
  if (length)
    return ustring(buf, length - 1);
  return str;
#elif defined(ANDROID) || defined(WINDOWLESS)
  return str;
#else
  char cbuf[256];
  strfmon(cbuf, items_in(cbuf), "%n", wtof(str));
  return cbuf;
#endif
}

ustring format_date(const ustring &locale, date_time dt, const ustring &fmt) {
  wchar buf[256] = { 0 };

  enum { SHORT_FMT, LONG_FMT, CUSTOM_FMT} format = SHORT_FMT;
  if (fmt == WCHARS("short"))
    format = SHORT_FMT;
  else if (fmt == WCHARS("long"))
    format = LONG_FMT;
  else if (fmt.length())
    format = CUSTOM_FMT;

  ustring format_string;

#ifdef WINDOWS // for win32 and winCE
  SYSTEMTIME st = dt.ms_systemtime();
  DWORD flags = 0;
  switch (format) {
  case SHORT_FMT: flags = DATE_SHORTDATE; break;
  case LONG_FMT: flags = DATE_LONGDATE; break;
  case CUSTOM_FMT: format_string = fmt; break;
  }
  uint length = length = GetDateFormat(
    locale_name_2_LCID(locale.length()
      ? locale.c_str()
      : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
    flags,   // options
    &st, // input date
    format_string.length() ? format_string.c_str() : nullptr,
    buf,          // output buffer
    items_in(buf) // size of output buffer
  );
  if (length)
    return ustring(buf, length - 1);

#elif defined(OSX)

    CF::ref<CFStringRef> cflocname = cfstr(locale);
    CF::ref<CFLocaleRef> cfloc = CFLocaleCreate(kCFAllocatorDefault, cflocname);

    if( format == CUSTOM_FMT )
    {
        CF::ref<CFStringRef> dateAsString =
           CFDateFormatterCreateDateFormatFromTemplate(kCFAllocatorDefault, cfstr(format_string), 0, cfloc);
        return cvt(dateAsString);
    }
    else {
      CF::ref<CFDateFormatterRef> formatter = CFDateFormatterCreate(kCFAllocatorDefault, cfloc,
                                                                    format == SHORT_FMT ? kCFDateFormatterShortStyle : kCFDateFormatterMediumStyle,
                                                                    kCFDateFormatterNoStyle);

      CF::ref<CFTimeZoneRef> tz = CFTimeZoneCreateWithTimeIntervalFromGMT(kCFAllocatorDefault, 0);
      CFDateFormatterSetProperty(formatter,kCFDateFormatterTimeZone,tz);

      date_time start(2001,1,1);
      double sec = (dt.absolute_millis() - start.absolute_millis()) / 1000.0;

      CF::ref<CFStringRef> dateAsString = CFDateFormatterCreateStringWithAbsoluteTime(kCFAllocatorDefault, formatter, sec);

      return cvt(dateAsString);
    }

#else

  return dt.format(W("%x"));

#endif
  return buf;
}

ustring format_time(const ustring &locale, date_time dt, const ustring &fmt) {
  wchar buf[256] = {0};
  uint  length   = 0;

  enum { SHORT_FMT, LONG_FMT, CUSTOM_FMT } format = SHORT_FMT;
  if (fmt == WCHARS("short"))
    format = SHORT_FMT;
  else if (fmt == WCHARS("long"))
    format = LONG_FMT;
  else if (fmt.length())
    format = CUSTOM_FMT;

  ustring format_string;

#ifdef WINDOWS // for win32 and winCE
  // struct tm syst;
  // dt.systemtime ( syst );

  DWORD flags = 0;

  switch (format) {
  case SHORT_FMT: flags = TIME_NOSECONDS; break;
  case LONG_FMT: flags = 0; break;
  case CUSTOM_FMT: format_string = fmt; break;
  }

  SYSTEMTIME st = dt.ms_systemtime();
  length        = GetTimeFormat(
      locale_name_2_LCID(locale.length()
                             ? locale.c_str()
                             : LPCWSTR(LOCALE_NAME_USER_DEFAULT)), // locale
      flags,   // options
      &st, // input date
      format_string.length() ? format_string.c_str() : nullptr,
      buf,          // output buffer
      items_in(buf) // size of output buffer
  );
  if (length)
    return ustring(buf, length - 1);
#elif defined(OSX)

    CF::ref<CFStringRef> cflocname = cfstr(locale);
    CF::ref<CFLocaleRef> cfloc = CFLocaleCreate(kCFAllocatorDefault, cflocname);

    if( format == CUSTOM_FMT )
    {
        CF::ref<CFStringRef> dateAsString =
        CFDateFormatterCreateDateFormatFromTemplate(kCFAllocatorDefault, cfstr(format_string), 0, cfloc);
        return cvt(dateAsString);
    }
    else {
        CF::ref<CFDateFormatterRef> formatter = CFDateFormatterCreate(kCFAllocatorDefault, cfloc,
                                                                      kCFDateFormatterNoStyle,
                                                                      (format == SHORT_FMT ? kCFDateFormatterShortStyle : kCFDateFormatterMediumStyle));
        CF::ref<CFTimeZoneRef> tz = CFTimeZoneCreateWithTimeIntervalFromGMT(kCFAllocatorDefault, 0);
        CFDateFormatterSetProperty(formatter,kCFDateFormatterTimeZone,tz);

        date_time start(2001,1,1,dt.hours(),dt.minutes(),dt.seconds());
        double sec = start.absolute_millis() / 1000.0;

        CF::ref<CFStringRef> dateAsString = CFDateFormatterCreateStringWithAbsoluteTime(kCFAllocatorDefault, formatter, sec);

        return cvt(dateAsString);
    }

#else
  #pragma TODO("implement this!")
  return dt.format(W("%X"));
#endif
  return buf;
}

bool parse_number_format_def(wchars format, number_format_def &f) {

  value fv = xjson::parse(format, true);

  if (!fv.is_map())
    return false;
  f.currency_symbol = fv.get_prop("currency").get(W("$"));
  f.decimal_sep     = fv.get_prop("decimal-sep").get(W("."));
  f.grouping        = fv.get_prop("grouping").get(3);
  f.group_sep       = fv.get_prop("grouping-sep").get(W(","));
  f.leading_zero    = fv.get_prop("leading-zero").get(false);
  f.negative_order  = fv.get_prop("negative-order").get(0);
  f.positive_order  = fv.get_prop("positive-order").get(0);
  f.num_digits      = fv.get_prop("precision").get(2);
  return true;
}

#ifdef NATIVE_WINDOWS_BACKEND
DLOADV(_GetComputerNameEx, GetComputerNameEx, kernel32.dll,
       BOOL(WINAPI *)(COMPUTER_NAME_FORMAT NameType, LPTSTR lpBuffer,
                      LPDWORD lpnSize));
#endif

namespace environment {

  ustring _lang;
  ustring _country;

  bool set_used_lang_country(const ustring &lang, const ustring &country) {
    _lang    = lang;
    _country = country;
    return true;
  }

  bool used_lang_country(ustring &lang, ustring &country, bool for_the_user) {
    critical_section _(tool::lock);
    if (_lang.is_undefined())
      get_lang_country(_lang, _country, for_the_user);
    lang = _lang;
    country = _country;
    return lang.is_defined();
  }

  const char *platform_name() {
    #if defined(WINDOWS)
      return "Windows";
    #elif defined(OSX)
      return "OSX";
    #elif defined(LINUX)
      return "Linux";
    #elif defined(ANDROID)
      return "Android";
    #elif defined(WINDOWLESS)
      return "IoT";
    #else
      return "undefined";
    #endif
  }

#ifdef NATIVE_WINDOWS_BACKEND
ustring user_name() {
  wchar buffer[1024];
  ULONG size = 1023;
#ifdef UNDER_CE
  if (GetUserNameExW(NameSamCompatible, buffer, &size))
#else
  if (GetUserNameW(buffer, &size))
#endif
  {
    buffer[size] = 0;
    return buffer;
  }
  return ustring();
}
#endif

#ifdef NATIVE_WINDOWS_BACKEND
ustring machine_name(bool full) {
  {
    wchar buffer[1025];
    buffer[0]  = 0;
    DWORD size = 1024;

    if (full) {
      if (_GetComputerNameEx) {
        if (_GetComputerNameEx()(ComputerNameDnsFullyQualified, buffer,
                                 &size)) {
          buffer[size] = 0;
          return buffer;
        }
      }
    }
    if (GetComputerName(buffer, &size)) {
      buffer[size] = 0;
      return buffer;
    }
  }
  return ustring();
}
#endif

#ifdef NATIVE_WINDOWS_BACKEND
bool get_lang_country(ustring &lang, ustring &country, bool for_user) {
  long  langId = for_user ? GetUserDefaultLCID() : GetSystemDefaultLCID();
  WCHAR buf[256];
  buf[0] = 0;
  GetLocaleInfoW(MAKELCID(langId, SORT_DEFAULT), LOCALE_SISO639LANGNAME, buf,
                 items_in(buf));
  lang = buf;
  GetLocaleInfoW(MAKELCID(langId, SORT_DEFAULT), LOCALE_SISO3166CTRYNAME, buf,
                 items_in(buf));
  country = buf;
  return true;
}
#endif

#ifdef NATIVE_WINDOWS_BACKEND

#define WINDOWS_ACRYLIC_BUILD 17134
#define WINDOWS_1903_BUILD 18362

int _get_os_version() {
  if (IsWindows10OrGreater()) {
    OSVERSIONINFOEX osvi = { sizeof(osvi), 0, 0, 0, 0,{ 0 }, 0, 0 };
    DWORDLONG const  dwlConditionMask = VerSetConditionMask(0, VER_BUILDNUMBER, VER_GREATER_EQUAL);

    osvi.dwBuildNumber = WINDOWS_1903_BUILD;
    if (VerifyVersionInfo(&osvi, VER_BUILDNUMBER, dwlConditionMask))
      return WIN_10_2; // Win10 with broken Acrylic :)
    osvi.dwBuildNumber = WINDOWS_ACRYLIC_BUILD;
    if(VerifyVersionInfo(&osvi, VER_BUILDNUMBER,dwlConditionMask))
      return WIN_10_1; // Win10 with Acrylic
    return WIN_10;
  }
  if (IsWindows8Point1OrGreater())
    return WIN_8_1;
  if (IsWindows8OrGreater())
    return WIN_8;
  if (IsWindows7SP1OrGreater())
    return WIN_7_1;
  if (IsWindows7OrGreater())
    return WIN_7;
  if (IsWindowsVistaOrGreater())
    return WIN_VISTA;
  if (IsWindowsXPOrGreater())
    return WIN_XP;
  return WIN_2000;

  /*
      OSVERSIONINFOEX osvi;

      // Try calling GetVersionEx using the OSVERSIONINFOEX structure.
      //
      // If that fails, try using the OSVERSIONINFO structure.

      ZeroMemory(&osvi, sizeof(OSVERSIONINFOEX));
      osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);

      BOOL bOsVersionInfoEx = GetVersionEx ((OSVERSIONINFO *) &osvi);

      if( !bOsVersionInfoEx)
      {
          // If OSVERSIONINFOEX doesn't work, try OSVERSIONINFO.

         osvi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
         if (! GetVersionEx ( (OSVERSIONINFO *) &osvi) )
             return 0;
      }

      switch (osvi.dwPlatformId)
      {
        case VER_PLATFORM_WIN32_NT:

        // Test for the product.
           if ( osvi.dwMajorVersion <= 4 )
             return WIN_NT4;
           if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 0 )
             return WIN_2000;
           if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 1 )
             return WIN_XP;
           if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 2 )
             return WIN_2003;
           if ( osvi.dwMajorVersion == 6 && osvi.dwMinorVersion == 0 )
             return WIN_VISTA;
           if ( osvi.dwMajorVersion == 6 && osvi.dwMinorVersion == 1 )
             return WIN_7;
           if ( osvi.dwMajorVersion == 6 && osvi.dwMinorVersion == 2 )
             return WIN_8;

           return ABOVE_WIN_8;

        case VER_PLATFORM_WIN32_WINDOWS:

           if (osvi.dwMajorVersion == 4 && osvi.dwMinorVersion == 0)
           {
               if ( osvi.szCSDVersion[1] == 'C' )
                 return WIN_95_OSR2;
               else
                 return WIN_95;
           }
           if (osvi.dwMajorVersion == 4 && osvi.dwMinorVersion == 10)
           {
               if ( osvi.szCSDVersion[1] == 'A' )
                 return WIN_98_SE;
               else
                 return WIN_98;
           }
           if (osvi.dwMajorVersion == 4 && osvi.dwMinorVersion == 90)
           {
               return WIN_ME;
           }
           break;

        case VER_PLATFORM_WIN32s:

           return WIN_32S;
           break;
     }

     return 0;
  */
}

int get_os_version() {
#if defined(WINDOWS)
  static int osv = 0;
  if (osv == 0)
    osv = _get_os_version();
  return osv;
#else
  return SOME_LINUX;
#endif
}

const char *get_os_version_name() {
  const char *osname = "Unknown";
  switch (get_os_version()) {
/*case WIN_32S:         osname = "Windows-3.11"; break;
  case WIN_95:          osname = "Windows-95"; break;
  case WIN_95_OSR2:     osname = "Windows-95-OSR2"; break;
  case WIN_98:          osname = "Windows-98"; break;
  case WIN_98_SE:       osname = "Windows-98-SE"; break;
  case WIN_ME:          osname = "Windows-ME"; break;
  case WIN_CE:          osname = "Windows-CE"; break;
  case WIN_NT4:         osname = "Windows-NT4"; break; */
  case WIN_2000:
    osname = "Windows-2000";
    break;
  case WIN_2003:
    osname = "Windows-2003";
    break;
  case WIN_XP:
    osname = "Windows-XP";
    break;
  case WIN_VISTA:
    osname = "Windows-Vista";
    break;
  case WIN_7:
    osname = "Windows-7";
    break;
  case WIN_7_1:
    osname = "Windows-7.1";
    break;
  case WIN_8:
    osname = "Windows-8";
    break;
  case WIN_8_1:
    osname = "Windows-8.1";
    break;
  case WIN_10:
    osname = "Windows-10";
    break;
  case WIN_10_1:
    osname = "Windows-10.1";
    break;
  case WIN_10_2:
    osname = "Windows-10.2";
    break;

    // case ABOVE_WIN_8:     osname = "above-Windows-8"; break;
  }
  return osname;
}
#endif
}


static int index_of_eol(wchars text, bool &n) {
  for (uint_ptr i = 0; i < text.length; ++i) {
    if (text.start[i] == '\r') {
      n = false;
      return int(i);
    }
    if (text.start[i] == '\n') {
      n = true;
      return int(i);
    }
  }
  return -1;
}

bool chopline(wchars &text,wchars& line) {
  if (text.length == 0)
    return false;
  bool   n_seen = false;
  int    d      = index_of_eol(text, n_seen);
  wchars head;
  if (d < 0) {
    line = text;
    text.start += text.length;
    text.length = 0;
  } else if (n_seen) {
    line = wchars(text.start, d);
    text.start += d + 1;
    text.length -= d + 1;
  } else {
    line = wchars(text.start, d);
    text.start += d + 1;
    text.length -= d + 1;
    if (!n_seen && text[0] == '\n') // \r\n sequence
      text.prune(1);
  }
  return true;
}

bool chopline(wchars &text, wchars& line, bool& crlf_seen) {
  if (text.length == 0)
    return false;
  bool n_seen = false;
  int  d = index_of_eol(text, n_seen);
  crlf_seen = d >= 0;
  wchars head;
  if (d < 0) {
    line = text;
    text.start += text.length;
    text.length = 0;
  }
  else if (n_seen) {
    line = wchars(text.start, d);
    text.start += d + 1;
    text.length -= d + 1;
  }
  else {
    line = wchars(text.start, d);
    text.start += d + 1;
    text.length -= d + 1;
    if (!n_seen && text[0] == '\n') // \r\n sequence
      text.prune(1);
  }
  return true;
}


ustring function_value::to_string() const {
  array<wchar> buf;
  buf.push(name);
  buf.push('(');
  for (int n = 0; n < params.size(); ++n) {
    ustring s = params.key(n).to_string();
    if (s.length()) {
      buf.push(s());
      buf.push(':');
    }
    buf.push(xjson::emit(params.value(n), xjson::XJSON, false)());
    buf.push(',');
  }
  if (params.size())
    buf.pop();
  buf.push(')');
  return buf();
}

bool crack_data_url(chars u, string &mime_type, array<byte> &data) {
  chars cs_head, cs = u;
  cs = cs.chop(CHARS(","), cs_head);
  if (cs.length == 0) return true;
  cs_head.prune(5); // remove data:
  mime_type = cs_head.chop(';');
  if (cs_head.like("*base64*")) {
    if (base64_decode(cs, data)) return true;
  }
  else {
    ustring wd = url::unescape(cs.start);
    string  u8d = u8::cvt(wd);
    data = u8d.chars_as_bytes();
    return true;
  }
  return false;
}

bool launch(ustring what)
{
#if defined(WINDOWS)
  tool::tstring    tname = what;
  SHELLEXECUTEINFO sei;
  memzero(sei);
  sei.cbSize = sizeof(sei);

  sei.lpVerb = TEXT("open");
  sei.lpFile = tname;
  sei.nShow = SW_SHOWNORMAL;

  ShellExecuteEx(&sei);

  return (uint_ptr)sei.hInstApp > 32 ? true : false;
#elif defined(OSX)
  string u8 = u8::cvt(what);
  return system(string::format("open \"%s\"", u8.c_str())) >= 0
    ? true
    : false;
#elif defined(WINDOWLESS)
  string u8 = u8::cvt(what);
  return system(string::format("open \"%s\"", u8.c_str())) >= 0
    ? true
    : false;
#else
  // return system( ustring::utf8(wchars(path,path_len)) ) >= 0 ?
  // TRUE_VALUE: FALSE_VALUE;
  string path = string(what);
  GdkScreen *screen = gdk_screen_get_default();
  GError *   perr = 0;
  bool r = gtk_show_uri(screen, path, GDK_CURRENT_TIME, &perr);
  if (!r)
    return system(path + " &") >= 0 ? true : false;
  return true;
#endif
}


} // namespace tool

extern "C" int filename_match(const char* name, const char* mask) {
  return tool::chars_of(name).like(mask);
}

std::string get_home(const char *relpath) {
  tool::ustring r = tool::get_home_dir(tool::ustring(relpath));
  return std::string(tool::string(r));
}

