
//|
//|
//| Copyright (c) 2001-2013
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//|

#pragma once

#ifndef __tl_timer_h
#define __tl_timer_h

#include "tl_basic.h"

#ifdef WIN32 // Windows system specific
#else // Unix based system specific
#include <sys/time.h>
#endif

namespace tool {
// high resolution timer
class timer {
public:
  timer(bool auto_start = false); // default constructor
  ~timer();                       // default destructor

  void   start();                   // start timer
  void   stop();                    // stop the timer
  double get_elapsed_ticks_f();     // get elapsed time in milliseconds
  uint   get_elapsed_ticks();       // get elapsed time in milliseconds
  uint64 get_elapsed_micro_ticks(); // get elapsed time in microseconds

  static uint   get_ticks() { return uint(get_micro_ticks() / 1000LL); }
  static double get_ticks_f() { return double(get_micro_ticks()) / 1000.0; }
  static uint64 get_micro_ticks();

private:
  bool stopped; // stop flag
#ifdef WIN32
  LARGE_INTEGER frequency;   // ticks per second
  LARGE_INTEGER start_count; //
  LARGE_INTEGER end_count;   //
#else
  timeval start_count; //
  timeval end_count;   //
#endif
};

inline timer::timer(bool auto_start) {
#ifdef WIN32
  QueryPerformanceFrequency(&frequency);
  start_count.QuadPart = 0;
  end_count.QuadPart   = 0;
#else
  start_count.tv_sec = start_count.tv_usec = 0;
  end_count.tv_sec = end_count.tv_usec = 0;
#endif
  stopped = 0;
  if (auto_start)
    start();
}

inline timer::~timer() {}

inline void timer::start() {
  stopped = false; // reset stop flag
#ifdef WIN32
  QueryPerformanceCounter(&start_count);
#else
  gettimeofday(&start_count, NULL);
#endif
}

inline void timer::stop() {
  stopped = true; // set timer stopped flag
#ifdef WIN32
  QueryPerformanceCounter(&end_count);
#else
  gettimeofday(&end_count, NULL);
#endif
}

inline uint64 timer::get_elapsed_micro_ticks() {
#ifdef WIN32
  if (!stopped)
    QueryPerformanceCounter(&end_count);

  uint64 start_ticks = (start_count.QuadPart * 1000000LL) / frequency.QuadPart;
  uint64 end_ticks   = (end_count.QuadPart * 1000000LL) / frequency.QuadPart;

#else
  if (!stopped)
    gettimeofday(&end_count, NULL);

  uint64 start_ticks = (start_count.tv_sec * 1000000LL) + start_count.tv_usec;
  uint64 end_ticks   = (end_count.tv_sec * 1000000LL) + end_count.tv_usec;
#endif

  return end_ticks - start_ticks;
}

inline uint timer::get_elapsed_ticks() // milliseconds
{
  return uint(get_elapsed_micro_ticks() / 1000LL);
}

inline double timer::get_elapsed_ticks_f() // milliseconds
{
  return double(get_elapsed_micro_ticks()) / 1000.0;
}

inline uint64 timer::get_micro_ticks() {
  static timer tmr(true);
  return tmr.get_elapsed_micro_ticks();
}

} // namespace tool

#endif