#ifndef __tl_scope_h_
#define __tl_scope_h_

#include "tl_config.h"
#include <type_traits>

#define CONCAT_IMPL(x, y) x##y
#define CONCAT(x, y) CONCAT_IMPL(x, y)

namespace tool {
// using std::forward;
// using std::remove_reference;

// ON_SCOPE_EXIT implementation (D alike)

template <class F> class on_scope_exit_ctx {
public:
  on_scope_exit_ctx(F &&f) throw() : _f(std::forward<F>(f)), _is_owner(true) {}
  on_scope_exit_ctx(on_scope_exit_ctx &&x) throw()
      : _f(std::move(x._f)), _is_owner(true) {
    x._is_owner = false;
  }
  ~on_scope_exit_ctx() {
    if (_is_owner)
      _f();
  }

private:
  typename std::remove_reference<F>::type _f;
  bool                                    _is_owner;
};

template <class F> on_scope_exit_ctx<F> make_on_scope_exit(F &&f) throw() {
  return on_scope_exit_ctx<F>(std::forward<F>(f));
}

  //#define ON_SCOPE_EXIT(...) auto CONCAT(_on_scope_exit_, __LINE__) =
  // tool::make_on_scope_exit(__VA_ARGS__)

#define ON_SCOPE_EXIT(ST)                                                      \
  auto CONCAT(_on_scope_exit_, __LINE__) =                                     \
      tool::make_on_scope_exit([=]() { ST; })

/*#include <iostream>

int main()
{
    std::cout << "initialization 1\n";
    ON_SCOPE_EXIT( std::cout << "destruction 1\n"; );
    std::cout << "initialization 2\n";
    ON_SCOPE_EXIT( std::cout << "destruction 2\n"; );
    std::cout << "processing\n";
}*/

// counts levels of function call nesting.
struct nesting_counter {
  int &_cnt;
  nesting_counter(int &cnt) : _cnt(cnt) { ++_cnt; }
  ~nesting_counter() { --_cnt; }

private:
  nesting_counter(const nesting_counter &nc) : _cnt(nc._cnt) {}
  nesting_counter &operator=(const nesting_counter & /*nc*/) { return *this; }
};

// State setter, on scope exit restore the state automatically.
template <typename T> struct auto_state {
  T  value;
  T &_state;
  auto_state(T &state, T init_val) : _state(state) {
    value = state;
    state = init_val;
  }
  auto_state(T &state) : _state(state) { value = state; }
  ~auto_state() { _state = value; }

  NONCOPYABLE(auto_state)
};

// static_if, "D" style, by jazzer & Evgeny.Panasyuk, see:
// http://rsdn.ru/forum/cpp/5816748.1
namespace details {
struct noop {
  template <class F> noop operator*(const F &f) const { return noop(); }
};

struct yes {
  template <class F> noop operator*(const F &f) const {
    f(0);
    return noop();
  }
};

struct no {
  yes                   operator*(yes) const { return yes(); }
  template <class F> no operator*(const F &f) const { return no(); }
};

inline yes static_if(std::true_type) { return yes(); }
inline no  static_if(std::false_type) { return no(); }
} // namespace details

#define IF(...) tool::details::static_if(__VA_ARGS__()) *[&](auto)
#define ELIF(...) *tool::details::static_if(__VA_ARGS__()) *[&](auto)
#define ELSE *tool::details::yes{} *[&](auto)

#if 0
    // demo:
    template<typename T>
    void check(const T& t)
    {
        IF( std::is_same<T, int> )
        {
            f_int(t);
        }
        ELIF( std::is_constructible<std::string, const T&> )
        {
            f_string(t);
        }
        ELSE
        {
            std::cout<<"no idea: "<<t<<std::endl;
        };
    }
#endif

} // namespace tool

#endif
