//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terra-informatica.org
//|
//| hash functions
//|
//|

#ifndef __tl_hash_h
#define __tl_hash_h

#include "tl_slice.h"
#include "tl_string.h"

namespace tool {

template <typename T> inline unsigned int hash(const T &t) { return t.hash(); }
template <typename T, size_t size> inline unsigned int hash(const T(&arr)[size]) {
  unsigned int h = 0;
  for (int n = 0; n < size; ++n)
    hash_combine(h, tool::hash(arr[n]));
  return h;
}


/*typedef const char *const_char_ptr_t;
template <>
inline unsigned int hash<const_char_ptr_t>(const const_char_ptr_t &p_string) {
  unsigned int h  = 0, g;
  const char * pc = p_string;
  while (*pc) {
    h = (h << 4) + *pc++;
    if ((g = h & 0xF0000000) != 0)
      h ^= g >> 24;
    h &= ~g;
  }
  return h;
}*/

#if 1

inline uint32_t murmur3_32(const uint8_t* key, size_t len, uint32_t seed = 37)
{
  uint32_t h = seed;
  if (len > 3) {
    const uint32_t* key_x4 = (const uint32_t*)key;
    size_t i = len >> 2;
    do {
      uint32_t k = *key_x4++;
      k *= 0xcc9e2d51;
      k = (k << 15) | (k >> 17);
      k *= 0x1b873593;
      h ^= k;
      h = (h << 13) | (h >> 19);
      h = h * 5 + 0xe6546b64;
    } while (--i);
    key = (const uint8_t*)key_x4;
  }
  if (len & 3) {
    size_t i = len & 3;
    uint32_t k = 0;
    do {
      k <<= 8;
      k |= key[i - 1];
    } while (--i);
    k *= 0xcc9e2d51;
    k = (k << 15) | (k >> 17);
    k *= 0x1b873593;
    h ^= k;
  }
  h ^= len;
  h ^= h >> 16;
  h *= 0x85ebca6b;
  h ^= h >> 13;
  h *= 0xc2b2ae35;
  h ^= h >> 16;
  return h;
}

inline unsigned int hash_value(const chars &value) {
  return murmur3_32((const uint8_t*)value.start,value.length);
}
inline unsigned int hash_value(const wchars &value) {
  return murmur3_32((const uint8_t*)value.start, value.length * sizeof(wchar));
}
inline unsigned int hash_value(const bytes &value) {
  return murmur3_32(value.start, value.length);
}


#else
inline unsigned int hash_value(const chars &value) {
  unsigned int h   = unsigned(value.length), g;
  const char * pc  = value.start;
  const char * end = value.end();
  while (pc != end) {
    h = (h << 4) + *pc++;
    if ((g = h & 0xF0000000) != 0)
      h ^= g >> 24;
    h &= ~g;
  }
  return h;
}
inline unsigned int hash_value(const wchars &value) {
  unsigned int h   = unsigned(value.length), g;
  const wchar *pc  = value.start;
  const wchar *end = value.end();
  while (pc != end) {
    h = (h << 4) + *pc++;
    if ((g = h & 0xF0000000) != 0)
      h ^= g >> 24;
    h &= ~g;
  }
  return h;
}
inline unsigned int hash_value(const bytes &value) {
  unsigned int h   = unsigned(value.length), g;
  const byte * pc  = value.start;
  const byte * end = value.end();

  while (pc != end) {
    h = (h << 4) + *pc++;
    if ((g = h & 0xF0000000) != 0)
      h ^= g >> 24;
    h &= ~g;
  }
  return h;
}
#endif

inline unsigned int hash_value(uint value) {
  value += ~(value << 16);
  value ^= (value >> 5);
  value += (value << 3);
  value ^= (value >> 13);
  value += ~(value << 9);
  value ^= (value >> 17);
  return value;
}

inline unsigned int hash_value(int value) { return hash_value(uint(value)); }

inline void hash_combine(uint &seed, uint v) {
  seed = v + 0x9e3779b9 + (seed << 6) + (seed >> 2);
}


template <> inline unsigned int hash<chars>(const chars &value) {
  return hash_value(value);
}
template <> inline unsigned int hash<wchars>(const wchars &value) {
  return hash_value(value);
}
template <> inline unsigned int hash<bytes>(const bytes &value) {
  return hash_value(value);
}

/*
template <> inline unsigned int hash<long>(const long &the_long) {
  return (unsigned int)the_long;
}

template <>
inline unsigned int hash<unsigned long>(const unsigned long &the_long) {
  // return (unsigned int) the_long;
  return hash_value(uint(the_long));
}
*/

template <> inline unsigned int hash<int>(const int &the_int) {
  return hash_value(uint(the_int));
}

template <> inline unsigned int hash<uint>(const uint &the_int) {
  return hash_value(the_int);
}

template <> inline unsigned int hash<word>(const word &the_word) {
  return (unsigned int)the_word;
}

template <> inline unsigned int hash<uint64>(uint64 const &v) {
  uint seed = hidword(v);
  hash_combine(seed, lodword(v));
  return seed;
}

template <> inline unsigned int hash<int64>(int64 const &v) {
   return hash(uint64(v));
}

inline unsigned int hash_value(const slice<uint> &value) {
  unsigned int h   = unsigned(value.length);
  const uint * pi  = value.start;
  const uint * end = value.end();
  while (pi < end)
    hash_combine(h, hash(*pi++));
  return h;
}
template <> inline unsigned int hash<slice<uint>>(const slice<uint> &value) {
  return hash_value(value);
}

inline unsigned int hash_value(const float &value) {
  return hash_value(*((uint *)&value));
}

template <> inline unsigned int hash<float>(const float &value) {
  return hash(*((uint *)&value));
}

template <> inline unsigned int hash<double>(const double &value) {
  return hash(*((uint64 *)&value));
}

#define mix(a, b, c)                                                           \
  {                                                                            \
    a -= b;                                                                    \
    a -= c;                                                                    \
    a ^= (c >> 13);                                                            \
    b -= c;                                                                    \
    b -= a;                                                                    \
    b ^= (a << 8);                                                             \
    c -= a;                                                                    \
    c -= b;                                                                    \
    c ^= (b >> 13);                                                            \
    a -= b;                                                                    \
    a -= c;                                                                    \
    a ^= (c >> 12);                                                            \
    b -= c;                                                                    \
    b -= a;                                                                    \
    b ^= (a << 16);                                                            \
    c -= a;                                                                    \
    c -= b;                                                                    \
    c ^= (b >> 5);                                                             \
    a -= b;                                                                    \
    a -= c;                                                                    \
    a ^= (c >> 3);                                                             \
    b -= c;                                                                    \
    b -= a;                                                                    \
    b ^= (a << 10);                                                            \
    c -= a;                                                                    \
    c -= b;                                                                    \
    c ^= (b >> 15);                                                            \
  }

inline uint32 hash_uint32(const uint32 *k, uint32 length, uint32 initval)
// k - the key
// length - the length of the key, in uint32s
// initval - the previous hash, or an arbitrary value
// src: http://burtleburtle.net/bob/hash/index.html#lookup
{
  uint32 a, b, c, len;

  /* Set up the internal state */
  len = length;
  a = b = 0x9e3779b9; /* the golden ratio; an arbitrary value */
  c     = initval;    /* the previous hash value */

  /*---------------------------------------- handle most of the key */
  while (len >= 3) {
    a += k[0];
    b += k[1];
    c += k[2];
    mix(a, b, c);
    k += 3;
    len -= 3;
  }

  /*-------------------------------------- handle the last 2 ub4's */
  c += length;
  switch (len) /* all the case statements fall through */
  {
    /* c is reserved for the length */
  case 2:
    b += k[1];
  case 1:
    a += k[0];
    /* case 0: nothing left to add */
  }
  mix(a, b, c);
  /*-------------------------------------------- report the result */
  return c;
}

#undef mix

}; // namespace tool

#endif
