//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| file system primitives
//|
//|

#ifndef __tl_filesystem_h
#define __tl_filesystem_h

#include "tl_async_io.h"
#include "tl_basic.h"
#include "tl_ustring.h"

//#include <thread>
//#include <mutex>
//#include <condition_variable>
#include <deque>

#include <sys/stat.h>
#include <sys/types.h>

#include "external/uv/include/uv.h"

#ifndef WINDOWS
#include <dirent.h>
#endif

namespace tool {

namespace filesystem 
{
  using async::result_t;

  bool mime_type(const wchar *path_or_ext, ustring &out,
    bool from_content = false);

  enum FILE_ATTRIBUTES {

    FA_READONLY = 1,
    FA_DIR      = 2,
    FA_HIDDEN   = 4,
    FA_SYSTEM   = 8,
  };

  typedef function<bool(wchars name, wchars caption, uint flags)> scan_callback;

  int scan(const wchar *root, const scan_callback &cb);

  bool is_file(const char *path);
  bool is_file(const wchar *path);

  enum DIRENT {
    DIRENT_UNKNOWN = UV_DIRENT_UNKNOWN,
    DIRENT_FILE = UV_DIRENT_FILE,
    DIRENT_DIR = UV_DIRENT_DIR,
    DIRENT_LINK = UV_DIRENT_LINK,
    DIRENT_FIFO = UV_DIRENT_FIFO,
    DIRENT_SOCKET = UV_DIRENT_SOCKET,
    DIRENT_CHAR = UV_DIRENT_CHAR,
    DIRENT_BLOCK = UV_DIRENT_BLOCK
  };
  
  struct dir_entry {
    ustring   name;
    ustring   caption;
    DIRENT    type;
    uv_stat_t stat;
  };

  inline date_time uv_time_cvt(uv_timespec_t uvt)
  {
    double time = uvt.tv_sec + double(uvt.tv_nsec) / 1000000000ULL;
    date_time dt = ((datetime_t)((time) * 10000000ULL)) + 116444736000000000ULL;
    return dt;
  }
  inline uv_timespec_t uv_time_cvt(date_time dt)
  {
    auto FILETIME_TO_UINT = [](uint64_t filetime) { return (*((uint64_t*) &(filetime)) - 116444736000000000ULL); };
    auto FILETIME_TO_TIME_T = [&](uint64_t filetime) { return (FILETIME_TO_UINT(filetime) / 10000000ULL); };
    auto FILETIME_TO_TIME_NS = [&](uint64_t filetime, long secs) { return ((FILETIME_TO_UINT(filetime) - (secs * 10000000ULL)) * 100); };
    uv_timespec_t ts;
    ts.tv_sec = (long) FILETIME_TO_TIME_T(dt.time());
    ts.tv_nsec = (long) FILETIME_TO_TIME_NS(dt.time(), (ts).tv_sec);
    return ts;
  }

  typedef function<void(const dir_entry& de)> scan_dir_callback;
  result_t scan_dir(ustring dirpath, const scan_dir_callback& cb, bool wants_stats);

  class monitor: public async::entity
  {
    uv_fs_event_t _evt = { 0 };
    ustring       _path;
    static void on_event(uv_fs_event_t *handle, const char *filename, int events, int status);
    static void on_close(uv_handle_t* handle);

  public:
    
    monitor(async::dispatch* pd);
    virtual ~monitor();

    virtual bool is_live() const;   
    virtual bool is_dead() const;  

    const ustring& get_path() const { return _path; }

    int watch(const ustring& path);

    virtual void final_release() {
      close();
    }
    
    void close();

    virtual void handle_change(const ustring& filename, bool changed, bool renamed) = 0;
    virtual void handle_error(int status) = 0;
    virtual void handle_close() { _evt = { 0 }; release(); }
    
  };


} // namespace filesystem

// child process with stdin/stdout/stderr piping

class process : public async::entity {

public:

  //enum STATE {
  //  INITIAL,
  //  STARTING,
  //  OPERATIONAL,
  //  CLOSING,
  //  CLOSED,
  //};

  //typedef function<void(uint status, chars data)> callback;
  process(async::dispatch* pd = async::dispatch::current());
  virtual ~process() { 
    assert(is_dead());
  }

  bool exec(ustring path, slice<ustring> argv, bool detached);
  bool send(wchars data);

  bool is_sending() const { return stdin_buf.length() > 0; }

  bool terminate() { 
    if (!is_live()) return false;
    uv_process_kill(&uv_proc, SIGKILL);
    return true;
  }
  bool close();

  virtual void on_stdout(wchars text) {}
  virtual void on_stderr(wchars text) {}
  virtual void on_stdin(int r) {} // data to send is delivered
  virtual void on_terminate(int status) {}

  //int  get_status() const { return status; }
  //void set_status(int s) { status = s; }

  virtual void stop() override { 
    terminate(); 
  }
  virtual bool is_live() const override;
  virtual bool is_dead() const override;

  //STATE state() const { return st; }

protected:

  static void write_cb(uv_write_t* req, int status);

  uv_process_t         uv_proc = { 0 };
  uv_process_options_t uv_options = { 0 };
  array<string>        uv_args;
  array<char*>         arg_ptrs;
  uv_pipe_t            uv_stdout = { 0 };
  uv_pipe_t            uv_stderr = { 0 };
  uv_pipe_t            uv_stdin = { 0 };
  uv_write_t           uv_stdin_write = { 0 };

  //STATE                st = INITIAL; 
  int                  channels = 0; // n of open subchannels
  bool                 closing = false;

  int         status = 0;
  array<byte> stdin_buf; // data to send to their stdin
  array<char> stdout_buf;
  array<char> stderr_buf;

  void handle_stdout(chars data);
  void handle_stderr(chars data);
  void handle_stdout(int r) { stdin_buf.clear(); on_stdin(r); }

  static void exit_cb(uv_process_t* proc, int64_t exit_status, int term_signal);
  static void close_cb(uv_handle_t* handle);
  static void on_stdout_alloc(uv_handle_t* handle, size_t suggested_size, uv_buf_t* buf);
  static void on_stderr_alloc(uv_handle_t* handle, size_t suggested_size, uv_buf_t* buf);
  static void on_stdout_read(uv_stream_t* str, ssize_t nread, const uv_buf_t* rdbuf);
  static void on_stderr_read(uv_stream_t* str, ssize_t nread, const uv_buf_t* rdbuf);

};

} // namespace tool

#endif
