
#include "config.h"
#include "tool.h"
#include "tl_filesystem.h"

#if defined(WINDOWS)
#include <shlobj.h>
#include <urlmon.h>

#elif defined(OSX)
#include <errno.h>
#include <libproc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#elif defined(LINUX)
#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/wait.h>
#include <unistd.h>
#endif

#include <limits.h>

//extern char** environ;

#if defined(WINDOWS)
#define CVT(str) u16::cvt(CP_OEMCP,str)
#else
#define CVT(str) u8::cvt(str)
#endif

namespace tool {
namespace filesystem {
int scan(const wchar *folder_path, const scan_callback &cb) {
  if (folder_path == 0 || folder_path[0] == 0)
    return 0;

  ustring wpath = chars_of(folder_path);
  wpath = url::file_url_to_path(wpath);
  wchars root = wpath();

  int counter = 0;
#ifdef WINDOWS

  if (root == WCHARS("/*.*")) {
    WCHAR  buffer[2048];
    DWORD  used = GetLogicalDriveStringsW(items_in(buffer), buffer);
    wchars list(buffer, used);
    while (!!list) {
      wchars path = list.chop('\0');

      WCHAR caption[256] = {0};

      DWORD fsFlags = 0;
      BOOL  r       = GetVolumeInformation(
          path.start,
          caption,           //_Out_opt_  LPTSTR lpVolumeNameBuffer,
          items_in(caption), //_In_       DWORD nVolumeNameSize,
          NULL,              //_Out_opt_  LPDWORD lpVolumeSerialNumber,
          NULL,     //_Out_opt_  LPDWORD lpMaximumComponentLength,
          &fsFlags, //_Out_opt_  LPDWORD lpFileSystemFlags,
          NULL,     //_Out_opt_  LPTSTR lpFileSystemNameBuffer,
          0         //_In_       DWORD nFileSystemNameSize
      );
      if (!r)
        continue;

      ++counter;

      path.prune(0, 1);
      ustring zpath = path;

      uint attr = FA_DIR;
      if (fsFlags & FILE_READ_ONLY_VOLUME)
        attr |= FA_READONLY;

      // if( path.li )
      if (!cb(path, chars_of(caption), attr))
        break;
    }

  } else {
    WIN32_FIND_DATAW file_data;
    if (root[0] == '/')
      ++root;
    HANDLE hSearch = FindFirstFileW(root.start, &file_data);
    if (hSearch == INVALID_HANDLE_VALUE)
      return counter;
    for (;;) {
      ++counter;
      uint attr = 0;
      if (file_data.dwFileAttributes & FILE_ATTRIBUTE_READONLY)
        attr |= FA_READONLY;
      if (file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
        attr |= FA_DIR;
      if (file_data.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM)
        attr |= FA_SYSTEM;
      if (file_data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
        attr |= FA_HIDDEN;
      if (!cb(chars_of(file_data.cFileName), wchars(), attr))
        break;
      if (!FindNextFileW(hSearch, &file_data))
        break;
    }
    FindClose(hSearch);
  }
#else
  //#pragma TODO("this needs to be defined using POSIX opendir and friends")

  // if( root[0] == '/' )
  //    ++root;

  string uroot = u8::cvt(root);
  chars  path  = uroot();

  string name_ext_filter = path.r_tail('/');
  path.length -= name_ext_filter.length();

  bool   has_dot     = name_ext_filter().contains('.');
  string name_filter = name_ext_filter().r_head('.');
  string ext_filter  = name_ext_filter().r_tail('.');

  DIR *dir = opendir(string(path));
  if (!dir)
    return counter;
  dirent *dp;
  while ((dp = readdir(dir)) != nullptr) {
    uint attr = 0;
    if (dp->d_type == DT_DIR)
      attr |= FA_DIR;
    else if (dp->d_type != DT_REG)
      attr |= FA_HIDDEN;

    if (dp->d_name[0] == '.')
      attr |= FA_SYSTEM;

    if (name_ext_filter.length()) {
      chars nameext = chars_of(dp->d_name);
      // attempt to reproduce logic of FindFirstFileW
      if (has_dot) {
        chars name = nameext.r_head('.');
        chars ext  = nameext.r_tail('.');
        if (!name.like(name_filter) || !ext.like(ext_filter))
          continue;
      } else {
        if (!nameext.like(name_ext_filter))
          continue;
      }
    }
    ++counter;
    if (!cb(ustring(dp->d_name), wchars(), attr))
      break;
  }
  closedir(dir);
#endif
  return counter;
}

bool is_file(const char *path) {
#ifdef WINDOWS
  WIN32_FIND_DATAW file_data;
  HANDLE           hSearch = FindFirstFileW(ustring(path), &file_data);
  if (hSearch == INVALID_HANDLE_VALUE)
    return false;
  bool r = (file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0;
  FindClose(hSearch);
  return r;
#else
  struct stat st;
  if (0 == stat(path, &st)) {
    return (st.st_mode & S_IFREG) != 0;
  }
  return false;
#endif
}

bool is_file(const wchar *path) {
#ifdef WINDOWS
  WIN32_FIND_DATAW file_data;
  HANDLE           hSearch = FindFirstFileW(path, &file_data);
  if (hSearch == INVALID_HANDLE_VALUE)
    return false;
  bool r = (file_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0;
  FindClose(hSearch);
  return r;
#else
  struct stat st;
  if (0 == stat(string(path), &st)) {
    return (st.st_mode & S_IFREG) != 0;
  }
  return false;
#endif
}

bool mime_type(const wchar *path_or_ext, ustring &out, bool from_content) {
// from_content;
#ifdef WINDOWS
  if (!path_or_ext || !path_or_ext[0])
    return false;
  ustring buf;
  if (path_or_ext[0] != '.') {
    ustring drive, dir, name, ext;
    split_path(path_or_ext, drive, dir, name, ext);
    buf         = ext;
    path_or_ext = buf;
  }
  HKEY hKey;
  LONG rv =
      ::RegOpenKeyExW(HKEY_CLASSES_ROOT, path_or_ext, 0, KEY_EXECUTE, &hKey);
  if (rv != ERROR_SUCCESS)
    return false;
  wchar outbuf[1024] = {0};
  DWORD dwtype       = REG_SZ;
  DWORD dwsize       = sizeof(outbuf);
  rv = ::RegQueryValueExW(hKey, L"Content Type", 0, &dwtype, (LPBYTE)outbuf,
                          &dwsize);
  if (rv != ERROR_SUCCESS || dwsize == 0)
    return false;
  out = outbuf;
  //::FindMimeFromData() and from_content anyone?
  return true;
#else
  assert(false);
  return false;
#endif

}

ustring url_to_path(ustring path) {
  if (path().starts_with(WCHARS("file://")))
    return url::file_url_to_path(path);
  //  return url::unescape(path()(7));
  return path;
}

result_t scan_dir(ustring dirpath, const scan_dir_callback& cb, bool wants_stats)
{
  dirpath = url_to_path(dirpath);
  if (!dirpath().ends_with(WCHARS("/")))
    dirpath += "/";
  uv_fs_t req = { 0 };

  uv_loop_t* loop = async::dispatch::current()->uv_loop();

  string u8dirpath = u8::cvt(dirpath);

  int r = uv_fs_scandir( loop, &req, u8dirpath, O_RDONLY, nullptr);
  if (r < 0) return r;
  if( req.result < 0 ) return result_t(req.result);

  uv_dirent_t dent = { 0 };
  uv_fs_t stat_req = { 0 };

  int count = 0;

  while (0 == uv_fs_scandir_next(&req, &dent))
  {
    dir_entry de;
    de.name = u8::cvt(chars_of(dent.name));
    de.type = DIRENT(dent.type);
    if (wants_stats)
    {
      string u8filepath = u8dirpath + dent.name;
      int r = uv_fs_stat(loop, &stat_req, u8filepath, nullptr);
      if (r < 0)
        continue;
      de.stat = stat_req.statbuf;
      uv_fs_req_cleanup(&stat_req);
    }
    cb(de);
    ++count;
  }
  uv_fs_req_cleanup(&req);
  return count;
}

void monitor::on_event(uv_fs_event_t *handle, const char *filename, int events, int status)
{
  monitor* self = (monitor*)(handle->data);
  if (status != 0) {
    self->close();
    self->handle_error(status);
  }
  else {
    ustring us = u8::cvt(chars_of(filename));
    self->handle_change(us, !!(events & UV_CHANGE), !!(events & UV_RENAME));
  }
}

void monitor::on_close(uv_handle_t* handle)
{
  // never being called on Windows, why?
  handle->data = nullptr;
}

monitor::monitor(async::dispatch* pd): async::entity(pd)
{
  uv_fs_event_init(pd->uv_loop(), &_evt);
  _evt.data = this;
}

monitor::~monitor()
{
  close();
}

int monitor::watch(const ustring& path)
{
  close();
  _path = path;
  return uv_fs_event_start(&_evt, &on_event, u8::cvt(path), 0 /*UV_FS_EVENT_RECURSIVE*/);
}

void monitor::close()
{
  if (is_live()) {
    //uv_fs_event_stop(&_evt);
    uv_close((uv_handle_t*)&_evt, on_close);
    _evt.data = nullptr;
  }
}

bool monitor::is_live() const {
  bool r = uv_is_active((uv_handle_t*)&_evt) != 0;
  return r;
}

bool monitor::is_dead() const {
  return !is_live() && !_evt.data;
}

//virtual void on(wchars filename, bool changed, bool renamed) = 0;


} // namespace filesystem

#if defined(PROCESS_SUPPORT)

void process::handle_stdout(chars data) {
  on_stdout(CVT(data));
}
void process::handle_stderr(chars data) {
  on_stderr(CVT(data));
}

process::process(async::dispatch* pd): async::entity(pd) {
  memzero(uv_proc);
  memzero(uv_options);
  uv_disable_stdio_inheritance();
}

void process::exit_cb(uv_process_t* proc,
  int64_t exit_status,
  int term_signal)
{
  process* pp = static_cast<process*>(proc->data);
  pp->status = int(exit_status);
  pp->close();
}

void process::on_stdout_alloc(uv_handle_t* handle, size_t suggested_size, uv_buf_t* buf)
{
  process* pp = static_cast<process*>(handle->data);
  pp->stdout_buf.length(suggested_size);
  buf->base = pp->stdout_buf.begin();
  buf->len = pp->stdout_buf.size();
}

void process::on_stderr_alloc(uv_handle_t* handle, size_t suggested_size, uv_buf_t* buf)
{
  process* pp = static_cast<process*>(handle->data);
  pp->stderr_buf.length(suggested_size);
  buf->base = pp->stderr_buf.begin();
  buf->len = pp->stderr_buf.size();
}

void process::on_stdout_read(uv_stream_t* str, ssize_t nread, const uv_buf_t* rdbuf) {
  process* pp = static_cast<process*>(str->data);
  if (nread > 0)
    pp->handle_stdout(chars(rdbuf->base, min(size_t(nread),rdbuf->len)));
  pp->stdout_buf.clear();
}

void process::on_stderr_read(uv_stream_t* str, ssize_t nread, const uv_buf_t* rdbuf) {
  process* pp = static_cast<process*>(str->data);
  if (nread > 0)
    pp->handle_stderr(chars(rdbuf->base, min(size_t(nread),rdbuf->len)));
  pp->stderr_buf.clear();
}

bool process::exec(ustring path, slice<ustring> argv, bool detached) {

  uv_args.length(argv.length + 1);
  arg_ptrs.length(argv.length + 2);

  uv_args[0] = CVT(path());
  arg_ptrs[0] = (char*)uv_args[0].c_str();

  int na = 1;
  for (size_t n = 0; n < argv.length; ++n, ++na) {
    uv_args[na] = CVT(argv[n]);
    arg_ptrs[na] = (char*)uv_args[na].c_str();
  }
  arg_ptrs[na] = nullptr;

  auto disp = async::dispatch::current()->uv_loop();

  uv_pipe_init(disp, &uv_stdout, false); uv_stdout.data = this;
  uv_pipe_init(disp, &uv_stderr,false); uv_stderr.data = this;
  uv_pipe_init(disp, &uv_stdin, false); uv_stdin.data = this;

  uv_proc.data = this; ++channels;

  uv_stdio_container_t child_stdio[3];

  child_stdio[0].flags = uv_stdio_flags(UV_CREATE_PIPE | UV_READABLE_PIPE); child_stdio[0].data.stream = (uv_stream_t*)&uv_stdin; ++channels;
  child_stdio[1].flags = uv_stdio_flags(UV_CREATE_PIPE | UV_WRITABLE_PIPE); child_stdio[1].data.stream = (uv_stream_t*)&uv_stdout; ++channels;
  child_stdio[2].flags = uv_stdio_flags(UV_CREATE_PIPE | UV_WRITABLE_PIPE); child_stdio[2].data.stream = (uv_stream_t*)&uv_stderr; ++channels;

  uv_options.stdio = child_stdio;
  uv_options.stdio_count = items_in(child_stdio);
  uv_options.flags = UV_PROCESS_WINDOWS_HIDE;
  if (detached)
    uv_options.flags |= UV_PROCESS_DETACHED;

  uv_options.file = arg_ptrs[0];
  uv_options.args = arg_ptrs.begin();
  //uv_options.env = environ;
  uv_options.exit_cb = &exit_cb;

  //  char** ppc = environ;
#ifdef DEBUG
  //  while(ppc) {
  //      printf("env=%s\n",*(ppc++));
  //  }
#endif

  int r = uv_spawn(disp, &uv_proc, &uv_options);
  if (r < 0) {
    close();
    status = r;
    return false;
  }

  r = uv_read_start((uv_stream_t*)&uv_stdout, on_stdout_alloc, on_stdout_read);
  if (r < 0) {
    close();
    status = r;
    return false;
  }

  r = uv_read_start((uv_stream_t*)&uv_stderr, on_stderr_alloc, on_stderr_read);
  if (r < 0) {
    close();
    status = r;
    return false;
  }

  return true;
}

void process::write_cb(uv_write_t* req, int status) {
  auto proc = static_cast<process*>(req->data);
  proc->handle_stdout(status);
}

bool process::send(wchars data) {

  if (stdin_buf.length())
    return false; // still sending
  stdin_buf = CVT(data).chars_as_bytes();

  uv_buf_t buf = uv_buf_init((char*)stdin_buf.cbegin(), uint(stdin_buf.length()));
  uv_stdin_write.data = this;
  int r = uv_write(&uv_stdin_write, (uv_stream_t*)&uv_stdin, &buf, 1, &write_cb);
  if (r < 0)
    return false;
  return true;
}

void process::close_cb(uv_handle_t* handle) {
  process* pp = static_cast<process*>(handle->data);
  --pp->channels;
}

bool process::close() {

  if (!closing) {
    closing = true;
    if (uv_stdout.data)
      uv_close((uv_handle_t*)&uv_stdout, &close_cb);
    if (uv_stderr.data)
      uv_close((uv_handle_t*)&uv_stderr, &close_cb);
    if (uv_stdin.data)
      uv_close((uv_handle_t*)&uv_stdin, &close_cb);
    if (uv_proc.data)
      uv_close((uv_handle_t*)&uv_proc, &close_cb);
  }

  on_terminate(status);

  if (!is_live())
    return false;

  return true;
}

bool process::is_live() const {
  if (uv_is_active((const uv_handle_t*)&uv_proc)) return true;
  if (uv_is_active((const uv_handle_t*)&uv_stdout)) return true;
  if (uv_is_active((const uv_handle_t*)&uv_stderr)) return true;
  if (uv_is_active((const uv_handle_t*)&uv_stdin)) return true;
  return false;
}

bool process::is_dead() const {
  return channels == 0;
    //&& !uv_has_ref((const uv_handle_t*)&uv_proc)
    //&& !uv_has_ref((const uv_handle_t*)&uv_stdout)
    //&& !uv_has_ref((const uv_handle_t*)&uv_stderr)
    //&& !uv_has_ref((const uv_handle_t*)&uv_stdin);
}


#endif // PROCESS_SUPPORT

} // namespace tool

#if defined(WINDOWS)
FILE *wfopen(LPCWSTR filename, LPCWSTR mode) {
  return _wfopen(filename, mode);
}
int   wopen(LPCWSTR filename, int flags, int mode) {
  return _wopen(filename, flags,mode);
}
#elif defined(LINUX)
FILE *wfopen(LPCWSTR filename, LPCWSTR mode) {
  return fopen(tool::u8::cvt(tool::chars_of(filename)), tool::u8::cvt(tool::chars_of(mode)));
}
int   wopen(LPCWSTR filename, int flags, int mode) {
  return open(tool::u8::cvt(tool::chars_of(filename)), flags, mode);
}
#endif


