//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terra-informatica.org
//|
//| dictionary - hash map of key/value pairs backed by array
//|
//|

#ifndef __cs_dictionary_h
#define __cs_dictionary_h

#include "tl_array.h"
#include "tl_basic.h"
#include "tl_string.h"

namespace tool {
  template <class c_key, class c_element, int HTS = 11> class dictionary
  {
    enum {
      HT_THRESHOLD = 6  // less than this - linear lookup table, more - will create hash table infrastructure. 
    };

  public:
    dictionary() : _table(nullptr) { init(0); }

    dictionary(dictionary &&rs) : _table(nullptr) {
      swap(_table, rs._table);
      swap(_array, rs._array);
    }

    dictionary(const dictionary &copy) : _table(nullptr) {
      init(copy.size());
      copy_items(copy);
    }

    dictionary &operator=(const dictionary &copy) {
      clear();
      init(copy.size());
      copy_items(copy);
      return *this;
    }

    void copy_items(const dictionary &copy);

    virtual ~dictionary() { clear(); }
    bool exists(const c_key &the_key) const;

    //
    // Add to hash table association of object with specified name.
    //
    c_element &operator[](const c_key &the_key);
    c_element  operator()(const c_key &the_key) const {
      c_element el;
      find(the_key, el);
      return el;
    }

    //
    // Search for object with specified name in hash table.
    // If object is not found false is returned.
    //
    bool find(const c_key &the_key, c_element &the_element) const;
    bool find(const c_key &the_key, c_element *&the_element_ptr) const;
    //
    // Remove object with specified name from hash table.
    // If there are several objects with the same name the one last inserted
    // is removed. If such name was not found 'false' is returned.
    //
    bool remove(const c_key &the_key);

    void clear();

    index_t size() const { return _array.size(); };
    size_t  length() const { return _array.size(); };

    bool is_empty() const { return _array.size() == 0; };

    bool equal(const dictionary &rd) const {
      if (size() != rd.size()) return false;
      for (int n = 0; n < size(); ++n) {
        c_key k = key(n);
        if (k == c_key()) { // keys are optional, and no key 
          if (key(n) != rd.key(n)) return false;
          if (value(n) != rd.value(n)) return false;
        }
        else {
          c_element v;
          if (!rd.find(k, v)) return false;
          if (v != value(n)) return false;
        }
      }
      return true;
    }

    bool operator==(const dictionary &rd) const { return equal(rd); }
    bool operator!=(const dictionary &rd) const { return !equal(rd); }

    struct key_value {
      c_key     key;
      c_element val;

      // key_value() {}
      // key_value(const key_value& rs): key(rs.key), val(rs.val) {}
      // key_value& operator=(const key_value& rs) { key = rs.key; val =
      // rs.val; return *this; }

      bool operator==(const key_value &rd) const {
        return key == rd.key && val == rd.val;
      }
      bool operator!=(const key_value &rd) const {
        return key != rd.key || val != rd.val;
      }
      unsigned int hash() const {
        uint r = 33;
        hash_combine(r, tool::hash(key));
        hash_combine(r, tool::hash(val));
        return r;
      }
    };

    // push unnamed value
    void push(const c_element &v) {
      key_value it;
      it.val = v;
      _array.push(it);
    }

    unsigned int hash() const { return _array.hash(); }

    index_t get_index(const c_key &the_key, bool create = false);

    array<key_value>& elements() { return _array; }
    const array<key_value>& elements() const { return _array; }

    const c_key&     key(index_t i) const;
    c_element&       value(index_t i);
    const c_element& value(index_t i) const;

    bool inherit(const dictionary &rs) {
      int changes = 0;
      for (auto& pair : rs.elements()) {
        c_element e = operator()(pair.key);
        if (e != pair.val) {
          ++changes;
          operator[](pair.key) = pair.val;
        }
      }
      return changes > 0;
    }

  protected:
    struct hash_item {
      index_t    _index;
      hash_item *_next;

      hash_item() { _next = nullptr; }

      hash_item(index_t the_index, hash_item *the_next) {
        _index = the_index;
        _next = the_next;
      }
    };

    hash_item **_table;

    array<key_value> _array;

    void init(size_t nelements) {
      if (nelements >= HT_THRESHOLD) {
        _table = new hash_item*[HTS];
        for (unsigned i = 0; i < HTS; ++i) _table[i] = nullptr;
      }
      else
        _table = nullptr;
      _array.reserve(nelements);
    }
  };


  template <class c_key, class c_element, int HTS>
  inline c_element &dictionary<c_key, c_element, HTS>::operator[](const c_key &the_key) {
    index_t index = get_index(the_key, true);
    return _array[index].val;
  }

  template <class c_key, class c_element, int HTS>
  inline const c_key &dictionary<c_key, c_element, HTS>::key(index_t i) const {
    return _array[i].key;
  }

  template <class c_key, class c_element, int HTS>
  inline c_element &dictionary<c_key, c_element, HTS>::value(index_t i) {
    return _array[i].val;
  }

  template <class c_key, class c_element, int HTS>
  inline const c_element &dictionary<c_key, c_element, HTS>::value(index_t i) const {
    return _array[i].val;
  }

  template <class c_key, class c_element, int HTS>
  inline bool dictionary<c_key, c_element, HTS>::find(const c_key &the_key,
    c_element &  the_element) const {
    index_t index = const_cast<dictionary<c_key, c_element, HTS> *>(this)->get_index(
      the_key, false);

    if (index < 0)
      return false;

    the_element = _array[index].val;
    return true;
  }

  template <class c_key, class c_element, int HTS>
  inline bool
    dictionary<c_key, c_element, HTS>::find(const c_key &the_key,
      c_element *& the_element_ptr) const {
    index_t index = const_cast<dictionary<c_key, c_element, HTS> *>(this)->get_index(
      the_key, false);

    if (index < 0)
      return false;

    the_element_ptr = const_cast<c_element *>(&_array[index].val);
    return true;
  }

  template <class c_key, class c_element, int HTS>
  inline index_t dictionary<c_key, c_element, HTS>::get_index(const c_key &the_key, bool create) {
    size_t h = 0; bool h_valid;
    if (_table) {
      h = tool::hash(the_key) % HTS;
      h_valid = true;
      for (hash_item *ip = _table[h]; ip != nullptr; ip = ip->_next)
        if (_array[ip->_index].key == the_key)
          return ip->_index;
    }
    else { // no hash table yet, linear lookup
      h_valid = false;
      for (index_t i = 0; i < _array.size(); ++i) {
        if (_array[i].key == the_key)
          return i;
      }
    }
    if (!create)
      return -1;

    index_t ni = _array.size();
    key_value di; di.key = the_key;
    _array.push(di);
    //_array.size(ni+1);
    //_array[ni].key = the_key;

    if (_array.size() >= HT_THRESHOLD)
    {
      if (!_table) {
        _table = new hash_item*[HTS];
        for (unsigned i = 0; i < HTS; ++i) _table[i] = nullptr;
        for (unsigned n = 0; n < _array.length(); ++n) {
          size_t th = tool::hash(_array[n].key) % HTS;
          _table[th] = new hash_item(n, _table[th]);
        }
      }
      if (!h_valid) h = tool::hash(the_key) % HTS;
      _table[h] = new hash_item(ni, _table[h]);
    }
    return ni;

  }

  template <class c_key, class c_element, int HTS>
  inline bool dictionary<c_key, c_element, HTS>::exists(const c_key &the_key) const {
    return (const_cast<dictionary<c_key, c_element, HTS> *>(this)->get_index(
      the_key, false) >= 0);
  }

  template <class c_key, class c_element, int HTS>
  inline void dictionary<c_key, c_element, HTS>::clear() {
    if (_table) {
      for (int i = HTS; --i >= 0;) {
        hash_item *ip = _table[i];
        while (ip != nullptr) {
          hash_item *_next = ip->_next;
          delete ip;
          ip = _next;
        }
      }
      delete[] _table;
      _table = nullptr;
    }
    _array.clear();
  }

  template <class c_key, class c_element, int HTS>
  inline bool dictionary<c_key, c_element, HTS>::remove(const c_key &the_key) 
  {
#if 1
    int idx = get_index(the_key);
    if (idx < 0) return false;
    dictionary ni;
    for (int n = 0; n < size(); ++n) {
      if (n != idx)
        ni[key(n)] = value(n);
    }
    swap(_array, ni._array);
    swap(_table, ni._table);
    return true;
#else
    if (_table) {
      size_t     h = tool::hash(the_key) % HTS;
      hash_item *curr = _table[h], *prev = nullptr;
      while (curr) {
        if (_array[curr->_index].key == the_key) 
        {
          if (prev == nullptr) 
            _table[h] = curr->_next; 
          else 
            prev->_next = curr->_next;
          _array.remove(curr->_index);
          for (size_t cn = 0; cn < HTS; ++cn) {
            for (hash_item *t = _table[cn]; t; t = t->_next) {
              if (t->_index > curr->_index)
                --t->_index;
            }
          }
          delete curr;
          return true;
        }
        prev = curr;
        curr = curr->_next;
      }
    }
    else {
      for (index_t i = 0; i < _array.size(); ++i) {
        if (_array[i].key == the_key) {
          _array.remove(i);
          return true;
        }
      }
    }
    return false;
#endif
  }

  template <class c_key, class c_element, int HTS>
  inline void dictionary<c_key, c_element, HTS>::copy_items(const dictionary &copy) {
    for (int i = 0; i < copy._array.size(); i++)
    {
      const key_value& it = copy._array[i];
      if (it.key == c_key())
        this->push(it.val);
      else
        this->operator[](it.key) = it.val;
    }

  }

}; // namespace tool

#endif
