//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terra-informatica.org
//|
//| Date & time class
//|
//|

#ifndef __tl_datetime_h__
#define __tl_datetime_h__

#include "tl_slice.h"
#include "tl_string.h"
#include "tl_ustring.h"
#include <time.h>

namespace tool 
{

typedef int64 datetime_t; // 64-bit value representing the number of 100-nanosecond intervals since January 1, 1601 (UTC).

class date_time {
public:
  static date_time  now(bool utc = true);
  static datetime_t local_offset();
  static datetime_t local_offset_ms();
  static int        first_day_of_week(
             ustring locale =
                 ustring()); // returns first day of week for the current locale
  static ustring
  week_day_name(int n, int maxlength = 0,
                ustring locale =
                    ustring()); // returns week day name  for the current locale

  enum date_format_order { MDY = 0, DMY = 1, YMD = 2 };

  enum time_format_hours { H_12 = 0, H_24 = 1 };

  enum time_format_marker_pos {
    MP_AFTER  = 0,
    MP_BEFORE = 1,
  };

  static void date_format(date_format_order &order, wchar &separator,
                          ustring locale = ustring());
  static void time_format(time_format_hours &     hours,
                          time_format_marker_pos &marker_pos, ustring &am_text,
                          ustring &pm_text, ustring locale = ustring());

public:
  date_time();
  date_time(const date_time &dt);
  date_time(datetime_t dt);
  date_time(const struct tm &syst);

  // date_time ( time_t time );
  date_time(int year, int month, int day, int hours = 0, int minutes = 0,
            int seconds = 0, int milli = 0, int micro = 0, int nano = 0);

  void to_local();
  void to_utc();
  void to_timezone(datetime_t shift);
  void from_timezone(datetime_t shift);
  // operators
public:
  date_time &operator=(const date_time &dt);
  date_time &operator=(datetime_t dts);
  // const date_time& operator= ( const struct tm& syst );

  bool operator==(const date_time &date) const;
  bool operator!=(const date_time &date) const;
  bool operator<(const date_time &date) const;
  bool operator>(const date_time &date) const;
  bool operator<=(const date_time &date) const;
  bool operator>=(const date_time &date) const;

  static bool is_leap_year(int year);
  static int  days_in_month(int year, int month);

  string  format(const char *fmt) const;
  ustring format(const wchar *fmt) const;
  ustring locale_format(const wchar *fmt, ustring locale = ustring()) const;
  ustring locale_format_time(const wchar *fmt,
                             ustring      locale = ustring()) const;

  enum PARTS {
    DATE = 1,
    TIME = 2,
    DATE_TIME = 3
  };

  ustring default_format(bool full, PARTS parts,
                         ustring locale = ustring()) const;

  enum type {
    DT_UNDEFINED   = 0x00,
    DT_HAS_DATE    = 0x01,
    DT_HAS_TIME    = 0x02,
    DT_HAS_SECONDS = 0x04,
    DT_UTC         = 0x10,
  };

  static date_time parse_iso /*8601*/ (tool::wchars str, uint &dt);
  static date_time parse_iso /*8601*/ (
      tool::chars str, uint &dt); // utc == true if it is known to be UTC time

  string emit_iso /*8601*/ (
      uint dt) const; // utc == true if it is known to be UTC time

  // Operations
public:
  bool set(int year, int month, int day, int hours, int minutes, int seconds,
           int milli = 0, int micro = 0, int nano = 0);
  bool set_date(int year, int month, int day);
  bool set_time(int hours, int minutes, int seconds);
  bool set_frac_time(int millis, int micros, int nanos);

  bool systemtime(struct tm &syst) const;

#ifdef WINDOWS
  SYSTEMTIME ms_systemtime() const {
    date_time::datetime_s dts;
    cvt(dts, _time);
    ::SYSTEMTIME st  = {0};
    st.wYear         = WORD(dts.year);
    st.wMonth        = WORD(dts.month);
    st.wDay          = WORD(dts.day);
    st.wHour         = WORD(dts.hour);
    st.wMinute       = WORD(dts.minute);
    st.wSecond       = WORD(dts.second);
    st.wMilliseconds = WORD(dts.millis);
    st.wDayOfWeek    = WORD(dts.day_of_week);
    // FileTimeToSystemTime((::FILETIME*)&_time,&st);
    return st;
  }
#endif

  // getters
  int year() const;
  int month() const;       // month of year (1 = Jan)
  int day() const;         // day of month (1-31)
  int hours() const;       // hour in day (0-23)
  int minutes() const;     // minute in hour (0-59)
  int seconds() const;     // second in minute (0-59)
  int millis() const;      // millisecond in minute (0-999)
  int micros() const;      // microsecond in minute (0-999)
  int nanos() const;       // nanosecond in minute (0-999), step of 100ns
  int day_of_week() const; // (mon=0...sun=6)
  int day_of_year() const; // days since start of year, Jan 1 = 1

  bool has_date() const;
  bool has_time() const;

  // setters
  void nanos(int nv);
  void micros(int nv);
  void millis(int nv);
  void seconds(int nv);
  void minutes(int nv);
  void hours(int nv);
  void day(int nv);
  void month(int nv);
  void year(int nv);

  datetime_t absolute_millis() const;
  void       absolute_millis(datetime_t t);

  datetime_t absolute_seconds() const { return absolute_millis() / 1000; }
  datetime_t absolute_minutes() const { return absolute_seconds() / 60; }
  datetime_t absolute_hours() const { return absolute_minutes() / 60; }
  datetime_t absolute_days() const { return absolute_hours() / 24; }

  datetime_t absolute_millis_since_1970() const {
    return absolute_millis() - 11644473600000LL;
  }

  double seconds_since_1970() const {
    return absolute_millis_since_1970() / 1000.0;
  }

  static date_time from_absolute_millis_since_1970(double ms) {
    datetime_t t = (datetime_t(ms) + 11644473600000LL) * 10000L;
    return date_time(t);
  }

/*  int64 normalized_seconds() const {
    int64 n = year() * 60 * 60 * 24 * 30 * 12;
    n += month() * 60 * 60 * 24 * 30;
    n += day() * 60 * 60 * 24;
    n += hours() * 60 * 60;
    n += minutes() * 60;
    n += seconds();
    return n;
  } */

  int64 normalized_months() const { int64 n = year() * 12; n += month();  return n; }
  int64 normalized_days() const { int64 n = normalized_months() * 30; n += day(); return n; }
  int64 normalized_hours() const { int64 n = normalized_days() * 24; n += hours(); return n;}
  int64 normalized_minutes() const { int64 n = normalized_hours() * 60; n += minutes(); return n; }
  int64 normalized_seconds() const { int64 n = normalized_minutes() * 60; n += seconds(); return n;}

  enum DIFF {
    DIFF_SECONDS,
    DIFF_MINUTES,
    DIFF_HOURS,
    DIFF_DAYS,
    DIFF_MONTHS,
    DIFF_YEARS,
  };
    
  static int64 diff(date_time t1, date_time t2, DIFF d) {
    switch (d) 
    {
      default:
      case DIFF_SECONDS: return t2.absolute_seconds() - t1.absolute_seconds();
      case DIFF_MINUTES: return t2.absolute_minutes() - t1.absolute_minutes();
      case DIFF_HOURS:   return t2.absolute_hours() - t1.absolute_hours();
      case DIFF_DAYS:    return t2.absolute_days() - t1.absolute_days();
      case DIFF_MONTHS:  return t2.normalized_months() - t1.normalized_months();
      case DIFF_YEARS:   return t2.year() - t1.year();
    }
  }

  // formatting
  // ....

public:
  datetime_t time() const { return _time; }

private:
  datetime_t _time;

public:
  struct datetime_s {
    int          year;
    unsigned int month;
    unsigned int day;
    unsigned int hour;
    unsigned int minute;
    unsigned int second;
    unsigned int millis;
    unsigned int micros;
    unsigned int nanos;
    unsigned int day_of_year;
    unsigned int day_of_week;
  };
  static int month_day_in_year[13];

  static bool cvt(datetime_t &dst, const datetime_s &src);
  static bool cvt(datetime_s &dst, const datetime_t &src);
  static bool cvt(struct tm &dst, const datetime_s &src);
  static bool cvt(datetime_s &dst, const struct tm &src);
};

datetime_t time_zone_shift(bool plus, uint hours, uint minutes);
datetime_t time_zone_shift(chars txt);

inline date_time::date_time() : _time(0) {
  ;
} // 1 Jan-1601, 00:00, 100ns clicks

inline date_time::date_time(const date_time &src) { _time = src._time; }

inline date_time::date_time(datetime_t t) { _time = t; }

/*
inline
  date_time::date_time ( time_t timeSrc )
{
  struct tm _tm = *( localtime ( &timeSrc ) );
  *this = _tm;
}*/

inline date_time &date_time::operator=(const date_time &src) {
  _time = src._time;
  return *this;
}

inline date_time &date_time::operator=(const datetime_t src) {
  _time = src;
  return *this;
}

inline bool date_time::set_date(int year, int month, int day) {
  if (year == 0 && month == 0 && day == 0) {
    *this = now();
    return true;
  }
  return set(year, month, day, 0, 0, 0, 0, 0, 0);
}

inline bool date_time::set_time(int hour, int min, int sec) {
  return set(1601, 1, 1, hour, min, sec, 0, 0, 0);
}

inline bool date_time::set_frac_time(int millis, int micros, int nanos) {
  datetime_s d;
  cvt(d, _time);
  return set(d.year, d.month, d.day, d.hour, d.minute, d.second, millis, micros,
             nanos);
}

inline bool date_time::operator==(const date_time &date) const {
  return (_time == date._time);
}

inline bool date_time::operator!=(const date_time &date) const {
  return (_time != date._time);
}

inline bool date_time::operator<(const date_time &date) const {
  return (_time < date._time);
}

inline bool date_time::operator>(const date_time &date) const {
  return (_time > date._time);
}

inline bool date_time::operator<=(const date_time &date) const {
  return (_time <= date._time);
}

inline bool date_time::operator>=(const date_time &date) const {
  return (_time >= date._time);
}

} // namespace tool

#endif
