#pragma once

#include "ObjBase.h"
#include "tl_handle.h"

namespace com {

  struct exception
  {
    HRESULT result;
    exception(HRESULT const value) : result(value) {}
  };

  inline void HR(HRESULT const result)
  {
    if (FAILED(result))
      throw exception(result);
  }


// asset - yet another smart pointer
template <class T> class asset {
protected:
  T *p;

public:
  typedef T asset_t;

  asset() : p(0) {}
  asset(T *lp) : p(0) {
    if (lp)
      (p = lp)->AddRef();
  }
  asset(const asset<T> &cp) : p(0) {
    if (cp.p)
      (p = cp.p)->AddRef();
  }

  ~asset() {
    if (p) {
      p->Release();
      p = 0;
    }
  }
     operator T *() const { return p; }
  T *operator->() const {
    assert(p != 0);
    return p;
  }
  T *ptr() const {
    assert(p != 0);
    return p;
  }

  // used as target T** pointer to pointer - in places receiving newly created
  // objects (initially add-refed)
  T **target() {
    release();
    return &p;
  }

  bool operator!() const { return p == 0; }
       operator bool() const { return p != 0; }
  bool operator!=(T *pT) const { return p != pT; }
  bool operator==(T *pT) const { return p == pT; }

  // release the interface and set it to NULL
  void release() {
    if (p) {
      T *pt = p;
      p     = 0;
      pt->Release();
    }
  }

  // attach to an existing interface (does not AddRef)
  void attach(T *p2) {
    release();
    p = p2;
  }
  // detach the interface (does not Release)
  T *detach() {
    T *pt = p;
    p     = 0;
    return pt;
  }

  static T *assign(T *&pp, T *lp) {
    if (lp != 0)
      lp->AddRef();
    if (pp)
      pp->Release();
    pp = lp;
    return lp;
  }

  T *operator=(T *lp) {
    if (p != lp)
      assign(p, lp);
    return p;
  }
  asset<T> &operator=(const asset<T> &lp) {
    if (p != lp)
      assign(p, lp.p);
    return *this;
  }

  T *acquire() {
    if (p)
      p->AddRef();
    return p;
  }

  HRESULT CoCreateInstance(REFCLSID classUUID,
                           DWORD    dwClsContext = CLSCTX_INPROC_SERVER) {
    HRESULT hr = ::CoCreateInstance(classUUID, 0, dwClsContext, __uuidof(T),
                                    (void **)target());
    assert(hr != CO_E_NOTINITIALIZED); // You haven't called CoInitialize for
                                       // the current thread!
    return hr;
  }

  template <class OTHER_COM_CLASS>
  HRESULT QueryInterface(REFCLSID                classUUID,
                         asset<OTHER_COM_CLASS> &dest_object) const {
    if (p == 0)
      return E_POINTER;
    return p->QueryInterface(classUUID, (void **)dest_object.target());
  }

  template <class OTHER_COM_CLASS>
  HRESULT QueryInterface(asset<OTHER_COM_CLASS> &dest_object) const {
    return this->QueryInterface(__uuidof(OTHER_COM_CLASS), dest_object);
  }
};

// Generic COM base implementation for classes, since DirectWrite uses
// callbacks for several different kinds of objects, particularly the
// script analysis source/sink.
//
// Example:
//
//
template <typename Resource, typename InterfaceChain>
class base : public InterfaceChain {
public:
  explicit base() throw() {}

  // IUnknown interface
  IFACEMETHOD(QueryInterface)(IID const &iid, OUT void **ppObject) {
    *ppObject = NULL;
    InterfaceChain::QueryInterfaceInternal(iid, ppObject);
    if (*ppObject == NULL)
      return E_NOINTERFACE;

    AddRef();
    return S_OK;
  }

  IFACEMETHOD_(ULONG, AddRef)() {
    // return InterlockedIncrement(&refValue_);
    return static_cast<Resource *>(this)->add_ref();
  }

  IFACEMETHOD_(ULONG, Release)() {
    // ULONG newCount = InterlockedDecrement(&refValue_);
    // if (newCount == 0)
    //    delete this;
    //
    // return newCount;
    return static_cast<Resource *>(this)->release();
  }

  virtual ~base() {}

private:
  // No copy construction allowed.
  base(const base &b);
  base &operator=(base const &);
};

struct list_nil {};

// When the QueryInterface list refers to itself as class,
// which hasn't fully been defined yet.
template <typename InterfaceName, typename InterfaceChain>
class list_self : public InterfaceChain {
public:
  inline void QueryInterfaceInternal(IID const &iid,
                                     OUT void **ppObject) throw() {
    if (iid != __uuidof(InterfaceName))
      return InterfaceChain::QueryInterfaceInternal(iid, ppObject);

    *ppObject = static_cast<InterfaceName *>(this);
  }
};

// When this interface is implemented and more follow.
template <typename InterfaceName, typename InterfaceChain = list_nil>
class list : public InterfaceName, public InterfaceChain {
public:
  inline void QueryInterfaceInternal(IID const &iid,
                                     OUT void **ppObject) throw() {
    if (iid != __uuidof(InterfaceName))
      return InterfaceChain::QueryInterfaceInternal(iid, ppObject);

    *ppObject = static_cast<InterfaceName *>(this);
  }
};

// When the this is the last implemented interface in the list.
template <typename InterfaceName>
class list<InterfaceName, list_nil> : public InterfaceName {
public:
  inline void QueryInterfaceInternal(IID const &iid,
                                     OUT void **ppObject) throw() {
    if (iid != __uuidof(InterfaceName))
      return;

    *ppObject = static_cast<InterfaceName *>(this);
  }
};

struct i_unknown_resource : public IUnknown, public virtual tool::resource {
  // IUnknown impl
  unsigned long STDMETHODCALLTYPE AddRef() {
    return add_ref();
    // return InterlockedIncrement(&_ref_count);
  }
  unsigned long STDMETHODCALLTYPE Release() {
    // unsigned long new_count = InterlockedDecrement(&_ref_count);
    // if (new_count == 0) { delete this; return 0; }
    // return new_count;
    return release();
  }
  HRESULT STDMETHODCALLTYPE QueryInterface(IID const &riid, void **ppvObject) {
    if (__uuidof(IUnknown) == riid)
      *ppvObject = static_cast<IUnknown *>(this);
    else {
      *ppvObject = NULL;
      return E_FAIL;
    }
    return S_OK;
  }
  // private:
  //   unsigned long _ref_count;
};

} // namespace com
