//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terra-informatica.org
//|
//|
//|
//|

#include "config.h"
#include "tool.h"

#include <stdio.h>

#if defined(WINDOWS) && !defined(PLATFORM_WINCE)
#include <mmsystem.h>
#pragma comment(lib, "winmm.lib")
#endif

namespace tool {

void play_sound(bytes sound_data) {
#if defined(WINDOWS)
  PlaySound((LPCTSTR)sound_data.start, 0, SND_ASYNC | SND_MEMORY | SND_NOWAIT);
#endif
}

}; // namespace tool

/*
Binary data search using Boyer-Moore algorithm, modified by R. Nigel Horspool
see:
  http://www-igm.univ-mlv.fr/~lecroq/string/node18.html
SYNOPSIS:
   where - source data, src_length - source data byte length (bytes)
   what - data to be sought, what_length - sought data length (bytes)
RETURNS:
   address of 'what' data in 'where' block or 0 if not found.
   c adjustment made by: andrew@terrainformatica.com
*/
const void *mem_lookup(const void *where, size_t where_length, const void *what,
                       size_t what_length) {
#define WHAT ((const byte *)what)

  int m1 = int(what_length) - 1;

  /* Preprocessing */
  int i, bm_bc[256];
  for (i = 0; i < 256; i++)
    bm_bc[i] = int(what_length);
  for (i = 0; i < m1; i++)
    bm_bc[WHAT[i]] = m1 - i;

  /* Searching */
  byte b, lastb = WHAT[m1];

  const byte *p   = (const byte *)where;
  const byte *end = p + where_length - what_length;
  for (; p <= end; p += bm_bc[b]) {
    b = p[m1];
    if ((b == lastb) && (memcmp(p, what, m1) == 0))
      return p;
  }
#undef WHAT
  return 0;
}

const char *str_in_str(const char *where, const char *what) {
  if (where && what)
    return (const char *)mem_lookup(where, strlen(where), what, strlen(what));
  return 0;
}

//#ifdef _DEBUG
extern "C" void _dprintf(const char *fmt, ...) {
  char buffer[2049];
  buffer[0] = 0;
  va_list args;
  va_start(args, fmt);
#ifdef WINDOWS
  _vsnprintf(buffer, 2048, fmt, args);
#else
  vsprintf(buffer, fmt, args);
  // puts(buffer);
  for (const char *pc = buffer; *pc; ++pc)
    putc(*pc, stderr);
#endif
  va_end(args);
#ifdef UNICODE
  wchar wbuffer[2049];
  for (int i = 0; i < 2048; ++i) {
    wbuffer[i] = buffer[i];
    if (!buffer[i])
      break;
  }
  OutputDebugStringW(wbuffer);
#else
#ifdef WINDOWS
  OutputDebugStringA(buffer);
#endif
#endif
}
  //#endif

#ifdef WINDOWS

static void CALLBACK _debug_output_func(void * /*param*/, uint /*subsystem*/,
                                        uint /*severity*/, const wchar *text,
                                        uint text_length) {
  if (text[text_length])
    OutputDebugStringW(tool::ustring(text, text_length));
  else
    OutputDebugStringW(text);
}
#else
static void _debug_output_func(void * /*param*/, uint /*subsystem*/,
                               uint /*severity*/, const wchar *text,
                               uint text_length) {
  //#ifdef LINUX
  //
  //  GtkDialogFlags flags = GTK_DIALOG_DESTROY_WITH_PARENT;
  //  GtkWidget * dialog = gtk_message_dialog_new (NULL,
  //                                 flags,
  //                                 GTK_MESSAGE_ERROR,
  //                                 GTK_BUTTONS_CLOSE,
  //                                 tool::string(text, text_length));
  //  gtk_dialog_run (GTK_DIALOG (dialog));
  //  gtk_widget_destroy (dialog);
  //
  //#else

  tool::array<byte> bytes;
  tool::u8::from_utf16(tool::wchars(text, text_length), bytes);

  // bytes.ch

  tool::tokens<byte> msg(bytes(), tool::bytes((const byte *)"\r\n",2));

  tool::array<byte> line;

  for (tool::bytes span; msg.next(span);) {
    line = span;
    line.push(0);
    puts((const char *)line.cbegin());
  }

  // if( bytes.size() && bytes.last() == '\n' ) {
  //  bytes.pop();
  //  bytes.push(0);
  //}
  // puts( (const char*) bytes.cbegin() );

  // for(uint n = 0; n < text_length; ++n)
  //  if( text[n] != '\n')
  //    putchar(text[n]);
  //#endif
}
#endif

static debug_output_func *_dof = _debug_output_func;
static void * _dof_prm = 0;

void setup_debug_output(void *p, debug_output_func *pf) {
  _dof = pf;
  _dof_prm = p;
}

/*void debug_printf(const char* fmt, ...)
{
  char buffer[2049]; buffer[1] = 0;
  va_list args;
  va_start ( args, fmt );
  do_vsnprintf( &buffer[0], 2048, fmt, args );
  va_end ( args );
  strcat(buffer,"\n");
  OutputDebugStringA(buffer);
}*/

void debug_printf(uint subsystem, uint severity, const char *fmt, ...) {
  struct os : public printf_output_stream {
    tool::array<char> buffer;
    virtual bool      out(int c) {
      buffer.push((char)c);
      return true;
    }
  } os;

  va_list args;
  va_start(args, fmt);
  do_vsprintf_os(&os, fmt, args);
  va_end(args);

  tool::ustring s = os.buffer();

  debug_print(subsystem, severity, s);
}

void debug_print(uint subsystem, uint severity, tool::wchars text) {
  //#ifdef WINDOWS
  if (!_dof)
    return;
  _dof(_dof_prm, subsystem, severity, text.start, uint(text.length));
  //#else

  //#endif
}
