
#ifndef __tl_async_task_h
#define __tl_async_task_h

#include "tl_config.h"
#include "tl_array.h"
#include "tl_basic.h"
#include "tl_handle.h"
#include "tl_slice.h"
#include "tl_string.h"
#include "tl_sync.h"

#ifdef TLS_SUPPORT
#include "external/uv-tls/include/uv_tls.h"
#endif 



#if defined(OSX)
#include "../osx/osx-ref.h"
#include <CoreFoundation/CoreFoundation.h>
#endif

struct timeval;

namespace tool {
namespace async {

  class dispatch;

  // async entity abstraction

  // life cycle of the entity:
  // - entity(dispatch* pd) :
  //   a) connects the entity to dispatch list of live entities
  //   b) add_refs the entity
  //
  // - when entity has ended

  class entity : public virtual resource, public l2elem<entity>
  {
    friend class dispatch;
  public:
    entity(dispatch* pd);
    virtual ~entity();

    virtual void stop() = 0;            // stop (cancel) IO operations on the entity
    virtual bool is_live() const = 0;   // operational && !closed
    virtual bool is_dead() const = 0;   // !operational && closed
    virtual bool is_used() const { return get_ref_count() > 0; } // something holds it

    uv_loop_t* loop() const { return pl; }

    // resource overridable
    virtual void finalize() override;

  protected:

    void unlink_from_dispatch() {
      if (!l2elem::is_empty())
        unlink();
    }

    uv_loop_t *pl;

    NONCOPYABLE(entity);
  };

  typedef int result_t; // < 0 - uv_error

  inline chars result_to_string(int r)
  {
    if (r < 0)
      return chars_of(uv_strerror(r));
    return chars();
  }

  struct func_block : resource
  {
    function<void()>    that;
    semaphore*          psem;
    func_block(function<void()> &&f, semaphore* ps) : that(std::move(f)), psem(ps) {}
    void exec() {
      that();
      if (psem) psem->signal();
    }
  };

  class dispatch : public resource
  {
    friend class entity;
    uv_loop_t  _loop = {0};
    mutex      _lock;
    void*      _timer_id = nullptr;
    uint_ptr   _thread_id = 0;
    entity*    _root_entity = nullptr;
    array<handle<func_block>> _funcs;

/*#if defined(WINDOWS)
    HWND      _message_window;
#elif defined(OSX)
    static void dispatch_timer_callback(CFRunLoopTimerRef timer, void *info);
#elif defined(LINUX)
    static gboolean dispatch::timer_callback(gpointer user_data);
#endif*/

    void start();
    void stop();

    void stop_timer();

    entity*  root_entity();

  public:
    void start_timer();

    bool heartbeat()
    {
      array<tool::handle<func_block>> funcs;
      {
        critical_section _(_lock);
        funcs.swap(_funcs);
      }
      for (int n = 0; n < funcs.size(); ++n)
        funcs[n]->exec();

      for (int n = 0; n < 16; ++n) {
        if (!uv_run(&_loop, UV_RUN_NOWAIT))
          break;
      }
      delete_closed_entities();
      return !!uv_loop_alive(&_loop) || _funcs.size() > 0;
    }

    dispatch();
    virtual ~dispatch();

    uv_loop_t* uv_loop() {
      if (!_timer_id)
        start();
      return  &_loop;
    }

    void each(function<void(entity *t)> receiver);
    void delete_closed_entities();

    static dispatch* current(bool create = true);
    static void shutdown_current();
    static void heartbit_current();

    void run_once();
    bool is_alive(); // true - has items in the queue

    // this function is called from worker threads to
    // execute the gui_block in GUI thread
    void exec(function<void()> &&f, bool wait);

  };

  class task: public entity // a.k.a. task executed in a thread of UV thread pool
  {
  protected:
    bool finished = false;
    uv_work_t w = { 0 };

    static void work_cb(uv_work_t* req) {
      task* pt = static_cast<task*>(req->data);
      pt->exec();
      //pt->finished = true;
    }
    static void after_work_cb(uv_work_t* req, int status) {
      task* pt = static_cast<task*>(req->data);
      pt->done();
      pt->unlink_from_dispatch();
      pt->finished = true;
      pt->release();
    }

  public:
    task(dispatch* queue = dispatch::current()) : async::entity(queue) {
       this->add_ref();
       memzero(w);
       w.data = (void*)this;
    }
    virtual ~task() {
      assert(finished);
    }
    virtual void exec() = 0;

    virtual void done() {};

    void start() {
      uv_queue_work(loop(), &w, &work_cb, &after_work_cb);
    }

    virtual void stop() override {
      uv_cancel((uv_req_t*)&w);
    }

    virtual bool is_live() const override { return !finished; /*!!uv_is_active((const uv_handle_t*)&w);*/ }
    virtual bool is_dead() const override { return finished; /*!uv_is_active((const uv_handle_t*)&w) && !uv_is_closing((const uv_handle_t*)&w); */ }

    //virtual void shutdown() { stop(); }

    NONCOPYABLE(task);
  };


class pipe_connection : public async::entity {
public:
  // typedef hash_table< socket_handle, handle<connection> > bag;

  enum STATE {
    INITIAL = 0,
    NAME_RESOLVING, // resolving name - TCP state
    LISTENING,
    CONNECTING,
    CONNECTED,
    CLOSING,
    CLOSED,
  };


  enum TRANSPORT {
    NP,
    TCP,
    TCP_TLS
  };

  STATE     _state;
  TRANSPORT _transport;

  union {
    uv_stream_s   stream;
    uv_pipe_t     nap;
    uv_tcp_t      tcp;
#if defined(TLS_SUPPORT)
    uv_tls_t      tcp_tls;
#endif
  } pipe;


  union {
    uv_req_t         req;
    uv_getaddrinfo_t addr;
    uv_connect_t     conn;
    uv_write_t       write;
  } req;

  array<byte>        data_to_send;
  array<byte>        data_to_read;

  array<array<byte>> send_queue;

  // constructor
  pipe_connection(async::dispatch* pd = async::dispatch::current()) : async::entity(pd) , _state(INITIAL) {
    memzero(pipe);
    pipe.stream.data = this;
#if defined(TLS_SUPPORT)
    pipe.tcp_tls.data = this;
#endif
    req.req.data = this;
  }

  ~pipe_connection() {
    assert(_state == CLOSED || _state == INITIAL);
  }

  STATE get_state() const { return _state; }
  void  set_state(STATE st);
  virtual void stop() override;

  virtual bool readable(void) const {
    return _state == CONNECTED || _state == LISTENING;
  }
  virtual bool writable(void) const {
    return _state == CONNECTED ||
           _state == CONNECTING; /* (!connected || write_blocked);*/
  }

  bool is_closed() const { return _state == INITIAL || _state == CLOSED; }

  //bool hostname_connect(string hostname, int port, bool to_listen = false);

  // --------------------------------------------------
  // socket methods
  // --------------------------------------------------

  // tcp
  bool listen(const string& name, int port, bool tls);
  bool connect(const string&  name, int port, bool tls);

  // named pipes
  bool listen(const string& name);
  bool connect(const string&  name);

  bool accept(pipe_connection *con);

  uint send(bytes data, int flags = 0);
  uint _send();
  virtual void close();

  // --------------------------------------------------
  // event handlers
  // --------------------------------------------------

  //void handle_read_event(void);
  //void handle_write_event(void);

  // These are meant to be overridden.
  virtual void handle_connect(void) {
  }
  virtual void handle_read(bytes data) {
    // cerr << fileno << ":unhandled read" << endl;
    //assert(0);
  }
  virtual void handle_write(void) {
    // cerr << fileno << ":unhandled write" << endl;
    //assert(0);
  }
  virtual void handle_close(void) {
    // cerr << fileno << ":unhandled close" << endl;
    //assert(0);
  }
  virtual void handle_accept(void) {
    // cerr << fileno << ":unhandled accept" << endl;
    //assert(0);
  }
  virtual void handle_error(int /*error*/) {
    //assert(0);
  }

  void shutdown(int r) {
    if(r < 0)
      handle_error(r);
    close();
  }

  virtual bool is_live() const override { return _state == LISTENING || _state == CONNECTING || _state == CONNECTED; }
  virtual bool is_dead() const override { return _state == INITIAL || _state == CLOSED; }

  target_bytes allocate_buffer(size_t suggested_size)
  {
    assert(data_to_read.size() == 0);
    data_to_read.length(suggested_size);
    return data_to_read.target();
  }
  static void getaddrinfo_cb_listen(uv_getaddrinfo_t* handle, int status, struct addrinfo* response);
  static void getaddrinfo_cb_connect(uv_getaddrinfo_t* handle, int status, struct addrinfo* response);
};


class data_connection : public pipe_connection {
  typedef pipe_connection super;
  enum { CHUNK_SIZE = 4096 };
  array<byte> data_in, data_out;
  // uint        data_in_length;
  // uint        data_out_pos;

  bool reading;
  bool sending;
  bool sent_notify;

  virtual void handle_connect(void) override { this->on_connected(); }
  virtual void handle_read(bytes data) override;
  virtual void handle_write(void) override;
  virtual void handle_close(void) override { this->on_closed(); }
  virtual void handle_error(int er) override { this->on_error(er); }

public:
  TOOL_INTERFACE_1(tool, data_connection, pipe_connection)

  data_connection() : reading(false), sending(false), sent_notify(false) {}
  virtual bool send_data(bytes out);
  virtual void on_connected() = 0;
  virtual void on_data_received(bytes data) = 0;
  virtual void on_data_sent()               = 0;
  virtual void on_closed()                  = 0;
  virtual void on_error(int er)             = 0;
};


class fs_req: public async::entity {
protected:
  uv_fs_t req = {0};

public:
  fs_req(async::dispatch* pd = async::dispatch::current()): async::entity(pd) {
    req.loop = pd->uv_loop();
    req.data = this;
  }
  virtual ~fs_req() {}

  virtual void stop() override {
    if ( req.cb )
      uv_cancel((uv_req_t*)&req);
  }

  virtual bool is_live() const override { return req.cb && req.data != 0; }
  virtual bool is_dead() const override { return !req.cb || req.data == 0; }

  virtual void on_done(const uv_fs_t& req) = 0;
  virtual void on_error(const uv_fs_t& req) = 0;

  static void fs_cb(uv_fs_t* req) {
    fs_req* self = (fs_req*)req->data;
    if (!self)
      return;
    if (req->result < 0)
      self->on_error(*req);
    else
      self->on_done(*req);
    uv_fs_req_cleanup(req);
    req->data = 0;
  }

};

} // namespace async
} // namespace tool
#endif // __tl_socket_srv_h
