#include "config.h"
#include "tool.h"

namespace tool {
namespace async {

  void dispatch::exec(function<void()> &&f, bool wait)
  {
    if (wait) {
      semaphore sm(0);
      handle<func_block> pf = new func_block(std::move(f), &sm);
      {
        critical_section _(_lock);
        _funcs.push(pf);
      }
      start_timer();
      sm.wait();
    }
    else {
      handle<func_block> pf = new func_block(std::move(f), nullptr);
      {
        critical_section _(_lock);
        _funcs.push(pf);
        start_timer();
      }
    }
  }

  static thread_context<dispatch*> spd;

  struct root_entity : public entity
  {
    root_entity() : entity(nullptr) {}
    virtual void stop() override {}
    virtual bool is_live() const override { return false; }
    virtual bool is_dead() const override { return true; }
  };

  entity*  dispatch::root_entity()
  {
    return _root_entity;
  }

  dispatch* dispatch::current(bool create) {
    dispatch* pd = spd.get();
    if (!pd && create) {
      spd.set(pd = new dispatch());
      pd->add_ref();
    }
    return pd;
  }

  void dispatch::shutdown_current() {
    dispatch* pd = spd.get();
    if (pd) {
      pd->stop();
      spd.set(nullptr);
      pd->release();
    }
  }

  void dispatch::heartbit_current() {
    dispatch* pd = spd.get();
    if (pd) {
      pd->heartbeat();
    }
  }


  void dispatch::run_once() {
    bool has_stuff_to_do = heartbeat();
    if (has_stuff_to_do)
      start_timer();
  }

  bool dispatch::is_alive() {
    return uv_loop_alive(&_loop) != 0 || _funcs.size() > 0;
  }

  void  dispatch::start() {
    start_timer();
  }

  void dispatch::stop() {

        heartbeat();

        stop_timer();

        each([](entity* pe) {
            if(pe->is_live())
                pe->stop();
        });

        heartbeat();

        //bool closed = UV_EBUSY != uv_loop_close(&_loop);
        //assert(closed);
        //closed = closed;

        delete_closed_entities();

#ifdef _DEBUG
        if (_root_entity != _root_entity->next()) {
          entity * first = _root_entity->next();
          printf("%s\n", first->resource_class_name().start);
        }
#endif
        assert(_root_entity == _root_entity->next() && _root_entity == _root_entity->prev());
  }

/*
#if defined(SCITER) 
  // nothing here - Sciter reuses request_idle/on_idle functionality on its windows
#elif defined(WINDOWS)

  #define USE_MESSAGE_WINDOWS
 
  LRESULT CALLBACK wndproc(HWND hwnd, UINT msg, WPARAM wp, LPARAM lp) {
    if (msg == WM_TIMER && wp == 1) {
      dispatch *pdc = reinterpret_cast<dispatch *>(GetWindowLongPtr(hwnd, GWLP_USERDATA));
      pdc->run_once();
    }
    return DefWindowProc(hwnd,msg,wp,lp);
  }

  static HWND init_window_for(dispatch* pd)
  {
    HWND message_hwnd = nullptr;
    static ATOM atom = 0;
    if (!atom) {
      WNDCLASSEX wx = {};
      wx.cbSize = sizeof(WNDCLASSEX);
      wx.lpfnWndProc = &wndproc;
      wx.hInstance = THIS_HINSTANCE;
      wx.lpszClassName = W("hsmile-msg-window");
      atom = RegisterClassExW(&wx);
      assert(atom);
    }
    if (atom) {
      message_hwnd = CreateWindowExW(0, LPCWSTR(atom), L"", 0, 0, 0, 0, 0, HWND_MESSAGE, nullptr, THIS_HINSTANCE, nullptr);
      assert(message_hwnd);
      SetWindowLongPtr(message_hwnd, GWLP_USERDATA, LONG_PTR(pd));
    }
    return message_hwnd;
  }

  void  dispatch::start_timer() {
    if (!_timer_id)
      _timer_id = (void*)SetTimer(_message_window, 1, USER_TIMER_MINIMUM, nullptr);
  }

  void dispatch::stop_timer() {
    if (_timer_id) {
      KillTimer(_message_window, 1);
      _timer_id = 0;
    }
  }

#elif defined (OSX)

    void dispatch::dispatch_timer_callback(CFRunLoopTimerRef timer, void *info)
    {
        dispatch* pd = static_cast<dispatch*>(info);

        bool has_stuff_to_do = pd->heartbeat();
        if (!has_stuff_to_do) {
            pd->stop_timer();
        }
    }

    void  dispatch::start_timer() {
        CFRunLoopTimerContext ctx = {0};
        ctx.info = this;

        const double idle_threshold = 0.016;

        CFRunLoopTimerRef tr = CFRunLoopTimerCreate(nullptr, CFAbsoluteTimeGetCurrent() + idle_threshold, idle_threshold, 0, 0, &dispatch_timer_callback,&ctx);
        _timer_id = tr;
        CFRunLoopAddTimer( CFRunLoopGetMain(), tr, kCFRunLoopCommonModes );
    }

    void  dispatch::stop_timer() {
        if(_timer_id) {
          CFRunLoopTimerRef timer = (CFRunLoopTimerRef) _timer_id;
          _timer_id = nullptr;
          CFRunLoopTimerInvalidate(timer);
          CFRunLoopRemoveTimer ( CFRunLoopGetMain() , timer, kCFRunLoopCommonModes );

        }
    }
#elif defined(LINUX)

    gboolean dispatch::timer_callback (gpointer user_data)
    {
        dispatch* pd = static_cast<dispatch*>(user_data);

        bool has_stuff_to_do = pd->heartbeat();
        if (!has_stuff_to_do) {
            //pd->stop_timer();
            pd->_timer_id = 0;
            return false;
        }
        return true;
    }

    void  dispatch::start_timer() {
       _timer_id = g_idle_add (timer_callback, this);
    }

    void  dispatch::stop_timer() {
        if(_timer_id) {
          _timer_id = 0;
          g_idle_remove_by_data (this);
        }
    }
#endif
 */

  dispatch::dispatch() : _timer_id(nullptr) {
#if defined(USE_MESSAGE_WINDOWS)    
    _thread_id = GetCurrentThreadId();
    _message_window = init_window_for(this);
#endif
    uv_loop_init(&_loop);
    _loop.data = this;
    _root_entity = new async::root_entity();
  }

  dispatch::~dispatch() {
#if defined(USE_MESSAGE_WINDOWS)    
    _thread_id = nullptr;
    DestroyWindow(_message_window);
#endif
    delete_closed_entities();
    assert(_root_entity == _root_entity->next() && _root_entity == _root_entity->prev());
    delete _root_entity;
    assert(spd.get() != this);
    /*dispatch* pd = spd.get();
    if (pd == this)
      spd.set(nullptr);*/
  }

  entity::entity(dispatch* pd) {
    if (pd)
    {
      pl = pd->uv_loop();
      link_after(pd->root_entity());
    }
  }
  entity::~entity() {
    unlink_from_dispatch();
  }

    // means no one is using it so
  void entity::finalize() {
    if (is_live())
      stop();
    if (is_dead()) {
      final_release();
      if (l2elem::is_empty())
        delete this;
    }
  }

  void dispatch::each(function<void(entity *t)> receiver) {
    if (!_root_entity) return;
    for (entity* pt = _root_entity->next(); pt != _root_entity; pt = pt->next())
      receiver(pt);
  }

  void dispatch::delete_closed_entities()
  {
    if (!_root_entity) return;
    for (entity* pt = _root_entity->next(); pt != _root_entity;)
    {
      if (pt->is_dead())
      {
        entity* next = pt->next();
        if (pt->is_used()) { // ref_count > 0, still used by script
          pt->unlink(); // so just remove it from list
        }
        else {
          delete pt;
        }
        pt = next;
      }
      else
        pt = pt->next();
    }
  }


static void uv_connection_cb(uv_stream_t* pipe, int status)
{
  static_cast<pipe_connection*>(pipe->data)->handle_accept();
}

static void uv_alloc_cb(uv_handle_t* handle, size_t suggested_size, uv_buf_t* buf)
{
  target_bytes buff = static_cast<pipe_connection*>(handle->data)->allocate_buffer(suggested_size);
  buf->base = (char*)buff.start;
#ifdef WINDOWS
  buf->len = ULONG(buff.length);
#else
  buf->len = buff.length;
#endif
}

string compose_pipe_name(const string& logical_name)
{
#ifdef WINDOWS
  return CHARS("\\\\.\\pipe\\sciter-") + logical_name;
#else
  return CHARS("/tmp/sciter-pipe-") + logical_name;
#endif
}

void pipe_connection::getaddrinfo_cb_listen(uv_getaddrinfo_t* handle, int status, struct addrinfo* response) {
  pipe_connection* self = (pipe_connection*)handle->data;

  if (status < 0 || !response)
  {
    self->shutdown(status);
    return;
  }

  int r = uv_tcp_bind(&self->pipe.tcp, (const struct sockaddr*) response->ai_addr, 0);
  if (r < 0) {
    self->shutdown(r);
    return;
  }
  uv_freeaddrinfo(response);
  r = uv_listen(&self->pipe.stream, 16, uv_connection_cb);
  if (r < 0) {
    self->shutdown(r);
    return;
  }
  self->set_state(LISTENING);

}

static void connect_cb(uv_connect_t* req, int status)
{
  auto conn = static_cast<pipe_connection*>(req->data);
  if (status < 0)
    conn->shutdown(status);
  else {
    conn->set_state(pipe_connection::CONNECTED);
    conn->handle_connect();
  }
}

void pipe_connection::getaddrinfo_cb_connect(uv_getaddrinfo_t* handle, int status, struct addrinfo* response) {
  pipe_connection* self = (pipe_connection*)handle->data;
  //uv_tcp_init(self->loop(), &self->pipe.tcp);

  if (status < 0 || !response)
  {
    self->shutdown(status);
    return;
  }

  int r = uv_tcp_connect(&self->req.conn, &self->pipe.tcp, (const struct sockaddr*) response->ai_addr, connect_cb);

  uv_freeaddrinfo(response);

  if (r < 0) {
    self->shutdown(r);
    return;
  }
  self->set_state(CONNECTING);

}

bool pipe_connection::connect(const string& name) {
  _transport = NP;
  uv_pipe_init(loop(), &pipe.nap, false);
  req.conn.data = this;
  set_state(CONNECTING);
  uv_pipe_connect(&req.conn, &pipe.nap, compose_pipe_name(name), &connect_cb);
  return true;
}

bool pipe_connection::listen(const string& name) {
  _transport = NP;
  uv_pipe_init(loop(), &pipe.nap, false);
  //&pipe.nap
  int r = uv_pipe_bind(&pipe.nap, compose_pipe_name(name));
  if (r < 0) {
    shutdown(r);
    return false;
  }
  r = uv_listen(&pipe.stream, 16, uv_connection_cb);
  if (r < 0) {
    shutdown(r);
    return false;
  }
  set_state(LISTENING);
  return true;
}


bool pipe_connection::connect(const string& name, int port, bool tls) 
{
  _transport = tls ? TCP_TLS : TCP;

  int r;
  itoa sport(port);
#if defined(TLS_SUPPORT)
  if (_transport == TCP_TLS) {
    set_state(CONNECTING);
    uv_tls_init(loop(), &pipe.tcp_tls);
    r = uv_tls_connect(&req.conn, &pipe.tcp_tls, name, port, connect_cb);
  }
  else
#endif
  {
    set_state(NAME_RESOLVING);
    uv_tcp_init(loop(), &pipe.tcp);
    struct addrinfo hints = { 0 };
    hints.ai_protocol = IPPROTO_TCP;
    hints.ai_family = name().contains(':') ? AF_INET6 : AF_INET; // do I need to specify this at all?
    hints.ai_socktype = SOCK_STREAM;

    r = uv_getaddrinfo(loop(), &req.addr, getaddrinfo_cb_connect, name, sport, &hints);
  }
  if (r < 0) {
    shutdown(r);
    return false;
  }
  return true;
}

bool pipe_connection::listen(const string& name, int port, bool tls)
{
  _transport = tls ? TCP_TLS : TCP;

  set_state(NAME_RESOLVING);
  itoa sport(port);

#if defined(TLS_SUPPORT)
  if (_transport == TCP_TLS)
    uv_tls_init(loop(), &pipe.tcp_tls);
  else
#endif
    uv_tcp_init(loop(), &pipe.tcp);

  struct addrinfo hints = { 0 };
  hints.ai_protocol = IPPROTO_TCP;
  hints.ai_family = AF_INET;
  hints.ai_socktype = SOCK_STREAM;

  int r = uv_getaddrinfo(loop(), &req.addr, getaddrinfo_cb_listen, name, sport, &hints);
  if (r < 0) {
    shutdown(r);
    return false;
  }
  return true;
}


bool pipe_connection::accept(pipe_connection *con) {

  if (pipe.stream.type == UV_NAMED_PIPE)
    uv_pipe_init(loop(),&con->pipe.nap, false);
#if defined(TLS_SUPPORT)
  else if( _transport == TCP_TLS )
    uv_tls_init(loop(), &con->pipe.tcp_tls);
#endif
  else 
    uv_tcp_init(loop(), &con->pipe.tcp);

  int r = uv_accept(&pipe.stream, &con->pipe.stream);
  if (r < 0) {
    shutdown(r);
    return false;
  }
  con->set_state(CONNECTED);
  return true;
}

//UV_EXTERN int uv_read_stop(uv_stream_t*);

void uv_read_cb(uv_stream_t* stream, ssize_t nread, const uv_buf_t* buf)
{
  auto conn = static_cast<pipe_connection*>(stream->data);

  if (nread == UV_EOF) {
    conn->close();
  }
  else if (nread < 0) {
    conn->shutdown(int(nread));
  }
  else {
    bytes data((const byte*)buf->base, nread);
    assert(buf->base == (char*)conn->data_to_read.cbegin());
    conn->handle_read(data);
    conn->data_to_read.clear();
  }
}

#if defined(TLS_SUPPORT)
static void uv_read_cb_tls(uv_tls_t* h, int nread, uv_buf_t* dcrypted_buf)
{
  auto conn = static_cast<pipe_connection*>(h->data);

  if (nread == UV_EOF) {
    conn->close();
  }
  else if (nread < 0) {
    conn->shutdown(int(nread));
  }
  else {
    bytes data((const byte*)dcrypted_buf->base, nread);
    //assert(dcrypted_buf->base == (char*)conn->data_to_read.cbegin());
    conn->handle_read(data);
    conn->data_to_read.clear();
  }
}
#endif


void pipe_connection::set_state(STATE st) {
  //STATE state = _state;
  switch (_state = st) {
    case LISTENING:
      break;
    case CONNECTED :
      {
        int r;
#if defined(TLS_SUPPORT)
        if( _transport == TCP_TLS )
          r = uv_tls_read(&pipe.tcp_tls, &uv_read_cb_tls);
        else 
#endif
          r = uv_read_start(&pipe.stream, &uv_alloc_cb, &uv_read_cb);
        if (r < 0)
          shutdown(r);
      }
      break;
    default:
      break;
  }
}

void pipe_connection::stop() {
  close();
}

static void write_cb(uv_write_t* req, int status) {
  auto conn = static_cast<pipe_connection*>(req->data);
  if (status < 0)
    conn->shutdown(status);
  else {
    conn->handle_write();
    conn->data_to_send.clear();
    if (conn->send_queue.size()) {
      conn->data_to_send = conn->send_queue.pop_front();
      conn->_send();
    }
  }
}

uint pipe_connection::send(bytes data, int flags) {

  if (data_to_send.size() > 0) {
    send_queue.push(array<byte>(data));
    return uint(data.length);
  }
  data_to_send = data;
  return _send();
}

uint pipe_connection::_send() {
  uv_buf_t buf = uv_buf_init((char*)data_to_send.begin(), uint(data_to_send.length()));
  int r;
#if defined(TLS_SUPPORT)
  if( _transport == TCP_TLS )
    r = uv_tls_write(&req.write, &pipe.tcp_tls, &buf, &write_cb);
  else
#endif
    r = uv_write(&req.write, &pipe.stream, &buf, 1, &write_cb);
  if (r < 0) {
    shutdown(r);
    return 0;
  }
  return (uint)data_to_send.length();
}


static void close_cb(uv_handle_t* handle) {
  tool::handle<pipe_connection> conn = static_cast<pipe_connection*>(handle->data);
  conn->handle_close();
  conn->set_state(pipe_connection::CLOSED);
}

#if defined(TLS_SUPPORT)
static void close_cb_tls(uv_tls_t* handle) {
  tool::handle<pipe_connection> conn = static_cast<pipe_connection*>(handle->data);
  conn->handle_close();
  conn->set_state(pipe_connection::CLOSED);
}
#endif

void pipe_connection::close(void) {
  if (is_closed() || _state == CLOSING)
    return;
  _state = CLOSING;
#if defined(TLS_SUPPORT)
  if(_transport == TCP_TLS)
    uv_tls_close(&pipe.tcp_tls, &close_cb_tls);
  else 
#endif
    uv_close((uv_handle_t*)&pipe.stream, &close_cb);
  uv_cancel(&req.req);
}

template <typename T> void pack_int(array<byte> &data, T n) {
  data.push((byte)(0xff & (n >> 24)));
  data.push((byte)(0xff & (n >> 16)));
  data.push((byte)(0xff & (n >> 8)));
  data.push((byte)(0xff & (n)));
}

template <typename T> bool unpack_int(bytes in, T &n) {
  if (in.length < 4)
    return false;
  n = 0;
  n |= (T)in[0] << 24;
  n |= (T)in[1] << 16;
  n |= (T)in[2] << 8;
  n |= (T)in[3];
  return true;
}

void data_connection::handle_read(bytes data) {

  data_in.push(data);

  reading = true;

  for (;;) {

    if (data_in.size() <= 4)
      return;

    uint package_sz = 0;
    unpack_int(data_in(), package_sz);

    assert(package_sz);

    if (data_in.length() < package_sz + 4)
      return;

    array<byte> recived = data_in(4, 4 + package_sz);

    if (data_in.length() == package_sz + 4) {
      data_in.clear();
      reading = false;
    } else
      data_in.remove(0, 4 + package_sz);

    this->on_data_received(recived());
  }
}

#if 0
  void data_connection::handle_read (void)
  {
    byte buffer[CHUNK_SIZE];

    uint n = this->recv(target(buffer));
    if( n == 0 )
      return;

    bytes r(buffer,n);

    data_in.push(r);

    for(;;)
    {

      if(!reading) // initial packet
      {
        if(data_in.size() >= 4) {
          reading = true;
          uint sz = data_in[0];
              sz |= data_in[1] << 8;
              sz |= data_in[2] << 16;
              sz |= data_in[3] << 24;
          data_in.remove(0,4);
          data_in_length = sz;
        } else
          return;
      }

      if( uint(data_in.size()) >= data_in_length )
      {
        this->reading = false;
        this->on_data_received(data_in(0,data_in_length));
        data_in.remove(0,data_in_length);
        //if(data_in.size())
        //  continue;
      }
    }
  }

#endif

bool data_connection::send_data(bytes out) {
  // if(sending)
  //  return false; // didn't finish with previous package
  uint sz = (uint)out.length;

  // data_out.length(4);

  // data_out.push(byte(sz & 0xFF));  sz >>= 8;
  // data_out.push(byte(sz & 0xFF));  sz >>= 8;
  // data_out.push(byte(sz & 0xFF));  sz >>= 8;
  // data_out.push(byte(sz & 0xFF));
  pack_int(data_out, sz);
  data_out.push(out);

  uint n = this->send(data_out());
  // data_out_pos = n;
  if (n == out.length + 4) {
    data_out.clear();
    sent_notify = true;
  } else {
    data_out.remove(0, n);
  }

  sending = data_out.length() > 0;

  /*    {
          sent_notify = true;
          sending = false;
          data_out.clear();
          on_data_sent();
        } */

  return true;
}

void data_connection::handle_write(void) {
  if (_state != CONNECTED) {
    data_out.clear();
    sending = false;
    return;
  }

  if (data_out.size() == 0) {
    this->on_data_sent();
    return;
  }

  uint n = this->send(data_out());

  if (n == data_out.length()) {
    data_out.clear();
    sent_notify = true;
    sending     = false;
  } else {
    data_out.remove(0, n);
  }
}

#if defined(OSX) && 0

static void socketCallback(CFSocketRef          theSocketRef,
                           CFSocketCallBackType theCallbackType,
                           CFDataRef theAddress, const void *theData,
                           void *theInfo) {
  socket_task *st = (socket_task *)theInfo;
  if (st)
    st->poll();
}

void socket_task::bind_socket(socket_handle s) {
  if (!s) {
    if (runLoopSourceRef) {
      CFRunLoopRemoveSource(CFRunLoopGetMain(), runLoopSourceRef,
                            kCFRunLoopDefaultMode);
      CFSocketInvalidate(socketRef);
      runLoopSourceRef = nullptr;
      socketRef        = nullptr;
    }
  }

  CFSocketContext context = {0};

  context.info = this;

  socketRef = CFSocketCreateWithNative(
      nullptr, s, kCFSocketReadCallBack | kCFSocketWriteCallBack, socketCallback,
      &context);
  if (socketRef == nullptr) {
    return;
  }
  runLoopSourceRef = CFSocketCreateRunLoopSource(nullptr, socketRef, 0);
  if (runLoopSourceRef == nullptr) {
    return;
  }
  CFRunLoopAddSource(CFRunLoopGetMain(), runLoopSourceRef,
                     kCFRunLoopDefaultMode);
}
socket_task::~socket_task() {
  if (runLoopSourceRef) {
    CFRunLoopRemoveSource(CFRunLoopGetMain(), runLoopSourceRef,
                          kCFRunLoopDefaultMode);
    CFSocketInvalidate(socketRef);
    runLoopSourceRef = nullptr;
    socketRef        = nullptr;
  }
}

#else

#endif

} // namespace async
} // namespace tool
