#include "tool.h"

#include "../snprintf.h"

#include "tl_eval.h"
#include <math.h>
#include <string.h>
#include <wchar.h>

namespace tool {
namespace eval {

void fetch_literal(parser &p, uint n);
void fetch_const(parser &p, uint n);
void fetch_var(parser &p, uint n);
void store_var(parser &p, uint n);
void fetch_attr(parser &p, uint n);
void store_attr(parser &p, uint n);
void fetch_style_attr(parser &p, uint n);
void store_style_attr(parser &p, uint n);
void push(parser &p, uint n);
void fetch_state(parser &p, uint n);
void store_state(parser &p, uint n);

void parser::raise_error(parse_error_e n, ...) {
  va_list args;
  va_start(args, n);
  char        buf[1024];
  parse_error er;
  er.no      = n;
  er.line_no = 1 + line_no;
  switch (n) {
  case PERR_UNKNOWN_CHARACTER:
    // er.msg = string::format("is not a number");
    do_vsnprintf(buf, 1023, "unknown character with code 0x%x", args);
    break;
  case PERR_UNEXPECTED_TOKEN:
    do_vsnprintf(buf, 1023, "unexpected token '%S'", args);
    break;
  case PERR_EXPECTED_TOKEN:
    do_vsnprintf(buf, 1023, "got '%S' but required %S", args);
    break;
  case PERR_IS_NOT_LVALUE:
    do_vsnprintf(buf, 1023, "is not an l-value", args);
    break;
  case PERR_BAD_NAME_TOKEN:
    do_vsnprintf(buf, 1023, "bad name token '%S'", args);
    break;
  case PERR_UNKNOWN_VARIABLE:
    do_vsnprintf(buf, 1023, "unknown variable '%S'", args);
    break;
  }
  va_end(args);
  er.msg = buf;
  throw er;
}

bool is_nmtoken_char(wchar c) {
  return isalnum(c) || (c == '_') || (c == '$') || (c == '-');
}

parser::token_t parser::get_token() {
  if (saved_token) {
    token_t t   = saved_token;
    saved_token = T_END;
    return t;
  }

  token_value.size(0);

  if (pos >= end)
    return T_END;

  for (;;) {
    while (pos < end) {
      if (*pos == '\n')
        ++line_no;
      else if (!is_space(*pos))
        break;
      ++pos;
    }

    if (pos >= end)
      return T_END;

    switch (*pos) {
    case '+':
      return parser::token_t(*pos++);
    case '-':
      if (*(pos + 1) == '>') {
        // token_value.push(L"->");
        pos += 2;
        return T_INVOKE;
      }
      if (is_alpha(*(pos + 1))) // css allows '-' in nmtokens
        return scan_nmtoken();
      return parser::token_t(*pos++);

    case '/':
      if (*(pos + 1) == '/') {
        while (pos < end)
          if (*pos++ == '\n')
            break;
        continue;
      }
      if (*(pos + 1) == '*') {
        pos += 2;
        while (pos < end - 2)
          if (*pos == '*' && *(pos + 1) == '/')
            break;
          else
            ++pos;
        pos += 2;
        continue;
      }

    case '<':
      if (*(pos + 1) == '=') {
        pos += 2;
        return T_LE;
      }
      return parser::token_t(*pos++);

    case '>':
      if (*(pos + 1) == '=') {
        pos += 2;
        return T_GE;
      }
      return parser::token_t(*pos++);
    case '=':
      if (*(pos + 1) == '=') {
        pos += 2;
        return T_EQ;
      }
      return parser::token_t(*pos++);

    case '|':
      if (*(pos + 1) == '|') {
        pos += 2;
        return T_OR;
      }
      return parser::token_t(*pos++);
    case '&':
      if (*(pos + 1) == '&') {
        pos += 2;
        return T_AND;
      }
      return parser::token_t(*pos++);
    case '!':
      if (*(pos + 1) == '=') {
        pos += 2;
        return T_NE;
      }
      return parser::token_t(*pos++);

    case ':':
      if (*(pos + 1) == ':') {
        pos += 2;
        return T_SATTR;
      }
      return parser::token_t(*pos++);

    case ',':
    case '*':
    case '%':
    case '^':
    case '?':
    case '#':

    case ';':
    case '[':
    case ']':
    case '{':
    case '}':
    case '(':
    case ')':
      return parser::token_t(*pos++);

    case '.':
      if (*(pos + 1) == '.') {
        pos += 2;
        return T_RANGE;
      }
      return parser::token_t(*pos++);

    case '\"':
      return scan_string(*pos++);
    case '\'':
      ++pos;
      return scan_char();

    case '@':
      if (*(pos + 1) == '(') {
        pos += 2;
        return T_FUNC_DECL;
      }
      ++pos;
      return scan_attribute();

    default:
      if (is_digit(*pos))
        return scan_number();
      if (is_nmtoken_char(*pos))
        return scan_nmtoken();
      else
        raise_error(PERR_UNKNOWN_CHARACTER, *pos);
    }
  }
  // return T_END;
}

wchars parser::parse(wchars in /*, const wchar** endptr*/) {
  input = in.start;
  pos   = input;
  end   = in.end();
  token_value.size(0);
  saved_token = T_END;
  // line_no = 0;
  last_line_no = uint(-1);

  push_code(BC_FRAME);
  uint frm_sz_a = push_uint(0); // at 1

  statements();

  put_uint(frm_sz_a, local.size());

  push_code(BC_RETURN);

  in.prune(pos - input);
  
  return in;
}

// parses list of statments, used in media queries to parse things like:
// media="screen, 3d-glasses, print and resolution > 90dpi"
// list items here are comma separated parts.

wchars parser::parse_mediaq(wchars list) {
  input = list.start;
  pos   = input;
  end   = list.end();
  token_value.size(0);
  saved_token = T_END;
  // line_no = 0;
  last_line_no = uint(-1);

  push_code(BC_FRAME);
  uint frm_sz_a = push_uint(0); // at 1

  /*token_t tok;
  while((tok = get_token()) != 0)
  {
     if(tok == ';')
       break;
     if(tok == ';')
       break;
     if(tok == ',')
       continue;
     else
     {
       push_back(tok);
       expr();
     }
  }*/
  token_t tok;
  uint    bc_nxt, bc_end = 0;
  pval    pv;
  expr_or(pv);
  while ((tok = get_token()) == ',') {
    pv.fetch(*this);
    push_code(BC_BRT);
    bc_nxt = push_uint(bc_end);
    expr_or(pv);
    bc_end = bc_nxt;
  }
  push_back(tok);
  pv.fetch(*this);

  fixup(bc_end, code_addr());
  put_uint(frm_sz_a, local.size());
  push_code(BC_RETURN);
  // if( endptr ) *endptr = pos;
  list.prune(pos - input);
  return list;
}

void parser::statements() {
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == ';')
      break;
    if (tok == ',')
      continue;
    else {
      push_back(tok);
      expr();
    }
  }
}

void parser::statement_list() {
  // caller consumed '('
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == ')') {
      push_back(tok);
      break;
    }
    if (tok == ',') {
      token_t ntok = get_token();
      push_back(ntok);
      if (ntok == ')')
        break;
      expr();
    } else {
      push_back(tok);
      expr();
    }
  }
}

void parser::expr() {
  pval pv;
  expr_q(pv);
  token_t tok;
  while (true) {
    tok = get_token();
    if (tok == '=') {
      pval pv2;
      pv.push(*this);
      expr_q(pv2);
      pv2.fetch(*this);
      pv.store(*this);
    } else if (tok == T_INVOKE) {
      // require_token();
      pv.fetch(*this);
      push_code(BC_PUSH);
      token_t ntok = get_token();
      if (ntok == T_NAME) {
        push_back(ntok);
        expr_primary(pv);
      } else if (ntok == T_FUNC_DECL) {
        expr_func(pv);
      } else
        raise_error(PERR_EXPECTED_TOKEN, token_name(ntok).c_str(),
                    L"function name or function declaration: '(' params ')' "
                    L"statement ");
      pv.fetch(*this);
      push_code(BC_INVOKE);
      // push_uint( symbol_id(name));
    } else if (tok == T_END) {
      pv.fetch(*this);
      break;
    } else {
      pv.fetch(*this);
      push_back(tok);
      break;
    }
    pv.clear();
  }
}

void parser::expr_assign(pval &pv) {
  expr_or(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '=') {
      pval pv2;
      pv.push(*this);
      expr_or(pv2);
      pv2.fetch(*this);
      pv.store(*this);
    } else {
      pv.fetch(*this);
      push_back(tok);
      break;
    }
    pv.clear();
  }
}

void parser::expr_q(pval &pv) // handle the '?:' operator
{
  int     nxt, end;
  token_t tok;
  expr_or(pv);
  while ((tok = get_token()) == '?') {
    pv.fetch(*this);
    push_code(BC_BRF);
    nxt = push_uint(0);
    expr_assign(pv);
    pv.fetch(*this);
    if ((tok = get_token()) == '#') {
      push_code(BC_BR);
      end = push_uint(0);
      fixup(nxt, code_addr());
      expr_q(pv);
      pv.fetch(*this);
      fixup(end, code_addr());
    } else {
      push_back(tok);
      fixup(nxt, code_addr());
    }
  }
  push_back(tok);
}

void parser::expr_or(pval &pv) {
  token_t tok;
  uint    nxt, end = 0;
  expr_and(pv);
  while ((tok = get_token()) == T_OR) {
    pv.fetch(*this);
    push_code(BC_BRT);
    nxt = push_uint(end);
    expr_and(pv);
    pv.fetch(*this);
    end = nxt;
  }
  fixup(end, code_addr());
  push_back(tok);
}

void parser::expr_and(pval &pv) {
  token_t tok;
  uint    nxt, end = 0;
  expr_range(pv);
  while ((tok = get_token()) == T_AND) {
    pv.fetch(*this);
    push_code(BC_BRF);
    nxt = push_uint(end);
    expr_range(pv);
    pv.fetch(*this);
    end = nxt;
  }
  fixup(end, code_addr());
  push_back(tok);
}

void parser::expr_range(pval &pv) {
  token_t tok;
  expr_bor(pv);
  if ((tok = get_token()) == T_RANGE) {
    pv.fetch(*this);
    push_code(BC_PUSH);
    expr_bor(pv);
    pv.fetch(*this);
    push_code(BC_RANGE);
  } else
    push_back(tok);
}

void parser::expr_bor(pval &pv) {
  expr_band(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '|') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr_band(pv);
      pv.fetch(*this);
      push_code(BC_OR);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr_band(pval &pv) {
  expr_comp(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '&') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr_comp(pv);
      pv.fetch(*this);
      push_code(BC_AND);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr_comp(pval &pv) {
  expr0(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '<') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_LT);
    } else if (tok == T_LE) {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_LE);
    } else if (tok == '>') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_GT);
    } else if (tok == T_GE) {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_GE);
    } else if (tok == T_EQ) {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_EQ);
    } else if (tok == T_NE) {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_NE);
    }
    else if (tok == T_LIKE) {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_LIKE);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr0(pval &pv) {
  expr1(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '+') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr1(pv);
      pv.fetch(*this);
      push_code(BC_ADD);
    } else if (tok == '-') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr1(pv);
      pv.fetch(*this);
      push_code(BC_SUB);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr1(pval &pv) {
  expr2(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '*') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr2(pv);
      pv.fetch(*this);
      push_code(BC_MUL);
    } else if (tok == '/') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr1(pv);
      pv.fetch(*this);
      push_code(BC_DIV);
    } else if (tok == '%') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr1(pv);
      pv.fetch(*this);
      push_code(BC_MOD);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr2(pval &pv) {
  expr3(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '^') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr2(pv);
      pv.fetch(*this);
      push_code(BC_POWER);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr3(pval &pv) // unary operators
{
  token_t tok;
  switch (uint(tok = get_token())) {
  case '!':
    expr4(pv);
    pv.fetch(*this);
    push_code(BC_NOT);
    break;
  case '-':
    expr4(pv);
    pv.fetch(*this);
    push_code(BC_NEG);
    break;
  case '+':
    expr4(pv);
    pv.fetch(*this);
    break;
  default:
    push_back(tok);
    expr4(pv);
    break;
  }
}

void parser::expr4(pval &pv) {
  expr_primary(pv);
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == '.') {
      token_t ntok = get_token();
      if (ntok != T_NAME)
        raise_error(PERR_UNEXPECTED_TOKEN, token_name(ntok).c_str());

      pv.fetch(*this);

      wchars name = token_value();
      ntok        = get_token();
      if (ntok == '(') {
        expr_call(name, true);
      } else {
        push_back(ntok);
        pv = pval(symbol_id(name), fetch_attr, store_attr, push);
      }
    } else if (tok == ':') {
      token_t ntok = get_token();
      if (ntok != T_NAME)
        raise_error(PERR_UNEXPECTED_TOKEN, token_name(ntok).c_str());

      pv.fetch(*this);
      // push_code( BC_PUSH );

      wchars name = token_value();
      pv          = pval(symbol_id(name), fetch_state, store_state, push);
    } else if (tok == T_SATTR) {
      token_t ntok = get_token();
      if (ntok != T_NAME)
        raise_error(PERR_UNEXPECTED_TOKEN, token_name(ntok).c_str());

      pv.fetch(*this);
      // push_code( BC_PUSH );

      wchars name = token_value();
      pv = pval(symbol_id(name), fetch_style_attr, store_style_attr, push);
    } else {
      push_back(tok);
      break;
    }
  }
}

void parser::expr_func(pval & /*pv*/) {
  // caller consumend '@('
  variables local_vars(this, vars);
  vars = &local_vars;

  // parsing list of parameters
  token_t tok;
  while ((tok = get_token()) != 0) {
    if (tok == T_NAME) {
      wchars name = token_value();
      local_vars.add_name(
          symbol_id(name)); // paramters are just first vars in the local table.
      if ((tok = get_token()) != ',')
        push_back(tok);
    } else if (tok == ')')
      break;
    else
      raise_error(PERR_EXPECTED_TOKEN, token_name(tok).c_str(), L"')'");
  }
  // done, we have list of params at this moment.

  push_code(BC_BR); // jump over the function body
  uint end = push_uint(0);

  // uint num_of_args = local_vars.size(); ha, all locals are like args, let's
  // assume so.

  uint start = code_addr();
  push_code(BC_FRAME);      // function body is a frame
  uint nvars = code_addr(); // num of vars "addr"
  push_uint(0);             // num of vars

  // here goes an expression - body of the function
  expr();

  // done with it, fixup number of local vars uses.
  put_uint(nvars, local_vars.size());

  // push_code(BC_NULL);
  push_code(BC_RETURN); // return that we've got in val register

  fixup(end, code_addr()); // finalize that BC_BR above

  // make a closure of it:
  push_code(BC_CLOSE);
  push_uint(start);
}

void fetch_literal(parser &p, uint n) {
  p.push_code(BC_LITERAL);
  p.push_uint(n);
}

void fetch_const(parser &p, uint n) {
  p.push_code(BC_CONST_GET);
  p.push_uint(n);
}
/*void store_const(parser& p, uint n)
{
  p.push_code( BC_ATTR_SET );
  p.push_uint( n );
}*/

void fetch_var(parser &p, uint name_sym) {
  uint name_idx;
  if (p.vars->is_name(name_sym, name_idx)) {
    p.push_code(BC_VAR_GET);
    p.push_uint(name_idx);
  } else {
    p.push_code(BC_EXT_CONST);
    p.push_uint(name_sym);
  }
  // p.raise_error(PERR_UNKNOWN_VARIABLE, symbol_name(name_sym).c_str());
}
void store_var(parser &p, uint name_sym) {
  uint name_idx;
  if (!p.vars->is_name(name_sym, name_idx))
    name_idx = p.vars->add_name(name_sym);
  p.push_code(BC_VAR_SET);
  p.push_uint(name_idx);
}

void fetch_attr(parser &p, uint n) {
  p.push_code(BC_ATTR_GET);
  p.push_uint(n);
}
void store_attr(parser &p, uint n) {
  p.push_code(BC_ATTR_SET);
  p.push_uint(n);
}

void fetch_style_attr(parser &p, uint n) {
  p.push_code(BC_SATTR_GET);
  p.push_uint(n);
}
void store_style_attr(parser &p, uint n) {
  p.push_code(BC_SATTR_SET);
  p.push_uint(n);
}

void fetch_state(parser &p, uint n) {
  p.push_code(BC_STATE_GET);
  p.push_uint(n);
}
void store_state(parser &p, uint n) {
  p.push_code(BC_STATE_SET);
  p.push_uint(n);
}

void push(parser &p, uint /*n*/) { p.push_code(BC_PUSH); }

void parser::expr_primary(pval &pv) {
  token_t tok = get_token();
  switch (uint(tok)) {
  case T_END:
    pv.clear();
    // push_code( BC_FALSE );
    break;

  case T_STRING: {
    value val = value(ustring(token_value()));
    pv        = pval(prg.set_literal(val), fetch_literal);
  } break;

  case T_CHAR: {
    int cc = 0;
    if (token_value.size() == 2)
      cc = token_value[1] | (token_value[0] == '^' ? 0x80000000 : 0);
    else if (token_value.size() == 1)
      cc = token_value[0];
    else {
      // assert(false);
    }
    value val = value(cc);
    pv        = pval(prg.set_literal(val), fetch_literal);
  } break;

  case T_NUMBER: {
    value val = value::parse(ustring(token_value()));
    pv        = pval(prg.set_literal(val), fetch_literal);
  } break;

  case T_LENGTH: {
    value val = value::parse_numeric_units(ustring(token_value()));
    pv        = pval(prg.set_literal(val), fetch_literal);
  } break;
  case T_DURATION: {
    value val = value::make_duration(str_to_d(token_value().start, (wchar **)0));
    pv = pval(prg.set_literal(val), fetch_literal);
  } break;

  case T_CONST: {
    value val;
    if (!constants->get_const(token_value(), val))
      raise_error(PERR_UNKNOWN_VARIABLE, token_name(tok).c_str());
    pv = pval(prg.set_literal(val), fetch_literal);
  } break;

    /*          case T_CONST:
                {
                  uint attr_sym = symbol_id(token_value());
                  pv = pval( attr_sym, fetch_const );
                } break;*/

    /*case T_CONST:
      {
        uint attr_sym = symbol_id(token_value());
        pv = pval( attr_sym, fetch_attribute, store_attribute );
      } break; */

  case T_TRUE:
    pv.clear();
    push_code(BC_TRUE);
    break;
  case T_FALSE:
    pv.clear();
    push_code(BC_FALSE);
    break;
  case T_NULL:
    pv.clear();
    push_code(BC_NULL);
    break;
  case T_CANCEL:
    pv.clear();
    push_code(BC_CANCEL);
    break;
  case T_SELF:
    pv.clear();
    push_code(BC_SELF);
    break;
  case T_NAME: {
    ustring  name = token_value();
    token_t  ntok = get_token();
    if (ntok == '(') {
      pv.clear();
      if (name == WCHARS("url")) {
        scan_string(')');
        //value val = value(ustring(token_value()));
        //val.units(value::uri);
        value val = value::make_url(token_value());
        pv = pval(prg.set_literal(val), fetch_literal);
      } else {
        push_code(BC_GLOBAL); // global function call
                              // pv = pval( , fetch_var, store_var );
        // fetch_var(*this,symbol_id(name));
        // pv.fetch(*this);
        expr_call(name, false);
      }
    } else {
      push_back(ntok);
      pv = pval(symbol_id(name), fetch_var, store_var);
    }
  } break;
    // case ':':

  case '(':
    pv.clear();
    statement_list();
    require_token(token_t(')'));
    break;

  case T_FUNC_DECL:
    pv.clear();
    expr_func(pv);
    break;

  case T_RETURN: {
    token_t ntok = get_token();
    push_back(ntok);
    if (ntok == ',' || ntok == ';')
      push_code(BC_NULL);
    else
      expr(); // return something
    push_code(BC_RETURN);
  } break;
  default:
    raise_error(PERR_UNEXPECTED_TOKEN, token_name(tok).c_str());
  }
}

void parser::expr_call(const ustring &name, bool method_call) {
  // caller consumed '('
  if (name.like(W("$*"))) {
    expr_quasi_call(name);
    return;
  }

  uint name_id = symbol_id(name);
  if (!method_call) {
    if (vars->is_defined(name_id))
      pval(name_id, fetch_var, store_var).fetch(*this);
  }

  push_code(BC_PUSH); // val contains the object

  token_t tok   = get_token();
  uint    nargs = 1; // always 'this'
  if (tok != ')')
    while (tok) {
      push_back(tok);
      pval pv;
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_PUSH);
      ++nargs;

      tok = get_token();
      if (tok == ',') {
        tok = get_token();
        continue;
      } else if (tok == ')')
        break;
      else
        raise_error(PERR_UNEXPECTED_TOKEN, token_name(tok).c_str());
    }
  push_code(BC_CALL);
  push_uint(name_id);
  push_uint(nargs);
}

void parser::expr_quasi_call(const ustring &name) {
  /*
         This piece parses selector alike functions that always have single
     parameter:
         

           $( div#my ) -> $("div#my")
         

         And with inclusion (marked by '<' and '>' pair):

           $( div:nth-child(<self.index>) ) -> $("div:nth-child(" + self.index +
     ")" )
           $( div[name="<self.name>"]) -> $('div[name="' + self.name + '"]')

   */

  push_code(BC_PUSH); // val contains the object
  int     level = 1;  // as we've got first '(' already
  token_t tok   = scan_selector_string(level);
  assert(tok == T_STRING);
  // uint    nargs = 1; // always 'this'
  push_back(tok);
  pval pv;
  expr0(pv);
  while ((tok = get_token()) != 0) {
    if (tok == '<') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_ADD);
    } else if (tok == '>') {
      pv.fetch(*this);
      push_code(BC_PUSH);
      tok = scan_selector_string(level);
      push_back(tok);
      expr0(pv);
      pv.fetch(*this);
      push_code(BC_ADD);
    } else if (tok == ')') {
      pv.fetch(*this);
      break;
    } else
      raise_error(PERR_UNEXPECTED_TOKEN, token_name(tok).c_str());
  }
  push_code(BC_PUSH);
  push_code(BC_CALL);
  push_uint(symbol_id(name));
  push_uint(2);
}

void parser::require_token(parser::token_t t) {
  //'('
  token_t tok = get_token();
  if (tok != t)
    raise_error(PERR_EXPECTED_TOKEN, token_name(tok).c_str(),
                token_name(t).c_str());
}

parser::token_t parser::scan_number() {
  int n_dots  = 0;
  int n_alpha = 0;
  token_value.push(*pos++);
  while (pos < end) {
    if (is_digit(*pos))
      token_value.push(*pos);
    else if (*pos == '.') {
      if (pos < (end - 1) && *(pos + 1) == '.')
        break; // range
      if (++n_dots > 1)
        break;
      token_value.push(*pos);
    } else if (*pos == 's') {
      ++pos;
      return T_DURATION;
    } else if (is_alpha(*pos) || *pos == '%' || *pos == '*' || *pos == '#') {
      token_value.push(*pos);
      ++n_alpha;
    } else
      break;
    ++pos;
  }
  return n_alpha ? T_LENGTH : T_NUMBER;
}
parser::token_t parser::scan_string(wchar delimeter) {
  while (pos < end) {
    if (*pos == delimeter) {
      ++pos;
      break;
    } else if (*pos == '\\') {
      switch (*(++pos)) {
      case 't':
        token_value.push('\t');
        break;
      case 'r':
        token_value.push('\r');
        break;
      case 'n':
        token_value.push('\n');
        break;
      default:
        token_value.push(*pos);
        break;
      }
    } else
      token_value.push(*pos);
    ++pos;
  }
  return T_STRING;
}

parser::token_t parser::scan_char() {
  scan_string('\'');
  if (token_value.size() > 1) {
    ustring tok = token_value();
    wchars  tv  = tok;

    token_value.clear();
    if (tv[0] == '^')
      token_value.push('^');

    // token_value.clear(); // clear
#if defined(KB_RETURN)	
    if (tv == WCHARS("RETURN")) {
      token_value.push(KB_RETURN);
    } else if (tv == WCHARS("LEFT")) {
      token_value.push(KB_LEFT);
    } else if (tv == WCHARS("RIGHT")) {
      token_value.push(KB_RIGHT);
    } else if (tv == WCHARS("UP")) {
      token_value.push(KB_UP);
    } else if (tv == WCHARS("DOWN")) {
      token_value.push(KB_DOWN);
    } else if (tv == WCHARS("PRIOR")) {
      token_value.push(KB_PRIOR);
    } else if (tv == WCHARS("NEXT")) {
      token_value.push(KB_NEXT);
    } else if (tv == WCHARS("TAB")) {
      token_value.push(KB_TAB);
    } else if (tv == WCHARS("HOME")) {
      token_value.push(KB_HOME);
    } else if (tv == WCHARS("END")) {
      token_value.push(KB_END);
    } else if (tv == WCHARS("DELETE")) {
      token_value.push(KB_DELETE);
    } else if (tv == WCHARS("INSERT")) {
      token_value.push(KB_INSERT);
    } else if (tv == WCHARS("BACK")) {
      token_value.push(KB_BACK);
    } else if (tv == WCHARS("ESCAPE")) {
      token_value.push(KB_ESCAPE);
    } else 
#endif
		if (tv.length == 2)
      token_value.push(tv[1]);
    else {
      token_value.clear();
      token_value.push(0);
    }
  }
  return T_CHAR;
}

parser::token_t parser::scan_selector_string(int &level) {
  while (pos < end) {
    if (*pos == '<') {
      break;
    }
    if (*pos == ')') {
      if (--level == 0)
        break;
      token_value.push(')');
    } else if (*pos == '(') {
      ++level;
      token_value.push('(');
    } else if (*pos == '\\') {
      switch (*(++pos)) {
      case 't':
        token_value.push('\t');
        break;
      case 'r':
        token_value.push('\r');
        break;
      case 'n':
        token_value.push('\n');
        break;
      default:
        token_value.push(*pos);
        break;
      }
    } else
      token_value.push(*pos);
    ++pos;
  }
  return T_STRING;
}

parser::token_t parser::scan_attribute() {
  while (pos < end) {
    if (is_nmtoken_char(*pos))
      token_value.push(*pos);
    else
      break;
    ++pos;
  }
  return T_CONST;
}

parser::token_t parser::scan_nmtoken() {
  token_value.push(*pos++);
  while (pos < end) {
    if (is_nmtoken_char(*pos))
      token_value.push(*pos);
    else
      break;
    ++pos;
  }

  wchars tv = token_value();
  if (tv == WCHARS("true"))
    return T_TRUE;
  else if (tv == WCHARS("null"))
    return T_NULL;
  else if (tv == WCHARS("self"))
    return T_SELF;
  else if (tv == WCHARS("false"))
    return T_FALSE;
  else if (tv == WCHARS("cancel"))
    return T_CANCEL;
  else if (tv == WCHARS("return"))
    return T_RETURN;
  else if (tv == WCHARS("and"))
    return T_AND;
  else if (tv == WCHARS("or"))
    return T_OR;
  else if (tv == WCHARS("like"))
    return T_LIKE;
  else 
    return T_NAME;
}
} // namespace eval
} // namespace tool
