#ifndef __tl_eval_parse_h__
#define __tl_eval_parse_h__

namespace tool {
namespace eval {
struct parser;

typedef void manipula_t(parser &p, uint n);

struct const_resolver {
  virtual bool get_const(wchars name, value &val) = 0;
};

struct parser {
private:
  parser(const parser &o); // we are not copyable, sorry.
  parser &operator=(const parser &);

public:
  conduit &prg;

  const wchar *input;
  const wchar *end;
  const wchar *pos;

  uint         line_no;
  uint         last_line_no;
  array<wchar> token_value;
  enum token_t {
    T_END    = 0,
    T_NUMBER = 256,
    T_LENGTH, // 12px, 1.2em, etc.
    T_STRING,
    T_URL,
    T_CHAR,      // 'ENTER', 'LEFT', 'RIGHT'
    T_CONST,     // e.g. @BACKGROUND
    T_NAME,      // e.g. abc12
    T_TRUE,      // true
    T_FALSE,     // false
    T_NULL,      // null
    T_CANCEL,    // cancel
    T_SELF,      // self
    T_INVOKE,    // obj -> (a,n) statement
    T_LE,        // <=
    T_GE,        // >=
    T_EQ,        // ==
    T_NE,        // !=
    T_FOR,       // for
    T_FUNC_DECL, // '@('
    T_AND,       // &&
    T_OR,        // ||
    T_RETURN,    // return
    T_SATTR,     // '::' - access to style attribute
    T_RANGE,     // '..'
    T_DURATION,  // 123.9s - duration in seconds.
    T_ANGLE,     // 45deg - angle.
    T_LIKE,      // str like "MacOS 10.7.*"

  };

  ustring token_name(token_t t) {
    if (t == T_END)
      return WCHARS("<eof>");
    else if (t >= 256)
      switch (t) {
      case T_NUMBER:
        return WCHARS("number");
      case T_LENGTH:
        return WCHARS("length");
      case T_STRING:
        return WCHARS("string");
      case T_CHAR:
        return WCHARS("char");
      case T_CONST:
        return WCHARS("const");
      case T_NAME:
        return WCHARS("name");
      case T_TRUE:
        return WCHARS("true");
      case T_FALSE:
        return WCHARS("false");
      case T_NULL:
        return WCHARS("null");
      case T_CANCEL:
        return WCHARS("cancel");
      case T_SELF:
        return WCHARS("self");
      case T_INVOKE:
        return WCHARS("->");
      case T_LE:
        return WCHARS("<=");
      case T_GE:
        return WCHARS(">=");
      case T_EQ:
        return WCHARS("==");
      case T_NE:
        return WCHARS("!=");
      case T_FUNC_DECL:
        return WCHARS("@(");
      case T_AND:
        return WCHARS("&&");
      case T_OR:
        return WCHARS("||");
      case T_RETURN:
        return WCHARS("return");
      case T_SATTR:
        return WCHARS("::");
      case T_RANGE:
        return WCHARS("..");
      case T_DURATION:
        return WCHARS("duration");
      case T_ANGLE:
        return WCHARS("angle");
      case T_LIKE:
        return WCHARS("like");
      default:
        return WCHARS("<?>");
      }
    return ustring(wchar(t), 1);
  }

  token_t saved_token;

  struct pval {
    uint        i;
    manipula_t *_fetch;
    manipula_t *_store;
    manipula_t *_push;
    pval() : i(0), _fetch(0), _store(0), _push(0) {}
    pval(uint idx, manipula_t *f, manipula_t *s = 0, manipula_t *p = 0)
        : i(idx), _fetch(f), _store(s), _push(p) {}
    void fetch(parser &p) {
      if (_fetch)
        _fetch(p, i);
      _fetch = 0;
    }
    void store(parser &p) {
      if (_store)
        _store(p, i);
      else
        p.raise_error(PERR_IS_NOT_LVALUE);
      _store = 0;
    }
    void push(parser &p) {
      if (_push)
        _push(p, i);
      _push = 0;
    }
    void clear() {
      _fetch = 0;
      _store = 0;
      _push  = 0;
    }
  };

  struct variables {
    variables *prev;
    parser *   self;

    array<uint> names;

    variables(parser *par = 0, variables *chain = 0) : prev(chain), self(par) {}
    ~variables();
    uint size() const { return (uint)names.length(); }
    bool is_name(uint sym, uint &idx) {
      uint       level = 0;
      variables *t     = this;
      while (t) {
        for (int i = 0; i < t->names.size(); ++i) {
          if (t->names[i] == sym) {
            idx = (level << 16) | i;
            return true;
          }
        }
        ++level;
        t = t->prev;
      }
      return false;
    }
    bool is_defined(uint sym) {
      uint idx;
      return is_name(sym, idx);
    }
    uint add_name(uint sym) {
      names.push(sym);
      return (uint)names.last_index();
    }
  };
  variables       local;
  variables *     vars;
  const_resolver *constants;

  void statements();
  void
       statement_list(); // a.k.a. block, ia a sequence of ',' separated statements.
  void expr();           // =
  void expr_assign(pval &pv); // =
  void expr_q(pval &pv);      // ?:
  void expr_comp(pval &pv);   // <,>, <=, >=, ==
  void expr_or(pval &pv);     // a || b
  void expr_and(pval &pv);    // a && b
  void expr_range(pval &pv);  // a .. b
  void expr_bor(pval &pv);    // a | b
  void expr_band(pval &pv);   // a & b
  void expr0(pval &pv);       // +,-
  void expr1(pval &pv);       // *,/
  void expr2(pval &pv);       // ^
  void expr3(pval &pv);       // unary -+
  void expr4(pval &pv);       // string, number, attribute, (...)
  void expr_call(const ustring &name, bool method_call); // function call
  void expr_primary(pval &pv);               // strings, numbers, etc
  void expr_quasi_call(const ustring &name); // quasi function call, this:
                                             //   $( sel1<stuff>sel2 )
                                             // is getting translated into:
                                             //   $( "sel1" + stuff + "sel2" )
  void expr_func(pval &pv);                  // function declaration:
                            //   '(' <params-list> ')' <statement>

  token_t get_token();
  void    push_back(token_t t) { saved_token = t; }
  void    require_token(token_t t);

  token_t scan_number();
  token_t scan_string(wchar delimeter);
  token_t scan_char();
  token_t scan_selector_string(int &level);
  token_t scan_attribute();
  token_t scan_nmtoken();

  void raise_error(parse_error_e err_num, ...);

  inline void push_code(byte b) {
    if (last_line_no != line_no) {
      prg.codes.push(BC_LINENO);
      push_uint(last_line_no = line_no);
    }
    prg.codes.push(b);
  }
  inline uint push_uint(uint ui) {
    uint  a = code_addr();
    byte *b = (byte *)&ui;
    prg.codes.push(b, 4);
    return a;
  }

  inline void put_uint(uint pos, uint val) { *((uint *)&prg.codes[pos]) = val; }

  inline uint get_uint(uint pos) { return *(uint *)&prg.codes[pos]; }

  uint code_addr() { return (uint)prg.codes.length(); }

  // fixup - fixup a reference chain
  void fixup(uint chn, uint val) {
    for (uint nxt = 0; chn != 0; chn = nxt) {
      nxt = get_uint(chn);
      put_uint(chn, val);
    }
  }

public:
  parser(conduit &c, const string &url, uint line_no, const_resolver *pcr)
      : prg(c), input(0), end(0), pos(0), vars(0), constants(pcr) {
    vars          = &local;
    c.url         = url;
    this->line_no = line_no;
  }
  ~parser() {}
  // return wchars left in expr after parsing
  wchars parse(wchars expr /*, const wchar** endptr = 0*/);
  wchars parse_mediaq(wchars expr); // parse list of expressions, as "exp1,
                                    // exp2, ..." as "(exp1) || (exp2) || ..."
                                    // used in media queries.
};

inline parser::variables::~variables() {
  if (self)
    self->vars = prev;
}

} // namespace eval
} // namespace tool

#endif
