#ifndef __tl_eval_h__
#define __tl_eval_h__

#include "../tl_array.h"
#include "../tl_hash_table.h"
#include "../tl_pool.h"
#include "../tl_string.h"
#include "../tl_ustring.h"
#include "../tl_value.h"
#include "config.h"

#include "tl_eval_bc.h"

#define ATTR_FIRST(NAME) if (attr == WCHARS(NAME)) {
#define ATTR_NEXT(NAME)                                                        \
  return true;                                                                 \
  }                                                                            \
  if (attr == WCHARS(NAME)) {
#define ATTR_LAST                                                              \
  return true;                                                                 \
  }

namespace tool {
namespace eval {
struct vm;

enum parse_error_e {
  PERR_UNKNOWN_CHARACTER,
  PERR_UNEXPECTED_TOKEN,
  PERR_EXPECTED_TOKEN,
  PERR_IS_NOT_LVALUE,
  PERR_BAD_NAME_TOKEN,
  PERR_UNKNOWN_VARIABLE,
};

struct parse_error {
  parse_error_e no;
  string        msg;
  string        url;
  uint          line_no;
  parse_error() : line_no(0), no() {}
  parse_error(const parse_error &e)
      : no(e.no), msg(e.msg), line_no(e.line_no) {}
};

enum runtime_error_e {
  ERR_IS_NOT_NUMBER = 1,
  ERR_ATTR_IS_READ_ONLY,
  ERR_ATTR_UNKNOWN,
  ERR_STATE_IS_READ_ONLY,
  ERR_STATE_UNKNOWN,
  ERR_FUNC_UNKNOWN,
  ERR_LENGTH_CONVERSION,
  ERR_EVENT_UNKNOWN,
  ERR_IS_NOT_ENUMERABLE,
  ERR_IS_NOT_FUNCTION,
  ERR_CONST_UNKNOWN,
  ERR_WRONG_ARGC,
  ERR_ATTR_ON_NULL,
  ERR_SATTR_ON_NULL,
  ERR_STATE_ON_NULL,
};

typedef uint symbol_t;

struct eval_runtime_error {
  runtime_error_e no;
  string          msg;
  string          url;
  uint            line_no;
  eval_runtime_error() : line_no(0) {}
  eval_runtime_error(const eval_runtime_error &e)
      : no(e.no), msg(e.msg), line_no(e.line_no), url(e.url) {}
};

struct conduit : object // compiled bytecodes - produced by the parser and
                        // interpretted by the vm
{
  pool<value>  literals;
  array<byte>  codes;
  mutable uint hash_value;
  mutable bool is_true;         // used in media expressions.
  mutable handle<conduit> prev; // chain, used in media expressions. 
  string       url;
#ifdef _DEBUG
  tristate_v trigger;
#endif

  // value<ustring>   var_names;
  // array<value>     vars;
  conduit(const conduit* previous = nullptr) : hash_value(0), is_true(false), prev(previous) {}
  inline uint  set_literal(const value &v) { return (uint)literals[v]; }
  inline value get_literal(uint id) { return literals(id); }

  unsigned int hash() const {
    if (!hash_value)
      hash_value = tool::hash(codes()) ^ literals.hash();
    if (prev)
      hash_value ^= prev->hash();
    return hash_value;
  }
  bool operator==(const conduit &rs) const {
    //return hash() == rs.hash() && codes == rs.codes && literals == rs.literals;
    if (hash() != rs.hash())
      return false;
    if (codes != rs.codes)
      return false;
    if (literals != rs.literals)
      return false;
    if (!!prev.ptr() != !!rs.prev.ptr())
      return false;
    if (prev.ptr() && (*prev.ptr() != *rs.prev.ptr()))
      return false;
    return true;
  }
  bool operator!=(const conduit &rs) const {  return !(operator==(rs)); }

  ~conduit() {}

  virtual uint type() const { return (uint)CONDUIT_OBJECT; }
};

extern uint    symbol_id(wchars name);
extern ustring symbol_name(uint id);

extern bool set_string_attribute(symbol_t attr_sym, const value &obj,
                                 const value &v);
extern bool get_string_attribute(symbol_t attr_sym, const value &obj, value &v);
extern bool call_string_function(symbol_t attr_sym, const value &obj, uint argn,
                                 const value *argv, value &retval);
extern bool call_intrinsic_function(vm &vm, symbol_t attr_sym, const value &obj,
                                    uint argn, const value *argv,
                                    value &retval);

struct vm {
  handle<conduit> prg;
  hobject         globals;
  hobject         self;

  array<value> _stack;
  uint         line_no;

  value val; // value register

  vm(conduit *c, object *g /*lobals*/, object *s /*elf*/)
      : prg(c), globals(g), self(s), line_no(), argv(), argn(), next() {}

  bool eval();

  void           raise_error(runtime_error_e n, ...);
  value          add(const tool::value &a1, const tool::value &a2);
  value          sub(const tool::value &a1, const tool::value &a2);
  value          div(const tool::value &a1, const tool::value &a2);
  value          mod(const tool::value &a1, const tool::value &a2);
  value          mul(const tool::value &a1, const tool::value &a2);
  value          power(const tool::value &a1, const tool::value &a2);
  value          neg(const tool::value &a1);
  value          bnot(const tool::value &a1);
  bool           to_bool(const tool::value &a1);
  value          cmp(uint code, const tool::value &a1, const tool::value &a2);

  bool to_pixels(const value &v, int &px);

private:
  vm(const vm &_) : prg(_.prg) {}

protected:
  bool _eval(byte *pc);

  struct frame : public resource {
  private:
    frame(const frame &c) : bc_off(c.bc_off), vars(c.vars) {}
    frame &operator=(const frame &c) {
      bc_off = c.bc_off;
      vars   = c.vars;
      return *this;
    }

  public:
    uint bc_off; // current offset to be saved.
    // uint          func_off; // offset of the func this frame belongs to.
    value clos; // current closure
    value vars;
    frame() : bc_off(0) {}
    virtual ~frame() { vars.clear(); }
  };
  struct closure : public resource {
    uint          func_off;
    handle<frame> context;
    virtual ~closure() { context = 0; }
  };
  array<handle<frame>> calls; // call stack
  // value     vars;

  bool set_attribute(symbol_t attr_sym, const value &obj, const value &v) {
    if (obj.is_string())
      return set_string_attribute(attr_sym, obj, v);
    object *po = obj.get_object();
    return po && po->set_attr(symbol_name(attr_sym), v);
  }
  bool get_attribute(symbol_t attr_sym, const value &obj, value &v) {
    if (obj.is_string())
      return get_string_attribute(attr_sym, obj, v);
    object *po = obj.get_object();
    return po && po->get_attr(symbol_name(attr_sym), v);
  }
  bool set_state(symbol_t attr_sym, const value &obj, const value &v) {
    object *po = obj.get_object();
    return po && po->set_state(symbol_name(attr_sym), v);
  }
  bool get_state(symbol_t attr_sym, const value &obj, value &v) {
    object *po = obj.get_object();
    return po && po->get_state(symbol_name(attr_sym), v);
  }
  bool set_style_attribute(symbol_t attr_sym, const value &obj,
                           const value &v) {
    object *po = obj.get_object();
    return po && po->set_style_attr(symbol_name(attr_sym), v);
  }
  bool get_style_attribute(symbol_t attr_sym, const value &obj, value &v) {
    object *po = obj.get_object();
    return po && po->get_style_attr(symbol_name(attr_sym), v);
  }

  bool get_const(symbol_t attr_sym, value &v) {
    return globals && globals->get_const(symbol_name(attr_sym), v);
  }
  bool call_function(symbol_t attr_sym, const value &obj, uint nargs,
                     const value *argv, value &retval) {
    if (obj.is_string())
      return call_string_function(attr_sym, obj, nargs, argv, retval);
    if (obj.is_object() || obj.is_proxy_of_object()) {
      object *po = obj.get_object();
      if (globals == po &&
          call_intrinsic_function(*this, attr_sym, obj, nargs, argv, retval))
        return true;
      return po && po->call(symbol_name(attr_sym), nargs, argv, retval);
    }
    return true;
  }

  bool raise_event(symbol_t attr_sym, const value &obj) {
    object *po = obj.get_object();
    return po && po->raise_event(symbol_name(attr_sym));
  }

  void stack_clear() {
    _stack.clear();
    _stack.size(128);
    _stack.size(0);
  }

  inline void  stack_push(const value &v) { _stack.push(v); }
  inline value stack_pop() {
    value v;
    _stack.pop(v);
    return v;
  }
  inline bool stack_pop(value &v) { return _stack.pop(v); }
  uint        stack_size() const { return (uint)_stack.length(); }

  static bool is_func_ref(const value &v) {
    return v.is_resource() && v.units() == 0xf000;
  }

  value make_func_ref(uint off) {
    closure *pcl  = new closure();
    pcl->context  = calls.last();
    pcl->func_off = off;
    value v;
    v.set_resource(pcl, 0xf000);
    return v;
  }

public:
  uint func_num_vars(const value &v);

  value call(const value &func_ref, uint argc, const value *argv);

  value *argv; //_stack.head();
  int    argn;
  byte * next;
};

} // namespace eval
} // namespace tool

#include "tl_eval_parse.h"

#endif