/* tiscript.c - the main routine */
/*
        Copyright (c) 2005, by Andrew Fedoniouk, Terra Informatica
        All rights reserved
*/

#include "config.h"
#include <cstdio>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "cs.h"
#include "cs_com.h"

/*wchar to_wchar(char c)
{
  wchar wc = '?';
  MultiByteToWideChar(CP_OEMCP,0,&c,1,&wc,1);
  return wc;
}

char to_char(wchar wc)
{
  char c = '?';
  WideCharToMultiByte(CP_OEMCP,0,&wc,1,&c,1,0,0);
  return c;
}*/

struct console_stream: public tis::stream
{
  virtual int  get() { int c = getwchar(); return c != WEOF? c : int(EOS); }
  virtual bool put(int ch) {
    putwchar(wchar(ch));
    return true;
  }
};

console_stream console;

/* prototypes */
static void ErrorHandler(tis::VM *c,int code, char* message, char *stackTrace);
static void CompileFile(tis::VM *c,char *inputName,char *outputName, int forceType);

enum FILE_TYPE
{
  CLIENT_SCRIPT = 1,
  SERVER_SCRIPT,
  BYTECODE,
};

static void LoadFile(tis::VM *c,char *name, int type );
static void ReadEvalPrint(tis::VM *c);
static void Usage(void);

static int sourceType(const char* t)
{
  if(tool::streqi(t, "c"))
    return CLIENT_SCRIPT;
  if(tool::streqi(t, "client"))
    return CLIENT_SCRIPT;
  if(tool::streqi(t, "s"))
    return SERVER_SCRIPT;
  if(tool::streqi(t, "server"))
    return SERVER_SCRIPT;
  if(tool::streqi(t, "b"))
    return BYTECODE;
  if(tool::streqi(t, "bc"))
    return BYTECODE;
  if(tool::streqi(t, "bytecode"))
    return BYTECODE;
  return 0;
}

tis::VM* pvm = 0;

#if defined(WINDOWS)
BOOL WINAPI ctrl_handler(DWORD dwCtrlType)
{
  if(!pvm) return FALSE;
  tis::value evt_sym = 0, vhandler = NULL_VALUE;
  switch(dwCtrlType)
  {
    case CTRL_C_EVENT:         evt_sym = tis::CsSymbolOf("onCtrlC"); break;
    case CTRL_BREAK_EVENT:     evt_sym = tis::CsSymbolOf("onCtrlBreak"); break;
    case CTRL_CLOSE_EVENT:     evt_sym = tis::CsSymbolOf("onClose"); break;
    case CTRL_LOGOFF_EVENT:    evt_sym = tis::CsSymbolOf("onLogoff"); break;
    case CTRL_SHUTDOWN_EVENT:  evt_sym = tis::CsSymbolOf("onExit"); break;
  }
  if(!evt_sym)
    return FALSE;

  if(!tis::CsGetGlobalValue(pvm,evt_sym,&vhandler) || !tis::CsMethodP(vhandler))
    return FALSE;

  tis::value r = FALSE_VALUE;
  try
  {
    r = tis::CsCallFunction(tis::CsCurrentScope(pvm),vhandler,0);
  }
  catch(tis::script_exception&) // uncaught error
  {
    //e;
    tis::CsDisplay(pvm,pvm->val,pvm->standardError);
    return FALSE;
  }
  return r == TRUE_VALUE? TRUE:FALSE;
}

#endif

// main - the main routine
int main(int argc,char **argv)
{
    bool interactiveP = true;
    int  forceType = 0;

    tis::VM vm;

    pvm = &vm;

    // setup standard i/o on console
    vm.standardInput = &console;
    vm.standardOutput = &console;
    vm.standardError = &console;
#if defined(WINDOWS)
    SetConsoleCtrlHandler(ctrl_handler,TRUE);
#endif

    try
    {
        char *inputName = nullptr,*outputName = nullptr;
        //bool verboseP = false;
        int i;

        /* process arguments */
        for (i = 1; i < argc; ++i) {
          if (argv[i][0] == '-')
          {
                switch (argv[i][1])
                {
                case '?':
                    Usage();
                    break;
                case 'c':   /* compile source file */
                    if (argv[i][2])
                        inputName = &argv[i][2];
                    else if (++i < argc)
                        inputName = argv[i];
                    else
                        Usage();
                    CompileFile(&vm,inputName,outputName, forceType);
                    interactiveP = false;
                    outputName = NULL;
                    break;
                case 'g':   /* emit debugging information when compiling */
                    vm.enableDebug = true;
                    break;
                case 'i':   /* enter interactive mode after loading */
                    interactiveP = true;
                    break;
                case 'o':   /* specify output filename when compiling */
                    if (argv[i][2])
                        outputName = &argv[i][2];
                    else if (++i < argc)
                        outputName = argv[i];
                    else
                        Usage();
                    interactiveP = false;
                    break;
                case 't':   /* display values of expressions loaded */
                    //verboseP = true;
                    if (argv[i][2])
                        forceType = sourceType(&argv[i][2]);
                    else if (++i < argc)
                        forceType = sourceType(argv[i]);
                    else
                        Usage();
                    break;
                default:
                    Usage();
                    break;
                }
            }
            else {
                LoadFile(&vm,argv[i],forceType);
                interactiveP = false;
            }
        }
    }
    catch(tis::script_exception&) // uncaught error
    {
      //e;
      tis::CsDisplay(&vm,vm.val,vm.standardError);
      if(!interactiveP)
        return -1;
    }

    /* read/eval/print loop */
    if (interactiveP)
        ReadEvalPrint(&vm);

    /* return successfully */

//FINISH:
#if defined(WINDOWS)
    SetConsoleCtrlHandler(ctrl_handler,FALSE);
#endif
    pvm = 0;

    return 0;
}

/* CompileFile - compile a single file */
static void CompileFile(tis::VM *c,char *inputName,char *outputName, int forceType)
{
    char oname[1024],*p;
    char *ext;

    /* determine the input filename */
    //if ((p = strrchr(inputName,'.')) == NULL) {
    //    strcpy(iname,inputName);
    //    strcat(iname,".js");
    //    inputName = iname;
    //}

    int type = 0;

    if ((ext = strrchr(inputName,'.')) != NULL)
    {
      if (tool::streqi(ext,".js"))
          type = CLIENT_SCRIPT;
      else if (tool::streqi(ext,".tis"))
          type = CLIENT_SCRIPT;
      else if (tool::streqi(ext,".jsp"))
          type = SERVER_SCRIPT;
      else if (tool::streqi(ext,".tsp"))
          type = SERVER_SCRIPT;
    }

    if( type == 0 && forceType == 0 )
    {
        fprintf(stderr,"Unknown file type '%s'\n",inputName);
        exit(1);
    }

    if( forceType )
      type = forceType;


    /* determine the output filename */
    if (outputName && strcmp(outputName, "null") == 0) {
       outputName = "null";
    }
    else if (outputName) {
        if ((p = strrchr(outputName,'.')) == NULL) {
            strcpy_s(oname,outputName);
            strcat_s(oname,".tsb");
            outputName = oname;
        }
    }
    /* construct an output filename */
    else {
        if ((p = strrchr(inputName,'.')) == NULL)
            strcpy_s(oname,inputName);
        else {
            int len = int(p - inputName);
            strncpy_s(oname,inputName,len);
            oname[len] = '\0';
        }
        strcat_s(oname,".tsb");
        outputName = oname;
    }

    /* compile the file */
    printf("Compiling '%s' -> '%s'\n",inputName,outputName);
    tis::CsCompileFile(tis::CsCurrentScope(c),tool::ustring(inputName),tool::ustring(outputName), type == SERVER_SCRIPT);
}

/* LoadFile - load a single file */
static void LoadFile(tis::VM *c,char *name, int forceType)
{
    //tis::stream *s = verboseP ? c->standardOutput : NULL;
    //tis::stream *s = c->standardOutput;

    const char *ext;

    int type = 0;

    if ((ext = strrchr(name,'.')) != NULL)
    {
      if (tool::streqi(ext,".js"))
          type = CLIENT_SCRIPT;
      else if (tool::streqi(ext,".tis"))
          type = CLIENT_SCRIPT;
      else if (tool::streqi(ext,".jsp"))
          type = SERVER_SCRIPT;
      else if (tool::streqi(ext,".tsp"))
          type = SERVER_SCRIPT;
      else if (tool::streqi(ext,".jsb"))
          type = BYTECODE;
      else if (tool::streqi(ext,".tsb"))
          type = BYTECODE;
    }

    if( type == 0 && forceType == 0 )
    {
        fprintf(stderr,"Unknown file type '%s'\n",name);
        exit(1);
    }

    if( forceType )
      type = forceType;

    switch( type )
    {
      default:
      case CLIENT_SCRIPT:
        tis::CsLoadFile(tis::CsGlobalScope(c),tool::ustring(name),0);
        break;
      case SERVER_SCRIPT:
        tis::CsLoadFile(tis::CsGlobalScope(c),tool::ustring(name),c->standardOutput);
        break;
      case BYTECODE:
        tis::CsLoadObjectFile(tis::CsGlobalScope(c),tool::ustring(name));
        break;
    }

}

/* ReadEvalPrint - enter a read/eval/print loop */
static void ReadEvalPrint(tis::VM *c)
{
    char lineBuffer[256];
    tis::pvalue val(c); // pinned value

    for (;;)
    {
      TRY
      {
        printf("\nExpr> "); fflush(stdout);
        if (fgets(lineBuffer,sizeof(lineBuffer),stdin))
        {
            tool::ustring line(lineBuffer);
            val = tis::CsEvalString(tis::CsCurrentScope(c),tis::CsCurrentScope(c)->globals,line, line.length());
            if (val) {
                printf("Value: ");
                tis::CsPrint(c,val,c->standardOutput);
            }
        }
        else
            break; // done
      }
      catch(tis::script_exception&) // uncaught error
      {
        //e;
        tis::CsDisplay(c,c->val,c->standardError);
      }
    }
}


/* Usage - display a usage message and exit */
static void Usage(void)
{
    fputs("\
usage: tiscript.exe \n\
       [-c file]     compile a source file\n\
       [-g]          include debugging information\n\
       [-i]          enter interactive mode after loading\n\
       [-o file]     bytecode output file name for compilation\n\
       [-o null]     output is set to dev/null \n\
       [-t typename] force type of the file (ignore extensions).\n\
                     where typename is one of:\n\
                     c[lient] - client script, \n\
                     s[erver] - server script - interprets <% %>\n\
                                script inclusions, \n\
                     b[ytecode] - load as compiled bytecode file.\n\
       [-?]          display (this) help information\n\
       [-v]          display version information\n\
       [file]        load a source or bytecode file, \n\
                     supported file extensions: \n\
                       tis, js - client script,\n\
                       tsp, jsp - server script,\n\
                       tsb, jsb - compiled script (bytecodes).\n\n\
See: http://terrainformatica.com/tiscript for more details."
       , stderr);

    exit(1);
}

namespace tool {
  namespace async {

    void dispatch::start_timer() {
      assert(false);
    }
    void dispatch::stop_timer() {
      assert(false);
    }
  }
}
