/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"

#include <stdlib.h>
#include <stdio.h>

//#ifndef WINDOWS
//#include <spawn.h>
//#endif

namespace tis {
  
  bool CsEntityMeta(VM* c,value o, value tag, value& val)
  {
    value pack;
    if (!CsEntityMeta(o, pack) || !CsObjectP(pack)) return false;
    return CsGetProperty1(c, pack, tag, &val);
  }
  bool CsSetEntityMeta(VM* c, value &o, value tag, value &val)
  {
    value pack;
    if (!CsEntityMeta(o, pack)) return false;
    PROTECT(o, tag, val);
    if (!CsObjectP(pack)) {
      pack = CsMakeObject(c, UNDEFINED_VALUE);
      CsSetEntityMeta(o, pack);
    }
    return CsSetProperty1(c, pack, tag, val);
  }

  value sym_instance = CsSymbolOf(CHARS("instance"));
  value sym_ref = CsSymbolOf(CHARS("@"));
  value sym_factory = CsSymbolOf(CHARS("factory"));
  
  static value CSF_addBreakpoint(VM *c);
  static value CSF_removeBreakpoint(VM *c);
  // static value CSF_listLoadedScripts(VM *c);
  static value CSF_foreachBreakpoint(VM *c);

  static value CSF_unhandledExceptionHandler(VM *c, value obj);
  static void  CSF_set_unhandledExceptionHandler(VM *c, value obj, value val);

  static value CSF_breakpointHitHandler(VM *c, value obj);
  static void  CSF_set_breakpointHitHandler(VM *c, value obj, value val);

  static value CSF_vm_id(VM *c, value obj);

  static value CSF_startStats(VM *c);
  static value CSF_getStats(VM *c);

  static value CSF_getProperty(VM *c);
  static value CSF_setProperty(VM *c);

  static value CSF_isolateValue(VM *c);
  static value CSF_evalAtBreakpoint(VM *c);

  value CSF_gc(VM *c);

  static value CSF_meta(VM *c);
  static value CSF_entityId(VM *c);

  static value CSF_pendingAsyncTasks(VM *c, value obj);

  /* VM methods */
  static c_method vm_methods[] = {
      C_METHOD_ENTRY("addBreakpoint", CSF_addBreakpoint),
      C_METHOD_ENTRY("removeBreakpoint", CSF_removeBreakpoint),
      C_METHOD_ENTRY("foreachBreakpoint", CSF_foreachBreakpoint),
      C_METHOD_ENTRY("startStats", CSF_startStats),
      C_METHOD_ENTRY("getStats", CSF_getStats),
      C_METHOD_ENTRY("getProperty", CSF_getProperty),
      C_METHOD_ENTRY("setProperty", CSF_setProperty),
      C_METHOD_ENTRY("isolateValue", CSF_isolateValue),
      C_METHOD_ENTRY("evalAtBreakpoint", CSF_evalAtBreakpoint),
      C_METHOD_ENTRY("gc", CSF_gc),
      C_METHOD_ENTRY("meta", CSF_meta),
      C_METHOD_ENTRY("entityId", CSF_entityId),
      //C_METHOD_ENTRY("getActiveAsyncTasks", CSF_getActiveAsyncTasks),
      C_METHOD_ENTRY(0, 0)};

  /* VM properties */
  static vp_method vm_properties[] = {
      VP_METHOD_ENTRY("unhandledExceptionHandler", CSF_unhandledExceptionHandler, CSF_set_unhandledExceptionHandler),
      VP_METHOD_ENTRY("breakpointHitHandler", CSF_breakpointHitHandler, CSF_set_breakpointHitHandler),
      VP_METHOD_ENTRY("pendingAsyncTasks", CSF_pendingAsyncTasks, 0),
      VP_METHOD_ENTRY("id", CSF_vm_id, 0), VP_METHOD_ENTRY(0, 0, 0)};

  static constant vm_constants[] = {
      // CONSTANT_ENTRY("IS_READONLY"  , int_value(tool::filesystem::FA_READONLY
      // )),
      CONSTANT_ENTRY(0, 0)};

  /* CsInitSystem - initialize the 'System' obj */
  void CsInitVM(VM *c) {
    /* create the 'VM' type */

    c->vmDispatch = CsEnterCPtrObjectType(CsGlobalScope(c), "VM", vm_methods, vm_properties);
    if (!c->vmDispatch)
      CsInsufficientMemory(c);
    else {
      CsEnterConstants(c, c->vmDispatch->obj, vm_constants);
    }
  }

  // VM namespace methods

  struct scriptable_debug_peer : debug_peer, gc_callback {
    value breakpointHitHandler;

    DEFINE_TYPE_ID(scriptable_debug_peer);

    scriptable_debug_peer(VM *c, value cb)
        : gc_callback(c), breakpointHitHandler(cb) {
      assert(CsMethodP(cb));
    }

    virtual void on_GC(VM *c) {
      breakpointHitHandler = CsCopyValue(c, breakpointHitHandler);
    }

    virtual SCRIPT_DEBUG_COMMANDS breakpoint_hit(VM *c, const wchar *fileUrl,
                                                 int                atLine,
                                                 const tool::value &envData) {
      tool::auto_state<bool> _(skipLineNoCheck, true);
      auto_scope as(c, CsMethodNamespace(breakpointHitHandler));
      TRY {
        value vurl = 0, venv = 0;
        PROTECT(vurl, venv);
        vurl    = CsMakeString(c, chars_of(fileUrl));
        venv    = value_to_value(c, envData);
        value r = CsCallFunction(&as, breakpointHitHandler, 3, vurl,
                                 CsMakeInteger(atLine), venv);
        assert(CsIntegerP(r));
        if (CsIntegerP(r)) 
          return (SCRIPT_DEBUG_COMMANDS)CsIntegerValue(r);
        else
          return tis::SCRIPT_DEBUG_CONTINUE;
      }
      CATCH_ERROR_NE {
        // assert(false);
        c->standardError->put_str(W("DEBUG_PEER:\n"));
        CsDisplay(c, c->val, c->standardError);
        c->standardError->put_str(W("\n"));
        tis::CsHandleUnhandledError(c);
      }
      return tis::SCRIPT_DEBUG_CONTINUE;
    }
    virtual void requested_data(VM *vm, uint command, const tool::value &data) {
      assert(false); // not used
    }
  };

  value CSF_addBreakpoint(VM *c) {
    if (!c->pdebug || !c->pdebug->is_of_type<scriptable_debug_peer>())
      CsThrowKnownError(c, CsErrGenericError,
                        "script debugging is not enabled");
    int_t  lineNo = 0;
    wchars url;
    CsParseArguments(c, "**S#i", &url.start, &url.length, &lineNo);
    c->pdebug.ptr_of<scriptable_debug_peer>()->add_breakpoint(url, lineNo);
    return UNDEFINED_VALUE;
  }
  value CSF_removeBreakpoint(VM *c) {
    if (!c->pdebug || !c->pdebug->is_of_type<scriptable_debug_peer>())
      CsThrowKnownError(c, CsErrGenericError,
                        "script debugging is not enabled");
    int_t  lineNo = 0;
    wchars url;
    CsParseArguments(c, "**S#i", &url.start, &url.length, &lineNo);
    c->pdebug.ptr_of<scriptable_debug_peer>()->remove_breakpoint(url, lineNo);
    return UNDEFINED_VALUE;
  }
  // value CSF_listLoadedScripts(VM *c) { return UNDEFINED_VALUE; }
  value CSF_foreachBreakpoint(VM *c) {
    if (!c->pdebug || !c->pdebug->is_of_type<scriptable_debug_peer>())
      CsThrowKnownError(c, CsErrGenericError,
                        "script debugging is not enabled");

    value cb = 0;
    CsParseArguments(c, "**V=", &cb, &CsMethodDispatch);
    PROTECT(cb);
    TRY {
      value vurl = 0, vlno = 0;

      array<breakpoint> &breakpoints =
          c->pdebug.ptr_of<scriptable_debug_peer>()->breakpoints;
      for (int n = 0; n < breakpoints.size(); ++n) {
        vurl = CsMakeString(c, CsSymbolName(breakpoints[n].fileUrlSym));
        vlno = CsMakeInteger(breakpoints[n].lineNo);
        CsCallFunction(CsCurrentScope(c), cb, 2, vurl, vlno);
      }
    }
    CATCH_ERROR_NE {
      assert(false);
      tis::CsHandleUnhandledError(c);
    }
    return UNDEFINED_VALUE;
  }

  value CSF_vm_id(VM *c, value obj) {
    i64tow sid(int64(c), 36);
    return CsMakeString(c, sid);
  }

  value CSF_breakpointHitHandler(VM *c, value obj) {
    if (!c->pdebug || !c->pdebug->is_of_type<scriptable_debug_peer>())
      return NULL_VALUE;
    return c->pdebug.ptr_of<scriptable_debug_peer>()->breakpointHitHandler;
  }
  void CSF_set_breakpointHitHandler(VM *c, value obj, value val) {
    if (CsMethodP(val))
      c->pdebug = new scriptable_debug_peer(c, val);
    else if (val == NULL_VALUE) {
      c->pdebug = nullptr;
    }
    else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, val,
                        "either function or null");
  }

  value CSF_startStats(VM *c) {
    if (!c->pdebug) return FALSE_VALUE;
    int argc = CsArgCnt(c);
    for (int narg = 3; narg <= argc; ++narg) {
      value arg = CsGetArg(c, narg);
      if (!CsSymbolP(arg))
        CsThrowKnownError(c, CsErrUnexpectedTypeError, arg, "symbols only");
      if (CsSymbolOf(WCHARS("coverage")) == arg) {
        c->pdebug->start_coverage_gathering();
      } else {
        CsThrowKnownError(c, CsErrGenericError, "unknown option");
      }
      // tool::ustring us = value_to_string(arg);
      // args.push(us);
    }
    return UNDEFINED_VALUE;
  }
  static value CSF_getStats(VM *c) {
    if (!c->pdebug) return FALSE_VALUE;
    value what = 0;
    CsParseArguments(c, "**V", &what);
    if (CsSymbolOf(WCHARS("coverage")) == what) {
      c->pdebug->stop_coverage_gathering();
      value map   = CsMakeObject(c, c->objectObject);
      value fnsym = 0;
      PROTECT(map, fnsym);
      for (int n = 0; n < c->pdebug->sourceFileDefs.size(); ++n) {
        auto def      = c->pdebug->sourceFileDefs.elements()[n];
        fnsym         = def->fileNameSym;
        ustring us    = value_to_string(fnsym);
        value   lines = NULL_VALUE;
        if (def->visitedLines.size()) {
          lines = CsMakeVector(c, def->visitedLines.size() - 1);
          for (int k = 1; k < def->visitedLines.size(); ++k)
            CsSetVectorElement(c, lines, k - 1,
                               CsMakeInteger(def->visitedLines[k]));
        }
        CsSetProperty(c, map, fnsym, lines);
      }
      return map;
    } else
      CsThrowKnownError(c, CsErrGenericError, "unknown option");

    return UNDEFINED_VALUE;
  }

  value CSF_unhandledExceptionHandler(VM *c, value obj) {
    return c->unhandledExceptionHandler;
  }
  void CSF_set_unhandledExceptionHandler(VM *c, value obj, value val) {
    if (CsMethodP(val)) c->unhandledExceptionHandler = val;
  }

  value CSF_getProperty(VM *c) {
    value obj, name;
    CsParseArguments(c, "**VV", &obj, &name);
    value val = UNDEFINED_VALUE;
    CsGetProperty1(c, obj, name, &val);
    return val;
  }

  value CSF_setProperty(VM *c) {
    value obj, name, val;
    CsParseArguments(c, "**VVV", &obj, &name);
    PROTECT(obj,val);
    CsSetProperty(c, obj, name, val);
    return val;
  }

  value CSF_isolateValue(VM *c) {
    value val;
    CsParseArguments(c, "**V", &val);
    tool::value tv = value_to_value_def(c, val);
    return value_to_value(c, tv);
  }

  value CSF_meta(VM *c) {
    value vnode, vname, vval = 0;
    CsParseArguments(c, "**V=V=|V=", &vnode,&CsTupleDispatch,&vname,&CsSymbolDispatch,&vval);
    if (vval) {
      if(!CsSetEntityMeta(c, vnode, vname, vval))
        CsThrowKnownError(c, CsErrUnexpectedTypeError, vnode, "- does not support meta data");
    }
    else {
      if (CsEntityMeta(c, vnode, vname, vval))
        return vval;
    }
    return UNDEFINED_VALUE;
  }

  value CSF_entityId(VM *c) {
    value entity = 0;
    CsParseArguments(c, "**V", &entity);
    if (is_ptr(entity)) {
      if (ptr<header>(entity)->id == 0) ptr<header>(entity)->id = ++c->sequentialId;
      if (ptr<header>(entity)->id == 0) ptr<header>(entity)->id = ++c->sequentialId; // to avoid zero id
      return CsMakeInteger(ptr<header>(entity)->id);
    }
    return CsHashValue(entity);
  }

  value* DebugLocalVarRef(VM *c, value name);

  /*bool localSetter(VM *c, value tag, value val)
  {
    value* pv = DebugLocalVarRef(c, tag);
    if (pv) {
      *pv = val;
      return true;
    }
    return false;
  }
  bool localGetter(VM *c, value tag, value *pval)
  {
    value* pv = DebugLocalVarRef(c, tag);
    if (pv) {
      *pval = *pv;
      return true;
    }
    return false;
  }*/

  bool CsDebugScope::setValue(value tag, value val) {
    value* pv = DebugLocalVarRef(c, tag);
    if (pv) {
      *pv = val;
      return true;
    }
    return false;
  }
  bool CsDebugScope::getValue(value tag, value *pval) {
    value* pv = DebugLocalVarRef(c, tag);
    if (pv) {
      *pval = *pv;
      return true;
    }
    return false;
  }

  value CSF_evalAtBreakpoint(VM *c) {
    value self;
    value ns = 0;
    wchars str;
    CsParseArguments(c, "V*S#|V", &self, &str.start,&str.length,&ns);
    CsDebugScope scope(c, ns ? ns : c->currentNS);
    return CsEvalString(&scope, self, str.start, str.length);
  }

  value CSF_pendingAsyncTasks(VM *c, value self) {
    value list = CsMakeVector(c,c->active_tasks.size());
    auto target = CsVectorTargetElements(list);
    target.copy(c->active_tasks());
    return list;
  }

} // namespace tis
