/* string.c - 'String' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include <stdlib.h>
#include <string.h>
#include "cs.h"
#include <wctype.h>
#include <cwchar>

namespace tis {

  /* method handlers */
  static value CSF_ctor(VM *c);
  static value CSF_toSymbol(VM *c);
  // static value CSF_Index(VM *c);
  // static value CSF_ReverseIndex(VM *c);
  static value CSF_substring(VM *c);
  static value CSF_substr(VM *c);
  static value CSF_splice(VM *c);
  static value CSF_distance(VM *c);

  static value CSF_toInteger(VM *c);
  static value CSF_toFloat(VM *c);
  static value CSF_toNumber(VM *c);

  static value CSF_charAt(VM *c);
  static value CSF_charCodeAt(VM *c);
  static value CSF_concat(VM *c);
  static value CSF_indexOf(VM *c);
  static value CSF_lastIndexOf(VM *c);
  static value CSF_compare(VM *c);
  static value CSF_lexicalCompare(VM *c);
  static value CSF_slice(VM *c);
  static value CSF_printf(VM *c);
  static value CSF_$(VM *c);
  static value CSF_scanf(VM *c);

  static value CSF_trim(VM *c);

  static value CSF_htmlEscape(VM *c);
  static value CSF_htmlUnescape(VM *c);

  static value CSF_urlEscape(VM *c);
  static value CSF_urlUnescape(VM *c);

  static value CSF_toString(VM *c);
  static value CSF_toCssString(VM *c);

  extern value CSF_head(VM *c);
  extern value CSF_tail(VM *c);

  /* from cs_regexp */
  extern value CSF_string_match(VM *c);
  extern value CSF_string_replace(VM *c);
  extern value CSF_string_search(VM *c);
  extern value CSF_string_split(VM *c);

  static value CSF_fromCharCode(VM *c);
  static value CSF_toLowerCase(VM *c);
  static value CSF_toUpperCase(VM *c);
  static value CSF_UID(VM *c);

  static value CSF_distance(VM *c);

  /* virtual property methods */
  static value CSF_length(VM *c, value obj);

  /* String methods */
  static c_method methods[] = {
      C_METHOD_ENTRY("this", CSF_ctor),
      C_METHOD_ENTRY("toLocaleString", CSF_std_toLocaleString),
      C_METHOD_ENTRY("toString", CSF_toString),
      C_METHOD_ENTRY("toHtmlString", CSF_htmlEscape),
      C_METHOD_ENTRY("toUrlString", CSF_urlEscape),
      C_METHOD_ENTRY("toCssString", CSF_toCssString),
      C_METHOD_ENTRY("toSymbol", CSF_toSymbol),

      C_METHOD_ENTRY("substring", CSF_substring),
      C_METHOD_ENTRY("substr", CSF_substr),
      C_METHOD_ENTRY("splice", CSF_splice),

      C_METHOD_ENTRY("charAt", CSF_charAt),
      C_METHOD_ENTRY("charCodeAt", CSF_charCodeAt),
      C_METHOD_ENTRY("concat", CSF_concat),
      C_METHOD_ENTRY("indexOf", CSF_indexOf),
      C_METHOD_ENTRY("lastIndexOf", CSF_lastIndexOf),
      C_METHOD_ENTRY("compare", CSF_compare),
      C_METHOD_ENTRY("lexicalCompare", CSF_lexicalCompare),
      C_METHOD_ENTRY("match", CSF_string_match),
      C_METHOD_ENTRY("replace", CSF_string_replace),
      C_METHOD_ENTRY("search", CSF_string_search),
      C_METHOD_ENTRY("split", CSF_string_split),
      C_METHOD_ENTRY("trim", CSF_trim),

      C_METHOD_ENTRY("htmlEscape", CSF_htmlEscape),
      C_METHOD_ENTRY("htmlUnescape", CSF_htmlUnescape),
      C_METHOD_ENTRY("urlEscape", CSF_urlEscape),
      C_METHOD_ENTRY("urlUnescape", CSF_urlUnescape),

      C_METHOD_ENTRY("slice", CSF_slice),
      C_METHOD_ENTRY("fromCharCode", CSF_fromCharCode),
      C_METHOD_ENTRY("toLowerCase", CSF_toLowerCase),
      C_METHOD_ENTRY("toUpperCase", CSF_toUpperCase),
      C_METHOD_ENTRY("valueOf", CSF_toString),

      C_METHOD_ENTRY("distance", CSF_distance),

      // C_METHOD_ENTRY( "head",             CSF_head ),
      // C_METHOD_ENTRY( "tail",             CSF_tail ),

      C_METHOD_ENTRY("UID", CSF_UID),

      // C_METHOD_ENTRY( "toStream",         CSF_toString ),

      C_METHOD_ENTRY("printf", CSF_printf), C_METHOD_ENTRY("$", CSF_$),
      C_METHOD_ENTRY("scanf", CSF_scanf),
      C_METHOD_ENTRY("toFloat", CSF_toFloat),
      C_METHOD_ENTRY("toInteger", CSF_toInteger),
      C_METHOD_ENTRY("toNumber", CSF_toNumber),

      C_METHOD_ENTRY(0, 0)};

  /* String properties */
  static vp_method properties[] = {VP_METHOD_ENTRY("length", CSF_length, 0),
                                   VP_METHOD_ENTRY(0, 0, 0)};

  /* CsInitString - initialize the 'String' obj */
  void CsInitString(VM *c) {
    c->stringObject =
        CsEnterType(CsGlobalScope(c), "String", &CsStringDispatch);
    CsEnterMethods(c, c->stringObject, methods);
    CsEnterVPMethods(c, c->stringObject, properties);

    // CsSetCObjectPrototype(c->stringObject,CsMakeObject(c,UNDEFINED_VALUE));
    // CsEnterMethods(c,CsCObjectPrototype(c->stringObject),prototype_methods);
    // CsEnterVPMethods(c,CsCObjectPrototype(c->stringObject),prototype_properties);
  }

  /* CSF_ctor - built-in method 'initialize' */
  static value CSF_ctor(VM *c) {
    long  size = 0;
    value init = 0;
    value obj;
    CsParseArguments(c, "V=*|i|V", &obj, &CsStringDispatch, &size, &init);

    wchar cinit = ' ';
    if (init) {
      if (CsStringP(init) && CsStringSize(init))
        cinit = CsStringChar(init, 0);
      else if (CsIntegerP(init))
        cinit = CsIntegerValue(init);
    }
    obj = CsMakeFilledString(c, cinit, size);
    CsCtorRes(c) = obj;
    return obj;
  }

  /* CSF_intern - built-in method 'toSymbol' */
  static value CSF_toSymbol(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsStringDispatch);
    return CsIntern(c, obj);
  }

  static value CSF_charAt(VM *c) {
    wchar *str;
    int    len, index;

    /* parse the arguments */
    CsParseArguments(c, "S#*i", &str, &len, &index);

    if (index >= 0 && index < len)
      return CsMakeSubString(c, CsGetArg(c, 1), index, 1);
    else
      return CsMakeCharString(c, NULL, 0);
  }

  static value CSF_charCodeAt(VM *c) {
    wchar *str;
    int    len, index;

    /* parse the arguments */
    CsParseArguments(c, "S#*i", &str, &len, &index);

    if (index >= 0 && index < len)
      return CsMakeInteger(str[index]);
    else
      return UNDEFINED_VALUE;
  }

  static value CSF_fromCharCode(VM *c) {
    value r;

    string_stream s(c->argc - 3);

    for (int i = 3; i <= c->argc; ++i) {
      value t = CsGetArg(c, i);
      if (CsIntegerP(t)) s.put(CsIntegerValue(t));
      // else
      //  CsDisplay(c,CsGetArg(c,i),&s);
    }

    r = s.string_o(c);

    s.close();

    return r;
  }

  static value CSF_concat(VM *c) {
    wchar *str;
    int    len, i;

    value r;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);
    string_stream s(len);

    s.put_str(str);

    for (i = 3; i <= c->argc; ++i)
      CsDisplay(c, CsGetArg(c, i), &s);

    r = s.string_o(c);

    s.close();

    return r;
  }

  value CSF_std_toLocaleString(VM *c) {
    value obj, r;

    /* parse the arguments */
    CsParseArguments(c, "V*", &obj);
    string_stream s(10);

    CsPrintValue(c, obj, &s, true);
    r = s.string_o(c);
    s.close();
    return r;
  }

  value CSF_std_toString(VM *c) {
    value obj, r;

    /* parse the arguments */
    CsParseArguments(c, "V*", &obj);
    string_stream s(10);

    CsPrintValue(c, obj, &s, false);
    r = s.string_o(c);
    s.close();
    return r;
  }
  value CSF_std_valueOf(VM *c) {
    value obj;
    /* parse the arguments */
    CsParseArguments(c, "V*", &obj);
    return obj;
  }

  /* CSF_indexOf - built-in method 'indexOf' */
  static value CSF_indexOf(VM *c) {
    wchars str;
    wchars str2;

    // wchar *p;
    // int len, len2;
    int startIndex = 0;

    /* parse the arguments */
    CsParseArguments(c, "S#*S#|i", &str.start, &str.length, &str2.start,
                     &str2.length, &startIndex);

    if (startIndex >= str.size() || startIndex < 0) return CsMakeInteger(-1);

    /* find the substr */
    str = str(startIndex);

    // p = wcsstr(str + startIndex,str2);

    // if (!p)
    //    return CsMakeInteger(-1);

    /* return the index of the substring */
    // return CsMakeInteger(int_t(p - str));
    int r = str.index_of(str2);
    if (r < 0)
      return CsMakeInteger(-1);
    else
      return CsMakeInteger(r + startIndex);
  }

  /* CSF_toLowerCase - built-in method 'toLowerCase' */
  static value CSF_toLowerCase(VM *c) {
    wchar *str;
    int    len;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);

    tool::ustring us(str, len);
    us.to_lower();
    return string_to_value(c, us);
  }

  /* CSF_trim - built-in method 'trim' */
  static value CSF_trim(VM *c) {
    static int s_all   = symbol_table()[WCHARS("all")];
    static int s_left  = symbol_table()[WCHARS("left")];
    static int s_right = symbol_table()[WCHARS("right")];

    wchar *str;
    int    len;
    int    flags = s_all;

    /* parse the arguments */
    CsParseArguments(c, "S#*|L", &str, &len, &flags);

    const wchar *p;
    const wchar *start = str;
    const wchar *end   = str + len;

    if (flags == s_all || flags == s_left)
      for (; start < end; ++start)
        if (!iswspace(*start)) break;

    if (flags == s_all || flags == s_right)
      for (p = end; --p >= start;) {
        if (!iswspace(*p))
          break;
        else
          end = p;
      }

    tool::wchars r = tool::wchars::range(start, end);

    if (r.size() == len) return CsGetArg(c, 1); // return this;
    return CsMakeString(c, r);
  }

  /* CSF_htmlEscape - built-in method 'htmlEscape' */
  static value CSF_htmlEscape(VM *c) {
    wchar *str;
    int    len;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);

    const wchar *cpstart = str;
    const wchar *cp      = cpstart;
    const wchar *cpend   = str + len;

    while (cp < cpend) {
      if (*cp == '<' || *cp == '>' || *cp == '&' || *cp == '"' || *cp == '\'' || *cp < ' ') {
        tool::array<wchar> buf;
        buf.push(cpstart, cp - cpstart);
        while (cp < cpend) {
          switch (*cp) {
          case '<': buf.push(WCHARS("&lt;")); break;
          case '>': buf.push(WCHARS("&gt;")); break;
          case '&': buf.push(WCHARS("&amp;")); break;
          case '"': buf.push(WCHARS("&quot;")); break;
          case '\'': buf.push(WCHARS("&apos;")); break;
          case '\t':
          case '\r':
          case '\n': buf.push(*cp); break;
          default:
            if (*cp < ' ') {
              tool::ustring es = tool::ustring::format(W("&#%d;"), *cp);
              buf.push(es.c_str(), es.length());
            } else
              buf.push(*cp);
            break;
          }
          ++cp;
        }
        return CsMakeCharString(c, buf.head(), buf.size());
      }
      ++cp;
    }
    return CsGetArg(c, 1);
  }

  inline ucode parse_entity(const wchar *cp, const wchar *cpend,
                            const wchar *&cpout) {
    // caller consumed '&'
    int last = min(16, (cpend - cp));
    for (int n = 2; n < last; ++n)
      if (cp[n] == ';') {
        cpout = &cp[n + 1];
        return tool::html_unescape(tool::string(cp, n));
      }
    return 0;
  }

  /* CSF_htmlEscape - built-in method 'htmlEscape' */
  static value CSF_htmlUnescape(VM *c) {
    wchar *str;
    int    len;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);

    const wchar *cpstart = str;
    const wchar *cp      = cpstart;
    const wchar *cpend   = str + len;

    while (cp < cpend) {
      if (*cp == '&') {
        tool::array<wchar> buf;
        buf.push(cpstart, cp - cpstart);
        while (cp < cpend) {
          if (*cp == '&') {
            const wchar *cpnext = cp + 1;
            ucode        xc     = parse_entity(cp + 1, cpend, cpnext);
            if (xc) {
              if (xc > 0xffff) {
                wchar w2[2];
                u16::putc(xc, w2);
                buf.push(w2[0]);
                buf.push(w2[1]);
              } else
                buf.push(wchar(xc));
              cp = cpnext;
              continue;
            } else
              buf.push('&');
          } else
            buf.push(*cp);
          ++cp;
        }
        return CsMakeCharString(c, buf.head(), buf.size());
      }
      ++cp;
    }
    return CsGetArg(c, 1);
  }

  /* CSF_htmlEscape - built-in method 'htmlEscape' */
  static value CSF_urlEscape(VM *c) {
    wchars str;
    value  how = 0;
    /* parse the arguments */
    CsParseArguments(c, "S#*|V", &str.start, &str.length,&how);

    tool::string r;
    
    if(how == CsSymbolOf(WCHARS("as-parameter")))
      r = tool::url::escape_param(str);
    else
      r = tool::url::escape(str);

    return CsMakeCString(c, r);
  }

  /* CSF_htmlEscape - built-in method 'htmlEscape' */
  static value CSF_urlUnescape(VM *c) {
    wchar *str;
    int    len;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);

    tool::ustring r = tool::url::unescape(tool::string(str, len));

    return CsMakeCString(c, r);
  }

  /* CSF_toUpperCase - built-in method 'toUpperCase' */
  static value CSF_toUpperCase(VM *c) {
    wchar *str;
    int    len;

    /* parse the arguments */
    CsParseArguments(c, "S#*", &str, &len);

    tool::ustring us(str, len);
    us.to_upper();
    return string_to_value(c, us);
  }

  static value CSF_toString(VM *c) {
    value obj;
    CsParseArguments(c, "V=*", &obj, &CsStringDispatch);
    return obj;
  }


  void escape_c_string(wchars us, array<wchar> &out) {
    const wchar *p = us.start;
    const wchar *end = us.end();
    for (; p < end; ++p) {
      switch (*p) {
      case '"':
        out.push('\\');
        out.push('"');
        continue;
      case '\\':
        out.push('\\');
        out.push('\\');
        continue;
      case '\b':
        out.push(WCHARS("\\b"));
        continue;
      case '\f':
        out.push(WCHARS("\\f"));
        continue;
      case '\n':
        out.push(WCHARS("\\n"));
        continue;
      case '\r':
        out.push(WCHARS("\\r"));
        continue;
      case '\t':
        out.push(WCHARS("\\t"));
        continue;
      default:
        if (*p < ' ') {
          out.push(WCHARS("\\u")); // case \u four-hex-digits
          out.push(ustring::format(W("%.4x"), *p));
          continue;
        }
        break;
      }
      out.push(*p);
    }
  }
  
  static value CSF_toCssString(VM *c) {
    wchars str;
    CsParseArguments(c, "S#*", &str.start, &str.length);
    array<wchar> buf;
    escape_c_string(str, buf);
    return CsMakeString(c, buf());
  }

  /*static wchar *wcsrnstr(const wchar *sbStr, size_t sbStrLen, const wchar
  *sbSub)
  {
    wchar ch, *p, *pSub = (wchar *)sbSub;
    int wLen;
    ch = *pSub++;
    if (ch == '\0') return (wchar *)sbStr; // arbitrary return (undefined)
    wLen = (int) str_len(pSub);
    for (p=(wchar *)sbStr + sbStrLen - 1; p >= sbStr; --p)
    {
       if (*p == ch && wcsncmp(p+1, pSub, wLen) == 0) return p;  // found
    }
    return NULL;
  }*/

  /* CSF_lastIndexOf - built-in method 'indexOf' */
  static value CSF_lastIndexOf(VM *c) {
    wchars str, str2;
    int    startIndex = -1;

    /* parse the arguments */
    CsParseArguments(c, "S#*S#|i", &str.start, &str.length, &str2.start,
                     &str2.length, &startIndex);

    if (startIndex < 0)
      startIndex = str.size();
    else if (startIndex > str.size())
      startIndex = str.size();
    else if (startIndex < str.size())
      str = str(startIndex);

    int r = str.last_index_of(str2);

    if (r < 0) return CsMakeInteger(-1);

    return CsMakeInteger(r);

    /* find the substr */
    // p = wcsrnstr(str, startIndex,str2);

    // if (!p)
    //    return CsMakeInteger(-1);

    /* return the index of the substring */
    // return CsMakeInteger(int_t(p - str));
  }

  value CsStringHead(VM *c, value s, value d) {
    tool::wchars ss(CsStringAddress(s), CsStringSize(s));
    tool::wchars sr;
    if (CsIntegerP(d))
      sr = ss.head((wchar)to_int(d));
    else if (CsStringP(d)) {
      tool::wchars sd(CsStringAddress(d), CsStringSize(d));
      sr = ss.head(sd);
    } else
      CsUnexpectedTypeError(c, d, "string or char code");
    if (sr.start == 0) return s;
    return CsMakeString(c, sr);
  }

  value CsStringTail(VM *c, value s, value d) {
    tool::wchars ss(CsStringAddress(s), CsStringSize(s));
    tool::wchars sr;
    if (CsIntegerP(d))
      sr = ss.tail((wchar)to_int(d));
    else if (CsStringP(d)) {
      tool::wchars sd(CsStringAddress(d), CsStringSize(d));
      sr = ss.tail(sd);
    } else
      CsUnexpectedTypeError(c, d, "string or char code");
    // if( sr.start == 0 )
    //  return CsMakeString(c,sr); - it returns empty string in this case
    return CsMakeString(c, sr);
  }

  value CsStringHeadR(VM *c, value s, value d) {
    tool::wchars ss(CsStringAddress(s), CsStringSize(s));
    tool::wchars sr;
    if (CsIntegerP(d))
      sr = ss.r_head((wchar)to_int(d));
    else if (CsStringP(d)) {
      tool::wchars sd(CsStringAddress(d), CsStringSize(d));
      sr = ss.r_head(sd);
    } else
      CsUnexpectedTypeError(c, d, "string or char code");
    if (sr.start == 0) return s;
    return CsMakeString(c, sr);
  }

  value CsStringTailR(VM *c, value s, value d) {
    tool::wchars ss(CsStringAddress(s), CsStringSize(s));
    tool::wchars sr;
    if (CsIntegerP(d))
      sr = ss.r_tail((wchar)to_int(d));
    else if (CsStringP(d)) {
      tool::wchars sd(CsStringAddress(d), CsStringSize(d));
      sr = ss.r_tail(sd);
    } else
      CsUnexpectedTypeError(c, d, "string or char code");
    // if( sr.start == 0 )
    //  return s; - it returns empty string in this case
    return CsMakeString(c, sr);
  }

#if 0
/* CSF_localeCompare - built-in method 'localeCompare' */
static value CSF_localeCompare(VM *c)
{
    wchar *str = 0,*str2 = 0;
    int len, len2;

    /* parse the arguments */
    CsParseArguments(c,"S#*S#",&str,&len,&str2,&len2);

    if ( !str || !str2)
        return UNDEFINED_VALUE;
//#if defined(PLATFORM_WINCE)        
    return CsMakeInteger(wcscmp(str,str2));
//#else
//  return CsMakeInteger(wcscoll(str,str2));
//#endif

}

#endif

  /* CSF_lexicalCompare - built-in method 'lexicalCompare', compares strings in
   * lexical order */
  static value CSF_lexicalCompare(VM *c) {
    tool::wchars s1;
    tool::wchars s2;
    bool         case_insesitive = false;

    /* parse the arguments */
    CsParseArguments(c, "S#*S#|B", &s1.start, &s1.length, &s2.start, &s2.length,
                     &case_insesitive);

    if (!s1.start || !s2.start) return UNDEFINED_VALUE;

    int r = case_insesitive ? tool::lexical::ci::cmp(s1, s2, c->lang()())
                            : tool::lexical::cs::cmp(s1, s2, c->lang()());

    return CsMakeInteger(r);
  }

  /* CSF_compare - built-in method 'compare', strcmp */
  static value CSF_compare(VM *c) {
    tool::wchars s1;
    tool::wchars s2;

    /* parse the arguments */
    CsParseArguments(c, "S#*S#", &s1.start, &s1.length, &s2.start, &s2.length);

    if (!s1.start || !s2.start) return UNDEFINED_VALUE;

    if (s1.length > s2.length) return CsMakeInteger(1);
    if (s1.length < s2.length) return CsMakeInteger(-1);

    return CsMakeInteger(memcmp(s1.start, s2.start, s1.length * sizeof(wchar)));
  }

  static value CSF_slice(VM *c) {
    int    len, start, end = -1;
    wchar *str;

    /* parse the arguments */
    CsParseArguments(c, "S#*i|i", &str, &len, &start, &end);

    /* handle indexing from the left */
    if (start > 0) {
      if (start > len) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (start < 0) {
      if ((start = len + start) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (end < 0) {
      end = len + end + 1;
      if (end < 0) end = 0;
    } else if (end > len)
      end = len;

    if (start > end) return CsMakeFilledString(c, 0, 0);

    /* return the substring */
    return CsMakeCharString(c, str + start, end - start);
  }

  static value CSF_printf(VM *c) {
    value         r;
    string_stream s(10);
    s.printf_args(c);
    r = s.string_o(c);
    s.close();
    // CsParseArguments(c,"S#*i|i",&str,&len,&start,&end);
    return r;
  }

  // simple and straightforward stringizer method, example:
  // var msg = String.$(XML error at line {scanner.lineNo});
  static value CSF_$(VM *c) {
    string_stream s;
    int           i, argc = CsArgCnt(c);
    for (i = 3; i <= argc; ++i)
      CsToString(c, CsGetArg(c, i), s);
    return s.string_o(c);
  }

  static value CSF_scanf(VM *c) {
    tool::wchars str;
    wchar *      fmt = 0;

    /* parse the arguments */
    CsParseArguments(c, "S#*S", &str.start, &str.length, &fmt);

    string_i_stream s(str);
    return s.scanf(c, fmt);
  }

  /* CsStringSlice - make substring */
  value CsStringSlice(VM *c, value s, int start, int end) {
    int len = CsStringSize(s);
    // wchar *str = CsStringAddress(s);

    /* handle indexing from the left */
    if (start > 0) {
      if (start > len) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (start < 0) {
      if ((start = len + start) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (end < 0) {
      end = len + 1 + end;
      if (end < 0) end = 0;
    } else if (end > len)
      end = len;

    if (start > end) tool::swap(start, end);

    /* return the substring */
    return CsMakeSubString(c, s, start, end - start);
  }

  /* CSF_substring - built-in method 'substring' */
  static value CSF_substring(VM *c) {
    int    len, start, end = -1;
    wchar *str;

    /* parse the arguments */
    CsParseArguments(c, "S#*i|i", &str, &len, &start, &end);

    /* handle indexing from the left */
    if (start > 0) {
      if (start > len) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (start < 0) {
      if ((start = len + start) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (end < 0) {
      end = len + 1 + end;
      if (end < 0) end = 0;
    } else if (end > len)
      end = len;

    if (start > end) tool::swap(start, end);

    /* return the substring */
    // return CsMakeCharString(c,str + start, end - start);
    return CsMakeSubString(c, CsGetArg(c, 1), start, end - start);
  }

  /* CSF_substr - built-in method 'substr' */
  static value CSF_substr(VM *c) {
    int    len, i, cnt = -1;
    wchar *str;

    /* parse the arguments */
    CsParseArguments(c, "S#*i|i", &str, &len, &i, &cnt);

    /* handle indexing from the left */
    if (i > 0) {
      if (i > len) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (i < 0) {
      if ((i = len + i) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (cnt < 0) {
      cnt = len - i;
    } else if (i + cnt > len)
      cnt = len - i;

    if (cnt < 0) return UNDEFINED_VALUE;

    /* return the substring */
    return CsMakeSubString(c, CsGetArg(c, 1), i, cnt);
  }

  
  static value CSF_distance(VM *c) {
    /* parse the arguments */
    wchars s1, s2;
    CsParseArguments(c, "S#*S#", &s1.start,&s1.length, &s2.start,&s2.length);
    return CsMakeInteger(tool::levenshtein_distance(s1, s2));
  }

  static value CSF_splice(VM *c) {
    value stro = 0;

    int start, cnt = -1;

    /* parse the arguments */
    CsParseArguments(c, "V=*i|i|", &stro, &CsStringDispatch, &start, &cnt);

    wchars str = CsStringChars(stro);

    /* handle indexing from the left */
    if (start > 0) {
      if (start > str.size()) return UNDEFINED_VALUE;
    }

    /* handle indexing from the right */
    else if (start < 0) {
      if ((start = str.size() + start) < 0) return UNDEFINED_VALUE;
    }

    /* handle the count */
    if (cnt < 0)
      cnt = str.size() - start;
    else if (start + cnt > str.size())
      cnt = str.size() - start;

    if (cnt < 0) return UNDEFINED_VALUE;

    int_t counter = str.size() - cnt;

    for (int n = 5; n <= CsArgCnt(c); ++n) {
      value v = CsGetArg(c, n);
      if (CsStringP(v))
        counter += CsStringSize(v);
      else
        counter += 1;
    }

    // allocate the result string:

    CsPush(c, stro);
    value outs = CsMakeFilledString(c, 0, counter);
    stro       = CsPop(c);
    str        = CsStringChars(stro);

    tslice<wchar> dst = CsStringTargetChars(outs);

    dst = dst.copy(str.start, size_t(start));

    str.prune(start + cnt);

    /* return the slice */

    for (int n = 5; n <= CsArgCnt(c); ++n) {
      value v = CsGetArg(c, n);
      if (CsStringP(v)) {
        wchars other = CsStringChars(v);
        dst          = dst.copy(other);
      } else {
        dst = dst.copy(wchar('?'));
      }
    }

    dst.copy(str);
    return outs;
  }

  /* CSF_toInteger - built-in method 'toInteger' */
  static value CSF_toInteger(VM *c) {
    value obj;
    value dv    = UNDEFINED_VALUE;
    int_t radix = 0;
    // wchar *pend;
    CsParseArguments(c, "V=*|V|i", &obj, &CsStringDispatch, &dv, &radix);
    tool::wchars s = CsStringChars(obj);
    s              = tool::trim(s);
    int_t i        = 0;
    if (!tool::parse_int(s, i, radix)) // wcstol(s.start,&pend,radix);
      return dv;
    return CsMakeInteger(i);
  }

  /* CSF_toFloat - built-in method 'toFloat' */
  static value CSF_toFloat(VM *c) {
    value obj;
    value dv = UNDEFINED_VALUE;
    // wchar *pend;
    CsParseArguments(c, "V=*|V", &obj, &CsStringDispatch, &dv);
    tool::wchars s = CsStringChars(obj);
    s              = tool::trim(s);
    double d       = 0; // wcstod(s.start,&pend);
    if (!tool::parse_real(s, d)) return dv;
    return CsMakeFloat(d);
  }

  /* CSF_toNumber - built-in method 'toNumber', returns either float or integer
   * or dv or undefined */
  static value CSF_toNumber(VM *c) {
    value  obj;
    value  dv = 0;
    wchar *pend;
    CsParseArguments(c, "V=*|V", &obj, &CsStringDispatch, &dv);
    tool::wchars s  = CsStringChars(obj);
    s               = tool::trim(s);
    tool::wchars s1 = s;
    // double d = (s.start,&pend);
    int_t i = str_to_i(s, 0);
    if (s.length != 0) {
      double d = str_to_d(s1.start, &pend);
      if (s.end() != pend) return dv ? dv : UNDEFINED_VALUE;
      return CsMakeFloat(d);
    }
    return CsMakeInteger(i);
  }

  /* CSF_size - built-in property 'length' */
  static value CSF_length(VM *c, value obj) {
    return CsMakeInteger(CsStringSize(obj));
  }

  static value CSF_UID(VM *c) {
    bool sequential = false;
    CsParseArguments(c, "**|B", &sequential);

    tool::string uid = tool::unique_id();

    if (sequential) {
      tool::datetime_t t = tool::date_time::now().time();
      // this ensures that two consequtive calls of Date.now(true) return
      // distinct values.
      static tool::mutex      guard;
      static tool::datetime_t last_t = 0;
      tool::critical_section  _(guard);
      if (t <= last_t) t = last_t + 1;
      last_t = t;
      return CsMakeCString(c, tool::string::format("%08X%08X.%s", hidword(t),
                                                   lodword(t), uid.c_str()));
    } else
      return CsMakeCString(c, uid);
  }

  /* String handlers */
  static bool  GetStringProperty(VM *c, value &obj, value tag, value *pValue);
  static bool  SetStringProperty(VM *c, value obj, value tag, value value);
  static value StringNewInstance(VM *c, value proto);
  static bool  StringPrint(VM *c, value val, stream *s, bool toLocale);
  static long  StringSize(value obj);
  static int_t StringHash(value obj);
  static value StringCopy(VM *c, value obj);

  static value CsStringGetItem(VM *c, value obj, value tag);
  static void  CsStringSetItem(VM *c, value obj, value tag, value value);

  static value StringNextElement(VM *c, value *index, value str, int nr) {
    if (*index == NOTHING_VALUE) // first
    {
      if (CsStringSize(str)) {
        wchars utf16 = CsStringChars(str);
        uint   ch    = u16::getc(utf16);
        *index       = CsMakeInteger(int_t(utf16.start - CsStringAddress(str)));
        //CsSetRVal(c, 1, *index);
        //return CsMakeInteger(int_t(ch));
        CS_RETURN2(c, *index, CsMakeInteger(int_t(ch)));
      }
    } else if (CsIntegerP(*index)) {
      int_t  i     = CsIntegerValue(*index);
      wchars utf16 = CsStringChars(str);
      utf16.prune(i);
      if (utf16.length) {
        uint ch = u16::getc(utf16);
        *index  = CsMakeInteger(int_t(utf16.start - CsStringAddress(str)));
        //CsSetRVal(c, 1, *index);
        //return CsMakeInteger(ch);
        CS_RETURN2(c, *index, CsMakeInteger(int_t(ch)));
      }
    } else
      assert(false);
    //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
    return NOTHING_VALUE;
  }

  /* String pdispatch */
  dispatch CsStringDispatch = {
      "String",          &CsStringDispatch, GetStringProperty,
      SetStringProperty, StringNewInstance, StringPrint,
      StringSize,        StringCopy,        CsDefaultScan,
      StringHash,        CsStringGetItem,   CsStringSetItem,
      StringNextElement};

  static value CsStringGetItem(VM *c, value obj, value tag) {
    if (CsIntegerP(tag)) {
      int_t i;
      if ((i = CsIntegerValue(tag)) < 0 || (size_t)i >= CsStringSize(obj))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      //return CsMakeInteger(CsStringChar(obj, i));
      return CsMakeString(c, CsStringChars(obj)(i, i + 1));
    }
    return UNDEFINED_VALUE;
  }
  static void CsStringSetItem(VM *c, value obj, value tag, value value) {
    /*if (CsIntegerP(tag)) {
      int_t i;
      if (!CsIntegerP(value)) CsTypeError(c, value);
      if ((i = CsIntegerValue(tag)) < 0 || (size_t)i >= CsStringSize(obj))
        CsThrowKnownError(c, CsErrIndexOutOfBounds, tag);
      //CsSetStringChar(obj, i, (wchar)CsIntegerValue(value));
    }*/
    CsThrowKnownError(c, CsErrIsFrozen, obj);
  }

  /* GetStringProperty - String get property handler */
  static bool GetStringProperty(VM *c, value &obj, value tag, value *pValue) {
    return CsGetVirtualProperty(c, obj, c->stringObject, tag, pValue);
  }

  /* SetStringProperty - String set property handler */
  static bool SetStringProperty(VM *c, value obj, value tag, value value) {
    return CsSetVirtualProperty(c, obj, c->stringObject, tag, value);
  }

  /* StringNewInstance - create a new string */
  static value StringNewInstance(VM *c, value proto) {
    return CsMakeCharString(c, NULL, 0);
  }

  /* StringPrint - String print handler */
  static bool StringPrint(VM *c, value val, stream *s, bool toLocale) {
    wchar *p    = CsStringAddress(val);
    long   size = CsStringSize(val);
    if (!s->put('"')) return false;
    while (--size >= 0)
      if (!s->put(*p++)) return false;
    // return true;
    return s->put('"');
  }

  /* StringSize - String size handler */
  static long StringSize(value obj) {
    return sizeof(CsString) +
           CsRoundSize((CsStringSize(obj) + 1) * sizeof(wchar));
  }

  static value StringCopy(VM *c, value obj) {
    value robj = CsDefaultCopy(c, obj);
    CsSetStringAllocatedSize(robj, CsStringSize(robj));
    return robj;
  }

  /* StringHash - String hash handler */
  static int_t StringHash(value obj) {
    CsString *ps = ptr<CsString>(obj);
    if (ps->hash == 0) {
      // WARNING, this may lead to overflow of symbol table when number string
      // keys exceeds 65538
      //    ps->hash = (int_t)CsSymbolIdx(CsStringChars(obj)); // hash value of
      //    string shall match hash value of symbol having same chars
      ps->hash = CsHashString(CsStringChars(obj));
    }
    return ps->hash;
    // return CsHashString(CsStringAddress(obj),CsStringSize(obj));
  }

  /* CsMakeString - make and initialize a new string value */
  value CsMakeCharString(VM *c, const wchar *data, size_t size) {
    long allocSize =
        sizeof(CsString) +
        CsRoundSize((size + 1) * sizeof(wchar)); /* space for zero terminator */
    value newo = CsAllocate(c, allocSize);
    CsSetDispatch(newo, &CsStringDispatch);
    CsSetStringSize(newo, size);
    CsSetStringAllocatedSize(newo, size);
    if (data)
      target(CsStringAddress(newo), size + 1).copy(data, size).copy(wchar(0));
    // assert(allocSize == ValueSize(newo));
    return newo;
  }

  /* CsMakeString - make and initialize a new string value */
  value CsMakeFilledString(VM *c, wchar fill, size_t size) {
    long allocSize =
        sizeof(CsString) +
        CsRoundSize((size + 1) * sizeof(wchar)); /* space for zero terminator */
    value  newo = CsAllocate(c, allocSize);
    wchar *p    = CsStringAddress(newo);
    CsSetDispatch(newo, &CsStringDispatch);
    CsSetStringSize(newo, size);
    CsSetStringAllocatedSize(newo, size);
    for (size_t n = 0; n < size; ++n)
      *p++ = fill;
    *p = '\0'; /* in case we need to use it as a C string */
    // assert(allocSize == ValueSize(newo));
    return newo;
  }

  value CsMakeSubString(VM *c, value s, int start, int length) {
    CsCPush(c, s);
    long allocSize = sizeof(CsString) +
                     CsRoundSize((length + 1) *
                                 sizeof(wchar)); /* space for zero terminator */
    value newo = CsAllocate(c, allocSize);
    CsSetDispatch(newo, &CsStringDispatch);
    CsSetStringSize(newo, length);
    CsSetStringAllocatedSize(newo, length);
    s          = CsPop(c);
    wchars src = CsStringChars(s)(start, start + length);
    target(CsStringAddress(newo), length + 1).copy(src).copy(wchar(0));
    // assert(allocSize == ValueSize(newo));
    // assert(CsStringP(newo));
    return newo;
  }

  value CsConcatStrings(VM *c, value str, value other, bool append) {

    size_t needed;

    tool::wchars        s = CsStringChars(str);
    tool::wchars        o = CsStringChars(other);
    tool::tslice<wchar> t;

    if (append) {
      t = CsStringAllocatedChars(str);
      if (s.length + o.length < t.length) {
        t.prune(s.length);
        *(t.copy(o).start) = '\0';
        CsSetStringSize(str, s.length + o.length);
        return str;
      }
      needed = max((s.length + o.length) * 4 / 3, 16);
    } else
      needed = s.length + o.length;

    PROTECT(str, other);
    value res                  = CsMakeCharString(c, nullptr, needed);
    t                          = CsStringAllocatedChars(res);
    s                          = CsStringChars(str);
    o                          = CsStringChars(other);
    *(t.copy(s).copy(o).start) = '\0';
    CsSetStringSize(res, s.length + o.length);
    return res;
  }

  value string_to_value(VM *c, const tool::ustring &s) {
    return CsMakeCharString(c, s, s.size());
  }
  tool::ustring value_to_string(value v) {
    if (CsStringP(v)) return tool::ustring(CsStringAddress(v), CsStringSize(v));
    if (CsSymbolP(v)) return CsSymbolName(v);
    // return
    // tool::ustring::utf8(CsSymbolPrintName(v),CsSymbolPrintNameLength(v));
    return tool::ustring();
  }

  tool::ustring value_to_string(VM* c, value v) {
    string_stream ss;
    CsDisplay(c, v, &ss);
    return ss.to_ustring();
  }


  tool::wchars value_to_wchars(value v) {
    if (CsStringP(v)) return tool::wchars(CsStringAddress(v), CsStringSize(v));
    if (v == UNDEFINED_VALUE) return tool::wchars();
    if (CsSymbolP(v)) return CsSymbolName(v)();
    return tool::wchars();
  }

  tool::string utf8_string(value o) {
    tool::ustring us = value_to_string(o);
    return u8::cvt(us);
  }

  /* CsMakeCString - make a string value from a C string */
  value CsMakeCString(VM *c, const char *str) {
    // tool::ustring us = tool::ustring::utf8(str,strlen(str));
    tool::ustring us = str;
    return CsMakeCharString(c, us, us.size());
  }

  /* CsMakeCString - make a string value from a C wide string */
  value CsMakeCString(VM *c, const wchar *str) {
    return CsMakeCharString(c, str, (int_t)str_len(str));
  }

  /* CsMakeCString - make a string value from a C wide string */
  value CsMakeString(VM *c, tool::wchars str) {
    if (str.start >= (wchar *)c->newSpace->base &&
        str.start <= (wchar *)c->newSpace->top) {
      tool::ustring buf(str);
      return CsMakeCharString(c, buf, buf.size());
    } else
      return CsMakeCharString(c, str.start, str.size());
  }

  value CsMakeString(VM *c, tool::chars str) {
    tool::ustring us = u8::cvt(str);
    return CsMakeCharString(c, us, us.size());
  }

} // namespace tis
