/* object.c - 'Object' handler */
/*
        Copyright (c) 2001-2004 Terra Informatica Software, Inc.
        and Andrew Fedoniouk andrew@terrainformatica.com
        All rights reserved
*/

#include "cs.h"
#include "cs_int.h"

namespace tis {

#pragma warning(push)
#pragma warning(                                                               \
    disable : 4706) // warning C4706: assignment within conditional expression

  /* method handlers */
  static value CSF_ctor(VM *c);
  // static value CSF_prototype(VM *c);
  static value CSF_clone(VM *c);
  static value CSF_exists(VM *c);
  static value CSF_remove(VM *c);
  static value CSF_call(VM *c);
  static value CSF_show(VM *c);
  static value CSF_store(VM *c);
  static value CSF_restore(VM *c);
  value        CSF_propertyAt(VM *c);
  value        CSF_addObserver(VM *c);
  value        CSF_removeObserver(VM *c);
  value        CSF_referenceOf(VM *c);
  value        CSF_eachObserver(VM *c);
  static value CSF_extend(VM *c);
  static value CSF_assign(VM *c);

  value CSF_isFrozen(VM *c);
  value CSF_freeze(VM *c);
  value CSF_isSealed(VM *c);
  value CSF_seal(VM *c);

  value CSF_keys(VM *c);
  value CSF_create(VM *c);
  
  static value CSF_length(VM *c, value obj);
  static value CSF_persistent(VM *c, value obj);
  // static value CSF_prototype(VM *c,value obj);
  // static void CSF_set_prototype(VM *c,value obj, value pro);

  static value CSF_class_name(VM *c, value obj);

  static value CSF_class_class_name(VM *c, value obj);
  static void  CSF_set_class_class_name(VM *c, value obj, value pro);

#if defined(TISCRIPT_USE_PERSISTENCE)
#define FETCH(c, obj)                                                          \
  if (_CsIsPersistent(obj)) obj = CsFetchObjectData(c, obj);
#define FETCH_P(c, obj, p1)                                                    \
  if (_CsIsPersistent(obj)) {                                                  \
    CsPush(c, p1);                                                             \
    obj = CsFetchObjectData(c, obj);                                           \
    p1  = CsPop(c);                                                            \
  }
#define FETCH_PP(c, obj, p1, p2)                                               \
  if (_CsIsPersistent(obj)) {                                                  \
    CsPush(c, p2);                                                             \
    CsPush(c, p1);                                                             \
    obj = CsFetchObjectData(c, obj);                                           \
    p1  = CsPop(c);                                                            \
    p2  = CsPop(c);                                                            \
  }
#else
#define FETCH(c, obj)
#define FETCH_P(c, obj, p1)
#define FETCH_PP(c, obj, p1, p2)
#endif

  /* CSF_ctor - built-in method 'initialize' */
  static value CSF_ctor(VM *c) {
    CsCheckArgCnt(c, 2);
    CsCheckType(c, 1, CsObjectP);
    return CsGetArg(c, 1);
  }

  /* CSF_Class - built-in function 'Class' */
  /* CSF_size - built-in property 'length' */
  static value CSF_length(VM *c, value obj) {
    FETCH(c, obj);
    return CsMakeInteger(CsObjectPropertyCount(obj));
  }

  static value CSF_persistent(VM *c, value obj) {
#if defined(TISCRIPT_USE_PERSISTENCE)
    return CsIsPersistent(c, obj) ? TRUE_VALUE : FALSE_VALUE;
#else
    return FALSE_VALUE;
#endif
  }

  static value CSF_class_name(VM *c, value obj) {
    value cls = obj;
    if (!CsClassP(obj) && !CsNamespaceP(obj)) cls = CsObjectClass(obj);
    if (CsClassP(cls)) return CsClassName(cls);
    return cls;
  }

  static value CSF_class_class_name(VM *c, value obj) {
    if (CsClassP(obj))
      return CsClassName(obj);
    else {
      value cls = CsObjectClass(obj);
      if (CsClassP(cls)) return CsClassName(cls);
      return cls;
    }
  }
  static void CSF_set_class_class_name(VM *c, value obj, value v) {
    if (CsClassP(obj) && CsStringP(v)) {
      value n = CsClassName(obj);
      if (n == UNDEFINED_VALUE) CsSetClassName(obj, v);
    }
  }

  value CsEventObjectAdd(VM *c, value obj, value fun, wchars name_namespace,
                         wchars selector) {
    wchars name = name_namespace.chop('.');
    PROTECT(obj, fun);
    value v_selector = selector.length ? CsSymbolOf(selector) : UNDEFINED_VALUE;
    value v_name     = name.length ? CsSymbolOf(name) : UNDEFINED_VALUE;
    value v_ns =
        name_namespace.length ? CsSymbolOf(name_namespace) : UNDEFINED_VALUE;
    return CsEventObjectAdd(c, obj, fun, v_name, v_ns, v_selector);
  }

#ifdef _DEBUG
  int CsObjectEventsCount(VM *c, value obj) {
    value erec = CsObjectEvents(obj);
    value curr;

    PROTECT(erec, curr);
    int n = 0;

    while (erec && CsFixedVectorP(erec)) {
      ++n;
      curr = erec;
      erec = CsFixedVectorElement(curr, EVENT_DEF_LENGTH - 1);
    }
    return n;
  }

#endif

  value CsEventObjectAdd(VM *c, value obj, value fun, value ename,
                         value enamespace, value eselector) {
    FETCH(c, obj);
    PROTECT(obj, fun, ename, enamespace, eselector);
    value rec = CsMakeFixedVector(c, EVENT_DEF_LENGTH);
    // dispatch *pobj = CsGetDispatch(obj);
    // dispatch *pfun = CsGetDispatch(fun);
    assert(ename);
    assert(enamespace);
    assert(eselector);
    CsSetFixedVectorElement(rec, 0, fun);
    CsSetFixedVectorElement(rec, 1, ename);
    CsSetFixedVectorElement(rec, 2, enamespace);
    CsSetFixedVectorElement(rec, 3, eselector);
    CsSetFixedVectorElement(rec, 4, CsObjectEvents(obj));
    CsSetObjectEvents(obj, rec);

#ifdef _DEBUG
//    int cnt = CsObjectEventsCount(c, obj);
//    assert(cnt < 50);
#endif
    return obj;
  }

  value CsEventObjectAddEF(VM *c, value obj, value fun) {
    if (!CsMethodEventP(fun))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, fun, "event function");

    /*
        tool::wchars eventspec = value_to_wchars(CsMethodName(fun));
        tool::wchars name_and_namespace = eventspec.chop('|');
        tool::wchars selector = eventspec;

        tool::wchars name = name_and_namespace.chop('.');
        tool::wchars ns   = name_and_namespace;

        value ename = UNDEFINED_VALUE;
        value enamespace = UNDEFINED_VALUE;
        value eselector = UNDEFINED_VALUE;

        PROTECT(obj,fun,ename,enamespace,eselector);

        ename = CsSymbolOf(name);
        enamespace = ns.length == 0 ? UNDEFINED_VALUE : CsSymbolOf(ns);
        eselector = selector.length == 0 ? UNDEFINED_VALUE:
       CsMakeString(c,selector);
    */
    value mnt = CsMethodName(fun);
    assert(CsTupleP(mnt));
    return CsEventObjectAdd(c, obj, fun, CsTupleElement(mnt, 0),
                            CsTupleElement(mnt, 1), CsTupleElement(mnt, 2));
  }

  value CsEventObjectRemoveV(VM *c, value obj, value v) {
    PROTECT(obj);
    if (CsStringP(v) || CsSymbolP(v)) {
      wchars ns = value_to_wchars(v);
      wchars nm = ns.chop('.');
      return CsEventObjectRemove(
          c, obj, UNDEFINED_VALUE,
          (nm.length ? CsSymbolOf(nm) : UNDEFINED_VALUE),
          (ns.length ? CsSymbolOf(ns) : UNDEFINED_VALUE));
    } else if (CsMethodP(v)) {
      return CsEventObjectRemove(c, obj, v);
    } else
      CsThrowKnownError(c, CsErrUnexpectedTypeError, v, "function or string");
    return obj;
  }

  value CsEventObjectRemove(VM *c, value obj, value fun, value ename,
                            value enamespace, value eselector) {
    value erec = CsObjectEvents(obj);
    value curr;
    value prev = 0;

    PROTECT(obj,erec, curr, prev, eselector);

    auto filter = [&](value f, value n, value ns, value s) -> bool {
      if (fun != UNDEFINED_VALUE && f != fun) return false;
      if (ename != UNDEFINED_VALUE && n != ename) return false;
      if (enamespace != UNDEFINED_VALUE && ns != enamespace) return false;
      if (eselector != UNDEFINED_VALUE && !CsEqualOp(c, s, eselector))
        return false;
      return true;
    };

    while (erec && CsFixedVectorP(erec)) {
      curr = erec;
      erec = CsFixedVectorElement(curr, EVENT_DEF_LENGTH - 1);
      if (filter(CsEventDefFunction(curr), CsEventDefName(curr),
                 CsEventDefNamespace(curr), CsEventDefSelector(curr))) {
        if (prev)
          CsSetFixedVectorElement(prev, EVENT_DEF_LENGTH - 1, erec);
        else
          CsSetObjectEvents(obj, erec);
      } else
        prev = curr;
    }
    return obj;
  }

  value CsEventObjectFire(VM *c, value obj, value ename, value param) {
    value erec = CsObjectEvents(obj);
    value curr = 0;

    PROTECT(erec, curr, obj, ename, param);
    int n_consumed = 0;
    int n_found = 0;

    while (erec && CsFixedVectorP(erec)) {
      curr = erec;
      erec = CsFixedVectorElement(curr, EVENT_DEF_LENGTH - 1);
      if (CsEventDefName(curr) == ename) {
        value      fun = CsEventDefFunction(curr);
        if (CsMethodP(fun)) {
          ++n_found;
          PROTECT(fun);
          CsSetEventDefFunction(curr, UNDEFINED_VALUE); // to prevent recursive event firing
          auto_scope as(c, CsMethodNamespace(fun));
          TRY{
            value r = param ? CsCallMethod(c, obj, fun, obj, 1, param)
                            : CsCallMethod(c, obj, fun, obj, 0);
            if (r == TRUE_VALUE) ++n_consumed;
            CsSetEventDefFunction(curr, fun);
          }
          CATCH_ERROR(e) {
            CsSetEventDefFunction(curr, fun);
            CsEventObjectRemoveV(c, obj, fun);
            if (e.number != 0) // close() called
              tis::CsHandleUnhandledError(c);
          }
        }
      }
    }
    if (n_found == 0)  return NULL_VALUE;
    return n_consumed > 0 ? TRUE_VALUE : FALSE_VALUE;
  }

  value CsEventObjectFire(VM *c, value obj, value ename,
                          tool::function<value()> param) {
    value erec = CsObjectEvents(obj);
    value curr = 0;
    value fun  = 0;

    PROTECT(erec, curr, fun, obj, ename);
    int n_consumed = 0;

    while (erec && CsFixedVectorP(erec)) {
      curr = erec;
      erec = CsFixedVectorElement(curr, EVENT_DEF_LENGTH - 1);
      if (CsEventDefName(curr) == ename) {
        fun = CsEventDefFunction(curr);
        if (CsMethodP(fun)) {
          PROTECT(fun);
          CsSetEventDefFunction(curr, UNDEFINED_VALUE); // to prevent recursive event firing
          auto_scope as(c, CsMethodNamespace(fun));
          TRY{
            value r;
            if (param) {
              value p = param();
              r = CsCallMethod(c, obj, fun, obj, 1, p);
            } else {
              r = CsCallMethod(c, obj, fun, obj, 1, UNDEFINED_VALUE);
            }
            if (r == TRUE_VALUE) ++n_consumed;
            CsSetEventDefFunction(curr, fun);
          }
          CATCH_ERROR(e) {
            CsSetEventDefFunction(curr, fun);
            CsEventObjectRemoveV(c, obj, fun);
            if (e.number != 0) // close() called
              tis::CsHandleUnhandledError(c);
          }
        }
      }
    }
    return n_consumed > 0 ? TRUE_VALUE : FALSE_VALUE;
  }

  value CsDefaultObjectBinOp(VM *c, int op, value obj, value operand) {
    if (!CsDerivedFromObjectP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object");

    switch (op) {
    case BC_ADD:
    case BC_SHL: return CsEventObjectAddEF(c, obj, operand);
    case BC_SUB:
    case BC_SHR: return CsEventObjectRemoveV(c, obj, operand);
    default: CsUnsupportedBinaryOp(c, op, obj, operand);
    }

    return obj;
  }

  void CsMergeObjects(VM* c, value& to, value from) {
    each_property  gen(c, from);
    for (value key, val; gen(key, val);) {
      if (val == NOTHING_VALUE || val == UNDEFINED_VALUE) {
        if (CsStringP(key) || CsSymbolP(key))
          CsSetProperty1(c, to, key, TRUE_VALUE);
        else if (CsObjectP(key))
          CsMergeObjects(c, to, key);
        //else 
      }
      else
        CsSetProperty1(c, to, key, val);
    }
  };

  /*value CsFlattenObject(VM *c, value obj) {
    value nobj = CsMakeObject(c, c->objectObject);
    PROTECT(nobj);
    CsMergeObjects(c, nobj, obj);
    return nobj;
  }*/

  /*value CsStringifyCollection(VM *c, value obj) {
    PROTECT(obj);
    if (CsObjectP(obj)) {
      each_property  gen(c, obj);
      for (value key, val; gen(key, val);) {
        if (!CsStringP(val)) {
          val = CsToString(c, val);
          CsSetPropertyValue(gen.prop, val);
        }
      }
    }
    else if (CsVectorP(obj)) {
      int sz = CsVectorSizeI(obj);
      for (int i = 0; i < sz; ++i) {
        value val = CsVectorElementI(obj, i);
        if (!CsStringP(val)) {
          val = CsToString(c, val);
          CsSetVectorElementI(obj,i,val);
        }
      }
    }
    return obj;
  }*/
  
  value CsAddObserver(VM *c, value obj, value fun) 
  {
    if (!CsEntityP(obj))
      CsUnexpectedTypeError(c, obj, "Object or Array");
    //FETCH_P(c, obj, fun); -- no need to fetch persistent data here 
    if (value observer = CsEntityObserver(obj)) {
      if (CsMethodP(observer))
      // so we need to make list
      {
        if (observer != fun) {
          PROTECT(obj,observer,fun);
          value newobserver = CsMakeVector(c, 2);
          CsSetVectorElement(c, newobserver, 0, observer);
          CsSetVectorElement(c, newobserver, 1, fun);
          CsSetEntityObserver(obj, newobserver);
        }
      } else if (CsVectorP(observer)) {
        if (!CsVectorElements(c, observer).contains(fun)) {
          //dbg_printf("CsAddObserver chain %d\n", CsVectorSize(c, observer));
          CsVectorPush(c, observer, fun);
        }
      }
    } else
      CsSetEntityObserver(obj, fun);
    return UNDEFINED_VALUE;
  }

  value CSF_addObserver(VM *c) {
    value obj, fun;
    if(CsArgCnt(c) == 4)
      CsParseArguments(c, "**VM", &obj, &fun);
    else {
      CsWarning(c, "OBSOLETE addObserver call, update +plus/+vlist libraries");
      CsParseArguments(c, "V*M", &obj, &fun);
    }
    return CsAddObserver(c, obj, fun);
  }
    
  value CsEachObserver(VM *c, value obj, value fun) {
    value observer = 0;
    if (CsEntityP(obj))
      observer = CsEntityObserver(obj);
    else
      CsUnexpectedTypeError(c, obj, "Object or Array");

    if (!observer) return UNDEFINED_VALUE;

    if (CsMethodP(observer)) {
      value r = CsCallMethod(c, obj, fun, obj, 1, observer);
      return r;
    } else if (CsVectorP(observer)) {
      int length = CsVectorSize(c, observer);
      PROTECT(observer, obj, fun);
      for (int n = length - 1; n >= 0; --n) {
        value sub = CsVectorElement(c, observer, n);
        if (CsMethodP(sub)) {
          value r = CsCallMethod(c, obj, fun, obj, 1, sub);
          if (r == TRUE_VALUE) return r;
        }
      }
    }
    return FALSE_VALUE;
  }

  value CSF_eachObserver(VM *c) {
    value obj, fun;
    if (CsArgCnt(c) == 4)
      CsParseArguments(c, "**VM", &obj, &fun);
    else {
      CsWarning(c, "OBSOLETE eachObserver call, update +plus/+vlist libraries");
      CsParseArguments(c, "V*M", &obj, &fun);
    }
    return CsEachObserver(c, obj, fun);
  }
  
  value CsRemoveObserver(VM *c, value obj, value fun) {

    if (!CsEntityP(obj))
      CsUnexpectedTypeError(c, obj, "Object or Array");

    FETCH(c, obj);

    if (CsSymbolP(fun)) {
      auto cmp = [fun](value val) { return CsMethodShortName(val) == fun; };

      if (value observer = CsEntityObserver(obj)) {
        if (CsVectorP(observer)) {
          int idx = CsVectorElements(c, observer).index_by(cmp);
          if (idx >= 0) {
            CsVectorRemove(c, observer, idx);
            if (CsVectorSize(c, observer) == 1)
              CsSetEntityObserver(obj, CsVectorElement(c, observer, 0));
          }
        }
        else if (cmp(observer))
          CsSetEntityObserver(obj, 0);
      }
    }
    else {
      if (value observer = CsEntityObserver(obj)) {
        if (CsVectorP(observer)) {
          int idx = CsVectorElements(c, observer).index_of(fun);
          if (idx >= 0) {
            CsVectorRemove(c, observer, idx);
            if (CsVectorSize(c, observer) == 1)
              CsSetEntityObserver(obj, CsVectorElement(c, observer, 0));
          }
        }
        else if (observer == fun)
          CsSetEntityObserver(obj, 0);
      }
    }
    return UNDEFINED_VALUE;
  }

  value CSF_removeObserver(VM *c) {
    value obj, fun;
    if (CsArgCnt(c) == 4)
      CsParseArguments(c, "**VV", &obj, &fun);
    else {
      CsWarning(c, "OBSOLETE removeObserver call, update +plus/+vlist libraries");
      CsParseArguments(c, "V*V", &obj, &fun);
    }
    return CsRemoveObserver(c, obj, fun);
  }
  
  /*value CsGetClassInstancePrototype(VM* c, value o) {
    value prototype = CsClassInstancePrototype(o);
    if (!CsObjectP(prototype)) {
      PROTECT(o);
      prototype = CsMakeObject(c, UNDEFINED_VALUE);
      CsSetClassInstancePrototype(o, prototype);
    }
    return prototype;
  }*/

  pair<value, value> CsReferenceOf(VM *c, value ns, wchars path) 
  {

    tool::xjson::scanner s(path, false);

    tool::wchars part;
    value        obj = ns;
    value        collection = ns;
    value        tag = UNDEFINED_VALUE;
    while (!s.at_end()) {

      if (CsDerivedFromObjectP(obj)) { // last name part

        tool::xjson::scanner::token_t t = s.get_index_token(part);
        if (t != tool::xjson::scanner::T_NAME &&
          t != tool::xjson::scanner::T_INDEX_NAME)
          goto PROBLEM;
        tag = CsMakeSymbol(c, part.start, part.size());

        if (s.at_end()) {
          // last name part
          collection = obj;
          value v = UNDEFINED_VALUE;
          if (CsGetProperty(c, obj, tag, &v))
            collection = obj;
          break;
        }
        else {
          value v = UNDEFINED_VALUE;
          CsGetProperty(c, obj, tag, &v);
          obj = v;
          continue;
        }
      }
      else if (CsVectorP(obj)) {
        tool::xjson::scanner::token_t t = s.get_index_token(part);
        if (t == tool::xjson::scanner::T_NAME ||
          t == tool::xjson::scanner::T_INDEX_NAME) {
          if (part == WCHARS("first") || part == WCHARS("last") ||
            part == WCHARS("length"))
            tag = CsMakeSymbol(c, part.start, part.size());
          else
            goto PROBLEM;
        }
        else if (t == tool::xjson::scanner::T_INDEX_NUMBER) {
          int index = parse_int(part);
          tag = CsMakeInteger(index);
        }

        if (s.at_end()) {
          collection = obj; // last one break
          break;
        }
        else {
          if (CsIntegerP(tag)) {
            obj = CsVectorElement(c, obj, CsIntegerValue(tag));
            continue;
          }
          else if (CsGetProperty(c, obj, tag, &obj)) {
            continue;
          }
          else
            goto PROBLEM;
        }
      }
      else {
      PROBLEM:
        collection = NULL_VALUE;
        tag = NULL_VALUE; // to indicate problem, and break anyway
        break;
      }
    }
    assert(tag);
    assert(collection);
    return pair<value, value>(collection, tag);
  }

  // note: static method of Object class
  value CSF_referenceOf(VM *c) {
    value  ns;
    wchars path;
    CsParseArguments(c, "**V=S#", &ns, &CsObjectDispatch, &path.start, &path.length);

    auto ct = CsReferenceOf(c, ns, path);

    CS_RETURN2(c, ct.first, ct.second);
  }

  /* CSF_clone - built-in method 'Clone' */
  static value CSF_clone(VM *c) {
    value obj, deep = 0;
    CsParseArguments(c, "V=*|V", &obj, &CsObjectDispatch, &deep);
    FETCH(c, obj);
    return CsCloneObject(c, obj, deep == TRUE_VALUE);
  }

  /* CSF_Exists - built-in method 'exists' */
  static value CSF_hasOwnProperty(VM *c) {
    /*
    value obj, tag;
    bool  deep = false;
    CsParseArguments(c, "V=*V|B", &obj, &CsObjectDispatch, &tag, &deep);
    FETCH_P(c, obj, tag);

    while (CsObjectP(obj)) {
      if (CsFindProperty(c, obj, tag, NULL, NULL)) return TRUE_VALUE;
      if (!deep) break;
      obj = CsObjectClass(obj);
    }*/

    value obj, tag;
    CsParseArguments(c, "V=*V", &obj, &CsObjectDispatch, &tag);
    FETCH_P(c, obj, tag);

    if (CsFindProperty(c, obj, tag, NULL, NULL)) return TRUE_VALUE;
    return FALSE_VALUE;
  }

  /* CSF_propertyAt - built-in method 'val = Object.propertyAt(sym)'
     It is a direct equivalent of val = Object.sym;
  */
  value CSF_propertyAt(VM *c) {
    value obj, tag, val;
    CsParseArguments(c, "V*V", &obj, &tag);
    FETCH_P(c, obj, tag);
    //CsWarning(c, "OBSOLETE propertyAt call, use obj.(propName) construct");
    if (CsGetProperty(c, obj, tag, &val)) return val;

    return UNDEFINED_VALUE;
  }

  /* CSF_keys - built-in method 'list = Object.keys()' */

  value CSF_keys(VM *c) {
    value obj;
    CsParseArguments(c, "**V=", &obj, &CsObjectDispatch);
    FETCH(c, obj);

    PROTECT(obj);

    int_t count = CsObjectPropertyCount(obj);

    value list = CsMakeVector(c, count);

    struct pd {
      uint  ordinal;
      value k;
    };
    buffer<pd, 32> ordered((size_t)count);
    each_property  gen(c, obj);
    uint           n = 0;
    for (value key, val; gen(key, val);) {
      pd d         = {gen.ordinal, key};
      ordered[n++] = d;
    }
    tool::sort(ordered.begin(), ordered.length(),
               [](const pd &l, const pd &r) { return l.ordinal < r.ordinal; });

    for (int_t n = 0; n < count; ++n)
      CsSetVectorElementNoLoad(c, list, n, ordered[n].k);

    return list;
  }

  value CSF_create(VM *c) {
    value obj = c->objectObject;
    value props = 0;
    CsParseArguments(c, "**V|V=", &obj, &props, &CsObjectDispatch);
    FETCH(c, obj);

    if (obj == NULL_VALUE)
      obj = UNDEFINED_VALUE;
    else if(!CsObjectP(obj) && !CsClassP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object or null");

    if (props) {
      PROTECT(obj);
      value r = CsCloneObject(c, props);
      CsSetObjectClass(r, obj);
      return r;
    }
    return CsMakeObject(c,obj);
  }

  value CsClone(VM *c, value obj, bool deep) {
    if (CsObjectP(obj))
      obj = CsCloneObject(c, obj, deep);
    else if (CsVectorP(obj))
      obj = CsCloneVector(c, obj, deep);
    return obj;
  }

  value CsExtendObject(VM *c, value obj, value other, bool deep) {
    assert(CsDerivedFromObjectP(obj));
    //dispatch* pd = CsGetDispatch(other);
    assert(CsDerivedFromObjectP(other));
    value key = 0, val = 0;
    PROTECT(obj, other, key, val);
    FETCH(c, obj);
    FETCH(c, other);
    each_property gen(c, other);
    for (; gen(key, val);) {
      if (deep) val = CsClone(c, val, true);
      CsSetProperty(c, obj, key, val);
    }
    return obj;
  }

  value CSF_extend(VM *c) {
    value obj, other = 0, deep = 0;
    CsParseArguments(c, "V=*V", &obj, &CsObjectDispatch, &deep);
    FETCH(c, obj);

    PROTECT(obj, other);
    int i = 3;
    if (deep == TRUE_VALUE) i = 4;

    for (; i <= c->argc; ++i) {
      other = CsGetArg(c, i);
      if (!CsDerivedFromObjectP(other))
        CsThrowKnownError(c, CsErrUnexpectedTypeError, other, "object");
      obj   = CsExtendObject(c, obj, other, deep == TRUE_VALUE);
    }
    return obj;
  }

  value CSF_assign(VM *c) {
    value obj, other = 0;
    CsParseArguments(c, "**V", &obj);
    FETCH(c, obj);

    if (!CsDerivedFromObjectP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object");

    PROTECT(obj, other);

    for (int i = 3; i <= c->argc; ++i) {
      other = CsGetArg(c, i);
      obj = CsExtendObject(c, obj, other, false);
    }
    return obj;
  }

  value CSF_getPrototypeOf(VM *c) {
    value obj;
    CsParseArguments(c, "**V", &obj);

    if(!CsEntityP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object, array or function");
    //FETCH(c, obj); ????
    
    value prototype = CsEntityClass(obj);
    return prototype;
  }

  value CSF_setPrototypeOf(VM *c) {
    value obj, proto;
    CsParseArguments(c, "**VV", &obj, &proto);
    
    if (!CsEntityP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object, array or function");
    //FETCH(c, obj); ????
    if ((proto == NULL_VALUE) || CsEntityP(proto))
      CsSetEntityClass(obj,proto);
    else 
      CsThrowKnownError(c, CsErrUnexpectedTypeError, proto, "class, object or null");

    return obj;
  }

  value CSF_fireEvent(VM *c) {
    value obj, sym, data = 0;
    CsParseArguments(c, "**VV=|V", &obj, &sym, &CsSymbolDispatch,&data);

    if (!CsDerivedFromObjectP(obj))
      CsThrowKnownError(c, CsErrUnexpectedTypeError, obj, "object or function");

    CsEventObjectFire(c, obj, sym, data);

    return obj;
  }

  


  value CSF_isFrozen(VM *c) {
    value obj = CsGetArg(c, 3);
    if (CsEntityP(obj)) {
      bool strictly_sealed = CsArgCnt(c) >= 4 && CsGetArg(c, 4) == TRUE_VALUE;
      if (strictly_sealed)
        return ((CsEntityFlags(obj) & (IS_FROZEN | IS_STRONG_SEALED)) ==
                (IS_FROZEN | IS_STRONG_SEALED))
                   ? TRUE_VALUE
                   : FALSE_VALUE;
      else
        return ((CsEntityFlags(obj) & IS_FROZEN) != 0) ? TRUE_VALUE
                                                       : FALSE_VALUE;
    }
    return UNDEFINED_VALUE;
  }
  value CSF_freeze(VM *c) {
    value obj = CsGetArg(c, 3);
    if (CsEntityP(obj)) {
      bool strictly_sealed = CsArgCnt(c) >= 4 && CsGetArg(c, 4) == TRUE_VALUE;
      if (strictly_sealed)
        CsEntityFlags(obj) |= (IS_FROZEN | IS_SEALED | IS_STRONG_SEALED);
      else
        CsEntityFlags(obj) |= (IS_FROZEN | IS_SEALED);
    }
    return obj;
  }
  value CSF_isSealed(VM *c) {
    value obj = CsGetArg(c, 3);
    if (CsEntityP(obj)) {
      bool strictly_sealed = CsArgCnt(c) >= 4 && CsGetArg(c, 4) == TRUE_VALUE;
      if (strictly_sealed)
        return ((CsEntityFlags(obj) & (IS_SEALED | IS_STRONG_SEALED)) ==
                (IS_SEALED | IS_STRONG_SEALED))
                   ? TRUE_VALUE
                   : FALSE_VALUE;
      else
        return ((CsEntityFlags(obj) & IS_SEALED) != 0) ? TRUE_VALUE
                                                       : FALSE_VALUE;
    }
    return UNDEFINED_VALUE;
  }
  value CSF_seal(VM *c) {
    value obj = CsGetArg(c, 3);
    if (CsEntityP(obj)) {
      bool strictly_sealed = CsArgCnt(c) >= 4 && CsGetArg(c, 4) == TRUE_VALUE;
      if (strictly_sealed)
        CsEntityFlags(obj) |= (IS_SEALED | IS_STRONG_SEALED);
      else
        CsEntityFlags(obj) |= IS_SEALED;
    }
    return obj;
  }

  /* CSF_remove - built-in method 'Remove' */
  static value CSF_remove(VM *c) {
    value obj, tag;
    CsParseArguments(c, "V=*V", &obj, &CsObjectDispatch, &tag);
    FETCH_P(c, obj, tag);
    CsWarning(c, "OBSOLETE obj.remove(key) call, use: delete obj[key]");
    CsRemoveObjectProperty(c, obj, tag);
    return UNDEFINED_VALUE;
  }

  /* CSF_Exists - built-in method 'exists' */
  static value CSF_exists(VM *c) {
    value obj, tag;
    bool  deep = false;
    CsParseArguments(c, "V=*V|B", &obj, &CsObjectDispatch, &tag, &deep);
    FETCH_P(c, obj, tag);

    CsWarning(c, "OBSOLETE obj.exists(key) call, use: key in obj");

    while (CsObjectP(obj)) {
      if (CsFindProperty(c, obj, tag, NULL, NULL)) return TRUE_VALUE;
      if (!deep) break;
      obj = CsObjectClass(obj);
    }
    return FALSE_VALUE;
  }


  bool CsObjectsEqual(VM *c, value v1, value v2, array<value> &stack) {
    if (v1 == v2) return true;

    if (CsObjectPropertyCount(v1) != CsObjectPropertyCount(v2)) return false;

    if (stack.get_index(v1) >= 0 || stack.get_index(v2) >= 0)
      CsThrowKnownError(c, CsErrGenericError,
                        "comparison of objects with loops");
    // return false; // recursive data graphs are non-comparable

    stack.push(v1);
    stack.push(v2);

    PROTECT(v1, v2);
    FETCH(c, v1);
    FETCH(c, v2);

    bool r = false;

    each_property props(c, v1);
    for (value key, tval; props(key, tval);) {
      value rval = NOTHING_VALUE;
      if (!CsGetProperty(c, v2, key, &rval)) goto DONE;
      if (!CsEqualOp(c, tval, rval, stack)) goto DONE;
    }
    r = true;
  DONE:
    stack.pop();
    stack.pop();
    return r;
  }

  /* CSF_call - built-in method 'call'
     calls method in context of 'this' */
  /*value CSF_call(VM *c) {
    int_t i, vcnt, argc;
    value argv;

    if (CsArgCnt(c) == 3) {
      CsCheckType(c, 1, CsObjectOrMethodP);
      CsCheckType(c, 3, CsMethodP);
      CsCheck(c, 3);
      CsPush(c, CsGetArg(c, 1));
      CsPush(c, CsGetArg(c, 3));
      CsPush(c, CsGetArg(c, 1));
      return CsInternalSend(c, 2);
    }
    CsCheckArgMin(c, 4);
    CsCheckType(c, 1, CsObjectOrMethodP);
    CsCheckType(c, CsArgCnt(c), CsVectorP);
    argv = CsGetArg(c, CsArgCnt(c));
    vcnt = CsVectorSizeI(argv);
    argc = CsArgCnt(c) + vcnt - 2;
    CsCheck(c, argc + 1);
    CsPush(c, CsGetArg(c, 1));
    CsPush(c, CsGetArg(c, 3));
    CsPush(c, CsGetArg(c, 1));
    for (i = 4; i < CsArgCnt(c); ++i)
      CsPush(c, CsGetArg(c, i));
    for (i = 0; i < vcnt; ++i)
      CsPush(c, CsVectorElementI(argv, i));
    return CsInternalSend(c, argc);
  }*/

  /* CSF_show - built-in method 'Show' */
  static value CSF_show(VM *c) {
    stream *s = c->standardOutput;
    value   obj, props;
    CsParseArguments(c, "V=*|P=", &obj, &CsObjectDispatch, &s, c->fileDispatch);

    CsWarning(c, "OBSOLETE object.show call(), consider alternative");

    FETCH(c, obj);
    props = CsObjectProperties(obj);
    s->put_str("Class: ");
    CsPrint(c, CsObjectClass(obj), s);
    s->put('\n');
    if (CsObjectPropertyCount(obj)) {
      s->put_str("Properties:\n");
      if (CsHashTableP(props)) {
        int_t cnt = CsHashTableSize(props);
        int_t i;
        for (i = 0; i < cnt; ++i) {
          value prop = CsHashTableElement(props, i);
          for (; prop != UNDEFINED_VALUE; prop = CsPropertyNext(prop)) {
            s->put_str("  ");
            CsPrint(c, CsPropertyTag(prop), s);
            s->put_str(": ");
            CsPrint(c, CsPropertyValue(prop), s);
            s->put('\n');
          }
        }
      } else {
        for (; props != UNDEFINED_VALUE; props = CsPropertyNext(props)) {
          s->put_str("  ");
          CsPrint(c, CsPropertyTag(props), s);
          s->put_str(": ");
          CsPrint(c, CsPropertyValue(props), s);
          s->put('\n');
        }
      }
    }
    return obj;
  }

  /* CSF_store - built-in method 'store' for persistent objects */
  value CSF_store(VM *c) {
#if defined(TISCRIPT_USE_PERSISTENCE)
    value obj;
    CsParseArguments(c, "**V", &obj);

    if (!CsIsPersistent(c, obj)) return FALSE_VALUE;
    if (!CsIsModified(obj)) return TRUE_VALUE;

    CsStoreObjectData(c, obj);
    return TRUE_VALUE;
#else
    return FALSE_VALUE;
#endif
  }

  /* CSF_restore, aka rollback for persistent objects  */
  /* usage: obj.restore(); */
  value CSF_restore(VM *c) {
    value obj;
    CsParseArguments(c, "**V", &obj);
#if defined(TISCRIPT_USE_PERSISTENCE)
    if (!CsIsPersistent(c, obj))
      CsThrowKnownError(c, CsErrPersistError, "is not persistent");

    FETCH(c, obj);
    return obj;

#else
    CsThrowKnownError(c, CsErrPersistError, "is not persistent");
#endif
  }

  /* Object methods */
  static c_method methods[] = {
    C_METHOD_ENTRY("this", CSF_ctor),
    C_METHOD_ENTRY("toLocaleString", CSF_std_toLocaleString),
    C_METHOD_ENTRY("toString", CSF_std_toString),
    C_METHOD_ENTRY("valueOf", CSF_std_valueOf),

    C_METHOD_ENTRY("clone", CSF_clone),

    C_METHOD_ENTRY("hasOwnProperty", CSF_hasOwnProperty),

    C_METHOD_ENTRY("eval", CSF_eval),
    C_METHOD_ENTRY("extend", CSF_extend),

    // OBSOLETE {
    C_METHOD_ENTRY("propertyAt", CSF_propertyAt),
    C_METHOD_ENTRY("show", CSF_show),
    C_METHOD_ENTRY("remove", CSF_remove),
    C_METHOD_ENTRY("exists", CSF_exists),
    // OBSOLETE }

    // static methods
    C_METHOD_ENTRY("isFrozen", CSF_isFrozen),
    C_METHOD_ENTRY("freeze", CSF_freeze),
    C_METHOD_ENTRY("isSealed", CSF_isSealed),
    C_METHOD_ENTRY("seal", CSF_seal),

    C_METHOD_ENTRY("addObserver", CSF_addObserver),
    C_METHOD_ENTRY("removeObserver", CSF_removeObserver),
    C_METHOD_ENTRY("eachObserver", CSF_eachObserver),
    C_METHOD_ENTRY("referenceOf", CSF_referenceOf),

    C_METHOD_ENTRY("keys", CSF_keys),
    C_METHOD_ENTRY("create", CSF_create),
    C_METHOD_ENTRY("assign", CSF_assign),

    C_METHOD_ENTRY("getPrototypeOf", CSF_getPrototypeOf),
    C_METHOD_ENTRY("setPrototypeOf", CSF_setPrototypeOf),

    C_METHOD_ENTRY("store", CSF_store),
    C_METHOD_ENTRY("restore", CSF_restore),

    C_METHOD_ENTRY("fireEvent", CSF_fireEvent),

    C_METHOD_ENTRY(0, 0) };


  /* Object properties */
  static vp_method properties[] = {
    // VP_METHOD_ENTRY( "class",    CSF_prototype, CSF_set_prototype ),
    VP_METHOD_ENTRY("length", CSF_length, 0),
    VP_METHOD_ENTRY("className", CSF_class_name, 0),
    VP_METHOD_ENTRY("persistent", CSF_persistent, 0),

    VP_METHOD_ENTRY(0, 0, 0) };

  /* Object properties */
  static vp_method class_properties[] = {
    //VP_METHOD_ENTRY( "class",    CSF_prototype, CSF_set_prototype ),
    //VP_METHOD_ENTRY("length", CSF_length, 0),
    VP_METHOD_ENTRY("className", CSF_class_class_name, CSF_set_class_class_name),
    VP_METHOD_ENTRY(0, 0, 0) };

  /* CsInitObject - initialize the 'Object' obj */
  void CsInitObject(VM *c) {
    /* make the base of the obj inheritance tree */
    c->objectObject = CsEnterObject(CsGlobalScope(c), "Object", UNDEFINED_VALUE, methods, properties);
    c->classObject = CsEnterObject(CsGlobalScope(c), "Class", UNDEFINED_VALUE, nullptr, class_properties);
  }

  /* OBJECT */

  inline void IncObjectPropertyCount(value o) {
    //#ifdef  _DEBUG
    //    if (ptr<object>(o)->propertyCount > 2000)
    //      o = o;
    //#endif //  _DEBUG

    ptr<object>(o)->propertyCount += 1;
  }

  /* Object handlers */
  // static bool CsGetObjectProperty(VM *c,value obj,value tag,value *pValue);
  // static bool CsSetObjectProperty(VM *c,value obj,value tag,value value);
  static value CopyPropertyTable(VM *c, value table);
  static value CopyPropertyList(VM *c, value plist);
  static void  CreateHashTable(VM *c, value obj, value p);
  static int   ExpandHashTable(VM *c, value obj, int_t i);
  bool         CsAddObjectConstant(VM *c, value obj, value tag, value val);
  static bool  AddClassConstant(VM *c, value obj, value tag, value val);

  //    c->fileDispatch->binOp = FileBinOp;
  value CsObjectBinOp(VM *c, int op, value obj, value operand) {
    switch (op) {
    case BC_ADD:
    case BC_SHL: return CsEventObjectAddEF(c, obj, operand); 
    case BC_SHR:
    case BC_SUB: return CsEventObjectRemoveV(c, obj, operand); 
    }
    return obj;
  }

  value& CsObjectMetaAccessor(value tup) {
    return ptr<object>(tup)->meta;
  }
  
  /* Object pdispatch */
#if 0
  dispatch CsObjectDispatch = {
      "Object",
      &CsObjectDispatch,
      CsGetObjectProperty,
      CsSetObjectProperty,
      CsMakeObject,
      CsDefaultPrint,
      CsObjectSize,
      CsDefaultCopy,
      CsObjectScan,
      CsDefaultHash,
      CsObjectGetItem,
      CsObjectSetItem,
      CsObjectNextElement,
      CsAddObjectConstant,
      CsRemoveObjectProperty, // del_item_t                delItem;
      nullptr, // call_method_t             handleCall; // native call used by
               // e.g. Sciter behavior
      nullptr,       // dispatch**                interfaces;
      0,             // value                     obj; // a.k.a. class vtbl;
      0,             // long                      dataSize;
      0,             // destructor_t              destroy;
      nullptr,       // void*                     destroyParam;
      CsObjectBinOp, // binary_operator_t         binOp;   // address of
                     // function responsible for '<<' (BC_SHL) and '>>' (BC_SHR)
                     // operations.
      nullptr, // print_t
      CsObjectMetaAccessor,
      nullptr, // dispatch *next
  };
#else 


  BEGIN_DISPATCH(CsObjectDispatch)
    d.typeName        = "Object";
    d.baseType        = &CsObjectDispatch;
    d.getProperty     = CsGetObjectProperty;
    d.setProperty     = CsSetObjectProperty;
    d.newInstance     = CsMakeObject;
    d.print           = CsDefaultPrint;
    d.size            = CsObjectSize;
    d.copy            = CsDefaultCopy;
    d.scan            = CsObjectScan;
    d.hash            = CsDefaultHash;
    d.getItem         = CsObjectGetItem;
    d.setItem         = CsObjectSetItem;
    d.getNextElement  = CsObjectNextElement;
    d.addConstant     = CsAddObjectConstant;
    d.delItem         = CsRemoveObjectProperty;
    d.binOp           = CsObjectBinOp; // binary_operator_t         binOp;   // address of
                     // function responsible for '<<' (BC_SHL) and '>>' (BC_SHR)
                     // operations.
    d.metaAccessor    = CsObjectMetaAccessor;
  END_DISPATCH(CsObjectDispatch)


#endif 

  //#ifndef KEY_STRING_TO_SYMBOL
  // bool  CsGetJsonObjectProperty(VM *c,value& obj,value tag,value *pValue);
  // bool  CsSetJsonObjectProperty(VM *c,value obj,value tag,value val);
  // bool  CsRemoveJsonObjectProperty(VM *c,value obj, value tag);
  // value CsJsonObjectGetItem(VM *c,value obj,value tag);
  // void  CsJsonObjectSetItem(VM *c,value obj,value tag,value val);
  //
  ///* DataObject pdispatch (JSON primitive object) */
  // dispatch CsJsonObjectDispatch = {
  //    "DataObject",
  //    &CsObjectDispatch,
  //    CsGetJsonObjectProperty,
  //    CsSetJsonObjectProperty,
  //    CsMakeJsonObject,
  //    CsDefaultPrint,
  //    CsObjectSize,
  //    CsDefaultCopy,
  //    CsObjectScan,
  //    CsDefaultHash,
  //    CsJsonObjectGetItem,
  //    CsJsonObjectSetItem,
  //    CsObjectNextElement,
  //    CsAddObjectConstant,
  //    CsRemoveJsonObjectProperty,
  //};
  //#endif

  static void CsClassScan(VM *c, value obj);
  static long CsClassSize(value obj);
  static bool CsSetClassProperty(VM *c, value obj, value tag, value val);
  static bool CsGetClassProperty(VM *c, value &obj, value tag, value *pValue);
  static void CsClassSetItem(VM *c, value obj, value tag, value val);

  /* Class pdispatch */
  dispatch CsClassDispatch = {
      "Class",
      &CsObjectDispatch,
      //CsGetObjectProperty,
      CsGetClassProperty,
      CsSetClassProperty,
      CsMakeObject,
      CsDefaultPrint,
      CsClassSize,
      CsDefaultCopy,
      CsClassScan,
      CsDefaultHash,
      CsObjectGetItem,
      CsClassSetItem,
      CsObjectNextElement,
      AddClassConstant,
  };

  /* Namespace pdispatch */
  dispatch CsNamespaceDispatch = {
      "Namespace",         &CsObjectDispatch,    CsGetClassProperty,
      CsSetClassProperty,  CsDefaultNewInstance, CsDefaultPrint,
      CsClassSize,         CsDefaultCopy,        CsClassScan,
      CsDefaultHash,       CsObjectGetItem,      CsClassSetItem,
      CsObjectNextElement, AddClassConstant,
  };

  /* CsSetCObjectProperty - CObject set property handler */
  static bool AddClassConstant(VM *c, value obj, value tag, value val) {
    if (tag == UNDEFINED_VALUE) // special treatment for property undefined(n,v)
                                // handler
    {
      if (CsPropertyMethodP(val)) {
        CsSetClassUndefinedPropHandler(obj, val);
        return true;
      }
    }
    return CsAddObjectConstant(c, obj, tag, val);
  }

  bool CsFetchProperty(VM *c, value& obj, value& self, value tag, value *pValue);

  static bool CsGetClassProperty(VM *c, value &obj, value tag, value *pValue) {
    FETCH_P(c, obj, tag);

    if (CsSymbolP(tag) && tag == PROTOTYPE_SYM) {
      *pValue = CsObjectClass(obj);
      return true;
    }

    value self = obj;
    PROTECT(tag,self,obj);

    if (CsFetchProperty(c, obj, self, tag, pValue))
      return true;

    for (; obj;) {
      value next = CsObjectClass(obj);
      if (next == obj)
        break;
      if (!CsObjectOrMethodP(next))
        break;

      obj = next;

      if (CsFetchProperty(c, obj, self, tag, pValue))
        return true;

    }

    if (CsEntityFlags(self) & IS_STRONG_SEALED)
      CsThrowKnownError(c, CsErrNoProperty, self, tag);

    return false;
  }

  static void CsSetPrototype(VM *c, value obj, value proto);
  //bool CsTryStoreProperty(VM *c, value obj, value self, value tag, value val, int_t *pHashValue, int_t *pIndex);

  static bool CsSetClassProperty(VM *c, value obj, value tag, value val) 
  {
    if (tag == UNDEFINED_VALUE) // special treatment for property undefined(n,v)
                                // handler
    {
      if (CsPropertyMethodP(val)) {
        CsSetClassUndefinedPropHandler(obj, val);
        return true;
      }
    }

    int_t hashValue = 0, i = 0;

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    if (tag == PROTOTYPE_SYM) {
      CsWarning(c, "OBSOLETE obj.prototype call, use Object.setPrototypeOf()");
      CsSetPrototype(c, obj, val);
      return true;
    }

    tristate_v r = CsStoreProperty(c, obj, obj, tag, val, &hashValue, &i);
    if (r.is_defined()) return r.val(0) != 0;

    value self = obj;

    for (;;) {

      obj = CsObjectClass(obj);
      if (!obj)
        break;
      if (!CsObjectOrMethodP(obj))
        break;

      tristate_v r = CsStoreProperty(c, obj, obj, tag, val);
      if (r.is_defined()) return r.val(0) != 0;
    }

    if (value observer = CsEntityObserver(self)) {
      PROTECT(observer, self, tag, val);
      CsAddProperty(c, self, tag, val, hashValue, i);
      CsEnqueueNotification(c, observer, self, tag, val, UNDEFINED_VALUE,
        ADD_PROPERTY);
    }
    else
      CsAddProperty(c, self, tag, val, hashValue, i);
    return true;
  }
  
  static void CsClassSetItem(VM *c, value obj, value tag, value val) {
    if (tag == UNDEFINED_VALUE) // special treatment for property undefined(n,v)
                                // handler
    {
      if (CsPropertyMethodP(val)) {
        CsSetClassUndefinedPropHandler(obj, val);
        return;
      }
    }
    CsObjectSetItem(c, obj, tag, val);
  }

  void CsObjectSetItemNoLoad(VM *c, value obj, value tag, value val);

  void CsSetClassMemberTemplateVar(VM *c, value sym, value val) {
    assert(CsClassP(c->currentNS));
    PROTECT(sym, val);
    if (!CsClassP(c->currentNS)) return;

    value templ = CsClassThisVars(c->currentNS);
    if (templ == UNDEFINED_VALUE) {
      templ = CsMakeObject(c, UNDEFINED_VALUE); // raw typeless map
      CsSetClassThisVars(c->currentNS, templ);
    }
    assert(CsObjectP(templ));
    CsObjectSetItemNoLoad(c, templ, sym, val);
  }

  /*void CsSetClassInstancePrototypeMember(VM *c, value sym, value val) {
    assert(CsClassP(c->currentNS));
    PROTECT(sym, val);
    if (!CsClassP(c->currentNS)) return;

    value prototype = CsClassInstancePrototype(c->currentNS);
    if (prototype == UNDEFINED_VALUE) {
      prototype = CsMakeObject(c, c->currentNS); // raw typeless map
      CsEntityFlags(prototype) |= IS_PROTOTYPE_OBJECT;
      CsSetClassInstancePrototype(c->currentNS, prototype);
    }
    assert(CsObjectP(prototype));
    CsObjectSetItemNoLoad(c, prototype, sym, val);
  } */

  value CsObjectGetItem(VM *c, value obj, value tag) {
    FETCH_P(c, obj, tag);
    value p;
    if (tag == PROTOTYPE_SYM) { return CsObjectClass(obj); }
    //#ifdef KEY_STRING_TO_SYMBOL
    //    else if( CsStringP(tag) )
    //      tag = CsMakeSymbol(c,CsStringChars(tag));
    //#endif

    // if( CsStringP(tag) )
    //  tag = CsMakeSymbol(c,CsStringChars(tag));

    if ((p = CsFindProperty(c, obj, tag, NULL, NULL))) {
      value propValue = CsPropertyValue(p);
      return propValue;
    }
    return UNDEFINED_VALUE;
  }

  void CsObjectSetItemNoLoad(VM *c, value obj, value tag, value val) {
    int_t hashValue, i;
    value p;

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    if (tag == PROTOTYPE_SYM) {
      //if (!CsObjectP(val) && val != UNDEFINED_VALUE)
      //  CsUnexpectedTypeError(c, val, "Object class");
      //CsSetObjectClass(obj, val);
      CsWarning(c, "OBSOLETE obj.prototype call, use Object.setPrototypeOf()");
      CsSetPrototype(c, obj, val);
      return;
    }
    //#ifdef KEY_STRING_TO_SYMBOL
    //    else if( CsStringP(tag) )
    //      tag = CsMakeSymbol(c,CsStringChars(tag));
    //#endif

    if (!(p = CsFindProperty(c, obj, tag, &hashValue, &i))) {

      if (value observer = CsEntityObserver(obj)) {
        PROTECT(observer, obj, tag, val);
        CsAddProperty(c, obj, tag, val, hashValue, i);
        CsEnqueueNotification(c, observer, obj, tag, val, UNDEFINED_VALUE,
                              ADD_PROPERTY);
      } else
        CsAddProperty(c, obj, tag, val, hashValue, i);
    } else {
      value t = CsPropertyValue(p);
      if (CsPropertyMethodP(t)) {
        CsSendMessage(c, obj, t, 1, val);
      }
      else if (CsVirtualPropertyP(p)) {
        value fun = CsGetVirtualPropertySetter(p);
        if (CsAnyMethodP(fun))
          CsSendMessage(c, obj, fun, 1, val);
        else
          CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
      }
      else if (t != val) {
        CsSetPropertyValue(p, val);
        if (value observer = CsEntityObserver(obj))
          CsEnqueueNotification(c, observer, obj, tag, val, t, UPDATE_PROPERTY);
      }
    }
  }

  void CsObjectSetItem(VM *c, value obj, value tag, value val) {
    FETCH_PP(c, obj, tag, val);
    CsSetModified(obj, true);
    CsObjectSetItemNoLoad(c, obj, tag, val);
    // CsSetObjectProperty(c,obj,tag,value);
  }

  /* CsGetProperty - recursively search for a property value */
  bool CsGetProperty(VM *c, value obj, value tag, value *pValue) {
    return CsGetProperty1(c, obj, tag, pValue);
  }

  bool CsGetProperty(VM *c, value obj, const char *tag, ustring &val) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    /*if (!CsStringP(rv)) return false;
    val = tis::value_to_string(rv);*/
    string_stream s;
    CsToString(c, rv, s);
    val = s.to_ustring();
    return true;
  }
  bool CsGetProperty(VM *c, value obj, const char *tag, bool &val) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    if (CsStringP(rv)) {
      if (CsStringChars(rv) == WCHARS("true")) { val = true; return true; }
      else if (CsStringChars(rv) == WCHARS("false")) { val = false; return true; }
      return false;
    }

    rv = CsToBoolean(c, rv);

    if (CsBooleanP(rv)) {
      val = rv == TRUE_VALUE;
      return true;
    }
    CsThrowKnownError(c, CsErrUnexpectedTypeError, rv, "boolean");
    return false;
  }
  bool CsGetProperty(VM *c, value obj, const char *tag, int_t &val) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    if (CsIntegerP(rv)) {
      val = tis::to_int(rv);
      return true;
    } else if (CsFloatP(rv)) {
      val = int_t(tis::to_float(rv));
      return true;
    } else if (CsStringP(rv) && tool::try_parse_int(CsStringChars(rv),val,10u))
      return true;

    CsThrowKnownError(c, CsErrUnexpectedTypeError, rv, "integer");

    return false;
  }
  bool CsGetProperty(VM *c, value obj, const char *tag, float_t &val) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    if (CsIntegerP(rv)) {
      val = tis::to_int(rv);
      return true;
    } else if (CsFloatP(rv)) {
      val = tis::to_float(rv);
      return true;
    } else if (CsDurationP(rv)) {
      val = CsDurationSeconds(rv); // returns seconds
      return true;
    } else if (CsAngleP(rv)) {
      val = CsAngleRadians(rv); // returns radians
      return true;
    } else if (CsStringP(rv) && tool::try_parse_real(CsStringChars(rv), val))
      return true;
    CsThrowKnownError(c, CsErrUnexpectedTypeError, rv, "float");
    return false;
  }

  bool CsGetPropertyMilliseconds(VM *c, value obj, const char *tag, int_t &val) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    if (CsIntegerP(rv)) {
      val = tis::to_int(rv);
      return true;
    }
    else if (CsDurationP(rv)) {
      val = int(CsDurationSeconds(rv) * 1000.0); // returns milliseconds
      return true;
    }
    CsThrowKnownError(c, CsErrUnexpectedTypeError, rv, "duration");
    return false;
  }


  bool CsGetProperty(VM *c, value obj, const char *tag, value &val, dispatch *pd) {
    value rv = 0;
    if (!CsGetProperty(c, obj, CsSymbolOf(tag), &rv)) return false;
    if (CsIsType(rv, pd)) {
      val = rv;
      return true;
    }
    CsThrowKnownError(c, CsErrUnexpectedTypeError, rv, pd->typeName);
    return false;
  }

  bool CsSetProperty(VM *c, value obj, const char *tag, wchars str) {
    PROTECT(obj);
    value v = CsMakeString(c, str);
    return CsSetProperty(c, obj, CsSymbolOf(tag), v);
  }
  bool CsSetProperty(VM *c, value obj, const char *tag, chars str) {
    PROTECT(obj);
    value v = CsMakeString(c, str);
    return CsSetProperty(c, obj, CsSymbolOf(tag), v);
  }
  bool CsSetProperty(VM *c, value obj, const char *tag, bool val) {
    return CsSetProperty(c, obj, CsSymbolOf(tag),
                         val ? TRUE_VALUE : FALSE_VALUE);
  }
  bool CsSetProperty(VM *c, value obj, const char *tag, int_t val) {
    return CsSetProperty(c, obj, CsSymbolOf(tag), CsMakeInteger(val));
  }
  bool CsSetProperty(VM *c, value obj, const char *tag, float_t val) {
    return CsSetProperty(c, obj, CsSymbolOf(tag), CsMakeFloat(val));
  }
  bool CsSetProperty(VM *c, value obj, const char *tag, value val) {
    return CsSetProperty(c, obj, CsSymbolOf(tag), val);
  }

  bool CsFetchProperty(VM *c, value& obj, value& self, value tag, value *pValue)
  {
    value p = CsFindProperty(c, obj, tag, NULL, NULL);
    if (!p)
      return false;
    value propValue = CsPropertyValue(p);
    if (CsVPMethodP(propValue)) {
      PROTECT(obj,self);
      vp_method *method = ptr<vp_method>(propValue);
      if (method->get(c, self, *pValue))
        return true;
      else
        CsThrowKnownError(c, CsErrWriteOnlyProperty, tag);
    }
    else if (CsPropertyMethodP(propValue)) {
      PROTECT(obj, self);
      *pValue = CsSendMessage(c, self, propValue, 1, NOTHING_VALUE);
    }
    else if (CsVirtualPropertyP(propValue)) {
      PROTECT(obj, self);
      value fun = CsGetVirtualPropertyGetter(propValue);
      if (CsAnyMethodP(fun))
        *pValue = CsSendMessage(c, self, fun, 0);
      else
        CsThrowKnownError(c, CsErrWriteOnlyProperty, tag);
    }
    else
      *pValue = propValue;
    return true;
  }

  /* CsGetObjectProperty - Object get property handler */
  bool CsGetObjectProperty(VM *c, value &obj, value tag, value *pValue) {
    FETCH_P(c, obj, tag);

    if (tag == PROTOTYPE_SYM) {
      *pValue = CsObjectClass(obj);
      return true;
    }

    value self = obj;
    PROTECT(tag,self,obj); 

    if (CsFetchProperty(c, obj, self, tag, pValue))
      return true;

    value uph = UNDEFINED_VALUE; // undefined property handler
    PROTECT(uph);

    for (; obj;) {
      value next = CsObjectClass(obj);
      if (next == obj)
        break;
      if (next == UNDEFINED_VALUE || next == NOTHING_VALUE)
        break;
      if (next == NULL_VALUE)
        break;
            
      obj = next;
      
      /*value prototype;
      if (CsTryGetClassInstancePrototype(obj, prototype) && CsFetchProperty(c, prototype, self, tag, pValue))
        return true;*/

      if (CsFetchProperty(c, obj, self, tag, pValue))
        return true;

      if (CsClassP(obj)) {
        if (uph == UNDEFINED_VALUE) uph = CsClassUndefinedPropHandler(obj);
      }
    }

    if (uph != UNDEFINED_VALUE && CsPropertyMethodP(uph)) {
      PROTECT(self, tag);
      value v = CsSendMessage(c, self, uph, 2, tag, NOTHING_VALUE);
      if (v != NOTHING_VALUE) {
        *pValue = v;
        return true;
      }
    }

    if (CsEntityFlags(self) & IS_STRONG_SEALED)
      CsThrowKnownError(c, CsErrNoProperty, self, tag);

    return false;
  }

  void CsSetPrototype(VM *c, value obj, value proto) {
    //if (CsObjectClass(obj) == val)
    if (proto == NULL_VALUE || proto == UNDEFINED_VALUE)
      CsSetObjectClass(obj, proto);
    else if (CsDerivedFromObjectP(proto)) {
      for (value t = obj; t && CsDerivedFromObjectP(t); t = CsObjectClass(t))
      {
        if( t == proto )
          CsThrowKnownError(c, CsErrGenericError, "will create prototype chain loop");
      }
      CsSetObjectClass(obj, proto);
    } else 
      CsUnexpectedTypeError(c, proto, "instance of Object class");
  }

  bool CsTryStoreProperty(VM *c, value obj, value self, value tag, value val, int_t *pHashValue, int_t *pIndex) {

    value p = CsFindProperty(c, obj, tag, pHashValue, pIndex);

    if (!p)
      return false;

    value propValue = CsPropertyValue(p);

    if (CsVPMethodP(propValue)) {
      vp_method *method = ptr<vp_method>(propValue);
      return method->set(c, tag, self, val);
    }
    else if (CsPropertyMethodP(propValue)) {
      CsSendMessage(c, self, propValue, 1, val);
    }
    else if (CsVirtualPropertyP(propValue)) {
      value fun = CsGetVirtualPropertySetter(propValue);
      if (CsAnyMethodP(fun))
        CsSendMessage(c, self, fun, 1, val);
      else
        CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
    }
    else if (CsPropertyIsConst(p))
      CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
    else if (obj == self) {
      if (propValue != val) {
        CsSetPropertyValue(p, val);
        if (value observer = CsEntityObserver(obj))
          CsEnqueueNotification(c, observer, obj, tag, val, propValue, UPDATE_PROPERTY);
      }
    }
    else
      return false;
    return true;
  }

  // allows to override const property
  tristate_v CsStoreProperty(VM *c, value obj, value self, value tag, value val, int_t *pHashValue, int_t *pIndex) {

    value p = CsFindProperty(c, obj, tag, pHashValue, pIndex);

    if (!p)
      return tristate_v();

    value propValue = CsPropertyValue(p);

    if (CsVPMethodP(propValue)) {
      vp_method *method = ptr<vp_method>(propValue);
      return method->set(c, tag, self, val) ? TRUE : FALSE;
    }
    else if (CsPropertyMethodP(propValue)) {
      CsSendMessage(c, self, propValue, 1, val);
    }
    else if (CsVirtualPropertyP(propValue)) {
      value fun = CsGetVirtualPropertySetter(propValue);
      if (CsAnyMethodP(fun))
        CsSendMessage(c, self, fun, 1, val);
      else
        CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
    }
    //else if (CsPropertyIsConst(p))
    //  CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
    else if (obj == self) {
      if (propValue != val) {
        CsSetPropertyValue(p, val);
        if (value observer = CsEntityObserver(obj))
          CsEnqueueNotification(c, observer, obj, tag, val, propValue, UPDATE_PROPERTY);
      }
    }
    else
      return tristate_v();
    return TRUE;
  }
  
  bool CsSetObjectPropertyNoLoad(VM *c, value obj, value tag, value val) {
    int_t hashValue = 0, i = 0;

    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    if (tag == PROTOTYPE_SYM) {
      CsWarning(c, "OBSOLETE obj.prototype call, use Object.setPrototypeOf()");
      CsSetPrototype(c, obj, val);
      return true;
    }

    if (CsTryStoreProperty(c, obj, obj, tag, val, &hashValue, &i))
      return true;

    value self = obj;
    value uph  = UNDEFINED_VALUE;

    for (;;) {

      obj = CsObjectClass(obj);
      if (!obj)
        break;
      if (obj == UNDEFINED_VALUE)
        break;

      /*value prototype;
      if (CsTryGetClassInstancePrototype(obj, prototype) && CsTryStoreProperty(c, prototype, self, tag, val))
        return true;*/

      tristate_v r = CsStoreProperty(c, obj, self, tag, val);
      if (r.is_defined()) return r.val(0) != 0;

      if (CsClassP(obj) && uph == UNDEFINED_VALUE)
        uph = CsClassUndefinedPropHandler(obj);
    }

    if (CsPropertyMethodP(uph)) {
      PROTECT(self, tag, val);
      if (NOTHING_VALUE != CsSendMessage(c, self, uph, 2, tag, val))
        return true;
    }
    if (value observer = CsEntityObserver(self)) {
      PROTECT(observer, self, tag, val);
      CsAddProperty(c, self, tag, val, hashValue, i);
      CsEnqueueNotification(c, observer, self, tag, val, UNDEFINED_VALUE,
                            ADD_PROPERTY);
    } else
      CsAddProperty(c, self, tag, val, hashValue, i);
    return true;
  }

  bool CsSetObjectPersistentProperty(VM *c, value obj, value tag, value val) {
    int_t hashValue, i;
    value p;

    if (tag == PROTOTYPE_SYM) {
      CsWarning(c, "OBSOLETE obj.prototype call, use Object.setPrototypeOf()");
      CsSetPrototype(c, obj, val);
      return true;
    }
    p = CsFindProperty(c, obj, tag, &hashValue, &i);
    if (p) {
      value t = CsPropertyValue(p);
      CsSetPropertyValue(p, val);
      if (value observer = CsEntityObserver(obj))
        CsEnqueueNotification(c, observer, obj, tag, val, t, UPDATE_PROPERTY);
      return true;
    }

    value self = obj;

    for (;;) {

      obj = CsObjectClass(obj);
      if (!obj)
        break;
      if (obj == UNDEFINED_VALUE)
        break;
      
      if ((p = CsFindProperty(c, obj, tag, 0, 0))) {
        value propValue = CsPropertyValue(p);
        if (CsPropertyMethodP(propValue) || CsVirtualPropertyP(propValue)) {
          // CsSendMessage(c,self,propValue,1, val );
          return true;
        } else if (CsVPMethodP(propValue)) {
          // vp_method *method = ptr<vp_method>(propValue);
          // if (method->set(c,obj,val))
          return true;
          // else
          //  CsThrowKnownError(c,CsErrReadOnlyProperty,tag);
        } else if (CsPropertyIsConst(p))
          // CsThrowKnownError(c,CsErrReadOnlyProperty,tag);
          return true;
        break;
      }
    }

    if (value observer = CsEntityObserver(self)) {
      PROTECT(observer, self, tag, val);
      CsAddProperty(c, self, tag, val, hashValue, i);
      CsEnqueueNotification(c, observer, self, tag, val, UNDEFINED_VALUE,ADD_PROPERTY);
    } else
      CsAddProperty(c, self, tag, val, hashValue, i);
    return true;
  }

  /* CsSetObjectProperty - Object set property handler */
  bool CsSetObjectProperty(VM *c, value obj, value tag, value val) {
    FETCH_PP(c, obj, tag, val);
    CsSetModified(obj, true);
    return CsSetObjectPropertyNoLoad(c, obj, tag, val);
  }

  // inline  bool        CsSetProperty(VM* c,value o,value t, value v)  { return
  // CsGetDispatch(o)->setProperty(c,o,t,v); }

  ustring CsClassClassName(VM *c, value cls) {
    if (CsClassP(cls)) {
      value n = CsClassName(cls);
      if (CsSymbolP(n)) return CsSymbolName(n);
    }
    return ustring();
  }

  ustring CsClassClassFullName(VM *c, value cls) {
    ustring us;
    if (CsClassP(cls) || CsNamespaceP(cls)) {
      value ns = CsClassNamespace(cls);
      if (CsClassP(ns) || CsNamespaceP(ns))
        us = CsClassClassFullName(c, ns);
      value n = CsClassName(cls);
      if (n != NULL_VALUE && n != UNDEFINED_VALUE && CsSymbolP(n)) {
        if (us.length())
          us += WCHARS(".");
        us += CsSymbolName(n);
      }
    }
    return us;
  }

  ustring CsObjectClassName(VM *c, value obj) {
    value cls = CsObjectClass(obj);
    if (cls == UNDEFINED_VALUE) return ustring();
    return CsClassClassName(c, cls);
  }

  bool CsSetProperty1(VM *c, value obj, value tag, value val) {
    uint flags = CsEntityFlags(obj);
    if (flags & IS_FROZEN) CsThrowKnownError(c, CsErrIsFrozen, obj);

    int_t hashValue, i;
    value p;

    if (tag == PROTOTYPE_SYM) {
      //if (!CsObjectP(val) && !CsMethodP(val) && val != UNDEFINED_VALUE)
      //  CsUnexpectedTypeError(c, val, "instance of Object class");
      //CsSetObjectClass(obj, val);
      CsWarning(c, "OBSOLETE obj.prototype call, use Object.setPrototypeOf()");
      CsSetPrototype(c, obj, val);
    } else if (!(p = CsFindProperty(c, obj, tag, &hashValue, &i))) {
      // CsAddProperty(c,obj,tag,val,hashValue,i);
      dispatch *pd = CsGetDispatch(obj);
      pd->setItem(c, obj, tag, val);
    } else {
      CsSetPropertyValue(p, val);
    }
    return true;
  }

  /* CsObjectSize - Object size handler */
  long CsObjectSize(value obj) { return sizeof(object); }

  /* CsObjectScan - Object scan handler */
  void CsObjectScan(VM *c, value obj) {
    ptr<persistent_header>(obj)->vstorage =
        CsCopyValue(c, ptr<persistent_header>(obj)->vstorage);
    CsSetObjectClass(obj, CsCopyValue(c, CsObjectClass(obj)));
    CsSetObjectProperties(obj, CsCopyValue(c, CsObjectProperties(obj)));
    if (CsEntityObserver(obj))
      CsSetEntityObserver(obj, CsCopyValue(c, CsEntityObserver(obj)));
    if (CsObjectEvents(obj))
      CsSetObjectEvents(obj, CsCopyValue(c, CsObjectEvents(obj)));
    if (CsObjectMeta(obj))
      CsSetObjectMeta(obj, CsCopyValue(c, CsObjectMeta(obj)));
  }

  /* CsClassSize - Class size handler */
  long CsClassSize(value obj) { return sizeof(klass); }

  /* CsObjectScan - Object scan handler */
  void CsClassScan(VM *c, value obj) {
    CsObjectScan(c, obj);
    CsSetClassName(obj, CsCopyValue(c, CsClassName(obj)));
    CsSetClassNamespace(obj, CsCopyValue(c, CsClassNamespace(obj)));
    CsSetClassUndefinedPropHandler(
        obj, CsCopyValue(c, CsClassUndefinedPropHandler(obj)));
    CsSetClassThisVars(obj, CsCopyValue(c, CsClassThisVars(obj)));
  }

  void CsMergeThisVarsFromClass(VM *c, value obj, value proto) {
    value tmpl = UNDEFINED_VALUE;
    if (CsClassP(proto) && CsObjectP(tmpl = CsClassThisVars(proto))) {
      value key = UNDEFINED_VALUE, val = UNDEFINED_VALUE;
      PROTECT(obj, proto, tmpl, key, val);
      // newo = CsCloneObject(c,tmpl);
      each_property prop(c, tmpl);
      while (prop(key, val)) {
        if (CsObjectP(val))
          val = CsCloneObject(c, val);
        else if (CsVectorP(val))
          val = CsCloneVector(c, val);
        CsSetProperty1(c, obj, key, val);
      }
    }
  }

  /* CsMakeObject - make a new obj */
  value CsMakeObject(VM *c, value proto) {
    value newo;
    value tmpl = 0;

    PROTECT(proto);

    if (CsClassP(proto) && CsObjectP(tmpl = CsClassThisVars(proto)))
      newo = CsCloneObject(c, tmpl, true);
    else 
    {
      newo = CsAllocate(c, sizeof(object));
#if defined(TISCRIPT_USE_PERSISTENCE)
      _CsInitPersistent(newo);
#endif
      CsSetObjectProperties(newo, UNDEFINED_VALUE);
      CsSetObjectPropertyCount(newo, 0);
      CsSetObjectEvents(newo, NULL_VALUE);
    }

    CsSetDispatch(newo, &CsObjectDispatch);
    CsSetObjectClass(newo, proto);
    //--CsSetObjectObserver(newo,UNDEFINED_VALUE);

    assert(sizeof(object) == ValueSize(newo));

    return newo;
  }

  value CsNewClassInstance(VM *c, value parentClass, value nameSymbol) {
    value newo  = 0;
    value templ = UNDEFINED_VALUE;

    PROTECT(parentClass, nameSymbol, newo, templ);
    // CsCPush(c,parentClass);
    // CsCPush(c,nameSymbol);
    newo = CsAllocate(c, sizeof(klass));
    CsSetDispatch(newo, &CsClassDispatch);
    CsSetClassName(newo, nameSymbol);
    CsSetClassNamespace(newo, c->currentNS);
    CsSetObjectClass(newo, parentClass);
    CsSetClassUndefinedPropHandler(newo, UNDEFINED_VALUE);
    
    //CsSetClassInstancePrototype(newo, UNDEFINED_VALUE);

    if (CsClassP(parentClass))
      templ = CsCloneObject(c, CsClassThisVars(parentClass));
    CsSetClassThisVars(newo, templ);

    CsSetObjectProperties(newo, UNDEFINED_VALUE);
    CsSetObjectPropertyCount(newo, 0);
    CsSetObjectEvents(newo, NULL_VALUE);
    //--CsSetObjectObserver(newo,UNDEFINED_VALUE);
    assert(sizeof(klass) == ValueSize(newo));
    return newo;
  }

  value CsNewNamespaceInstance(VM *c, value parentNamespace, value nameSymbol) {
    value newo = 0;
    if (c->currentNS && CsNamespaceP(c->currentNS) &&
        CsGetProperty(c, c->currentNS, nameSymbol,
                      &newo)) // multiple namespace Foo are allowed
    {
      if (CsNamespaceP(newo) && CsObjectClass(newo) == parentNamespace)
        return newo;
    }
    CsCPush(c, parentNamespace);
    CsCPush(c, nameSymbol);
    newo = CsAllocate(c, sizeof(klass));
    CsSetDispatch(newo, &CsNamespaceDispatch);
    CsSetClassName(newo, CsPop(c));
    CsSetClassNamespace(newo, c->currentNS ? c->currentNS : UNDEFINED_VALUE);
    CsSetObjectClass(newo, CsPop(c));
    CsSetClassUndefinedPropHandler(newo, UNDEFINED_VALUE);
    CsSetNamespaceIncludedList(newo, UNDEFINED_VALUE);
    CsSetObjectProperties(newo, UNDEFINED_VALUE);
    CsSetObjectPropertyCount(newo, 0);
    CsSetObjectEvents(newo, NULL_VALUE);
    //--CsSetObjectObserver(newo,UNDEFINED_VALUE);
    assert(sizeof(klass) == ValueSize(newo));
    return newo;
  }

  /* CsCloneObject - clone an existing obj */
  value CsCloneObject(VM *c, value obj, bool deep) {
    if (!CsObjectP(obj)) return obj;
    FETCH(c, obj);

    value newobj = 0, key = 0, val = 0;
    PROTECT(obj, newobj, key, val);

    newobj = CsMakeObject(c, CsObjectClass(obj));

    each_property gen(c, obj);
    for (; gen(key, val);) {
      if (deep) val = CsClone(c, val, true);
      CsSetProperty1(c, newobj, key, val);
    }
    return newobj;
  }

  value CsCloneObject(VM *c, value obj, slice<value> inc_keys, slice<value> exc_keys) {
    if (!CsObjectP(obj)) return obj;
    FETCH(c, obj);

    value newobj = 0, key = 0, val = 0;
    PROTECT(obj, newobj, key, val);

    newobj = CsMakeObject(c, CsObjectClass(obj));

    each_property gen(c, obj);
    for (; gen(key, val);) {
      if (inc_keys.length && !inc_keys.contains(key)) continue;
      if (exc_keys.length && exc_keys.contains(key)) continue;
      CsSetProperty1(c, newobj, key, val);
    }
    return newobj;
  }

  static value CopyPropertyTableExcept(VM *c, value table, value tag, bool &r);
  static value CopyPropertyListExcept(VM *c, value table, value tag, bool &r);

  bool CsRemoveObjectProperty(VM *c, value obj, value tag) {
    FETCH_P(c, obj, tag);

    uint flags = CsEntityFlags(obj);
    if (flags & (IS_FROZEN | IS_SEALED))
      CsThrowKnownError(c, CsErrIsSealed, obj);

    value properties;
    CsCheck(c, 1);
    CsPush(c, obj);
    properties   = CsObjectProperties(obj);
    bool removed = false;
    if (CsHashTableP(properties)) {
      properties = CopyPropertyTableExcept(c, properties, tag, removed);
    } else {
      properties = CopyPropertyListExcept(c, properties, tag, removed);
    }
    if (removed) {
      CsSetModified(obj, true);
      CsSetObjectProperties(CsTop(c), properties);
      CsSetObjectPropertyCount(CsTop(c), CsObjectPropertyCount(CsTop(c)) - 1);
      if (value observer = CsEntityObserver(obj))
        CsEnqueueNotification(c, observer, obj, tag, UNDEFINED_VALUE,
                              UNDEFINED_VALUE, DELETE_PROPERTY);
    }
    CsPop(c);
    return removed;
  }

  // inline int_t CsKeyHashValue(value tag) {
  //
  //}

  /* CsFindProperty - find a property of a non-inherited obj property */
  value CsFindProperty(VM *c, value obj, value tag, int_t *pHashValue,
                       int_t *pIndex) {
#ifdef _DEBUG
   dispatch *pd = CsGetDispatch(obj);
   //string symname = CsSymbolName(tag);
#endif

    value p = CsObjectProperties(obj);
    if (CsHashTableP(p)) {
      int_t hashValue = CsHashValue(tag);
      int_t i         = hashValue & (CsHashTableSize(p) - 1);
      p               = CsHashTableElement(p, i);
      if (pHashValue) *pHashValue = hashValue;
      if (pIndex) *pIndex = i;
    } else {
      if (pIndex) *pIndex = -1;
    }
    for (; p != UNDEFINED_VALUE; p = CsPropertyNext(p)) {
      value ptag = CsPropertyTag(p);
      if (CsKeysAreEqual(c, ptag, tag)) return p;
    }
    return 0;
  }

  value FindFirstSymbol(VM *c, value obj, value &idx) {
    value p = CsObjectProperties(obj);
    if (CsHashTableP(p)) {
      for (int i = 0; i < CsHashTableSize(p); ++i) {
        value t = CsHashTableElement(p, i);
        if (t != UNDEFINED_VALUE) {
          idx = t;
          return CsPropertyTag(t);
        }
      }
    }
    if (p != UNDEFINED_VALUE) {
      idx = p;
      return CsPropertyTag(p);
    } else {
      idx = UNDEFINED_VALUE;
      return NOTHING_VALUE;
    }
  }

  value FindFirstSymbolValue(VM *c, value obj, value &idx) {
    value p = CsObjectProperties(obj);
    if (CsHashTableP(p)) {
      for (int i = 0; i < CsHashTableSize(p); ++i) {
        value t = CsHashTableElement(p, i);
        if (t != UNDEFINED_VALUE) {
          idx = t;
          //CsSetRVal(c, 1, CsPropertyTag(t));
          //return CsPropertyValue(t);
          CS_RETURN2(c, CsPropertyTag(t), CsPropertyValue(t));
        }
      }
    }
    if (p != UNDEFINED_VALUE) {
      idx = p;
      //CsSetRVal(c, 1, CsPropertyTag(p));
      //return CsPropertyValue(p);
      CS_RETURN2(c, CsPropertyTag(p), CsPropertyValue(p));
    } else {
      idx = UNDEFINED_VALUE;
      //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
      return NOTHING_VALUE;
    }
  }

  value FindNextSymbol(VM *c, value obj, value &idx) {
    if (idx == UNDEFINED_VALUE) return NOTHING_VALUE;

    value np = CsPropertyNext(idx);
    if (np != UNDEFINED_VALUE) {
      idx = np; // easy case
      return CsPropertyTag(np);
    }

    value props = CsObjectProperties(obj);
    value tag   = CsPropertyTag(idx);

    if (!CsHashTableP(props)) // this is simple prop list
                              // its end reached.
    {
      idx = UNDEFINED_VALUE;
      return NOTHING_VALUE;
    }

    int_t hashValue = CsHashValue(tag);
    int_t i         = hashValue & (CsHashTableSize(props) - 1);

    for (++i; i < CsHashTableSize(props); ++i) {
      value p = CsHashTableElement(props, i);
      if (p != UNDEFINED_VALUE) {
        idx = p;
        return CsPropertyTag(p);
      }
    }

    idx = UNDEFINED_VALUE; // end of table has been reached;
    return NOTHING_VALUE;
  }

  value FindNextSymbolValue(VM *c, value obj, value &idx) {
    if (idx == UNDEFINED_VALUE) 
      return NOTHING_VALUE;
      //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);

    value np = CsPropertyNext(idx);
    if (np != UNDEFINED_VALUE) {
      idx = np; // easy case
      //CsSetRVal(c, 1, CsPropertyTag(np));
      //return CsPropertyValue(np);
      CS_RETURN2(c, CsPropertyTag(np), CsPropertyValue(np));
    }

    value props = CsObjectProperties(obj);
    value tag   = CsPropertyTag(idx);

    if (!CsHashTableP(props)) // this is simple prop list
                              // its end reached.
    {
      idx = UNDEFINED_VALUE;
      //CsSetRVal(c, 1, NOTHING_VALUE);
      return NOTHING_VALUE;
      //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
    }

    int_t hashValue = CsHashValue(tag);
    int_t i         = hashValue & (CsHashTableSize(props) - 1);

    for (++i; i < CsHashTableSize(props); ++i) {
      value p = CsHashTableElement(props, i);
      if (p != UNDEFINED_VALUE) {
        idx = p;
        //CsSetRVal(c, 1, CsPropertyTag(p));
        //return CsPropertyValue(p);
        CS_RETURN2(c, CsPropertyTag(p), CsPropertyValue(p));
      }
    }
    idx = UNDEFINED_VALUE; // end of table has been reached;
    //CsSetRVal(c, 1, NOTHING_VALUE);
    return NOTHING_VALUE;
    //CS_RETURN2(c, NOTHING_VALUE, NOTHING_VALUE);
  }

  value CsObjectNextElement(VM *c, value *index, value obj, int nr) {
    if (*index == NOTHING_VALUE) // first
    {
      FETCH(c, obj);
      return nr > 1 ? FindFirstSymbolValue(c, obj, *index)
                    : FindFirstSymbol(c, obj, *index);
    } else
      return nr > 1 ? FindNextSymbolValue(c, obj, *index)
                    : FindNextSymbol(c, obj, *index);
  }

  /* CopyPropertyList - copy the property list of an obj */
  static value CopyPropertyList(VM *c, value plist) {
    CsCheck(c, 2);
    CsPush(c, UNDEFINED_VALUE);
    CsPush(c, plist);
    for (; CsTop(c) != UNDEFINED_VALUE; CsSetTop(c, CsPropertyNext(CsTop(c)))) {
      value tag  = CsPropertyTag(CsTop(c));
      value val  = CsPropertyValue(CsTop(c));
      value newo = CsMakeProperty(c, tag, val, CsPropertyFlags(CsTop(c)));
      CsSetPropertyNext(newo, c->sp[1]);
      c->sp[1] = newo;
    }
    CsDrop(c, 1);
    return CsPop(c);
  }

  // used for deletion of properties
  static value CopyPropertyListExcept(VM *c, value plist, value tag, bool &r) {
    value p = 0;
    value t = plist;

    for (; t != UNDEFINED_VALUE; t = CsPropertyNext(t)) {
      value pTag = CsPropertyTag(t);
      if (CsKeysAreEqual(c, tag, pTag)) {
        if (CsPropertyIsConst(t))
          CsThrowKnownError(c, CsErrReadOnlyProperty, tag);
        r = true;
        if (p)
          CsSetPropertyNext(p, CsPropertyNext(t));
        else
          plist = CsPropertyNext(t);
        break;
      }
      p = t;
    }
    return plist;
  }

  /*
  static value CopyPropertyListExcept(VM *c,value plist, value tag, bool& r)
  {
      CsCheck(c,2);
      CsPush(c,UNDEFINED_VALUE);
      CsPush(c,plist);
      for (; CsTop(c) != UNDEFINED_VALUE; CsSetTop(c,CsPropertyNext(CsTop(c))))
      {
          value pTag = CsPropertyTag(CsTop(c));
          if( tag == pTag)
          {
            r = true;
            continue;
          }
          value pValue = CsPropertyValue(CsTop(c));
          value newo = CsMakeProperty(c,pTag,pValue,CsPropertyFlags(CsTop(c)));
          CsSetPropertyNext(newo,c->sp[1]);
          c->sp[1] = newo;
      }
      CsDrop(c,1);
      return CsPop(c);
  }

  */

  /* CopyPropertyTable - copy the property hash table of an obj */
  static value CopyPropertyTable(VM *c, value table) {
    int_t size = CsHashTableSize(table);
    int_t i;
    CsCheck(c, 2);
    CsPush(c, CsMakeHashTable(c, size));
    CsPush(c, table);
    for (i = 0; i < size; ++i) {
      value properties = CopyPropertyList(c, CsHashTableElement(CsTop(c), i));
      CsSetHashTableElement(c->sp[1], i, properties);
    }
    CsDrop(c, 1);
    return CsPop(c);
  }

  /* CopyPropertyTable - copy the property hash table of an obj */
  static value CopyPropertyTableExcept(VM *c, value table, value tag, bool &r) {
    r          = false;
    int_t size = CsHashTableSize(table);
    int_t i;
    CsCheck(c, 1);
    // CsPush(c,CsMakeHashTable(c,size));
    // CsPush(c,table);
    CsPush(c, table);
    // CsPush(c,CsMakeHashTable(c,size));
    // tool::swap( c->sp[1], c->sp[0] );

    for (i = 0; i < size; ++i) {
      value properties =
          CopyPropertyListExcept(c, CsHashTableElement(CsTop(c), i), tag, r);
      CsSetHashTableElement(CsTop(c), i, properties);
    }
    // CsDrop(c,1);
    return CsPop(c);
  }

  /* CsSetCObjectProperty - CObject set property handler */
  bool CsAddObjectConstant(VM *c, value obj, value tag, value val) {
    int_t hashValue, i;
    value p;

    //#ifdef KEY_STRING_TO_SYMBOL
    //    if( CsStringP(tag) )
    //      tag = CsMakeSymbol(c,CsStringChars(tag));
    //#endif

    if ((p = CsFindProperty(c, obj, tag, &hashValue, &i))) {
      if (CsPropertyValue(p) == val) return true;
      CsAlreadyDefined(c, tag);
    }

    /* add a property */
    CsAddProperty(c, obj, tag, val, hashValue, i, PROP_CONST);
    return true;
  }

  /* CsAddProperty - add a property to an obj */
  void CsAddProperty(VM *c, value obj, value tag, value val, int_t hashValue,
                     int_t i, uint flags) {
    uint oflags = CsEntityFlags(obj);
    if (oflags & (IS_SEALED | IS_FROZEN))
      CsThrowKnownError(c, CsErrIsSealed, obj);

    value p = 0;
    PROTECT(obj, p, val, tag);

    flags |= (PROP_ORDINAL_MASK & CsObjectPropertyCount(obj));
    p = CsMakeProperty(c, tag, val, flags);
    if (i >= 0) {
      int_t currentSize = CsHashTableSize(CsObjectProperties(obj));
      if (CsObjectPropertyCount(obj) >=
          currentSize * CsHashTableExpandThreshold) {
        CsCheck(c, 1);
        i = ExpandHashTable(c, obj, hashValue);
      }
      CsSetPropertyNext(p, CsHashTableElement(CsObjectProperties(obj), i));
      CsSetHashTableElement(CsObjectProperties(obj), i, p);
    } else {
      if (CsObjectPropertyCount(obj) >= CsHashTableCreateThreshold)
        CreateHashTable(c, obj, p);
      else {
        CsSetPropertyNext(p, CsObjectProperties(obj));
        CsSetObjectProperties(obj, p);
      }
    }
    IncObjectPropertyCount(obj);
  }

  /* CreateHashTable - create an obj hash table and enter a property */
  static void CreateHashTable(VM *c, value obj, value prop) {
    int_t i;
    value table;
    // CsCheck(c,2);
    // CsPush(c,p);
    // CsPush(c,obj);
    value p = 0;
    PROTECT(obj, p, prop);
    table = CsMakeHashTable(c, CsHashTableCreateThreshold);
    // obj = CsPop(c);
    p = CsObjectProperties(obj);
    CsSetObjectProperties(obj, table);
    while (p != UNDEFINED_VALUE) {
      value next = CsPropertyNext(p);
      i = CsHashValue(CsPropertyTag(p)) & (CsHashTableCreateThreshold - 1);
      CsSetPropertyNext(p, CsHashTableElement(table, i));
      CsSetHashTableElement(table, i, p);
      p = next;
    }
    // p = CsPop(c);
    i = CsHashValue(CsPropertyTag(prop)) & (CsHashTableCreateThreshold - 1);
    CsSetPropertyNext(prop, CsHashTableElement(table, i));
    CsSetHashTableElement(table, i, prop);
  }

  /* ExpandHashTable - expand an obj hash table and return the new hash index */
  static int ExpandHashTable(VM *c, value obj, int_t hashValue) {
    int_t oldSize, newSize;
    PROTECT(obj);
    oldSize = CsHashTableSize(CsObjectProperties(obj));
    newSize = oldSize << 1;
    if (newSize <= CsHashTableExpandMaximum) {
      value oldTable, newTable;
      int_t j;
      // CsPush(c,obj);
      newTable = CsMakeHashTable(c, newSize);
      oldTable = CsObjectProperties(obj);
      for (j = 0; j < oldSize; ++j) {
        value p    = CsHashTableElement(oldTable, j);
        value new0 = UNDEFINED_VALUE;
        value new1 = UNDEFINED_VALUE;
        while (p != UNDEFINED_VALUE) {
          value next = CsPropertyNext(p);
          if (CsHashValue(CsPropertyTag(p)) & oldSize) {
            CsSetPropertyNext(p, new1);
            new1 = p;
          } else {
            CsSetPropertyNext(p, new0);
            new0 = p;
          }
          p = next;
        }
        CsSetHashTableElement(newTable, j, new0);
        CsSetHashTableElement(newTable, j + oldSize, new1);
      }
      CsSetObjectProperties(obj, newTable);
    }
    return hashValue & (newSize - 1);
  }

    /* PRBC_ERTY */

#define SetPropertyTag(o, v) CsSetFixedVectorElement(o, 0, v)

  /* Property handlers */
  static long PropertySize(value obj);
  static void PropertyScan(VM *c, value obj);

  /* Property pdispatch */
  dispatch CsPropertyDispatch = {
      "PropertyTuple",      &CsFixedVectorDispatch, CsDefaultGetProperty,
      CsDefaultSetProperty, CsDefaultNewInstance,   CsDefaultPrint,
      PropertySize,         CsDefaultCopy,          PropertyScan,
      CsDefaultHash,        CsDefaultGetItem,       CsDefaultSetItem};

  /* PropertySize - Property size handler */
  static long PropertySize(value obj) {
    return sizeof(CsFixedVector) + CsPropertySize * sizeof(value);
  }

  /* PropertyScan - Property scan handler */
  static void PropertyScan(VM *c, value obj) {
    int i;
    for (i = 0; i < CsPropertySize; ++i)
      CsSetFixedVectorElement(obj, i, CsCopyValue(c, CsFixedVectorElement(obj, i)));
  }

  /* CsMakeProperty - make a property */
  value CsMakeProperty(VM *c, value &key, value &val, uint flags) {
    value newo;
    CsCheck(c, 2);
    CsPush(c, val);
    CsPush(c, key);
    newo = CsMakeFixedVectorValue(c, &CsPropertyDispatch, CsPropertySize);
    key  = CsPop(c);
    val  = CsPop(c);
    SetPropertyTag(newo, key);
    CsSetPropertyValue(newo, val);
    CsSetPropertyFlags(newo, flags);
    return newo;
  }

#pragma warning(pop)
}
