
#include "cs.h"
#include "cs_async.h"
#include "tool/websockets/websockets.h"

namespace tis 
{
#ifdef NETWORK_SUPPORT

  // DataSocket

  struct data_connection : public async_object<data_connection,VM> , public tool::async::data_connection {
    typedef tool::async::data_connection super;

    data_connection(VM *c) {}

    TOOL_INTERFACE_2(tis, data_connection, tool::async::data_connection, gcable)
   
    bool send(value data) {
      if (!_this)
        return false;

      tis::binary_o_stream os;
      CsStoreValue(get_vm(), data, &os);
      return send_data(os.buf());
    }

    virtual void on_connected() override { 
      notify(WCHARS("connect")); 
    }
    virtual void on_error(int ern) override { 
      //notify(WCHARS("error"), CsMakeInteger(ern)); 
      notify(WCHARS("error"), CsSymbolOf(tool::async::result_to_string(ern)));
    }
    virtual void on_data_sent() override {
      notify(WCHARS("send"));
    }
    virtual void on_closed() override { 
      notify(WCHARS("close"));
      async_object<data_connection, VM>::detach();
    }
       
    virtual void on_data_received(bytes data) override {
      if (!_this)
        return;

      handle<VM> vm(get_vm());

      tis::binary_i_stream is(data, WCHARS("message"), false);
      value                r = NULL_VALUE;
      TRY {
        CsFetchValue(vm, &r, &is);
        notify(WCHARS("data"), r);
      }
      CATCH_ERROR(e) {
        e;
        CsDisplay(vm, vm->val, vm->standardError);
        notify(WCHARS("error"), vm->val);
      }
    }

    static value CSF_send(VM *c) {
      value data;
      value obj;
      CsParseArguments(c, "V=*V", &obj, this_dispatch(c), &data);
      data_connection *ws = data_connection::object_ptr(c, obj);
      if (!ws)
        CsThrowKnownError(c, CsErrGenericError, "inactive socket");
      else
        ws->send(data);
      return obj;
    }

    static value CSF_listen(VM *c);
    static value CSF_connect(VM *c);
    
    //traits:
    static dispatch*   this_dispatch(VM* c) { return c->datasocketDispatch; }
    static const char* type_name() { return "DataSocket"; }

    static c_method methods[];

  };

  c_method data_connection::methods[] = {
    C_METHOD_ENTRY("listen", CSF_listen),
    C_METHOD_ENTRY("connect", CSF_connect),
    C_METHOD_ENTRY("send", CSF_send),
    C_METHOD_ENTRY("close", CSF_close),
    C_METHOD_ENTRY(0, 0)
  };

  // connection*  accept    (function<connection*(chars host,int port)>
  // ctor);

  struct data_listener : public tis::data_connection {
    typedef tis::data_connection super;
    data_listener(VM *c, value acc) : super(c) {
      value obj = tis::data_connection::this_object(c, this);
      CsEventObjectAdd(c, obj, acc, WCHARS("accept"));
    }
    virtual ~data_listener() {
      //get_ref_count();
    }

    virtual void handle_accept(void) override {
      VM* c = get_vm();
      handle<tis::data_connection> con = new tis::data_connection(c);
      value obj = tis::data_connection::this_object(c, con);
      if (notify(WCHARS("accept"), obj)) 
        accept(con);
    }
  };

  value data_connection::CSF_listen(VM *c) {

    int    port     = 0;
    wchars host;  //= WCHARS("localhost");
    value  acceptor = 0;

    CsParseArguments(c, "**V=S#|i", &acceptor, &CsMethodDispatch,
                     &host.start, &host.length, &port);

    handle<data_listener> l = new data_listener(c, acceptor);

    //if (l->listen(pview, port, string(host))) return l->_this;
    if (l->listen(string(host)))
      return l->_this;

    return UNDEFINED_VALUE;
  }
  value data_connection::CSF_connect(VM *c) {

    int    port = 0;
    wchars host;// = WCHARS("localhost");

    CsParseArguments(c, "**S#|i", &host.start, &host.length, &port);

    handle<data_connection> con = new data_connection(c);
    if (con->connect(string(host)))
      return tis::data_connection::this_object(c, con);

    return NULL_VALUE;
  }

  // Socket - raw socket - either pipe or tcp socket

  struct connection : public async_object<connection, VM>, public tool::async::pipe_connection  {
    typedef tool::async::pipe_connection super;

    connection(VM *c) {}

    TOOL_INTERFACE_2(tis, connection, tool::async::pipe_connection, gcable)

    bool send(value data) {
      if (!_this)
        return false;

      //tis::binary_o_stream os;
      //CsStoreValue(get_vm(), data, &os);
      if (CsStringP(data))
        return super::send(u8::cvt(CsStringChars(data)).chars_as_bytes()) > 0;
      else if (CsByteVectorP(data))
        return super::send(CsByteVectorBytes(data)) > 0;
      else {
        CsThrowKnownError(get_vm(), CsErrUnexpectedTypeError, data, "string or bytes");
        return false;
      }
    }

    virtual void handle_connect() override {
      super::handle_connect();
      notify(WCHARS("connect"));
    }
    virtual void handle_error(int ern) override {
      super::handle_error(ern);
      notify(WCHARS("error"), CsSymbolOf( tool::async::result_to_string(ern)));
    }
    virtual void handle_write() override {
      super::handle_write();
      notify(WCHARS("send"));
    }
    virtual void handle_close() override {
      super::handle_close();
      notify(WCHARS("close"));
      async_object<connection, VM>::detach();
    }

    virtual void handle_read(bytes data) override {
      if (!_this)
        return;
      array<wchar> chars; u8::to_utf16(data,chars,false);
      notify(WCHARS("data"), CsMakeString(get_vm(),chars()));
    }

    static value CSF_send(VM *c) {
      value data;
      value obj;
      CsParseArguments(c, "V=*V", &obj, this_dispatch(c), &data);
      connection *ws = connection::object_ptr(c, obj);
      if (!ws)
        CsThrowKnownError(c, CsErrGenericError, "inactive socket");
      else
        ws->send(data);
      return obj;
    }

    static value CSF_listen(VM *c);
    static value CSF_connect(VM *c);

    //traits:
    static dispatch*   this_dispatch(VM* c) { return c->socketDispatch; }
    static const char* type_name() { return "Socket"; }

    static c_method methods[];

  };

  c_method connection::methods[] = {
    C_METHOD_ENTRY("listen", CSF_listen),
    C_METHOD_ENTRY("connect", CSF_connect),
    C_METHOD_ENTRY("send", CSF_send),
    C_METHOD_ENTRY("close", CSF_close),
    C_METHOD_ENTRY(0, 0)
  };

  struct listener : public tis::connection {
    typedef tis::connection super;
    listener(VM *c, value acc) : super(c) {
      value obj = tis::connection::this_object(c, this);
      CsEventObjectAdd(c, obj, acc, WCHARS("accept"));
    }
    virtual ~listener() {
      //get_ref_count();
    }

    virtual void handle_accept(void) override {
      VM* c = get_vm();
      handle<tis::connection> con = new tis::connection(c);
      value obj = tis::connection::this_object(c, con);
      if (notify(WCHARS("accept"), obj))
        accept(con);
    }
  };

  value connection::CSF_listen(VM *c) {
    int    port = 0;
    wchars host;  //= WCHARS("localhost");
    value  acceptor = 0;
    bool   tls = false;

    CsParseArguments(c, "**mS#|i|b", &acceptor, &host.start, &host.length, &port, &tls);

    handle<listener> l = new listener(c, acceptor);

    //if (l->listen(pview, port, string(host))) return l->_this;
    if (port) {
      if (!c->can_use_feature(FEATURE_SOCKET_IO))
        CsThrowKnownError(c, CsErrNotAllowed, "SOCKET IO");
      if (l->listen(string(host),port,tls))
        return l->_this;
    }
    else {
      if (l->listen(string(host)))
        return l->_this;
    }
    return UNDEFINED_VALUE;
  }
  value connection::CSF_connect(VM *c) {

    int    port = 0;
    wchars host;// = WCHARS("localhost");
    bool   tls = false;

    CsParseArguments(c, "**S#|i|b", &host.start, &host.length, &port, &tls);

    handle<connection> con = new connection(c);

    if (port) {
      if (!c->can_use_feature(FEATURE_SOCKET_IO))
        CsThrowKnownError(c, CsErrNotAllowed, "SOCKET IO");

      if (con->connect(string(host),port,tls))
        return tis::connection::this_object(c, con);
    }
    else {
      if (con->connect(string(host)))
        return tis::connection::this_object(c, con);
    }

    return NULL_VALUE;
  }


#ifdef WEBSOCKETS_SUPPORT
//
//
// WebSocket

  struct ws_connection : public async_object<ws_connection, VM>, public tool::async::websocket_connection {
    typedef tool::async::pipe_connection super;

    ws_connection(VM *c) {}

    TOOL_INTERFACE_2(tis, ws_connection, tool::async::websocket_connection, gcable)

    virtual void on_connected() override {
      notify(WCHARS("connect"));
    }
    virtual void on_error(chars msg) override {
      notify(WCHARS("error"), CsMakeString(get_vm(),msg));
    }
    virtual void on_closed() override {
      notify(WCHARS("close"));
      async_object<ws_connection, VM>::detach();
    }

    virtual void on_text(wchars data) override
    {
      notify(WCHARS("text"), CsMakeString(get_vm(), data));
    }

    virtual void on_data(bytes data) override
    {
      value b = CsMakeByteVector(get_vm(), data.start, int_t(data.length));
      notify(WCHARS("binary"), b);
    }

    static value CSF_send(VM *c) {
      value data;
      value obj;
      CsParseArguments(c, "V=*V", &obj, this_dispatch(c), &data);
      ws_connection *ws = ws_connection::object_ptr(c, obj);
      if (!ws)
        CsThrowKnownError(c, CsErrGenericError, "inactive socket");
      else if (CsStringP(data)) {
        tool::array<byte> bytes;
        u8::from_utf16(CsStringChars(data), bytes);
        ws->send_message(bytes(), ws_connection::TEXT_FRAME);
      }
      else if (CsByteVectorP(data)) {
        ws->send_message(CsByteVectorBytes(data), ws_connection::BINARY_FRAME);
      }
      else
        CsThrowKnownError(c, CsErrUnexpectedTypeError, data, "String or Bytes");
            
      return obj;
    }
        
    static value CSF_connect(VM *c) {

      if (!c->can_use_feature(FEATURE_SOCKET_IO))
        CsThrowKnownError(c, CsErrNotAllowed, "SOCKET IO");

      wchars url;
      int    timeout = 10000; // 10sec
      CsParseArguments(c, "**S#|T", &url.start, &url.length, &timeout);

      handle<ws_connection> con = new ws_connection(c);
      if (con->connect(tool::url(url.start)))
        return tis::ws_connection::this_object(c, con);
      return NULL_VALUE;
    }

    //traits:
    static dispatch*   this_dispatch(VM* c) { return c->websocketDispatch; }
    static const char* type_name() { return "WebSocket"; }

    static c_method methods[];

  };

  c_method ws_connection::methods[] = {
    C_METHOD_ENTRY("connect", CSF_connect),
    C_METHOD_ENTRY("send", CSF_send),
    C_METHOD_ENTRY("close", CSF_close),
    C_METHOD_ENTRY(0, 0)
  };
  
///
///
///
#endif

  /* element methods */
  c_method net_methods[] = {
    //C_METHOD_ENTRY("errorMessage", CSF_errorMessage),
    C_METHOD_ENTRY(0, 0)
  };

  /* Element properties */
  vp_method net_properties[] = {
    VP_METHOD_ENTRY(0, 0, 0) };

  /* CsInitNet- initialize the 'Net' namespace */
  void CsInitNet(VM *c) {
    /* create the 'System' type */
    //auto_scope asys(c, c->systemDispatch->obj);

    c->netDispatch = CsEnterCPtrObjectType(CsGlobalScope(c), "Net", net_methods, net_properties);
    
    if (!c->netDispatch)
      CsInsufficientMemory(c);
    else {
      CsAddConstant(c, c->systemDispatch->obj, CsSymbolOf("Net"), c->netDispatch->obj); // System.Net alias
      auto_scope as(c, c->netDispatch->obj);
      c->datasocketDispatch = data_connection::init_class(c, data_connection::methods, nullptr, nullptr);
      c->socketDispatch = connection::init_class(c, connection::methods, nullptr, nullptr);
#ifdef WEBSOCKETS_SUPPORT
        c->websocketDispatch = ws_connection::init_class(c, ws_connection::methods, nullptr, nullptr);
#endif
    }
  }

#else 
  void CsInitNet(VM *c) {}
#endif

} // namespace tis

